/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 *
 */

#ifndef __XITK_H_
#define __XITK_H_

#ifndef PACKAGE_NAME
#error config.h not included
#endif

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <inttypes.h>

#include "libcommon.h"
#include "xitk.h"

typedef struct {
  int first, last;
} xitk_range_t;

/** Convenience duplicate bit xitk_widget_t.type -> xitk_widget_t.state,
 *  copied inside _xitk_new_widget_apply (). Make sure not to #define new
 *  XITK_WIDGET_STATE_foo with this value. */
#define XITK_WIDGET_STATE_GROUP WIDGET_GROUP /** << 0x00000800 */

static inline int xitk_min (int a, int b) {
  int d = b - a;
  return a + (d & (d >> (8 * sizeof (d) - 1)));
}

static inline int xitk_max (int a, int b) {
  int d = a - b;
  return a - (d & (d >> (8 * sizeof (d) - 1)));
}

typedef struct {
  char *s, buf[64];
  size_t len;
} xitk_short_string_t;
void xitk_short_string_init (xitk_short_string_t *s);
#define xitk_short_string_init(_s) (_s)->s = (_s)->buf, (_s)->buf[0] = 0, (_s)->len = 0
/* return strlen (v), or (size_t)-1 if string is unchanged. */
size_t xitk_short_string_set (xitk_short_string_t *s, const char *v);
void xitk_short_string_deinit (xitk_short_string_t *s);
#define xitk_short_string_deinit(_s) if ((_s)->s != (_s)->buf) { free ((_s)->s); (_s)->s = NULL; (_s)->len = 0; }

typedef struct {
  uint32_t want, value;
  uint16_t r, g, b, a;
} xitk_color_info_t;

int xitk_color_db_query_value (xitk_t *xitk, xitk_color_info_t *info);

typedef struct xitk_font_cache_s xitk_font_cache_t;
struct xitk_tips_s;

struct xitk_s {
  struct xitk_backend_s *be;
  struct xitk_be_display_s *d, *d2;
  xitk_font_cache_t *font_cache;
  struct xitk_tips_s *tips;
  int verbosity; /** << 0 (quiet), 1 (base, errors), 2 (debug) */
};

#include "widget.h"
#include "xitkintl.h"

#if !defined(__GNUC__) || (defined(__STDC_VERSION__) && __STDC_VERSION__ >= 199901L)
#define	__FUNCTION__	__func__
#endif

#ifndef BUFSIZ
#define BUFSIZ 8192
#endif

#if defined(__GNUC__) && (!defined(__STDC_VERSION__) || __STDC_VERSION__ < 199901L)
#define XITK_DIE(FMT, ARGS...)                                        \
  do {                                                                \
    fprintf(stderr, "xiTK DIE: "FMT, ##ARGS);                         \
    exit(-1);                                                         \
  } while(0)

#define XITK_WARNING(FMT, ARGS...)                                    \
  do {                                                                \
    fprintf(stderr, "xiTK WARNING(%s:%d): ", __FUNCTION__, __LINE__); \
    fprintf(stderr, FMT, ##ARGS);                                     \
  } while(0)
#else	/* C99 version: */
#define XITK_DIE(...)                                                 \
  do {                                                                \
    fprintf(stderr, "xiTK DIE: "__VA_ARGS__);                         \
    exit(-1);                                                         \
  } while(0)

#define XITK_WARNING(...)                                             \
  do {                                                                \
    fprintf(stderr, "xiTK WARNING(%s:%d): ", __FUNCTION__, __LINE__); \
    fprintf(stderr, __VA_ARGS__);                                     \
  } while(0)
#endif

#define XITK_FREE(X) \
  do {               \
      free((X));     \
      (X) = NULL;    \
  } while(0)

/* apply nw-> tips, add_state, mode_mask, mode_value. */
xitk_widget_t *_xitk_new_widget_apply (const xitk_new_widget_t *nw, xitk_widget_t *w);

#define DIRECTION_UP     1
#define DIRECTION_DOWN   2
#define DIRECTION_LEFT   3
#define DIRECTION_RIGHT  4

#define CHECK_STYLE_OLD   1
#define CHECK_STYLE_CHECK 2
#define CHECK_STYLE_ROUND 3

const char *xine_get_homedir(void);

void xitk_set_xmb_enability(xitk_t *, int value);

void xitk_menu_auto_pop (xitk_widget_t *w);
/** << return 1 if a previous menu was closed. */
int xitk_set_current_menu (xitk_t *xitk, xitk_widget_t *menu);
/** << return 1 if menu _is_ the showing one. */
int xitk_unset_current_menu (xitk_t *, xitk_widget_t *menu);

typedef struct {
  int                               width;
  int                               height;
  int                               chars_per_row;
  int                               chars_total;
  int                               char_width;
  int                               char_height;
  xitk_point_t                      space;
  xitk_point_t                      asterisk;
  xitk_point_t                      unknown;
#define XITK_MAX_UNICODE_RANGES 16
  xitk_range_t                      unicode_ranges[XITK_MAX_UNICODE_RANGES + 1];
} xitk_pix_font_t;

typedef enum {
  /* states from left to right */
  XITK_IMG_STATE_NORMAL = 0,
  XITK_IMG_STATE_FOCUS,
  XITK_IMG_STATE_SELECTED,
  XITK_IMG_STATE_SEL_FOCUS,
  XITK_IMG_STATE_DISABLED_NORMAL,
  XITK_IMG_STATE_DISABLED_SELECTED,
  XITK_IMG_STATE_LAST
} xitk_img_state_t;

xitk_img_state_t xitk_image_find_state (xitk_img_state_t max, uint32_t state);

struct xitk_image_s {
  struct xitk_be_image_s *beimg;
  xitk_pix_font_t *pix_font;
  int width, height;
  xitk_img_state_t last_state;
  /* image private */
  xitk_t *xitk;
  xitk_widget_list_t *wl;
  int refs, max_refs;
  char key[32];
};

/* fix part_image->num_states, and fill in list with
 * {width, height}, {x[0], y[0]}, ... {x[part_image->num_states - 1], y[part_image->num_states - 1]},
 * {right, bottom}. */
#define XITK_IMG_STATE_LIST_SIZE ((unsigned int)XITK_IMG_STATE_LAST + 2)
void xitk_part_image_states (xitk_part_image_t *part_image, xitk_hv_t *list, int default_num_states);

void xitk_part_image_draw_menu_check (xitk_part_image_t *pi, int checked);
void xitk_part_image_draw_menu_arrow_branch (xitk_part_image_t *pi);

int xitk_shared_image (xitk_widget_list_t *wl, const char *key, int width, int height, xitk_image_t **image);
void xitk_shared_image_list_delete (xitk_widget_list_t *wl);

struct xitk_window_s {
  xitk_t                   *xitk;
  struct xitk_be_window_s  *bewin;
  xitk_image_t             *bg_image;
  int                       width;
  int                       height;
  xitk_wm_window_type_t     type;
  xitk_window_role_t        role;
  xitk_register_key_t       key;
  uint32_t                  flags; /** << the more or less up to date real state. */
  uint32_t                  app_flags; /** the state the application wants (without user -> window manager interference). */
  uint32_t                  last_cursor;
  xitk_widget_list_t       *widget_list;
};

void xitk_window_update_tree (xitk_window_t *xwin, uint32_t mask_and_flags);
xitk_window_t *xitk_window_wrap_native_window (xitk_t *xitk /* may be null if be_display given */,
                                               struct xitk_be_display_s *be_display /* may be NULL */,
                                               uintptr_t window);

xitk_widget_list_t *xitk_widget_list_get (xitk_t *xitk, xitk_window_t *xwin);

void xitk_widget_list_change_skin (xitk_widget_list_t *wl, xitk_skin_config_t *skonfig);

xitk_register_key_t xitk_get_focus_key (xitk_t *xitk);
xitk_register_key_t xitk_set_focus_key (xitk_t *xitk, xitk_register_key_t key, int focus);

/*
 * Helper function to free widget list inside callbacks.
 */
void xitk_widget_list_defferred_destroy(xitk_widget_list_t *wl);

void xitk_widget_list_window_focus (xitk_widget_list_t *wl, int focus);
int xitk_widget_list_anim (xitk_widget_list_t *wl, int start);

/* A few shortcuts for internal use. */
#ifndef XITK_WIDGET_C
#  define xitk_get_widget_width(w) ((w) ? XITK_HV_H ((w)->size) : 0)
#  define xitk_get_widget_height(w) ((w) ? XITK_HV_V ((w)->size) : 0)
#  define xitk_widget_user_id(w) ((w) ? (w)->user_id : 0)
#endif
#ifndef XITK_WINDOW_C
#  define xitk_window_get_background_image(_w) ((_w) ? (_w)->bg_image : NULL)
#endif
#ifndef _XITK_C_
#  define xitk_get_display_size(xitk,pw,ph) *(pw) = (xitk)->d->default_screen.r.width, *(ph) = (xitk)->d->default_screen.r.height
#  define xitk_get_wm_type(xitk) ((xitk)->d->wm_type)
#endif

#define xitk_widget_state_from_info(_w,info) do { \
  (_w)->state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE); \
  (_w)->state |= (info) && (info)->visibility ? XITK_WIDGET_STATE_VISIBLE : 0; \
  (_w)->state |= (info) && (info)->enability ? XITK_WIDGET_STATE_ENABLE : 0; \
  (_w)->saved_state = (_w)->state; \
} while (0)

void xitk_clipboard_unregister_widget (xitk_widget_t *w);
/* text == NULL: just tell length.
 * *text == NULL: return live buf.
 * *text != NULL: copy there.
 * return -1: wait for WIDGET_EVENT_CLIP_READY, then try again. */
int xitk_clipboard_get_text (xitk_widget_t *w, char **text, int max_len);

int xitk_image_temp_size (xitk_t *xitk, int width, int height);
xitk_image_t *xitk_image_temp_get (xitk_t *xitk);

/** keyboard nav helper. num_items < 0 means horizontal stacking.
 *  if item != NULL, also test for warp by initial letter.
 *  return >= 0 (new index), -1 (error), -2 (other XITK_KEY_* found), -3 (other string found). */
int xitk_browser_list_nav (const widget_event_t *event, const char * const *items, int num_items, int selected, int big_step);

xine_sarray_t **xitk_anim_threads (xitk_t *_xitk);

#endif

