/*
 * Copyright (C) 2000-2024 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <errno.h>
#include <pthread.h>
#include <time.h>

#include "common.h"
#include "stream_infos.h"
#include "videowin.h"
#include "actions.h"
#include "event.h"
#include "xine-toolkit/backend.h"
#include "xine-toolkit/label.h"
#include "xine-toolkit/labelbutton.h"
#include "recode.h"

#define WINDOW_WIDTH        570
#define WINDOW_HEIGHT       616

#define METAINFO_CHARSET    "UTF-8"

/* TEST: #undef XINE_QUERY_STREAM_INFO */

typedef enum {
  SINF_GROUP = 0, /** << super frames without display label. */
  SINF_g_general = SINF_GROUP,
  SINF_g_misc,
  SINF_g_video,
  SINF_g_audio,
  SINF_STRING, /** << string values. */
  SINF_title = SINF_STRING,
  SINF_comment,
  SINF_artist,
  SINF_genre,
  SINF_album,
  SINF_year,
  SINF_vcodec,
  SINF_acodec,
  SINF_systemlayer,
  SINF_input_plugin,
  SINF_BOOL, /** integer values 0 or != 0. */
  SINF_seekable = SINF_BOOL,
  SINF_has_chapters,
  SINF_has_still,
  SINF_has_audio,
  SINF_ignore_audio,
  SINF_audio_handled,
  SINF_has_video,
  SINF_ignore_video,
  SINF_video_handled,
  SINF_ignore_spu,
  SINF_INT, /** integer values. */
  SINF_bitrate = SINF_INT,
  SINF_video_ratio,
  SINF_video_channels,
  SINF_video_streams,
  SINF_video_bitrate,
  SINF_frame_duration,
  SINF_audio_channels,
  SINF_audio_bits,
  SINF_audio_samplerate,
  SINF_audio_bitrate,
  SINF_FOURCC, /** << integer values made from 4 chars. */
  SINF_video_fourcc = SINF_FOURCC,
  SINF_audio_fourcc,
  SINF_SPECIAL, /** << generated strings. */
  SINF_video_size = SINF_SPECIAL,
  SINF_END /** << that's it. */
} sinf_index_t;

#define _SF (XITK_DRAW_OUTTER) /* group frame */
#define _SM (XITK_DRAW_INNER | XITK_DRAW_R | XITK_DRAW_B | XITK_DRAW_SAT (255)) /* misc */
#define _SV (XITK_DRAW_INNER | XITK_DRAW_R | XITK_DRAW_G | XITK_DRAW_SAT (255)) /* video */
#define _SA (XITK_DRAW_INNER | XITK_DRAW_B | XITK_DRAW_SAT (255)) /* audio */
static const struct {
  uint16_t x, y, w, h;
  uint32_t style;
  uint16_t xine_type;
  char title[18];
} sinf_items[SINF_END] = {
  [SINF_g_general]        = {  15,  28, 540, 199, _SF, 0,                                 N_("General")},
  [SINF_g_misc]           = {  15, 230, 540, 109, _SF, 0,                                 N_("Misc")},
  [SINF_g_video]          = {  15, 342, 540, 109, _SF, 0,                                 N_("Video")},
  [SINF_g_audio]          = {  15, 454, 540, 109, _SF, 0,                                 N_("Audio")},
  [SINF_title]            = {  20,  43, 529,  42, _SM, XINE_META_INFO_TITLE,              N_("Title: ")},
  [SINF_comment]          = {  20,  88, 529,  42, _SM, XINE_META_INFO_COMMENT,            N_("Comment: ")},
  [SINF_artist]           = {  20, 133, 262,  42, _SM, XINE_META_INFO_ARTIST,             N_("Artist: ")},
  [SINF_genre]            = { 287, 133, 129,  42, _SM, XINE_META_INFO_GENRE,              N_("Genre: ")},
  [SINF_year]             = { 421, 133, 128,  42, _SM, XINE_META_INFO_YEAR,               N_("Year: ")},
  [SINF_album]            = {  20, 178, 262,  42, _SM, XINE_META_INFO_ALBUM,              N_("Album: ")},
  [SINF_input_plugin]     = {  20, 245, 128,  42, _SM, XINE_META_INFO_INPUT_PLUGIN,       N_("Input Plugin: ")},
  [SINF_systemlayer]      = { 153, 245, 129,  42, _SM, XINE_META_INFO_SYSTEMLAYER,        N_("System Layer: ")},
  [SINF_bitrate]          = { 287, 245, 129,  42, _SM, XINE_STREAM_INFO_BITRATE,          N_("Bitrate: ")},
  [SINF_frame_duration]   = { 421, 245, 128,  42, _SV, XINE_STREAM_INFO_FRAME_DURATION,   N_("Frame Duration: ")},
  [SINF_seekable]         = {  20, 290, 128,  42, _SM, XINE_STREAM_INFO_SEEKABLE,         N_("Is Seekable: ")},
  [SINF_has_chapters]     = { 153, 290, 129,  42, _SM, XINE_STREAM_INFO_HAS_CHAPTERS,     N_("Has Chapters: ")},
  [SINF_ignore_spu]       = { 287, 290, 129,  42, _SM, XINE_STREAM_INFO_IGNORE_SPU,       N_("Ignore Spu: ")},
  [SINF_has_still]        = { 421, 290, 128,  42, _SV, XINE_STREAM_INFO_VIDEO_HAS_STILL,  N_("Has Still: ")},
  [SINF_has_video]        = {  20, 357, 102,  42, _SV, XINE_STREAM_INFO_HAS_VIDEO,        N_("Has: ")},
  [SINF_video_handled]    = { 127, 357, 102,  42, _SV, XINE_STREAM_INFO_VIDEO_HANDLED,    N_("Handled: ")},
  [SINF_ignore_video]     = { 234, 357, 101,  42, _SV, XINE_STREAM_INFO_IGNORE_VIDEO,     N_("Ignore: ")},
  [SINF_vcodec]           = { 340, 357, 209,  42, _SV, XINE_META_INFO_VIDEOCODEC,         N_("Codec: ")},
  [SINF_video_fourcc]     = {  20, 402,  84,  42, _SV, XINE_STREAM_INFO_VIDEO_FOURCC,     N_("FourCC: ")},
  [SINF_video_channels]   = { 109, 402,  84,  42, _SV, XINE_STREAM_INFO_VIDEO_CHANNELS,   N_("Channel(s): ")},
  [SINF_video_bitrate]    = { 198, 402,  84,  42, _SV, XINE_STREAM_INFO_VIDEO_BITRATE,    N_("Bitrate: ")},
  [SINF_video_size]       = { 287, 402,  84,  42, _SV, 0,                                 N_("Resolution: ")},
  [SINF_video_ratio]      = { 376, 402,  84,  42, _SV, XINE_STREAM_INFO_VIDEO_RATIO,      N_("Ratio: ")},
  [SINF_video_streams]    = { 465, 402,  84,  42, _SV, XINE_STREAM_INFO_VIDEO_STREAMS,    N_("Stream(s): ")},
  [SINF_has_audio]        = {  20, 469, 102,  42, _SA, XINE_STREAM_INFO_HAS_AUDIO,        N_("Has: ")},
  [SINF_audio_handled]    = { 127, 469, 102,  42, _SA, XINE_STREAM_INFO_AUDIO_HANDLED,    N_("Handled: ")},
  [SINF_ignore_audio]     = { 234, 469, 101,  42, _SA, XINE_STREAM_INFO_IGNORE_AUDIO,     N_("Ignore: ")},
  [SINF_acodec]           = { 340, 469, 209,  42, _SA, XINE_META_INFO_AUDIOCODEC,         N_("Codec: ")},
  [SINF_audio_fourcc]     = {  20, 514, 102,  42, _SA, XINE_STREAM_INFO_AUDIO_FOURCC,     N_("FourCC: ")},
  [SINF_audio_channels]   = { 127, 514, 102,  42, _SA, XINE_STREAM_INFO_AUDIO_CHANNELS,   N_("Channel(s): ")},
  [SINF_audio_bitrate]    = { 234, 514, 101,  42, _SA, XINE_STREAM_INFO_AUDIO_BITRATE,    N_("Bitrate: ")},
  [SINF_audio_bits]       = { 340, 514, 102,  42, _SA, XINE_STREAM_INFO_AUDIO_BITS,       N_("Bits: ")},
  [SINF_audio_samplerate] = { 447, 514, 102,  42, _SA, XINE_STREAM_INFO_AUDIO_SAMPLERATE, N_("Samplerate: ")}
};

struct xui_sinfo_s {
  gui_new_window_t      nw;

  xitk_widget_t        *close;
  xitk_widget_t        *update;
  xitk_widget_t        *copy;

  xitk_widget_t        *w[SINF_END];

  struct timeval        update_time;

  xitk_recode_t        *xr;

  xitk_register_key_t   widget_key;

  const char           *frame_titles[SINF_END];
  const char            *yes, *no, *unavail;
  char                  temp[1024], buf[32];
};

static void sinf_query (xui_sinfo_t *sinfo, char *sbuf, size_t sblen, int *values) {
#ifdef XINE_QUERY_STREAM_INFO
  int tabs[SINF_BOOL - SINF_STRING + 1], tabi[SINF_SPECIAL - SINF_BOOL + 3];
  sinf_index_t i;

  for (i = 0; i < SINF_BOOL - SINF_STRING; i++)
    tabs[i] = sinf_items[SINF_STRING + i].xine_type;
  tabs[i] = -1;
  for (i = 0; i < SINF_SPECIAL - SINF_BOOL; i++)
    tabi[i] = sinf_items[SINF_BOOL + i].xine_type;
  tabi[i]     = XINE_STREAM_INFO_VIDEO_WIDTH;
  tabi[i + 1] = XINE_STREAM_INFO_VIDEO_HEIGHT;
  tabi[i + 2] = -1;
  xine_query_stream_info (sinfo->nw.gui->stream, sbuf, sblen, tabs, tabi);
  for (i = SINF_STRING; i < SINF_BOOL; i++)
    values[i] = tabs[i - SINF_STRING];
  for (i = SINF_BOOL; i < SINF_SPECIAL + 2; i++)
    values[i] = tabi[i - SINF_BOOL];
#else
  (void)sinfo;
  (void)sbuf;
  (void)sblen;
  (void)values;
#endif
}

static const char *sinf_get_string (xui_sinfo_t *sinfo, const char *sbuf, int *values, sinf_index_t type) {
  xitk_recode_string_t rs;
  const char *s;
#ifdef XINE_QUERY_STREAM_INFO
  s = values[type] ? sbuf + values[type] : NULL;
#else
  (void)sbuf;
  (void)values;
  s = xine_get_meta_info (sinfo->nw.gui->stream, sinf_items[type].xine_type);
#endif

  if (!s)
    return sinfo->unavail;

  rs.buf = sinfo->temp;
  rs.bsize = sizeof (sinfo->temp) - 1;
  rs.src = s;
  rs.ssize = strlen (s);
  xitk_recode2_do (sinfo->xr, &rs);
  if (rs.res == sinfo->temp)
    sinfo->temp[rs.rsize] = 0;
  return rs.res;
}

static const char *sinf_get_int (xui_sinfo_t *sinfo, int *values, sinf_index_t type) {
  int v;
  unsigned int u;
  char *q = sinfo->buf + sizeof (sinfo->buf);
#ifdef XINE_QUERY_STREAM_INFO
  v = values[type];
#else
  (void)values;
  v = xine_get_stream_info (sinfo->nw.gui->stream, sinf_items[type].xine_type);
#endif

  u = v < 0 ? -v : v;
  *--q = 0;
  do {
    *--q = u % 10u + '0';
    u /= 10u;
  } while (u);
  if (v < 0)
    *--q = '-';

  return q;
}

static const char *sinf_get_res (xui_sinfo_t *sinfo, int *values) {
  int v;
  unsigned int u;
  char *q = sinfo->buf + sizeof (sinfo->buf);
#ifdef XINE_QUERY_STREAM_INFO
  v = values[SINF_SPECIAL + 1];
#else
  (void)values;
  v = xine_get_stream_info (sinfo->nw.gui->stream, XINE_STREAM_INFO_VIDEO_HEIGHT);
#endif
  u = v < 0 ? -v : v;
  *--q = 0;
  do {
    *--q = u % 10u + '0';
    u /= 10u;
  } while (u);
  if (v < 0)
    *--q = '-';

  *--q = ' ';
  *--q = 'X';
  *--q = ' ';

#ifdef XINE_QUERY_STREAM_INFO
  v = values[SINF_SPECIAL];
#else
  v = xine_get_stream_info (sinfo->nw.gui->stream, XINE_STREAM_INFO_VIDEO_WIDTH);
#endif
  u = v < 0 ? -v : v;
  do {
    *--q = u % 10u + '0';
    u /= 10u;
  } while (u);
  if (v < 0)
    *--q = '-';

  return q;
}

static const char *sinf_get_bool (xui_sinfo_t *sinfo, int *values, sinf_index_t type) {
  int v;
#ifdef XINE_QUERY_STREAM_INFO
  v = values[type];
#else
  (void)values;
  v = xine_get_stream_info (sinfo->nw.gui->stream, sinf_items[type].xine_type);
#endif
  return v ? sinfo->yes : sinfo->no;
}

const char *get_fourcc_string (char *dst, size_t dst_size, uint32_t f) {
  static const char tab_hex[16] = "0123456789abcdef";
  union {
    uint32_t u;
    char z[4];
  } v;
  char *q;
  int i;

  if (!dst || (dst_size < 17))
    return NULL;

  v.u = f;
  q = dst + dst_size;
  *--q = 0;
  if (f < 0x10000) {
    do {
      *--q = tab_hex[f & 15];
      f >>= 4;
    } while (f);
    *--q = 'x';
    *--q = '0';
  } else {
    for (i = 3; i >= 0; i--) {
      if ((v.z[i] >= ' ') && (v.z[i] < 127)) {
        *--q = v.z[i];
      } else {
        *--q = tab_hex[v.z[i] & 15];
        *--q = tab_hex[v.z[i] >> 4];
        *--q = 'x';
        *--q = '\\';
      }
    }
  }
  return q;
}

static const char *sinf_get_4cc (xui_sinfo_t *sinfo, int *values, sinf_index_t type) {
  int v;
#ifdef XINE_QUERY_STREAM_INFO
  v = values[type];
#else
  (void)values;
  v = xine_get_stream_info (sinfo->nw.gui->stream, sinf_items[type].xine_type);
#endif
  return get_fourcc_string (sinfo->buf, sizeof (sinfo->buf), v);
}

static int stream_infos_dbl_click (xui_sinfo_t *sinfo) {
  struct timeval last = sinfo->update_time;

  xitk_gettime_tv (&sinfo->update_time);
  if (xitk_is_dbl_click (sinfo->nw.gui->xitk, &last, &sinfo->update_time)) {
    /* will call stream_infos_toggle_auto_update (). */
    config_update_num (sinfo->nw.gui->xine, "gui.sinfo_auto_update", !sinfo->nw.gui->stream_info_auto_update);
    return 1;
  }
  return 0;
}

static void stream_infos_update (xitk_widget_t *w, void *data, int state) {
  xui_sinfo_t *sinfo = (xui_sinfo_t *)data;

  (void)w;
  (void)state;
  if (!stream_infos_dbl_click (sinfo))
    stream_infos_update_infos (sinfo);
}

struct stream_infos_ident_s {
  xitk_vers_string_t raw_title, raw_artist, raw_album, ident;
  char buf[2048];
};

stream_infos_ident_t *stream_infos_ident_new (void) {
  stream_infos_ident_t *ident = malloc (sizeof (*ident));

  if (!ident)
    return NULL;
  xitk_vers_string_init (&ident->raw_title, NULL, 0);
  xitk_vers_string_init (&ident->raw_artist, NULL, 0);
  xitk_vers_string_init (&ident->raw_album, NULL, 0);
  xitk_vers_string_init (&ident->ident, ident->buf, sizeof (ident->buf));
  return ident;
}

static size_t _stream_infos_ident_make (char *buf, size_t bsize, const char *title, const char *artist, const char *album) {
  xitk_recode_t *xr;
  xitk_recode_string_t rs;
  char *q;
  size_t arlen, allen;

  xr = xitk_recode_init (METAINFO_CHARSET, NULL, 0);
  q = buf;
  rs.bsize = bsize - 8;

  rs.buf = q;
  rs.src = title ? title : "";
  rs.ssize = strlen (rs.src);
  xitk_recode2_do (xr, &rs);
  if (rs.res == rs.src) {
    rs.rsize = rs.ssize < rs.bsize ? rs.ssize : rs.bsize;
    if (rs.rsize > 0)
      memcpy (q, rs.src, rs.rsize);
  }
  q[rs.rsize] = 0;
  /* Since meta info can be corrupted/wrong/ugly
   * we need to clean and check them before using.
   * Note: str_unqote () may modify the string, so we work on a copy. */
  arlen = str_unquote (q);
  q += arlen;
  rs.bsize -= arlen;

  memcpy (q, " (", 2); q += 2;

  rs.buf = q;
  rs.src = artist ? artist : "";
  rs.ssize = strlen (rs.src);
  xitk_recode2_do (xr, &rs);
  if (rs.res == rs.src) {
    rs.rsize = rs.ssize < rs.bsize ? rs.ssize : rs.bsize;
    if (rs.rsize > 0)
      memcpy (q, rs.src, rs.rsize);
  }
  q[rs.rsize] = 0;
  arlen = str_unquote (q);
  q += arlen;
  rs.bsize -= arlen;

  if (arlen) {
    memcpy (q, " - ", 3); q += 3;
  }

  rs.buf = q;
  rs.src = album ? album : "";
  rs.ssize = strlen (rs.src);
  xitk_recode2_do (xr, &rs);
  if (rs.res == rs.src) {
    rs.rsize = rs.ssize < rs.bsize ? rs.ssize : rs.bsize;
    if (rs.rsize > 0)
      memcpy (q, rs.src, rs.rsize);
  }
  q[rs.rsize] = 0;
  allen = str_unquote (q);
  q += allen;
  rs.bsize -= allen;

  if (arlen && !allen)
    q -= 3;
  if (arlen + allen)
    *q++ = ')';
  else
    q -= 2;
  xitk_recode2_done (xr, &rs);
  xitk_recode_done (xr);
  *q = 0;
  return q - buf;
}

int stream_infos_ident_get (stream_infos_ident_t *ident, xitk_vers_string_t *to, xine_stream_t *stream) {
  const char *title, *artist, *album;
#ifdef XINE_QUERY_STREAM_INFO
  char sbuf[2048];
  int strings[4] = {XINE_META_INFO_TITLE, XINE_META_INFO_ARTIST, XINE_META_INFO_ALBUM, -1};
#endif
  int changed;

  if (!ident || !stream)
    return 0;

#ifdef XINE_QUERY_STREAM_INFO
  xine_query_stream_info (stream, sbuf, sizeof (sbuf), strings, NULL);
  title  = strings[0] ? sbuf + strings[0] : NULL;
  artist = strings[1] ? sbuf + strings[1] : NULL;
  album  = strings[2] ? sbuf + strings[2] : NULL;
#else
  title  = xine_get_meta_info (stream, XINE_META_INFO_TITLE);
  artist = xine_get_meta_info (stream, XINE_META_INFO_ARTIST);
  album  = xine_get_meta_info (stream, XINE_META_INFO_ALBUM);
#endif
  changed = xitk_vers_string_set (&ident->raw_title, title)
          + xitk_vers_string_set (&ident->raw_artist, artist)
          + xitk_vers_string_set (&ident->raw_album, album);

  if (changed) {
    ident->ident.version += 1;
    _stream_infos_ident_make (ident->buf, sizeof (ident->buf), ident->raw_title.s, ident->raw_artist.s, ident->raw_album.s);
  }

  return xitk_vers_string_get (to, &ident->ident);
}

void stream_infos_ident_delete (stream_infos_ident_t **ident) {
  stream_infos_ident_t *_ident;

  if (!ident)
    return;
  _ident = *ident;
  xitk_vers_string_deinit (&_ident->ident);
  xitk_vers_string_deinit (&_ident->raw_album);
  xitk_vers_string_deinit (&_ident->raw_artist);
  xitk_vers_string_deinit (&_ident->raw_title);
  free (_ident);
  *ident = NULL;
}

int stream_infos_get_ident_from_stream (xine_stream_t *stream, char *_buf, size_t bsize) {
  const char *title, *artist, *album;
  char buf[2048];
  size_t l;
#ifdef XINE_QUERY_STREAM_INFO
  char sbuf[2048];
  int strings[4] = {XINE_META_INFO_TITLE, XINE_META_INFO_ARTIST, XINE_META_INFO_ALBUM, -1};

  if (!_buf || !bsize)
    return 0;
  xine_query_stream_info (stream, sbuf, sizeof (sbuf), strings, NULL);
  title  = strings[0] ? sbuf + strings[0] : NULL;
  artist = strings[1] ? sbuf + strings[1] : NULL;
  album  = strings[2] ? sbuf + strings[2] : NULL;
#else
  title  = xine_get_meta_info (stream, XINE_META_INFO_TITLE);
  artist = xine_get_meta_info (stream, XINE_META_INFO_ARTIST);
  album  = xine_get_meta_info (stream, XINE_META_INFO_ALBUM);
#endif
  l = _stream_infos_ident_make (buf, sizeof (buf), title, artist, album) + 1;
  if (l < 2)
    return 0;
  if (l > bsize - 1)
    l = bsize - 1;
  memcpy (_buf, buf, l);
  _buf[l] = 0;
  return l;
}

static void stream_infos_copy (xitk_widget_t *w, void *data, int state) {
  xui_sinfo_t *sinfo = data;
  char *buf, *q, *e;

  if (!sinfo)
    return;
  (void)w;
  (void)state;
  buf = malloc (4096);
  if (!buf)
    return;
  q = buf;
  e = buf + 4096;

  gui_playlist_lock (sinfo->nw.gui);
  q += snprintf (q, e - q, "[%s]\n%s: %s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n",
    sinfo->frame_titles[SINF_g_general],
    _("Mrl"), sinfo->nw.gui->mmk.mrl ? sinfo->nw.gui->mmk.mrl : "",
    sinfo->frame_titles[SINF_title],            xitk_label_get_label (sinfo->w[SINF_title]),
    sinfo->frame_titles[SINF_artist],           xitk_label_get_label (sinfo->w[SINF_artist]),
    sinfo->frame_titles[SINF_album],            xitk_label_get_label (sinfo->w[SINF_album]),
    sinfo->frame_titles[SINF_year],             xitk_label_get_label (sinfo->w[SINF_year]),
    sinfo->frame_titles[SINF_genre],            xitk_label_get_label (sinfo->w[SINF_genre]),
    sinfo->frame_titles[SINF_comment],          xitk_label_get_label (sinfo->w[SINF_comment]));
  gui_playlist_unlock (sinfo->nw.gui);
  q += snprintf (q, e - q, "[%s]\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n",
    sinfo->frame_titles[SINF_g_misc],
    sinfo->frame_titles[SINF_input_plugin],     xitk_label_get_label (sinfo->w[SINF_input_plugin]),
    sinfo->frame_titles[SINF_systemlayer],      xitk_label_get_label (sinfo->w[SINF_systemlayer]),
    sinfo->frame_titles[SINF_seekable],         xitk_label_get_label (sinfo->w[SINF_seekable]),
    sinfo->frame_titles[SINF_bitrate],          xitk_label_get_label (sinfo->w[SINF_bitrate]),
    sinfo->frame_titles[SINF_has_chapters],     xitk_label_get_label (sinfo->w[SINF_has_chapters]),
    sinfo->frame_titles[SINF_ignore_spu],       xitk_label_get_label (sinfo->w[SINF_ignore_spu]));
  q += snprintf (q, e - q,
    "[%s]\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n",
    sinfo->frame_titles[SINF_g_video],
    sinfo->frame_titles[SINF_has_still],        xitk_label_get_label (sinfo->w[SINF_has_still]),
    sinfo->frame_titles[SINF_has_video],        xitk_label_get_label (sinfo->w[SINF_has_video]),
    sinfo->frame_titles[SINF_video_handled],    xitk_label_get_label (sinfo->w[SINF_video_handled]),
    sinfo->frame_titles[SINF_ignore_video],     xitk_label_get_label (sinfo->w[SINF_ignore_video]),
    sinfo->frame_titles[SINF_frame_duration],   xitk_label_get_label (sinfo->w[SINF_frame_duration]),
    sinfo->frame_titles[SINF_vcodec],           xitk_label_get_label (sinfo->w[SINF_vcodec]),
    sinfo->frame_titles[SINF_video_fourcc],     xitk_label_get_label (sinfo->w[SINF_video_fourcc]),
    sinfo->frame_titles[SINF_video_channels],   xitk_label_get_label (sinfo->w[SINF_video_channels]),
    sinfo->frame_titles[SINF_video_bitrate],    xitk_label_get_label (sinfo->w[SINF_video_bitrate]),
    sinfo->frame_titles[SINF_video_size],       xitk_label_get_label (sinfo->w[SINF_video_size]),
    sinfo->frame_titles[SINF_video_ratio],      xitk_label_get_label (sinfo->w[SINF_video_ratio]),
    sinfo->frame_titles[SINF_video_streams],    xitk_label_get_label (sinfo->w[SINF_video_streams]));
  q += snprintf (q, e - q,
    "[%s]\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n%s%s\n",
    sinfo->frame_titles[SINF_g_audio],
    sinfo->frame_titles[SINF_has_audio],        xitk_label_get_label (sinfo->w[SINF_has_audio]),
    sinfo->frame_titles[SINF_audio_handled],    xitk_label_get_label (sinfo->w[SINF_audio_handled]),
    sinfo->frame_titles[SINF_ignore_audio],     xitk_label_get_label (sinfo->w[SINF_ignore_audio]),
    sinfo->frame_titles[SINF_acodec],           xitk_label_get_label (sinfo->w[SINF_acodec]),
    sinfo->frame_titles[SINF_audio_fourcc],     xitk_label_get_label (sinfo->w[SINF_audio_fourcc]),
    sinfo->frame_titles[SINF_audio_channels],   xitk_label_get_label (sinfo->w[SINF_audio_channels]),
    sinfo->frame_titles[SINF_audio_bitrate],    xitk_label_get_label (sinfo->w[SINF_audio_bitrate]),
    sinfo->frame_titles[SINF_audio_bits],       xitk_label_get_label (sinfo->w[SINF_audio_bits]),
    sinfo->frame_titles[SINF_audio_samplerate], xitk_label_get_label (sinfo->w[SINF_audio_samplerate]));

  xitk_clipboard_set_text (sinfo->copy, buf, q - buf);
  free (buf);
}

static void stream_infos_exit (xitk_widget_t *w, void *data, int state) {
  xui_sinfo_t *sinfo = data;

  if (!sinfo)
    return;
  (void)w;
  (void)state;

  gui_save_window_pos (sinfo->nw.gui, "sinfos", sinfo->widget_key);

  xitk_unregister_event_handler (sinfo->nw.gui->xitk, &sinfo->widget_key);

  xitk_window_destroy_window (sinfo->nw.xwin);
  sinfo->nw.xwin = NULL;
  /* xitk_dlist_init (&sinfo->nw.wl.list); */

  xitk_recode_done (sinfo->xr);
  sinfo->xr = NULL;

  video_window_set_input_focus (sinfo->nw.gui->vwin);
  sinfo->nw.gui->streaminfo = NULL;
  free (sinfo);
}

static int stream_infos_event (void *data, const xitk_be_event_t *e) {
  xui_sinfo_t *sinfo = (xui_sinfo_t *)data;

  if (e->type == XITK_EV_KEY_DOWN) {
    if ((e->utf8[0] == XITK_CTRL_KEY_PREFIX)
        && !(e->qual & (MODIFIER_SHIFT | MODIFIER_CTRL | MODIFIER_META))) {
      if (e->utf8[1] == XITK_KEY_ESCAPE) {
        stream_infos_exit (NULL, sinfo, 0);
        return 1;
      } else if (e->utf8[1] == XITK_KEY_F5) {
        if (!stream_infos_dbl_click (sinfo))
          stream_infos_update_infos (sinfo);
        return 1;
      }
    } else if (e->utf8[0] == ('c' & 0x1f)) {
      stream_infos_copy (NULL, sinfo, 1);
      return 1;
    }
  } else if (e->type == XITK_EV_DEL_WIN) {
    stream_infos_exit (NULL, sinfo, 0);
    return 1;
  }
  return gui_handle_be_event (sinfo->nw.gui, e);
}

void stream_infos_toggle_auto_update (xui_sinfo_t *sinfo) {
  if (sinfo)
    xitk_labelbutton_set_state (sinfo->update, sinfo->nw.gui->stream_info_auto_update);
}

void stream_infos_update_infos (xui_sinfo_t *sinfo) {
  if (!sinfo)
    return;

  xitk_labelbutton_set_state (sinfo->update, sinfo->nw.gui->stream_info_auto_update);

  if (!sinfo->nw.gui->logo_mode) {
    char sbuf[2048];
    int values[SINF_SPECIAL + 2];
    sinf_index_t i;

    sinf_query (sinfo, sbuf, sizeof (sbuf), values);
    for (i = SINF_STRING; i < SINF_BOOL; i++)
      xitk_label_change_label (sinfo->w[i], sinf_get_string (sinfo, sbuf, values, i));
    for (; i < SINF_INT; i++)
      xitk_label_change_label (sinfo->w[i], sinf_get_bool (sinfo, values, i));
    for (; i < SINF_FOURCC; i++)
      xitk_label_change_label (sinfo->w[i], sinf_get_int (sinfo, values, i));
    for (; i < SINF_SPECIAL; i++)
      xitk_label_change_label (sinfo->w[i], sinf_get_4cc (sinfo, values, i));
    xitk_label_change_label (sinfo->w[SINF_video_size], sinf_get_res (sinfo, values));
  } else {
    sinf_index_t i;

    for (i = SINF_STRING; i < SINF_BOOL; i++)
      xitk_label_change_label (sinfo->w[i], sinfo->unavail);
    for (; i < SINF_SPECIAL; i++)
      xitk_label_change_label (sinfo->w[i], "---");
    xitk_label_change_label (sinfo->w[SINF_video_size], "--- X ---");
  }
}

void stream_infos_main (xitk_widget_t *mode, void *data) {
  int x, y;
  gGui_t *gui = data;
  xui_sinfo_t *sinfo;
  uint32_t bg_color[8] = {
    0xbebebe, 0xbebebe, 0xbebebe, 0xbebebe,
    0xbebebe, 0xbebebe, 0xbebebe, 0xbebebe
  };

  if (!gui)
    return;

  sinfo = gui->streaminfo;
  if (mode == XUI_W_OFF) {
    if (!sinfo)
      return;
    stream_infos_exit (NULL, sinfo, 0);
    return;
  } else if (mode == XUI_W_ON) {
    if (sinfo) {
      gui_raise_window (gui, sinfo->nw.xwin, 1, 0);
      xitk_window_set_input_focus (sinfo->nw.xwin);
      return;
    }
  } else { /* toggle */
    if (sinfo) {
      stream_infos_exit (NULL, sinfo, 0);
      return;
    }
  }

  sinfo = (xui_sinfo_t *)xitk_xmalloc (sizeof (*sinfo));
  if (!sinfo)
    return;
  sinfo->nw.gui = gui;

  /* Create window */
  sinfo->nw.title = _("Stream Information");
  sinfo->nw.id = "sinfos";
  if (xitk_init_NULL ()) {
    sinfo->nw.skin = NULL;
    sinfo->nw.wfskin = NULL;
    sinfo->nw.adjust = NULL;
  }
  sinfo->nw.wr.x = 80;
  sinfo->nw.wr.y = 80;
  sinfo->nw.wr.width = WINDOW_WIDTH;
  sinfo->nw.wr.height = WINDOW_HEIGHT;
  if (gui_window_new (&sinfo->nw) < 0) {
    free (sinfo);
    return;
  }
  sinfo->nw.gui->streaminfo = sinfo;

  sinfo->xr = xitk_recode_init (METAINFO_CHARSET, NULL, 0);
  sinfo->yes = _("Yes");
  sinfo->no = _("No");
  sinfo->unavail = _("Unavailable");

  {
    xitk_image_t *bg = xitk_window_get_background_image (sinfo->nw.xwin);
    uint32_t sat = XITK_DRAW_SAT (sinfo->nw.gui->gfx_saturation)
                 | XITK_DRAW_R | XITK_DRAW_G | XITK_DRAW_B | XITK_DRAW_INNER | XITK_DRAW_OUTTER;
    int i;
    for (i = 0; i < SINF_END; i++)
      bg_color[(sinf_items[i].style / XITK_DRAW_B) & 7] =
        xitk_image_draw_frame (bg, sinfo->frame_titles[i] = gettext (sinf_items[i].title), btnfontname,
        sinf_items[i].x, sinf_items[i].y, sinf_items[i].w, sinf_items[i].h, sinf_items[i].style & sat);
    xitk_window_set_background_image (sinfo->nw.xwin, bg);
  }

  {
    sinf_index_t i;
    char sbuf[2048];
    int values[SINF_SPECIAL + 2];
    xitk_label_widget_t lbl = {
        .nw = { .wl = sinfo->nw.wl, .userdata = sinfo, .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE }
    };

    sinf_query (sinfo, sbuf, sizeof (sbuf), values);

    for (i = SINF_STRING; i < SINF_BOOL; i++) {
      lbl.label = sinf_get_string (sinfo, sbuf, values, i);
      sinfo->w[i] = xitk_noskin_label_color_create (&lbl,
        sinf_items[i].x + 5, sinf_items[i].y + 15, sinf_items[i].w - 8, 20,
        fontname, XITK_NOSKIN_TEXT_NORM, bg_color[(sinf_items[i].style / XITK_DRAW_B) & 7]);
    }

    for (; i < SINF_INT; i++) {
      lbl.label = sinf_get_bool (sinfo, values, i);
      sinfo->w[i] = xitk_noskin_label_color_create (&lbl,
        sinf_items[i].x + 5, sinf_items[i].y + 15, sinf_items[i].w - 8, 20,
        fontname, XITK_NOSKIN_TEXT_NORM, bg_color[(sinf_items[i].style / XITK_DRAW_B) & 7]);
    }

    for (; i < SINF_FOURCC; i++) {
      lbl.label = sinf_get_int (sinfo, values, i);
      sinfo->w[i] = xitk_noskin_label_color_create (&lbl,
        sinf_items[i].x + 5, sinf_items[i].y + 15, sinf_items[i].w - 8, 20,
        fontname, XITK_NOSKIN_TEXT_NORM, bg_color[(sinf_items[i].style / XITK_DRAW_B) & 7]);
    }

    for (; i < SINF_SPECIAL; i++) {
      lbl.label = sinf_get_4cc (sinfo, values, i);
      sinfo->w[i] = xitk_noskin_label_color_create (&lbl,
        sinf_items[i].x + 5, sinf_items[i].y + 15, sinf_items[i].w - 8, 20,
        fontname, XITK_NOSKIN_TEXT_NORM, bg_color[(sinf_items[i].style / XITK_DRAW_B) & 7]);
    }

    lbl.label = sinf_get_res (sinfo, values);
    sinfo->w[SINF_video_size] = xitk_noskin_label_color_create (&lbl,
        sinf_items[i].x + 5, sinf_items[i].y + 15, sinf_items[i].w - 8, 20,
      fontname, XITK_NOSKIN_TEXT_NORM, bg_color[(sinf_items[i].style / XITK_DRAW_B) & 7]);
  }

  {
    uint32_t style = XITK_DRAW_SAT (sinfo->nw.gui->gfx_saturation);
    xitk_labelbutton_widget_t lb = {
      .nw = {
        .wl = sinfo->nw.wl,
        .userdata = sinfo,
        .add_state = gui->stream_info_auto_update
                   ? (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE | XITK_WIDGET_STATE_ON)
                   : (XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE)
      },
      .button_type = RADIO_BUTTON,
      .align       = ALIGN_CENTER
    };
    char buf[80];

    y = WINDOW_HEIGHT - (23 + 15);

    /* HACK: make this a radio button, and reset it inside callback.
     * exception: on double click, set it to "on", and enter auto update mode. */
    x = 15;
    lb.nw.tips  = _("F5; double click or double type to toggle auto update.");
    lb.label    = _("Update");
    lb.style    = XITK_DRAW_R | XITK_DRAW_B | style;
    lb.callback = stream_infos_update;
    sinfo->update = xitk_noskin_labelbutton_create (&lb,
      x, y, 100, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);

    lb.nw.add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE;
    lb.button_type = CLICK_BUTTON;
    x = (WINDOW_WIDTH - (100 + 2 * 15)) / 2;
    snprintf (buf, sizeof (buf), "%s-C", xitk_gettext ("Ctrl"));
    lb.nw.tips  = buf;
    lb.label    = xitk_gettext ("Copy");
    lb.style    = XITK_DRAW_R | XITK_DRAW_G | style;
    lb.callback = stream_infos_copy;
    sinfo->copy = xitk_noskin_labelbutton_create (&lb,
      x, y, 100, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);

    x = WINDOW_WIDTH - (100 + 15);
    lb.nw.tips  = "Esc";
    lb.label    = _("Close");
    lb.style    = XITK_DRAW_R | style;
    lb.callback = stream_infos_exit;
    sinfo->close = xitk_noskin_labelbutton_create (&lb,
      x, y, 100, 23, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_NORM, XITK_NOSKIN_TEXT_INV, btnfontname);
  }

  sinfo->widget_key = xitk_be_register_event_handler ("sinfos", sinfo->nw.xwin, stream_infos_event, sinfo, NULL, NULL);
  gui_raise_window (sinfo->nw.gui, sinfo->nw.xwin, 1, 0);
  xitk_window_set_input_focus (sinfo->nw.xwin);
}
