/*
 * Copyright (C) 2014 Canonical Ltd.
 *
 * This file is part of unity-chromium-extension
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "connection.h"

#include <QByteArray>
#include <QDebug>
#include <QFile>
#include <QJsonDocument>
#include <QSocketNotifier>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>

using namespace UnityWebapps;

namespace UnityWebapps {

class ConnectionPrivate: public QObject
{
    Q_OBJECT
    Q_DECLARE_PUBLIC(Connection)

public:
    inline ConnectionPrivate(Connection *connection);
    inline ~ConnectionPrivate();

    bool open();

private Q_SLOTS:
    void onReadyRead();

private:
    QFile m_readChannel;
    QFile m_writeChannel;
    QByteArray m_readBuffer;
    int m_expectedLength;
    mutable Connection *q_ptr;
};

} // namespace

ConnectionPrivate::ConnectionPrivate(Connection *connection):
    QObject(connection),
    m_expectedLength(0),
    q_ptr(connection)
{
}

ConnectionPrivate::~ConnectionPrivate()
{
}

bool ConnectionPrivate::open()
{
    bool ok;

    int flag = fcntl(STDIN_FILENO, F_GETFL);
    if (Q_UNLIKELY(flag == -1)) return false;

    fcntl(STDIN_FILENO, F_SETFL, flag | O_NONBLOCK);

    ok = m_readChannel.open(stdin, QIODevice::ReadOnly);
    if (Q_UNLIKELY(!ok)) return false;

    ok = m_writeChannel.open(stdout, QIODevice::WriteOnly);
    if (Q_UNLIKELY(!ok)) return false;

    QSocketNotifier *notifier = new QSocketNotifier(m_readChannel.handle(),
                                                    QSocketNotifier::Read,
                                                    this);
    QObject::connect(notifier, SIGNAL(activated(int)),
                     this, SLOT(onReadyRead()));

    // There might be already something to read
    onReadyRead();

    return true;
}

void ConnectionPrivate::onReadyRead()
{
    Q_Q(Connection);

    while (true) {
        if (m_expectedLength == 0) {
            /* We are beginning a new read */

            quint32 length;
            int bytesRead = m_readChannel.read((char *)&length,
                                               sizeof(length));
            if (bytesRead < int(sizeof(length))) break;
            m_expectedLength = length;
            m_readBuffer.clear();
        }

        int neededBytes = m_expectedLength - m_readBuffer.length();
        QByteArray buffer = m_readChannel.read(neededBytes);
        m_readBuffer += buffer;
        if (buffer.length() < neededBytes) break;
        if (m_readBuffer.length() == m_expectedLength) {
            QJsonDocument doc = QJsonDocument::fromJson(m_readBuffer);
            Q_EMIT q->messageReceived(doc.toVariant().toMap());
            m_expectedLength = 0;
        }
    }
}

Connection::Connection(QObject *parent):
    QObject(parent),
    d_ptr(new ConnectionPrivate(this))
{
}

Connection::~Connection()
{
}

bool Connection::open()
{
    Q_D(Connection);
    return d->open();
}

void Connection::postMessage(const QVariantMap &message)
{
    Q_D(Connection);

    QJsonDocument doc = QJsonDocument::fromVariant(message);
    QByteArray json = doc.toJson();
    quint32 length = json.length();
    d->m_writeChannel.write((char *)&length, sizeof(length));
    d->m_writeChannel.write(json);
    d->m_writeChannel.flush();
}

#include "connection.moc"
