/* This is part of tmview, a dvi previewer. (c) 1995 Thomas Moor         */
/*                                                                       */
/* This program may be used without any warranty. It may be modified and */
/* distributed without any restrictions.                                 */

#include "../src/defs.h"

#include <termios.h>
#include <errno.h>

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/mman.h>

#include <linux/fb.h>
#include <linux/vt.h>
#include <linux/kd.h> 

/* framebuffer globals  */  
static uchar *thefb=NULL;
static long thefblen;
static int thefbxdim;
static int thefbydim;
static int thefbbyteswidth;
static int thefbfd=-1;
static struct fb_cmap origpalette={0,0,NULL,NULL,NULL,NULL};
static unsigned short origreds[256];
static unsigned short origgreens[256];
static unsigned short origblues[256];
static unsigned short origtrans[256];

/* terminal related stuff */
static struct termios origtermattr;
static struct termios noblotermattr;
static int otermvalid=0;
static int thevtfd=-1;
static int themousefd=-1;


/* globals used by my painting stuff */  
static uchar *offscreen=NULL, *ptorigin;
static int ofscx1, ofscy1, ofscx2, ofscy2;
static int fontw, fonth, fonts;

/* Function prototypes for font-scaling */
void preparefont(void); 
void closefont(void);
char* expandedfont=NULL;

/*#define DEBUGFB */   

/* if POSIX O_NONBLOCK is not available, use O_NDELAY */
#if !defined(O_NONBLOCK) && defined(O_NDELAY)
#define O_NONBLOCK O_NDELAY
#endif


/*****************************************************************/
/* deal with framebuffer and virtual terminal                    */
/*                                                               */

void vgaclose(void){
  /* must be very robust, called by "atexit" at any time */
  /* or even by the signal catcher. issue: do each of the*/
  /* below actions only once .. if cleaning itself causes*/
  /* a serious error, the signal catcher will again cause*/
  /* cleaning and thus again an error ..                 */
  struct vt_mode vtmode;

  char* zwpt;
  int zwfd;

  pfprot("\nwritefb: restore console: ");

  if(otermvalid){
    pfprot("a");
    otermvalid=0;
    tcsetattr(STDIN_FILENO, TCSANOW, &origtermattr);
  }
  if(themousefd!=-1){
    pfprot("i");
    zwfd=themousefd;
    themousefd=-1;
    close(zwfd);
  } 
  if(thefbfd!=-1) {
    if(origpalette.len!=0) {
      pfprot("c");
      ioctl(thefbfd,FBIOPUTCMAP, &origpalette);
      origpalette.len=0;
    }
  }
  if(thefb!=NULL) { 
    pfprot("m");
    zwpt=thefb;
    thefb=NULL;
    munmap(zwpt,thefblen);
  }
  if(thefbfd!=-1){
    pfprot("f");
    zwfd=thefbfd;
    thefbfd=-1;
    close(zwfd);
  }
  if(thevtfd!=-1){
    pfprot("v");
    zwfd=thevtfd;
    thevtfd=-1;
    ioctl(zwfd, KDSETMODE, KD_TEXT); 
    if(ioctl(zwfd, VT_GETMODE, &vtmode)==0){
      vtmode.mode=VT_AUTO;
      ioctl(zwfd, VT_SETMODE, &vtmode);
    }
    signal(SIGUSR1, SIG_DFL);
    signal(SIGUSR2, SIG_DFL);
    close(zwfd);
  }
  freemem(&offscreen);                   
  closefont();
  pfprot("\n");
 }

void vgaerror(char* mssg){
   pfprot("\nfatal error: writefb: %s\n",mssg);   
   exit(1); 
}

/* what to do about all the signals that indicate abnormal   */
/* termination? need to clean up the console at e.g. SIGSEGV */
/* solution: we install svgalib style signal catcher         */

#define sizeofsig2catch (sizeof(sig2catch)/sizeof(*sig2catch))
static int sig2catch[]={
  SIGHUP, SIGINT, SIGQUIT, SIGILL,
  SIGTRAP, SIGIOT, SIGBUS, SIGFPE,
  SIGSEGV, SIGPIPE, SIGALRM, SIGTERM,
  SIGXCPU, SIGXFSZ, SIGVTALRM, SIGPWR};
static struct sigaction old_signal_handler[sizeofsig2catch];
static void catcher(int);  /* foreward */

void installcatch(void){
  struct sigaction siga;
  int i;
  for(i = 0; i < sizeofsig2catch; i++){
    siga.sa_handler = catcher;
    siga.sa_flags = 0;
    sigemptyset(&(siga.sa_mask));
    sigaction(sig2catch[i], &siga, old_signal_handler + i);
  }
}

static void catcher(int v){
  int i;
  pfprot("\nwritefb: \"%s\" received\n",strsignal(v));
  vgaclose();  
  for(i = 0; i < sizeofsig2catch; i++)
    if(sig2catch[i]==v){
     sigaction(v, old_signal_handler + i, NULL);
     raise(v);
     break;
  }
  if(i >= sizeofsig2catch){
    pfprot("writefb: cought a signal we shouldn't catch.\n");
    raise(SIGSEGV);
  }
}



/* the console switching of course is another quite   */
/* rather tricky issue. for the scope of dvifb there  */
/* is only the routine "myupdate" actually drawing    */
/* in the framebuffer ... giving me a chance          */

static volatile int updateinprogress=0;
static volatile int donotupdate=0;
static volatile int pendingswitch=0;

void setpalette(void); /* foreward */

static void vtrelease(int n){  /* switch away */
#ifdef DEBUGFB
  pfprot("(vtrelease)");
#endif
  donotupdate=1;        /* indicate that drawings are no more welcome */ 
  if(!updateinprogress) /* if we are not drawing "right now" its easy: */
    ioctl(thevtfd, VT_RELDISP, 1); /* just do it! */
  else                  /* if we are drawing "right now": */
    pendingswitch=1;    /* ask the drawinroutine to do it later on */ 
}

static void vtaquire(int n) {  /* come back */
#ifdef DEBUGFB
  pfprot("(vtaquire)");
#endif
  ioctl(thevtfd, VT_RELDISP, VT_ACKACQ);
  if(offscreen!=NULL && thefb!=NULL) {
     memcpy(thefb,offscreen,thefblen);
     setpalette();
  }
  pendingswitch=0;    /* discard a pending request ... */
  donotupdate=0;      /* drawwings welcome again */
}


void myupdate(int x1, int y1, int x2, int y2) {
   int h,w;
   uchar *src, *dst; 
 
   if(x1<0) x1=0; if(y1<0) y1=0;
   if(x2>truevgaxdim-1) x2=truevgaxdim-1;
   if(y2>truevgaydim-1) y2=truevgaydim-1;
   if(x2<x1 || y2<y1) return; 

   if(donotupdate) return; /* vt is switched away, dont draw on screen */
   updateinprogress=1;     /* indicate that there are drawings goning on */

#ifdef DEBUGFB
   pfprot("(myupdate");
#endif
   for(
     w=x2-x1+1,
     h=y2-y1+1,
     src=offscreen+thefbbyteswidth*y1+x1,   
     dst=thefb+thefbbyteswidth*y1+x1;
     h>0;
     h--,
     src+=thefbbyteswidth,
     dst+=thefbbyteswidth)
       memcpy(dst,src,w);

   updateinprogress=0;     /* all drawings done so far */
   if(pendingswitch){      /* aha: console switch requested while we were busy */
     pendingswitch=0;    
     ioctl(thevtfd, VT_RELDISP, 1);
   }
#ifdef DEBUGFB
   pfprot(")");
#endif
}



void fbsetpalettecolor(int num, unsigned short r, 
    unsigned short g, unsigned short b){
  struct fb_cmap fbc;
  fbc.start=num;
  fbc.len=1;
  fbc.red=&r;
  fbc.green=&g;
  fbc.blue=&b;
  fbc.transp=NULL;
  if(0!=ioctl(thefbfd,FBIOPUTCMAP, &fbc))
    vgaerror("\nerror: writefb: setpalette\n");
}

void setpalette(void) {
  int i;
  long gl;

  for(i=0;i<COLORS_PER_GREY;i++) {
    gl=(i==0) ? 0: 0xffff * exp(GAMMA*log(i/(float) (COLORS_PER_GREY-1)));
    fbsetpalettecolor(COLORS_PER_GREY-1-i,gl,gl,gl);
  }
  fbsetpalettecolor(SETPALTEXTCOL);           
  fbsetpalettecolor(SETPALTEXTBACKCOL);      
  fbsetpalettecolor(SETPALBORDERCOL);   
  fbsetpalettecolor(SETPALFRAMECOL);    
  fbsetpalettecolor(SETPALRECTCOL);     
  fbsetpalettecolor(SETPALTFMCOL);      
  fbsetpalettecolor(SETPALHREFCOL);     
  fbsetpalettecolor(SETPALMIXEDCOL);     
  fbsetpalettecolor(SETPALMARKSCOL);     
  fbsetpalettecolor(SETPALMARKHCOL);     
  fbsetpalettecolor(SETPALMARKDCOL);     
  fbsetpalettecolor(SETPALFOUNDCOL);     
}


void vgaopen(void) {  
  struct fb_fix_screeninfo fixinfo;
  struct fb_var_screeninfo varinfo;
  char vtdevice[50];
  char* str;
  int  ttyfd, vtno;
  struct vt_stat ttystat;
  struct vt_mode vtmode;

  pfprot("writefb: open: ");

  /* propper clean up console on normal exit */
  atexit(vgaclose);
  /* ... and any other terminating signals   */
  installcatch();  

  /* memorymap framebuffer */
  /* Open problem: any help for this one?                 */
  /* I want the framebuffer to be owned by root and to    */
  /* have crw--w--w- flags. So nobody can read my screen. */ 
  /* Now tmview will only write on the device, so tmview  */
  /* should open it write-only. Thus tmview would not     */
  /* require root provileges. But when openeing write-only*/
  /* I always get an error when memory-mapping the device.*/
  /* Why?                                                 */
  pfprot("f");
  str=getenv("FRAMEBUFFER");
  if(str==NULL) str=THEFBDEV;/* is this a security problem? */
  pfprot("(%s)",str);
  thefbfd=open(str,O_RDWR);  /* want write-only here */
  if(thefbfd==-1) 
    vgaerror("cannot open framebuffer device");
  if(0!=ioctl(thefbfd,FBIOGET_FSCREENINFO, &fixinfo))
    vgaerror("cannot get fixed info: its not a framebuffer?");
  pfprot("(%s)",fixinfo.id);
  if(0!=ioctl(thefbfd,FBIOGET_VSCREENINFO, &varinfo))
    vgaerror("cannot get var info");
  if(varinfo.xres!= varinfo.xres_virtual || varinfo.yres!= varinfo.yres_virtual)
    vgaerror("virtual res doesn't equal physical res");
  if(fixinfo.type!=FB_TYPE_PACKED_PIXELS || fixinfo.visual!= FB_VISUAL_PSEUDOCOLOR
     || varinfo.bits_per_pixel != 8)
    vgaerror("pseudocolor, packed-pixels, 8bpp required\n     ... try fbset utility?");
  thefbxdim=varinfo.xres;
  thefbydim=varinfo.yres;
  pfprot("(%dx%d)",thefbxdim,thefbydim);
  thefbbyteswidth=varinfo.xres;
  thefblen=((((long)thefbbyteswidth * (long)varinfo.yres * varinfo.bits_per_pixel+7)/8) 
             + 0x0fffL) & ~0x0fffL;
  if(thefblen>fixinfo.smem_len)
    pfprot("warning: writefb: mapping more than there is?\n");
  pfprot("m");
  thefb=mmap(NULL,thefblen, PROT_WRITE, MAP_SHARED, thefbfd, 0); /* get error here */
  if(thefb==NULL || (long int)thefb ==-1){
    thefb=NULL;
     vgaerror("memory mapping of framebuffer failed");
  }
  seteuid(getuid());  /* give back root permissions */
#ifdef DEBUGFB
  pfprot("(set euid %d uid %d)",geteuid(),getuid());
#endif

  /* .. some basic tests, copied from below: under certain */
  /* circumstances it is wise to bail out right here       */
  if(!isatty(STDIN_FILENO))
    vgaerror("stdin not a terminal");
  ttyfd=open("/dev/tty", O_RDWR);
  if(ttyfd < 0) 
    vgaerror("cannot open /dev/tty");
  if(ioctl(ttyfd, VT_GETSTATE, &ttystat)!=0)
    vgaerror("cannot get state of vt");
  close(ttyfd);


  /* save framebuffers palette */
  pfprot("c");
  origpalette.start=0;
  origpalette.len=255;
  origpalette.red=origreds;
  origpalette.green=origgreens;
  origpalette.blue=origblues;
  origpalette.transp=origtrans;
  if(0!=ioctl(thefbfd,FBIOGETCMAP, &origpalette)){
    origpalette.len=0;
    vgaerror("\nerror: writefb: readpalette\n");
  }

  /* accocaletmviews offscreenbuffer */
  allocmem(&offscreen,thefblen);

  /* open mouse deveice */
#ifdef HASMOUSE
  {
  uchar o;
  pfprot("i");
  pfprot("(%s)",THEMOUSEDEV);
  themousefd=open(THEMOUSEDEV,O_RDONLY | O_NONBLOCK);
  if(themousefd==-1)
    vgaerror("could not open mouse device");
  while(1==read(themousefd, &o, 1)); 
  }
#endif  

  /* install signal handlers for vt-switching */
  pfprot("v");
  ttyfd=open("/dev/tty", O_RDWR);
  if(ttyfd < 0) 
    vgaerror("cannot open /dev/tty");
  if(ioctl(ttyfd, VT_GETSTATE, &ttystat)!=0)
    vgaerror("cannot get state of vt");
  close(ttyfd);
  vtno=ttystat.v_active;
  sprintf(vtdevice,"/dev/tty%d",vtno);
  thevtfd=open(vtdevice, O_RDWR | O_NDELAY); 
  if(thevtfd<0) 
    vgaerror("cannot open vt");
  if(ioctl(thevtfd, VT_GETMODE, &vtmode)!=0)
    vgaerror("cannot get mode of vt");
  vtmode.mode  =VT_PROCESS; 
  vtmode.relsig=SIGUSR1;
  vtmode.acqsig=SIGUSR2;
  signal(SIGUSR1,vtrelease);
  signal(SIGUSR2,vtaquire);
  if(ioctl(thevtfd, VT_SETMODE, &vtmode)!=0)
    vgaerror("cannot install vtswitch handles");
  ioctl(thevtfd, KDSETMODE, KD_GRAPHICS); 

  /* set the terminal to noblocking/noecho */
  pfprot("a");
  if(!isatty(STDIN_FILENO))
    vgaerror("stdin not a terminal");
  tcgetattr(STDIN_FILENO, &origtermattr);
  otermvalid=1;
  tcgetattr(STDIN_FILENO, &noblotermattr);
  noblotermattr.c_lflag &= ~(ICANON|ECHO);
  noblotermattr.c_cc[VMIN] = 0;
  noblotermattr.c_cc[VTIME] = 0;
  tcsetattr(STDIN_FILENO,TCSAFLUSH, &noblotermattr);
  pfprot("\n");
}

void vgagraph(void) {
#ifdef DEBUGFB
  pfprot("(vgagraph ...");
#endif
  vgaxdim=truevgaxdim=thefbxdim;
  vgaydim=truevgaydim=thefbydim;
  ptorigin=offscreen;
  #define PTDELTAH (1)
  #define PTDELTAV (thefbbyteswidth)
  ofscx1=0; ofscy1=0; ofscx2=vgaxdim-1; ofscy2=vgaydim-1; 
  setpalette();
  preparefont();
  vgastatushight=fonth+fonts;
  vgamaxstatuslines=truevgaydim/vgastatushight;
  vgastatuslen=(truevgaxdim-1)/fontw;
#ifdef DEBUGFB
  pfprot("vgagraph: done)");
#endif
}


/**************************************************************************/
/* scrolling ..                                                           */
/*                                                                        */

void vgaxscroll(short dx) { 
  int i;
  uchar *src, *dst; 
  if(abs(dx) >= vgaxdim || dx==0) return;
  if(dx<0) 
    for(i=vgaydim,
      src=offscreen-dx,
      dst=offscreen;
      i>0;
      i--,
      src+=thefbbyteswidth,
      dst+=thefbbyteswidth)
        memmove(dst,src,vgaxdim-dx);
  else
    for(i=vgaydim,
      src=offscreen,
      dst=offscreen+dx;
      i>0;
      i--,
      src+=thefbbyteswidth,
      dst+=thefbbyteswidth)
        memmove(dst,src,vgaxdim-dx);
}   

void vgayscroll(short dy) {
  int i;
  uchar *src, *dst; 
  if(abs(dy) >= vgaydim || dy==0) return;   
  if(dy<0) 
    for(
      i=vgaydim+dy,
      src=offscreen+thefbbyteswidth*(-dy),   
      dst=offscreen;
      i>0;
      i--,
      src+=thefbbyteswidth,
      dst+=thefbbyteswidth)
        memcpy(dst,src,vgaxdim);
  else /* dy>0 */
    for(
      i=vgaydim-dy,
      src=offscreen+thefbbyteswidth*(vgaydim-dy-1),   
      dst=offscreen+thefbbyteswidth*(vgaydim-1);
      i>0;
      i--,
      src-=thefbbyteswidth,
      dst-=thefbbyteswidth)
        memcpy(dst,src,vgaxdim);
}   


/**************************************************************************/
/* keys (and later mice ..)                                               */
/*                                                                        */


/* mini key queue, holds at most one key */
int mode2char=-1; 

int vgawaitio(fd_set * fdsetin, fd_set * fdsetout, struct timeval *timeout) {
  fd_set myfdsetin;
  int res;
  if(mode2char>=0) return(STDIN_FILENO);
  if (fdsetin==NULL) {
      fdsetin = &myfdsetin;
      FD_ZERO(fdsetin);
  }
  FD_SET(STDIN_FILENO, fdsetin);
#ifdef HASMOUSE
  FD_SET(themousefd, fdsetin);
#endif
  res=select(FD_SETSIZE,fdsetin, fdsetout,NULL,timeout);
  return(res);
}

uchar getonechar(void){
  int res;
  if(1==read(STDIN_FILENO, &res, 1)) 
    return(res);
  return(0);
}

uchar vgagetchar(int mode) {  
  /*
    mode==0: get next key,  dont block
    mode==1: get next key,  wait
    mode==2: peek next key, dont block
  */
  uchar input,next1,next2,next3;
  uchar com=KEYNOP;

  if(mode2char>=0){
     com=mode2char;  
     if(mode!=2) mode2char=-1;
     return(com);
  }
  if(mode==1) /* hang until event occurs */
    vgawaitio(NULL, NULL, NULL); 
  input=getonechar(); 
  if(input >= ' ' && input < 128) com= input;
  else switch(input) {
    case 9: com=KEYTAB; break;
    case 10: com=KEYRET; break;
    case 27:                           /* escape */
      next1=0; next2=0; next3=0;
      next1= getonechar();
      switch(next1) {
      case 91:  
        next2=getonechar(); 
        switch(next2) {
        case '6': com= KEYNEXT;   break; 
        case '5': com= KEYPREV;   break;
        case 'A': com= KEYUP;     break;
        case 'B': com= KEYDOWN;   break;
        case 'D': com= KEYLEFT;   break;
        case 'C': com= KEYRIGHT;  break;
        case 'G': com= KEYCENTER; break;
        case '1': com= KEYHOME;   break;
        case '4': com= KEYEND;    break;
        } 
      break;
      case 0: case 27: com= KEYESC; break; /* allow wild escape abuse */
      default:
      }
    break;
  }  
  /*if(input !=0 && input != 255)
    fprintf(prot,"(vgacommeand: %d %d %d %d)",input,next1,next2,next3);*/
  while(input !=0 && input != 255) input=getonechar(); 
  if(mode==2 && com!=KEYNOP)
    mode2char=com;
  return(com);
}


/* now the mouse. we abuse "gpm -R" as a driver, i.e. we read from */
/* the pipe "/dev/gpmdata" and expect mouse-system protocol only   */

/* readbuffer for nousedevice 20 events a 5 bytes */
#define MAXMOUSEBUF 100 
#define PACKLEN 5

#ifdef HASMOUSE
static unsigned char mousebuf[MAXMOUSEBUF];
static int nmousebuf=0;
#endif
static int omx, omy;

void vgasetmouse(int x, int y) {
  omx=MIN(2*vgaxdim,MAX(-vgaxdim,x));
  omy=MIN(2*vgaydim,MAX(-vgaydim,y));
}  

void vgagetmouse(int *x, int *y, int *left, int* right) {
  int button=0,dx=0,dy=0; 
#ifdef HASMOUSE
  int p,rb;
  if(donotupdate==0){ /* only care about the mouse as long as the vt is active */
    while(1){ /* fill mousebuf */
      rb=read(themousefd, mousebuf+nmousebuf, MAXMOUSEBUF-nmousebuf);
      if(rb>=0) nmousebuf+=rb; 
      if(nmousebuf<PACKLEN) break; /* while ends here */ 
      p=0;
      while(1){ /* process mousebuf */
        while((mousebuf[p] & 0xf8) != 0x80 && (nmousebuf-p>=PACKLEN)) p++;
        if(nmousebuf-p<PACKLEN) break; /* while ends here */
        button = (~mousebuf[p]) & 0x07;
        dx =  (char)(mousebuf[p+1]);
        dx += (char)(mousebuf[p+3]);
        dy = -((char)(mousebuf[p+2]));
        dy -= (char)(mousebuf[p+4]);
        /*      
        pfprot("(p=%d; %#02x %#02x %#02x %#02x %#02x)",
          p, mousebuf[p],mousebuf[p+1],mousebuf[p+2],mousebuf[p+3],mousebuf[p+4]);
        pfprot("(dx %d dy %d but %#02x)",dx,dy,button);    
        pfprot("(%d)",p);
        */    

        p+=PACKLEN;
      }/*end: process mousebuf */
      if(p<nmousebuf)
        memmove(mousebuf,mousebuf+p,nmousebuf-p);      
      nmousebuf-=p;
    } /*end: fill mousebuf */
  } /* donotupdate */

  /* cheap acceleration. turn it of by MOUSESPEED=1 */
  if(abs(dx) > MOUSESPEED) dx*=MOUSESPEED;
  if(abs(dy) > MOUSESPEED) dy*=MOUSESPEED;
#endif
  *x=omx+dx;
  *y=omy+dy;
  vgasetmouse(*x,*y);
  *left= button & 0x04;
  *right=button & 0x01;
}

void vgascreen(int son) {  
}



/**************************************************************************/
/*
   now the pixel copy stuff. 
   destination is allways the offscreen context. 
   provide debugging makros to potential segfaults.
*/

/* #define DEBUGPIXEL */

#ifdef DEBUGPIXEL
#define CPT(apt,mess) \
     if(apt<offscreen || apt>=offscreen+vgaxdim*vgaydim ) \
     {pfprot("(ctp %s)\n",mess); exit(1);}

#define CXY(mess) \
      if(x2<=x1 || y2<=y1) \
        {pfprot("(cxy %s)\n",mess);}

#define CXW(mess) \
      if(w<0 || h<0) \
        {pfprot("(cxw %s)\n",mess);}\

#else /* no debugging */
#define CPT(apt,mess) 
#define CXW(mess) 
#define CXY(mess)
#endif


/****************************************************************/
/* set/reset clipping                                           */

void mysetclip(int x1, int y1, int x2, int y2) {
   if(x1<0) x1=0; if(y1<0) y1=0;
   if(x2>vgaxdim-1) x2=vgaxdim-1;
   if(y2>vgaydim-1) y2=vgaydim-1;
   if(x2<x1 || y2<y1) {
      CXY("clip") 
      x1=0;x2=0;x2=0;y1=0; /* dirty point? should not happen anyway */
   } 
   ofscx1=x1; 
   ofscy1=y1; 
   ofscx2=x2; 
   ofscy2=y2; 
}

void mysetclipof(void) {
   ofscx1=0; 
   ofscy1=0; 
   ofscx2=truevgaxdim-1; 
   ofscy2=truevgaydim-1; 
}


/* dimm colors for color special */
int mydimm(int color, int dimm) {
  return(color>>dimm);
}


/*****************************************************************/
/* actually draw something                                       */

void mycopybitsbw(int tx, int ty, int sw, int sh, void* src, int c) {
/* this is with self made clipping. no good  ? */
/* but it will do for all kind of BMUNIT       */
 
  int j,i,supi,supj,mini,minj,sx, skip;
  BMUNIT *srow, *sbmu;
  BMUNIT premask, firstpremask;
  register BMUNIT data, mask;
  short sbmu_wide;
  char *ydest, *xdest;
  char blackc; 

  /*fprintf(prot,"(vgacopybitmap1 : tx %d ty %d sw %d sh %d, ...\n",
       tx,ty,sw, sh); */                             
  mini=minj=0;
  if(ty<ofscy1) {mini= ofscy1-ty; ty=ofscy1;} 
  if(tx<ofscx1) {minj= ofscx1-tx; tx=ofscx1;} 
  supi= MIN(sh,ofscy2+1-ty+mini);
  supj= MIN(sw,ofscx2+1-tx+minj);
  if(supi <= mini || supj <= minj) {
     /*fprintf(prot,"clipped all away. no drawings)\n"); */
     return;
  }
  ydest=ptorigin+ty*PTDELTAV+tx;
  /* fprintf(prot,"clipped: tx %d ty %d mi %d si %d mj %d sj %d \n",
     tx,ty,mini,supi,minj,supj); */
  
  blackc=(*vgadimm)(BLACKCOL,c);
  sbmu_wide= ROUNDUP(sw,BITS_PER_BMUNIT);
  skip= minj & (BITS_PER_BMUNIT-1);
  sx= minj >> BITS_LOG2;
  premask = 1 << (BITS_PER_BMUNIT-1);
  firstpremask = premask >> skip;
  
  srow=((BMUNIT*)src)+mini*sbmu_wide+sx;
  for (i = mini;i<supi;i++) {
    xdest=ydest;
    mask = firstpremask;
    sbmu=srow;
    data=*sbmu;
    for (j = minj;j<supj;j++) {       
      CPT(xdest,"copybitbw") 
      if(!mask) {
        mask=premask;
        data= *(++sbmu);
      }
      if(data & mask) *xdest=  blackc;
      mask >>= 1;
      xdest++;
    }
    srow+=sbmu_wide;
    sbmu=srow;
    ydest+=PTDELTAV;
  }
}



void mycopybitsgs(int tx, int ty, int sw, int sh, void* src, int c) { 
/* this is with self made clipping. no good  ? */
/* but it will do for all kind of BMUNIT       */
/* compatible to gl_putbox                     */

  int j,i,supi,supj,mini,minj,sx, skip;
  BMUNIT *srow, *sbmu;
  BMUNIT premask, firstpremask;
  register BMUNIT data, mask;
  short sbmu_wide;
  int rs, firstrs;
  char *ydest, *xdest;
  char paintc, sourcec;

  mini=minj=0;
  if(ty<ofscy1) {mini= ofscy1-ty; ty=ofscy1;} 
  if(tx<ofscx1) {minj= ofscx1-tx; tx=ofscx1;} 
  supi= MIN(sh,ofscy2+1-ty+mini);
  supj= MIN(sw,ofscx2+1-tx+minj);
  if(supi <= mini || supj <= minj) return;

#ifdef DEBUGFB
  pfprot("(mycopybitsgs : ptr 0x%x tx %d ty %d sw %d sh %d, ...",
       src,tx,ty,sw, sh);                             
#endif

  ydest=ptorigin+ty*PTDELTAV+tx;  
  sbmu_wide= ROUNDUP(sw << GREYSCALE_LOG2,BITS_PER_BMUNIT);
  skip= minj & ((BITS_PER_BMUNIT>>GREYSCALE_LOG2)-1);
  sx= minj >> (BITS_LOG2-GREYSCALE_LOG2);
  premask = ((1<<GREYSCALE)-1) << (BITS_PER_BMUNIT-GREYSCALE);
  firstpremask = premask >> (skip<< GREYSCALE_LOG2);
  firstrs = BITS_PER_BMUNIT-GREYSCALE - skip*GREYSCALE;

  srow=((BMUNIT*)src)+mini*sbmu_wide+sx;
  for (i = mini;i<supi;i++) {
    xdest=ydest;
    mask = firstpremask;
    rs = firstrs;
    sbmu=srow;
    data=*sbmu;
    for (j = minj;j<supj;j++) {
      CPT(xdest,"copybitgs") 
      if(!mask) {
        mask=premask;
        rs=BITS_PER_BMUNIT-GREYSCALE;
        data= *(++sbmu); 
      }
      /* pfprot("- ptr 0x%x data %d -",xdest,((data &mask)>>rs));  */
#ifndef LETSTRYCOLOR
      (*xdest) |= ((data & mask) >> rs); /* sleezy but fast .. (rs+c) */
#else
      if((paintc=(data & mask) >> (rs+c))){
        if((sourcec=(*xdest) & (COLORS_PER_GREY-1))){
          if((paintc+=sourcec)>COLORS_PER_GREY-1) paintc=COLORS_PER_GREY-1;
          (*xdest) &= ~(COLORS_PER_GREY-1);
	}
        (*xdest) |=paintc;
      }
#endif
      mask >>= GREYSCALE;
      rs-=GREYSCALE;
      xdest++;
    }
    srow+=sbmu_wide;
    sbmu=srow;
    ydest+=PTDELTAV;
  }
#ifdef DEBUGFB
    pfprot(")"); 
#endif
}


void mycopybits88(int tx, int ty, int sw, int sh, char* src) { 
/* this is with self made clipping. no good  ? */
/* copy bytes to bytes                         */
/* compatible to gl_putbox                     */

  int j,i,supi,supj,mini,minj;
  char *srow, *sbmu;
  char *ydest, *xdest;


  mini=minj=0;
  if(ty<ofscy1) {mini= ofscy1-ty; ty=ofscy1;} 
  if(tx<ofscx1) {minj= ofscx1-tx; tx=ofscx1;} 
  supi= MIN(sh,ofscy2+1-ty+mini);
  supj= MIN(sw,ofscx2+1-tx+minj);
  if(supi <= mini || supj <= minj) {
     /* pfprot("clipped all away. no drawings)\n"); */
     return;
  }
  ydest=ptorigin+ty*PTDELTAV+tx;
  /* pfprot("(copy 88 clipped: tx %d ty %d mi %d si %d mj %d sj %d \n",
     tx,ty,mini,supi,minj,supj); */ 
  
  srow= src+mini*sw+minj;
  for (i = mini;i<supi;i++) {
    xdest=ydest;
    sbmu=srow;
    for (j = minj;j<supj;j++) {
      *xdest = *sbmu;
      xdest++;
      sbmu++;
    }
    srow+=sw;
    ydest+=PTDELTAV;
  }
  /* pfprot("(done 88)\n");*/
}


void myfillboxor(int tx, int ty, int w, int h, int c) {
 
  char *ydest, *xdest;
  char sourcec, paintc;
  int i;

  /* pfprot("(vgafillboxor : tx %d ty %d sw %d sh %d, ...\n",
       tx,ty,sw, sh); */                             
  if(ty<ofscy1) {h-=ofscy1-ty;  ty=ofscy1;} 
  if(tx<ofscx1) {w-=ofscx1-tx; tx=ofscx1;} 
  if(ty+h>ofscy2+1) h=ofscy2+1-ty;
  if(tx+w>ofscx2+1) w=ofscx2+1-tx;
  if(w <= 0 || h<= 0) {
     /*fprintf(prot,"clipped all away. no drawings)\n"); */
     return;
  }
  CXW("rector");
  ydest=ptorigin+ty*PTDELTAV+tx;
  /* pfprot("clipped or box: tx %d ty %d w %d h %d   \n",
     tx,ty,w,h); */ 
  paintc=0;
  for (;h>0;h--) {
    xdest=ydest; 
    for (i=w;i>0;i--) {
       CPT(xdest,"rector")                
#ifndef LETSTRYCOLOR
       (*xdest)|=c; /* sleezy but fast    */
#else
       if((sourcec=(*xdest) & (COLORS_PER_GREY-1))){
         if((paintc =c+sourcec)>COLORS_PER_GREY-1) paintc=COLORS_PER_GREY-1;
         (*xdest) &= ~(COLORS_PER_GREY-1);
         (*xdest) |=paintc;
       } else {
         (*xdest) |=c;
       }
#endif
       xdest++;
    }
    ydest+=PTDELTAV;
  }
}

void myfillboxbg(int tx, int ty, int w, int h, int c) {
 
  char *ydest;
  int i;
  register char *xdest;
  register char xdc;
  /* fprintf(prot,"(vgafillboxor : tx %d ty %d w %d h %d, ...\n",
       tx,ty,w,h); */                              
  if(ty<ofscy1) {h-=ofscy1-ty;  ty=ofscy1;} 
  if(tx<ofscx1) {w-=ofscx1-tx; tx=ofscx1;} 
  if(ty+h>ofscy2+1) h=ofscy2+1-ty;
  if(tx+w>ofscx2+1) w=ofscx2+1-tx;
  if(tx+w>vgaxdim) w=vgaxdim-tx;
  if(w <= 0 || h<= 0) {
     /*fprintf(prot,"clipped all away. no drawings)\n"); */
     return;
  }
  CXW("rectbg");
  ydest=ptorigin+ty*PTDELTAV+tx;

  /* fprintf(prot,"clipped bgbox : tx %d ty %d w %d h %d   \n",
     tx,ty,w,h); */

  for (;h>0;h--) {
    xdest=ydest;
    for (i=w;i>0;i--) {      
      CPT(xdest,"rectbg")
      xdc=*xdest; 
      if(xdc){      
      /*if(xdc>c) *xdest=c;  */                   /* transparent modus */     
        if(xdc>BACKCOLS && xdc!= c) *xdest=MIXEDCOL;   /* mixing modus */  
      } else *xdest=c;
      xdest++;
    }
    ydest+=PTDELTAV;
  }
}


void myfillbox(int tx, int ty, int w, int h, int c) {
 
  char *ydest;
  int i;
  register char *xdest;
  if(ty<ofscy1) {h-=ofscy1-ty;  ty=ofscy1;} 
  if(tx<ofscx1) {w-=ofscx1-tx; tx=ofscx1;} 
  if(ty+h>ofscy2+1) h=ofscy2+1-ty;
  if(tx+w>ofscx2+1) w=ofscx2+1-tx;
  if(tx+w>vgaxdim) w=vgaxdim-tx;
  if(w <= 0 || h<= 0) return;

#ifdef DEBUGFB
  pfprot("(vgafillbox : tx %d ty %d w %d h %d",
       tx,ty,w,h); 
#endif                              
  CXW("rectbg");
  ydest=ptorigin+ty*PTDELTAV+tx;
  for (;h>0;h--) {
    xdest=ydest;
    for (i=w;i>0;i--) {      
      CPT(xdest,"rectbg")
      (*(xdest))=c; 
      xdest++;
    }     
    ydest+=PTDELTAV;
  }
#ifdef DEBUGFB
  pfprot(")"); 
#endif                              
}

  
int vgasetstatuslines(int n) {
#ifdef DEBUGFB
  pfprot("(vgasetstatuslines %d ... ",n);
#endif 
  vgastatuslines=MAX(0,MIN(vgamaxstatuslines,n));
  vgaydim= truevgaydim- vgastatuslines* vgastatushight;
  vgaxdim= truevgaxdim;

  mysetclipof();   /* not clipped !! */
  if(vgastatuslines!=vgamaxstatuslines)
  myfillbox(0,vgaydim,truevgaxdim,vgastatuslines*vgastatushight,TEXTBACKCOL);
  else
  myfillbox(0,0,truevgaxdim,truevgaydim,TEXTBACKCOL);
  myupdate(0,0,vgaxdim-1,truevgaydim-1);
  mysetclip(0,0,vgaxdim-1,vgaydim-1);
#ifdef DEBUGFB
  pfprot("vgasetstatuslines: done)");
#endif 
  return(vgastatuslines);
}

void vgaupdatestatus(void) {
  mysetclipof();
  myupdate(0, vgaydim,truevgaxdim-1,truevgaydim-1); 
  mysetclip(0,0,vgaxdim-1,vgaydim-1);
}


void vgadrawstatus(char chr, int line, int pos) {
  if(line>=vgastatuslines || line <0) return;
  if(pos>=vgastatuslen || pos <0) return;
  mysetclipof();
  /* chr=(line&0x0f)*16 + (pos&0x0f); *//* test charset */
  mycopybits88(pos*fontw+1, vgaydim+(line+1)* vgastatushight-fonth, 
        fontw, fonth, expandedfont + (unsigned char) chr *fonth*fontw);
  mysetclip(0,0,vgaxdim-1,vgaydim-1); 
  /* pfprot("(vgadrawstatus [%c] l %d p %d )",chr,line,pos); */ 
}


void (*vgaupdate)(int, int, int, int) = myupdate;
void (*vgasetclip)(int, int, int, int) = mysetclip;
void (*vgadrawrect)(int x, int y, int w, int h, int c) = myfillbox; 
void (*vgadrawrector)(int x, int y, int w, int h, int c) = myfillboxor; 
void (*vgadrawrectbg)(int x, int y, int w, int h, int c) = myfillboxbg; 
int  (*vgadimm)(int, int) = mydimm;
void (*vgacopybitmapgs)(int x, int y, int w, int h, void* src, int c) = mycopybitsgs; 
void (*vgacopybitmapbw)(int x, int y, int w, int h, void* src, int c) = mycopybitsbw;


/***************************************************************************/
/* now the font stuff ...                                                  */

/* #include "9x14thin.h" */
/* #include "gekl9x16.h" */
#include "font9x16.h" 


void preparefont(void) {
  uchar *src;
  uchar *dst;
  int i,x,y,b=0;         

  fonth=FONTH;
  fontw=FONTW;
  fonts=FONTS;
  allocmem(&expandedfont,256L*fontw*fonth);
  src=tmviewfont;
  dst=expandedfont;
  for(i=0; i<256; i++){
    for(y=0; y<fonth; y++){
      for(x=0; x<fontw; x++){
        if(x % 8 == 0) b= *src++;
        if(b & (128 >> (x % 8)))  *(dst++) = TEXTCOL;
        else *(dst++) =TEXTBACKCOL;
      }
    }
  }
}

void closefont(void) {
  freemem(&expandedfont);
}


