[comment {-*- tcl -*- doctools manpage}]
[manpage_begin math::interpolate n 1.1]
[keywords interpolation]
[keywords math]
[keywords {spatial interpolation}]
[copyright {2004 Arjen Markus <arjenmarkus@users.sourceforge.net>}]
[copyright {2004 Kevn B. Kenny <kennykb@users.sourceforge.net>}]
[moddesc   {Tcl Math Library}]
[titledesc {Interpolation routines}]
[category  Mathematics]
[require Tcl [opt 8.4]]
[require struct]
[require math::interpolate [opt 1.1]]

[description]
[para]
This package implements several interpolation algorithms:

[list_begin itemized]
[item]
Interpolation into a table (one or two independent variables), this is useful
for example, if the data are static, like with tables of statistical functions.

[item]
Linear interpolation into a given set of data (organised as (x,y) pairs).

[item]
Lagrange interpolation. This is mainly of theoretical interest, because there is
no guarantee about error bounds. One possible use: if you need a line or
a parabola through given points (it will calculate the values, but not return
the coefficients).
[para]
A variation is Neville's method which has better behaviour and error
bounds.

[item]
Spatial interpolation using a straightforward distance-weight method. This procedure
allows any number of spatial dimensions and any number of dependent variables.

[item]
Interpolation in one dimension using cubic splines.

[list_end]

[para]
This document describes the procedures and explains their usage.

[section "INCOMPATIBILITY WITH VERSION 1.0.3"]

The interpretation of the tables in the [cmd ::math::interpolate::interpolate-1d-table] command
has been changed to be compatible with the interpretation for 2D interpolation in
the [cmd ::math::interpolate::interpolate-table] command. As a consequence this version is
incompatible with the previous versions of the command (1.0.x).

[section "PROCEDURES"]

The interpolation package defines the following public procedures:

[list_begin definitions]

[call [cmd ::math::interpolate::defineTable] [arg name] [arg colnames] [arg values]]

Define a table with one or two independent variables (the distinction is implicit in
the data). The procedure returns the name of the table - this name is used whenever you
want to interpolate the values. [emph Note:] this procedure is a convenient wrapper for the
struct::matrix procedure. Therefore you can access the data at any location in your program.

[list_begin arguments]
[arg_def string name in] Name of the table to be created

[arg_def list colnames in] List of column names

[arg_def list values in] List of values (the number of elements should be a
multiple of the number of columns. See [sectref EXAMPLES] for more information on the
interpretation of the data.

[para]
The values must be sorted with respect to the independent variable(s).

[list_end]
[para]

[call [cmd ::math::interpolate::interp-1d-table] [arg name] [arg xval]]

Interpolate into the one-dimensional table "name" and return a list of values, one for
each dependent column.

[list_begin arguments]
[arg_def string name in] Name of an existing table

[arg_def float xval in] Value of the independent [emph row] variable

[list_end]

[para]

[call [cmd ::math::interpolate::interp-table] [arg name] [arg xval] [arg yval]]

Interpolate into the two-dimensional table "name" and return the interpolated value.

[list_begin arguments]
[arg_def string name in] Name of an existing table

[arg_def float xval in] Value of the independent [emph row] variable

[arg_def float yval in] Value of the independent [emph column] variable

[list_end]

[para]

[call [cmd ::math::interpolate::interp-linear] [arg xyvalues] [arg xval]]

Interpolate linearly into the list of x,y pairs and return the interpolated value.

[list_begin arguments]

[arg_def list xyvalues in] List of pairs of (x,y) values, sorted to increasing x.
They are used as the breakpoints of a piecewise linear function.

[arg_def float xval in] Value of the independent variable for which the value of y
must be computed.

[list_end]

[para]

[call [cmd ::math::interpolate::interp-lagrange] [arg xyvalues] [arg xval]]

Use the list of x,y pairs to construct the unique polynomial of lowest degree
that passes through all points and return the interpolated value.

[list_begin arguments]

[arg_def list xyvalues in] List of pairs of (x,y) values

[arg_def float xval in] Value of the independent variable for which the value of y
must be computed.

[list_end]

[para]

[call [cmd ::math::interpolate::prepare-cubic-splines] [arg xcoord] [arg ycoord]]

Returns a list of coefficients for the second routine
[emph interp-cubic-splines] to actually interpolate.

[list_begin arguments]
[arg_def list xcoord] List of x-coordinates for the value of the
function to be interpolated is known. The coordinates must be strictly
ascending. At least three points are required.

[arg_def list ycoord] List of y-coordinates (the values of the
function at the given x-coordinates).

[list_end]

[para]

[call [cmd ::math::interpolate::interp-cubic-splines] [arg coeffs] [arg x]]

Returns the interpolated value at coordinate x. The coefficients are
computed by the procedure [emph prepare-cubic-splines].

[list_begin arguments]
[arg_def list coeffs] List of coefficients as returned by
prepare-cubic-splines

[arg_def float x] x-coordinate at which to estimate the function. Must
be between the first and last x-coordinate for which values were given.

[list_end]

[para]

[call [cmd ::math::interpolate::interp-spatial] [arg xyvalues] [arg coord]]

Use a straightforward interpolation method with weights as function of the
inverse distance to interpolate in 2D and N-dimensional space

[para]
The list xyvalues is a list of lists:
[example {
    {   {x1 y1 z1 {v11 v12 v13 v14}}
	{x2 y2 z2 {v21 v22 v23 v24}}
	...
    }
}]
The last element of each inner list is either a single number or a list in itself.
In the latter case the return value is a list with the same number of elements.

[para]
The method is influenced by the search radius and the power of the inverse distance

[list_begin arguments]
[arg_def list xyvalues in] List of lists, each sublist being a list of coordinates and
of dependent values.

[arg_def list coord in] List of coordinates for which the values must be calculated

[list_end]

[para]

[call [cmd ::math::interpolate::interp-spatial-params] [arg max_search] [arg power]]

Set the parameters for spatial interpolation

[list_begin arguments]
[arg_def float max_search in] Search radius (data points further than this are ignored)

[arg_def integer power in] Power for the distance (either 1 or 2; defaults to 2)

[list_end]

[call [cmd ::math::interpolate::neville] [arg xlist] [arg ylist] [arg x]]

Interpolates between the tabulated values of a function
whose abscissae are [arg xlist]
and whose ordinates are [arg ylist] to produce an estimate for the value
of the function at [arg x].  The result is a two-element list; the first
element is the function's estimated value, and the second is an estimate
of the absolute error of the result.  Neville's algorithm for polynomial
interpolation is used.  Note that a large table of values will use an
interpolating polynomial of high degree, which is likely to result in
numerical instabilities; one is better off using only a few tabulated
values near the desired abscissa.

[list_end]

[section EXAMPLES]

[emph "Example of using one-dimensional tables:"]
[para]
Suppose you have several tabulated functions of one variable:
[example {
    x     y1     y2
  0.0    0.0    0.0
  1.0    1.0    1.0
  2.0    4.0    8.0
  3.0    9.0   27.0
  4.0   16.0   64.0
}]
Then to estimate the values at 0.5, 1.5, 2.5 and 3.5, you can use:
[example {
   set table [::math::interpolate::defineTable table1 \
       {x y1 y2} {   -      1      2
                   0.0    0.0    0.0
                   1.0    1.0    1.0
                   2.0    4.0    8.0
                   3.0    9.0   27.0
                   4.0   16.0   64.0}]
   foreach x {0.5 1.5 2.5 3.5} {
       puts "$x: [::math::interpolate::interp-1d-table $table $x]"
   }
}]
For one-dimensional tables the first row is not used. For two-dimensional
tables, the first row represents the values for the second independent variable.
[para]

[emph "Example of using the cubic splines:"]
[para]
Suppose the following values are given:
[example {
    x       y
  0.1     1.0
  0.3     2.1
  0.4     2.2
  0.8     4.11
  1.0     4.12
}]
Then to estimate the values at 0.1, 0.2, 0.3, ... 1.0, you can use:
[example {
   set coeffs [::math::interpolate::prepare-cubic-splines \
                 {0.1 0.3 0.4 0.8  1.0} \
                 {1.0 2.1 2.2 4.11 4.12}]
   foreach x {0.1 0.2 0.3 0.4 0.5 0.6 0.7 0.8 0.9 1.0} {
      puts "$x: [::math::interpolate::interp-cubic-splines $coeffs $x]"
   }
}]
to get the following output:
[example {
0.1: 1.0
0.2: 1.68044117647
0.3: 2.1
0.4: 2.2
0.5: 3.11221507353
0.6: 4.25242647059
0.7: 5.41804227941
0.8: 4.11
0.9: 3.95675857843
1.0: 4.12
}]
As you can see, the values at the abscissae are reproduced perfectly.

[vset CATEGORY {math :: interpolate}]
[include ../doctools2base/include/feedback.inc]
[manpage_end]
