/* Copyright (C) 2016-2020 Open Information Security Foundation
 *
 * You can copy, redistribute or modify this Program under the terms of
 * the GNU General Public License version 2 as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * version 2 along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 */

/**
 * \file
 *
 * \author Victor Julien <victor@inliniac.net>
 *
 */

#include "suricata-common.h"
#include "suricata.h"
#include "detect.h"
#include "detect-parse.h"
#include "detect-content.h"
#include "output-json.h"
#include "util-buffer.h"
#include "util-print.h"
#include "detect-engine-profile.h"

#ifdef PROFILING
SCMutex g_rule_dump_write_m = SCMUTEX_INITIALIZER;

void RulesDumpMatchArray(const DetectEngineThreadCtx *det_ctx,
        const SigGroupHead *sgh, const Packet *p)
{
    JsonBuilder *js = CreateEveHeader(p, LOG_DIR_PACKET, "inspectedrules", NULL);
    if (js == NULL)
        return;

    jb_open_object(js, "inspectedrules");
    jb_set_uint(js, "rule_group_id", sgh->id);
    jb_set_uint(js, "rule_cnt", det_ctx->match_array_cnt);

    jb_open_array(js, "rules");
    for (uint32_t x = 0; x < det_ctx->match_array_cnt; x++) {
        const Signature *s = det_ctx->match_array[x];
        if (s == NULL)
            continue;
        jb_append_uint(js, s->id);

    }
    jb_close(js); // close array
    jb_close(js); // close inspectedrules object
    jb_close(js); // final close

    const char *filename = "packet_inspected_rules.json";
    const char *log_dir = ConfigGetLogDirectory();
    char log_path[PATH_MAX] = "";
    snprintf(log_path, sizeof(log_path), "%s/%s", log_dir, filename);

    SCMutexLock(&g_rule_dump_write_m);
    FILE *fp = fopen(log_path, "a");
    if (fp != NULL) {
        fwrite(jb_ptr(js), jb_len(js), 1, fp);
        fclose(fp);
    }
    SCMutexUnlock(&g_rule_dump_write_m);
    jb_free(js);
}
#endif /* PROFILING */