use ui;
use layout;
use core:geometry;
use core:lang;
use lang:bs:macro;

class TreeWindow extends Frame {
	private TreePainter painter;
	private Map<Key, fn()->void> keymap;
	private Bool mouseDown;

	init() {
		init("Treez", Size(300, 300)) {
			painter();
		}

		painter(painter);
		create();

		keymap.put(Key:i, &this.insert());
		keymap.put(Key:d, &this.delete());
		keymap.put(Key:c, &this.clear());
		keymap.put(Key:l, &this.setLeft());
		keymap.put(Key:r, &this.setRight());
		keymap.put(Key:j, &this.setLeft());
		keymap.put(Key:k, &this.setRight());
		keymap.put(Key:esc, &this.abort());
		keymap.put(Key:e, &this.loadExample());
		keymap.put(Key:q, &this.clearAll());
		keymap.put(Key(60), &this.insertPipe()); // < on swedish keyboards
		keymap.put(Key:z, &this.insertPipe());

		var me = this;
		keymap.put(Key:num0, () => me.numKey(0));
		keymap.put(Key:num1, () => me.numKey(1));
		keymap.put(Key:num2, () => me.numKey(2));
		keymap.put(Key:num3, () => me.numKey(3));
		keymap.put(Key:num4, () => me.numKey(4));
		keymap.put(Key:num5, () => me.numKey(5));
		keymap.put(Key:num6, () => me.numKey(6));
		keymap.put(Key:num7, () => me.numKey(7));
		keymap.put(Key:num8, () => me.numKey(8));
		keymap.put(Key:num9, () => me.numKey(9));
	}

	Bool onClick(Bool down, Point pt, MouseButton button) : override {
		mouseDown = down;
		painter.mouseClicked(down, pt, button);
		repaint();
		true;
	}

	Bool onMouseMove(Point pt) : override {
		lastMouse = pt;
		if (painter.mouseMoved(pt, mouseDown)) {
			repaint();
		}
		true;
	}

	void onMouseLeave() : override {
		mouseDown = false;
	}

	Bool onKey(Bool pressed, Key key, Modifiers modifiers) : override {
		if (!pressed)
			return false;

		if (modifiers == Modifiers:none) {
			if (keymap.has(key)) {
				keymap.get(key).call();
			}
			true;
		} else if (modifiers == Modifiers:shift) {
			if (key == Key:l | key == Key:j) {
				painter.rotateLeft(lastMouse);
				repaint();
			} else if (key == Key:r | key == Key:k) {
				painter.rotateRight(lastMouse);
				repaint();
			} else if (key == Key:c) {
				painter.cloneTree(lastMouse);
				repaint();
			}
		} else if (modifiers == Modifiers:ctrl) {
			if (key == Key:z) {
				painter.undo();
				repaint();
			}
		}

		false;
	}

	private Point lastMouse;

	private Nat number;

	private void insert() {
		painter.insert(number, lastMouse);
		number = 0;
		repaint();
	}

	private void insertPipe() {
		painter.insertPipe(lastMouse);
		repaint();
	}

	private void delete() {
		painter.delete(lastMouse);
		repaint();
	}

	private void clear() {
		painter.clear(lastMouse);
		repaint();
	}

	private void clearAll() {
		painter.clear();
		repaint();
	}

	private void abort() {
		number = 0;
		painter.abort();
		repaint();
	}

	private void setLeft() {
		painter.startDrag(lastMouse, DragFrom:left);
	}

	private void setRight() {
		painter.startDrag(lastMouse, DragFrom:right);
	}

	private void numKey(Nat num) {
		number = number * 10 + num;
	}

	private void loadExample() {
		Package here = named{};
		if (url = here.url) {
			url = url / "${number}.tree";
			try {
				if (!url.exists()) {
					showMessage(this, "File not found", "The file ${url} does not exist.");
				} else if (node = parseFile(url)) {
					painter.load(node);
					repaint();
				}
			} catch (Exception e) {
				showMessage(this, "File format error", "Failed to parse the file: ${e}");
			}
		}
		number = 0;
	}
}

enum DragFrom {
	left, right
}

class TreePainter extends Painter {
	init() {
		init {}
	}

	Point offset;

	Node[] nodes;

	// Last clicked.
	Node? clicked;

	// Mouse origin when dragging.
	Point mouseOrigin;

	// Dragging "from".
	DragFrom edgeFrom;
	Node? from;

	// Undo stack. We cap this at ~10-15 operations.
	Node[][] undoStack;

	Bool render(Size me, Graphics g) {
		g.transform(translate(-offset));
		g.lineWidth = 1.5;

		for (node in nodes) {
			node.draw(g);
		}

		if (from) {
			if (edgeFrom == DragFrom:left) {
				g.line(from.bottomLeft, mouseOrigin, nodeBrush);
			} else {
				g.line(from.bottomRight, mouseOrigin, nodeBrush);
			}
		}

		false;
	}

	void mouseClicked(Bool down, Point pt, MouseButton button) {
		pt += offset;

		if (down) {
			clicked = nodeAt(pt);
			if (clicked) {
				if (from) {
					saveUndo();
					if (edgeFrom == DragFrom:left) {
						from.left = clicked;
					} else {
						from.right = clicked;
					}
				}
				mouseOrigin = pt - clicked.at;
			} else {
				mouseOrigin = pt;
			}
			from = null;
		}
	}

	Bool mouseMoved(Point pt, Bool mouseDown) {
		pt += offset;

		if (from) {
			mouseOrigin = pt;
			return true;
		}

		if (mouseDown) {
			if (clicked) {
				clicked.at = pt - mouseOrigin;
				return true;
			} else {
				offset -= pt - mouseOrigin;
				return true;
			}
		}
		false;
	}

	void insert(Nat number, Point at) {
		at += offset;
		saveUndo();
		nodes << Node(at, number.toS);
	}

	void insertPipe(Point at) {
		saveUndo();
		nodes << Node(at, "|");
	}

	void delete(Point at) {
		at += offset;

		saveUndo();

		for (i, n in nodes) {
			if (n.contains(at)) {
				for (k in nodes) {
					if (k.left is n)
						k.left = null;
					if (k.right is n)
						k.right = null;
				}

				nodes.remove(i);
				break;
			}
		}
	}

	void clear() {
		nodes = Node[];
	}

	void clear(Point at) {
		at += offset;

		if (n = nodeAt(at)) {
			n.left = null;
			n.right = null;
		}
	}

	void startDrag(Point pt, DragFrom edge) {
		pt += offset;

		if (n = nodeAt(pt)) {
			from = n;
			edgeFrom = edge;
		}
	}

	void abort() {
		from = null;
	}

	void load(SimpleNode node) {
		clear();
		offset = Point();
		undoStack.clear();

		loadI(node, 20, 20);
	}

	void rotateLeft(Point pt) {
		pt += offset;

		if (n = nodeAt(pt)) {
			if (child = n.left) {
				saveUndo();

				n.left = child.right;

				for (node in nodes) {
					node.replace(n, child);
				}

				child.right = n;
			}
		}
	}

	void rotateRight(Point pt) {
		pt += offset;

		if (n = nodeAt(pt)) {
			if (child = n.right) {
				saveUndo();

				n.right = child.left;

				for (node in nodes) {
					node.replace(n, child);
				}

				child.left = n;
			}
		}
	}

	private value PosInfo {
		Node node;
		Float width;

		init(Node node) {
			init {
				node = node;
				width = node.text.size.w * 2;
			}
		}
	}

	private PosInfo loadI(SimpleNode node, Float x, Float y) {
		Node n(Point(x, y), node.text);
		nodes << n;
		y += n.text.size.h * 2;

		PosInfo info(n);

		if (left = node.left) {
			var l = loadI(left, x, y);
			info.width += l.width;
			n.at.x += l.width;
			n.left = l.node;
		}

		if (right = node.right) {
			var r = loadI(right, x + info.width, y);
			info.width += r.width;
			n.right = r.node;
		}

		info;
	}

	void cloneTree(Point pt) {
		pt += offset;
		unless (n = nodeAt(pt))
			return;

		saveUndo();

		Float maxY = 0;
		for (n in nodes) {
			maxY = max(maxY, n.bottomRight.y);
		}

		Float delta = maxY + 30 - n.at.y;

		cloneTree(n, delta);
	}

	private Node cloneTree(Node n, Float dy) {
		Node c(n.at + Point(0, dy), n.text.text);
		nodes << c;

		if (left = n.left)
			c.left = cloneTree(left, dy);
		if (right = n.right)
			c.right = cloneTree(right, dy);

		c;
	}

	Node? nodeAt(Point at) {
		for (n in nodes) {
			if (n.contains(at))
				return n;
		}
		null;
	}

	void undo() {
		if (undoStack.empty)
			return;

		nodes = undoStack.last;
		undoStack.pop();
	}

	void saveUndo() {
		// Keep from growing too much.
		if (undoStack.count > 20)
			undoStack.remove(0);

		Node->Node idMap;
		Node[] copy;
		for (i, n in nodes) {
			Node c = Node(n);
			idMap.put(n, c);
			copy << c;
		}

		for (c in copy) {
			c.left = update(c.left, idMap);
			c.right = update(c.right, idMap);
		}

		undoStack << copy;
	}

	private Node? update(Node? n, Node->Node idMap) : static {
		unless (n)
			return null;

		idMap.get(n, n);
	}
}


void main() {
	TreeWindow win;
	win.waitForClose();
}
