Examples
=========

Here, we take a closer look at the examples in the `demo` package. Each section below describes the
functionality of a file in the `demo` package, and what aspect of Storm it tries to illustrate.


Eval
-----

The files `eval.bs` and `eval.bnf` illustrates the tight interaction between different languages by
implementing a simple calculator that can evaluate simple expressions. Call `demo:eval("1 + 3")` to
evaluate expressions, or `demo:tree("1 + 3")` to see the syntax tree for expressions.

```
?Include:root/demo/eval.bnf?
```

The file `eval.bnf` contains the grammar describing the expressions supported by the
calculator. `Expr` is the rule that describes entire expressions, and it is therefore the start
rule. Aside from the language, the `bnf`-file also describes how to transform the syntax tree from a
successful parse into a representation that is more convenient for the user of the grammar. In this
particular case, we are only interested in evaluating the parsed expression, and as such we use the
transformations to evaluate the expression rather than constructing another representation. See
[BNF Syntax](md://BNF_Syntax) for details on the syntax and semantics of the grammar language in Storm.

```
?Include:root/demo/eval.bs?
```

The functions `eval` and `tree` are implemented in Basic Storm in the file `eval.bs`. The `tree`
function creates `Parser` instance which parses strings starting with the supplied `Expr` rule,
parses the string supplied and returns the syntax tree unless there is an error in the input. `eval`
simply calls `tree` to get the syntax tree for the parse, but then continues to `transform` the
syntax tree into the representation described in the `bnf`-file (which is implemented as evaluating
the expression).

In this file, we are using `Expr`, which is defined as a rule in the syntax language, as if it was a
regular type. Basic Storm has no special knowledge of rules and productions defined in the syntax
language, but since the syntax language stores rules and productions as types in the name tree Storm
provides, any language may access and use them. The types defined by rules and productions are used
to store the syntax tree that results from parsing a string, which is why the `tree` function
returns an `Expr`. Furthermore, `Expr` contains a function `transform`, which returns the type
specified when we declared `Expr` in the syntax language. This function is generated by the syntax
language, and it transforms the current syntax node according to the specification in the syntax
language. If we had specified names for the productions, we could have inspected the syntax tree
using constructs like `if (x as AddExpr)` and then inspecting the member variables in those
types. This illustrates the kind of close cooperation possible within Storm, without even losing
type safety.


Present
--------

This file serves to illustrate the fact that the syntax highlighting provided by Storm properly
supports language extensions. Open the file in Emacs in `storm-mode`, uncomment the `present`-block
and see what happens. Since the `present` package, which contains the syntax needed, is not yet
included, the syntax highlighting will be wrong. Now, uncomment the `use present;`-line in the top
of the file and see that the syntax highlighting is now correct since the proper grammar is now
included.

The grammar for the presentation language is available in the file `root/presentation/syntax.bnf`, which
is shown below:

```
?Include:root/presentation/syntax.bnf?
```

The presentation language also uses syntax from the layout language, which is implemented in
`root/layout/syntax.bnf` as follows:

```
?Include:root/layout/syntax.bnf?
```

Finally, these can be used as illustrated in the example in `root/presentation/test/simple.bs`:

```
?Include:root/presentation/test/simple.bs?
```

Reload
-------

This file shows that it is possible to reload code in an already running program in Storm (to
certain degrees, at least).

```
?Include:root/demo/reload.bs?
```

From the REPL, call `demo:reloadMain`, and you shall see that you are returned to the prompt, but
the numbers 1 to 10 are displayed in sequence in the background. While this is happening, change the
`myPrint` function by commenting the first `print` statement and replace it with the second one and
type `reload{demo}` into the REPL. Now, you shall see stars being displayed instead of numbers, even
when you reload the code in the middle of the running code!

Note that any changes made to the `slowFn` function while it is being executed will not be
visible. This is because code reloads replace entire functions, and since the call stack will still
contain a pointer to the old version of `slowFn`, the old version will be used until `slowFn` is
complete.

Thread
-------

The file `thread.bs` illustrates how the threading system in Storm works.

```
?Include:root/demo/thread.bs?
```

The `seq` function calls the function `threadDemo` twice. Since `threadDemo` is declared to be
executed on the `Demo` thread, this causes Storm to post a message to the `Demo` thread, asking for
the function to be executed there. This all happens behind the scenes, and the function call behaves
(almost) as if it was being a regular function.

The function `spawn`, on the other hand calls `threadDemo` using the `spawn` keyword. This causes
execution in `Spawn` to progress until `a.result + b.result` is being evaluated (`a` and `b` are
`Future<Int>` objects here). This time, we can see that the `A` and `B` outputs are
interleaved. However, the execution is still entirely deterministic. As both calls to `threadDemo`
are being executed on the same OS thread, no thread switching is performed until one of the calls
explicitly yields. `dbgSleep` is a version of `sleep` that blocks the entire OS thread while `sleep`
does not block the thread if there is other work to do. `dbgYield` performs an explicit yield. This
is not generally necessary, as any primitive in Storm that could block the current thread ensures
to perform a yield before attempting to block the thread.

See [Threads](md://Storm/Threads) for details on the semantics of the threading system in Storm.


Actor
------

The file `actor.bs` is another version of the example in `thread.bs`, using actors instead of plain
functions.

```
?Include:root/demo/actor.bs?
```

The `actorSeq` function creates two actors on different OS threads and calls `run` on
both of them. As in the previous example, Storm posts a message to the proper thread behind the
scenes, so calling the `run` function appears as a regular function call. As such, the output is
deterministic and should match the output of `seq` in the previous example.

The function `actorSpawn` is similar to `spawn` in the previous example. However, this example is
not deterministic as the actors are associated with different threads that are scheduled
independently by the operating system. Try running this function multiple times and see if you can
see different interleavings!

See [Threads](md://Storm/Threads) for details on the semantics of the threading system in Storm.
