/*
  watchlog -- reads lines from logfile. checks if name has changed
              and if file has new data.
  Copyright (C) 2003  Pedro Zorzenon Neto <pzn@autsens.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* NOTA: I tried to use "Interrupt I/O" described in libc info page
 * with SIGIO and O_ASYNC, but my try was not successful.
 * then I used a pool mode with file attributes, checking if file
 * size has changed or if file has changed by a logrotate
 * (device/inode) */

//#define WATCHLOG_DEBUG
//#define WATCHLOG_TEST

#include "watchlog.h"
#include "common.h"

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <unistd.h>

#define AT_BOF 1 /* begin-of-file */
#define AT_EOF 2 /* end-of-file */

#define WATCHLOG_INIT_CAN_ABORT /* watchlog_init will abort program
				 * if could not find the logfile.
				 * if not defined, watchlog_init will
				 * retry to open file every 10 senconds */

/* private functions */
int stat_withretry (char * fn, struct stat * fs);
FILE * logfile_open (watchlog_t * self, int where);
/* end of private functions */

watchlog_t * watchlog_init (char * logfilename) {
  watchlog_t * self;

  self = malloc (sizeof (watchlog_t));
  assert (self != NULL);

  self->fn = malloc (sizeof(char) * (strlen (logfilename) + 1));
  assert (self->fn != NULL);
  strcpy (self->fn, logfilename);

  self->fs = malloc (sizeof (struct stat));
  assert (self->fs != NULL);

  self->data = malloc (sizeof(char) * LOGLINEMAXSIZE);
  assert(self->data != NULL);
  strcpy(self->data,"");
  
  self->fh = NULL;

#ifdef WATCHLOG_INIT_CAN_ABORT
  if (stat(self->fn, self->fs) != 0) {
    fprintf(stderr,"logfile name: %s\n", self->fn);
    perror("logfile open: stat");
    abort();
  }
#endif

  while (self->fh == NULL)
    {
      stat_withretry (self->fn, self->fs);
      self->fh = logfile_open (self, AT_EOF);
      if (self->fh == NULL)
	{
	  /* open failed */
	  sleep (1);
	}
    }

  self->is_at_eof = 1;

#ifdef WATCHLOG_DEBUG
  fprintf(stderr,"watchlog_init\n");
#endif
  
  return self;
}

void watchlog_destroy (watchlog_t * self) {
  if (self != NULL)
    {
      if (self->fh != NULL)
	{
	  fclose (self->fh);
	}
      if (self->fn != NULL)
	{
	  free (self->fn);
	}
      if (self->data != NULL)
	{
	  free (self->data);
	}
      free(self);
    }
#ifdef WATCHLOG_DEBUG
  fprintf(stderr,"watchlog_destroy\n");
#endif
}

int stat_withretry (char * fn, struct stat * fs) {
  int firsttime = 1;
  int retval = -1;
  int no_need_to_retry = 1;

#ifdef WATCHLOG_DEBUG
  fprintf(stderr,"watchlog_stat_withretry begin\n");
#endif

  while (retval != 0)
    {
      retval=stat(fn, fs);
      if (retval != 0)
	{
	  /* something failed. wait sometime before retry */
#ifdef WATCHLOG_DEBUG
	  perror("stat");
	  fprintf(stderr,"filename %s\n",fn);
	  fprintf(stderr,"watchlog_stat_withretry retry\n");
#endif
	  no_need_to_retry = 0;
	  if (firsttime)
	    {
	      sleep(1);
	    }
	  else
	    {
	      sleep(10);
	    }
      }
      firsttime = 0;
    }
#ifdef WATCHLOG_DEBUG
  fprintf(stderr,"watchlog_stat_withretry end '%d'\n", no_need_to_retry);
#endif

  return no_need_to_retry;
}

FILE * logfile_open (watchlog_t * self, int where) {
  FILE * fh;
#ifdef WATCHLOG_DEBUG
  fprintf(stderr,"watchlog_logfile_open begin\n");
#endif
  /* open log file */
  fh = fopen(self->fn, "r");
  if (fh == NULL)
    {
      /* open failed */
#ifdef WATCHLOG_DEBUG
      fprintf(stderr,"watchlog_logfile_open failed\n");
#endif
      return NULL;
    }
  
  switch (where) {
  case AT_BOF:
    /* nothing to do, fopen already opens at begin-of-file */
    break;
  case AT_EOF:
    /* go to end-of-file */
    if ( fseek (fh, 0, SEEK_END) != 0 )
      {
	/* end-of-file failed */
#ifdef WATCHLOG_DEBUG
      fprintf(stderr,"watchlog_logfile_fseek failed\n");
#endif
	fclose (fh);
	fh = NULL;
      }
    break;
  default:
    assert(0);
  }
  return fh;
}

void watchlog_getline (watchlog_t * self) {
#ifdef WATCHLOG_DEBUG
      fprintf(stderr,"watchlog_getline begin\n");
#endif

  if (self->is_at_eof == 0)
    {
      /* there is data available */

      if (feof (self->fh))
	{ 
	  /* just a protection. should never enter here */
	  assert(0);
	  self->is_at_eof = 1;
	  clearerr (self->fh); /* clear eof flag */
	  watchlog_getline(self);
	  return;
	}

      if (fgets(self->data, LOGLINEMAXSIZE, self->fh) != NULL)
	{
	  self->is_at_eof = 0;
#ifdef WATCHLOG_DEBUG
	  printf("getline: %s",self->data);
#endif
	  return;
	}
      else
	{
	  /* could not read... then should be at eof */
	  self->is_at_eof = 1;
	  if (feof(self->fh))
	    {
	      clearerr (self->fh); /* clear eof flag */
	    }
	  watchlog_getline(self);
	  return;
	}
      return;

    }
  else
    {
      /* need to check if there is data available */

      struct stat * fsn;

      fsn = malloc(sizeof(struct stat));
      assert (fsn != NULL);

      while (1) {

	int need_reopen = 0;
	int has_newdata = 0;

	if (stat_withretry(self->fn, fsn) == 0)
	  {
	    need_reopen = 1;
	  }
	else if (memcmp(fsn, self->fs, sizeof(struct stat)) != 0)
	  {
	    /* something has changed */
	    if ( (fsn->st_dev != self->fs->st_dev) ||
		 (fsn->st_ino != self->fs->st_ino) ||
		 (fsn->st_size < self->fs->st_size) ) /* file is shorter */
	      {
		/* file device or inode has changed */
		need_reopen=1;
	      }
	    else if (fsn->st_size != self->fs->st_size)
	      {
		/* file size has changed */
		has_newdata = 1;
	      }
	  }
	memcpy(self->fs, fsn, sizeof(struct stat));
		
	if (need_reopen == 1)
	  {
	    /* we need to reopen the file */
	    fclose (self->fh);
	    self->fh = NULL;
	    while (self->fh == NULL)
	      {
		stat_withretry (self->fn, fsn);
		self->fh = logfile_open (self, AT_BOF);
		if (self->fh == NULL)
		  {
		    /* open failed */
		    sleep (1);
		  }
	      }
	    
	    fprintf(stderr,"WARNING: lost logfile handler. will reopen "
		    "logfile now -- this occours when file is truncated "
		    "or when file inode changes (maybe it was just "
		    "'logrotate' doing its job).\n");
	    
	    /* file is reopened now and fileposition is at BOF */
	    /* lets assume that the file is new, then we need to parse
	     * data from its begin */
	    has_newdata = 1;
	  }
    
	if (has_newdata == 1)
	  {
	    if (fgets(self->data, LOGLINEMAXSIZE, self->fh) != NULL)
	      {
		self->is_at_eof = 0;
#ifdef WATCHLOG_DEBUG
		fprintf(stderr,"watchlog_getline: %s",self->data);
#endif
		free(fsn);
		return;
	      }
	    else
	      {
		/* could not read... then should be at eof */
		if (feof(self->fh))
		  {
		    clearerr (self->fh); /* clear eof flag */
		  }
	      }
	  }

	sleep (2); /* prevents too much resource usage */
	
      }
    }
}

#ifdef WATCHLOG_TEST
int main() {
  watchlog_t * wl;
  wl = watchlog_init("/tmp/mylog");
  while (1) {
    watchlog_getline(wl);
    printf("gotdata: %s",wl->data);
  }
  return 0;
}
#endif
