/******************************************************************************
*       SOFA, Simulation Open-Framework Architecture, version 1.0 beta 4      *
*                (c) 2006-2009 MGH, INRIA, USTL, UJF, CNRS                    *
*                                                                             *
* This program is free software; you can redistribute it and/or modify it     *
* under the terms of the GNU General Public License as published by the Free  *
* Software Foundation; either version 2 of the License, or (at your option)   *
* any later version.                                                          *
*                                                                             *
* This program is distributed in the hope that it will be useful, but WITHOUT *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or       *
* FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for    *
* more details.                                                               *
*                                                                             *
* You should have received a copy of the GNU General Public License along     *
* with this program; if not, write to the Free Software Foundation, Inc., 51  *
* Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.                   *
*******************************************************************************
*                            SOFA :: Applications                             *
*                                                                             *
* Authors: M. Adam, J. Allard, B. Andre, P-J. Bensoussan, S. Cotin, C. Duriez,*
* H. Delingette, F. Falipou, F. Faure, S. Fonteneau, L. Heigeas, C. Mendoza,  *
* M. Nesme, P. Neumann, J-P. de la Plata Alcade, F. Poyer and F. Roy          *
*                                                                             *
* Contact information: contact@sofa-framework.org                             *
******************************************************************************/
#ifndef SOFA_GUI_QT_STRUCTDATAWIDGET_H
#define SOFA_GUI_QT_STRUCTDATAWIDGET_H

#include <sofa/gui/qt/SimpleDataWidget.h>
#include <sofa/defaulttype/RigidTypes.h>
#include <sofa/defaulttype/LaparoscopicRigidTypes.h>
#include <sofa/component/forcefield/SpringForceField.h>
#include <sofa/component/forcefield/JointSpringForceField.h>
/* #include <../../../projects/vulcain/lib/DiscreteElementModel.h> */
#include <sofa/helper/io/Mesh.h>

namespace sofa
{

namespace gui
{

namespace qt
{

////////////////////////////////////////////////////////////////
/// Generic data structures support
////////////////////////////////////////////////////////////////

template<class T>
class struct_data_trait
{
public:
    typedef T data_type;
    enum { NVAR = 1 };
    static void set( data_type& /*d*/ )
    {
    }
};

template<class T, int I>
class struct_data_trait_var
{
public:
    typedef T data_type;
    typedef T value_type;
    static const char* name() { return NULL; }
    static const value_type* get(const data_type& d) { return &d; }
    static void set( const value_type& v, data_type& d) { d = v; }
    static bool readOnly() { return false; }
    static bool isCheckable() { return false; }
    static bool isChecked(const data_type& /*d*/) { return true; }
    static void setChecked(bool /*b*/, data_type& /*d*/) {}
};

template<class T, int N = struct_data_trait<T>::NVAR >
class struct_data_widget_container
{
public:
    typedef T data_type;
    typedef struct_data_trait<data_type> shelper;
    typedef struct_data_trait_var<data_type,N-1> vhelper;
    typedef typename vhelper::value_type value_type;
    typedef data_widget_container<value_type> Container;
    typedef struct_data_widget_container<data_type,N-1> PrevContainer;
    PrevContainer p;
    Container w;
    QCheckBox* check;
    struct_data_widget_container() : check(NULL) {}
    template<class Dialog, class Slot>
    bool createWidgets(Dialog* dialog, Slot s, QWidget* parent, const data_type& d, bool readOnly)
    {
	if (!p.createWidgets(dialog, s, parent, d, readOnly))
	    return false;
	const char* name = vhelper::name();
	bool checkable = vhelper::isCheckable();
	if (checkable)
	{
	    check = new QCheckBox(parent);
	    if (name && *name)
	    {
		check->setText(QString(name));
	    }
	}
	else
	{
	    if (name && *name && N > 1)
		new QLabel(QString(name),parent);
	}
	if (!w.createWidgets(dialog, s, parent, *vhelper::get(d), readOnly || vhelper::readOnly()))
	    return false;
	if (checkable)
	{
	    bool isChecked = vhelper::isChecked(d);
	    check->setChecked(isChecked);
	    if (readOnly || vhelper::readOnly())
		check->setEnabled(false);
	    else
	    {
		if (!isChecked)
		    w.setReadOnly(true);
		dialog->connect(check, SIGNAL( toggled(bool) ), dialog, s);
	    }
	}
	return true;
    }
    void setReadOnly(bool readOnly)
    {
	p.setReadOnly(readOnly);
	w.setReadOnly(readOnly || vhelper::readOnly() || (check && !check->isOn()));
    }
    void readFromData(const data_type& d)
    {
	p.readFromData(d);
	if (check)
	{
	    bool wasChecked = check->isOn();
	    bool isChecked = vhelper::isChecked(d);
	    if (isChecked != wasChecked)
	    {
		check->setChecked(isChecked);
		if (check->isEnabled())
		    w.setReadOnly(!isChecked);
	    }
	}
	w.readFromData(*vhelper::get(d));
    }
    void readConstantsFromData(const data_type& d)
    {
	p.readConstantsFromData(d);
	if (vhelper::readOnly())
	{
	    if (check)
	    {
		check->setChecked(vhelper::isChecked(d));
	    }
	    w.readFromData(*vhelper::get(d));
	}
    }
    void writeToData(data_type& d)
    {
	p.writeToData(d);
	if (check)
	{
	    bool isChecked = check->isOn();
	    vhelper::setChecked(isChecked, d);
	}
	value_type v = *vhelper::get(d);
	w.writeToData(v);
	vhelper::set(v,d);
	if ( N == struct_data_trait<T>::NVAR )
	{
	    shelper::set(d);
	    readConstantsFromData(d); // reread constant fields
	}
    }
    bool processChange(const QObject* sender)
    {
	if (p.processChange(sender))
	    return true;
	if (check && sender == check)
	{
	    bool isChecked = check->isOn();
	    w.setReadOnly(!isChecked);
	    return true;
	}
	if (w.processChange(sender))
	    return true;
	return false;
    }
};

template<class T>
class struct_data_widget_container< T, 0 >
{
public:
    typedef T data_type;
    typedef struct_data_trait<data_type> shelper;
    struct_data_widget_container() {}
    template<class Dialog, class Slot>
    bool createWidgets(Dialog* /*dialog*/, Slot /*s*/, QWidget* /*parent*/, const data_type& /*d*/, bool /*readOnly*/)
    {
	return true;
    }
    void setReadOnly(bool /*readOnly*/)
    {
    }
    void readFromData(const data_type& /*d*/)
    {
    }
    void readConstantsFromData(const data_type& /*d*/)
    {
    }
    void writeToData(data_type& /*d*/)
    {
    }
    bool processChange(const QObject* /*sender*/)
    {
	return false;
    }
};

template<class T, int I>
class default_struct_data_trait_var
{
public:
    typedef T data_type;
    static const char* name() { return NULL; }
    static const char* shortname() { return NULL; }
    static bool readOnly() { return false; }
    static bool isCheckable() { return false; }
    static bool isChecked(const data_type& /*d*/) { return true; }
    static void setChecked(bool /*b*/, data_type& /*d*/) {}
};

#define STRUCT_DATA_VAR(parent, vid, vname, sname, vtype, var)	\
    class struct_data_trait_var < parent, vid > : public default_struct_data_trait_var < parent, vid > \
    { \
    public: \
        typedef parent data_type; \
	typedef vtype value_type; \
	static const char* name() { return vname; } \
	static const char* shortname() { return sname; } \
	static const value_type* get(const data_type& d) { return &(d.var); } \
	static void set( const value_type& v, data_type& d) { d.var = v; } \
    }

#define STRUCT_DATA_VAR_READONLY(parent, vid, vname, sname, vtype, var) \
    class struct_data_trait_var < parent, vid > : public default_struct_data_trait_var < parent, vid > \
    { \
    public: \
        typedef parent data_type; \
	typedef vtype value_type; \
	static const char* name() { return vname; } \
	static const char* shortname() { return sname; } \
	static bool readOnly() { return true; }	\
	static const value_type* get(const data_type& d) { return &(d.var); } \
	static void set( const value_type& v, data_type& d) { d.var = v; } \
    }

#define STRUCT_DATA_VAR_CHECK(parent, vid, vname, sname, vtype, var, check) \
    class struct_data_trait_var < parent, vid > : public default_struct_data_trait_var < parent, vid > \
    { \
    public: \
        typedef parent data_type; \
	typedef vtype value_type; \
	static const char* name() { return vname; } \
	static const char* shortname() { return sname; } \
	static const value_type* get(const data_type& d) { return &(d.var); } \
	static void set( const value_type& v, data_type& d) { d.var = v; } \
	static bool isCheckable() { return true; } \
	static bool isChecked(const data_type& d) { return d.check; } \
	static void setChecked(bool b, data_type& d) { d.check = b; } \
    }

// A comma can't appear in a macro argument...
#define COMMA ,


////////////////////////////////////////////////////////////////
/// Rigids (as data-structures) support
////////////////////////////////////////////////////////////////

template<int N, class T>
class struct_data_trait < sofa::defaulttype::RigidCoord<N, T> >
{
public:
    typedef sofa::defaulttype::RigidCoord<N, T> data_type;
    enum { NVAR = 2 };
    static void set( data_type& /*d*/ )
    {
    }
};
template<class T> STRUCT_DATA_VAR(sofa::defaulttype::RigidCoord<3 COMMA T>, 0, "Center", "", typename data_type::Vec3, getCenter());
template<class T> STRUCT_DATA_VAR(sofa::defaulttype::RigidCoord<3 COMMA T>, 1, "Orientation", "", typename data_type::Quat, getOrientation());

template<class T> STRUCT_DATA_VAR(sofa::defaulttype::RigidCoord<2 COMMA T>, 0, "Center", "", typename data_type::Vec2, getCenter());
template<class T> STRUCT_DATA_VAR(sofa::defaulttype::RigidCoord<2 COMMA T>, 1, "Orientation", "A", typename data_type::Real, getOrientation());

template<int N, class T>
class data_widget_container < sofa::defaulttype::RigidCoord<N, T> > : public struct_data_widget_container < sofa::defaulttype::RigidCoord<N, T> >
{};

template<int N, class T>
class struct_data_trait < sofa::defaulttype::RigidDeriv<N, T> >
{
public:
    typedef sofa::defaulttype::RigidDeriv<N, T> data_type;
    enum { NVAR = 2 };
    static void set( data_type& /*d*/ )
    {
    }
};
template<class T> STRUCT_DATA_VAR(sofa::defaulttype::RigidDeriv<3 COMMA T>, 0, "VCenter", "d", typename data_type::Vec3, getVCenter());
template<class T> STRUCT_DATA_VAR(sofa::defaulttype::RigidDeriv<3 COMMA T>, 1, "VOrientation", "w", typename data_type::Vec3, getVOrientation());

template<class T> STRUCT_DATA_VAR(sofa::defaulttype::RigidDeriv<2 COMMA T>, 0, "VCenter", "d", typename data_type::Vec2, getVCenter());
template<class T> STRUCT_DATA_VAR(sofa::defaulttype::RigidDeriv<2 COMMA T>, 1, "VOrientation", "dA", typename data_type::Real, getVOrientation());

template<int N, class T>
class data_widget_container < sofa::defaulttype::RigidDeriv<N, T> > : public struct_data_widget_container < sofa::defaulttype::RigidDeriv<N, T> >
{};


template<int N, class T>
class struct_data_trait < sofa::defaulttype::RigidMass<N, T> >
{
public:
    typedef sofa::defaulttype::RigidMass<N, T> data_type;
    enum { NVAR = 4 };
    static void set( data_type& d)
    {
	d.recalc();
    }
};

template<int N, class T> STRUCT_DATA_VAR(sofa::defaulttype::RigidMass<N COMMA T>, 0, "Mass", "Mass", T, mass);
template<int N, class T> STRUCT_DATA_VAR(sofa::defaulttype::RigidMass<N COMMA T>, 1, "Volume", "Vol", T, volume);
template<class T> STRUCT_DATA_VAR(sofa::defaulttype::RigidMass<2 COMMA T>, 2, "Inertia Matrix", "Inertia", T, inertiaMatrix);
template<class T> STRUCT_DATA_VAR(sofa::defaulttype::RigidMass<3 COMMA T>, 2, "Inertia Matrix", "Inertia", typename data_type::Mat3x3, inertiaMatrix);
template<class T> STRUCT_DATA_VAR_READONLY(sofa::defaulttype::RigidMass<2 COMMA T>, 3, "Inertia Mass Matrix", "InertialMass", T, inertiaMassMatrix);
template<class T> STRUCT_DATA_VAR_READONLY(sofa::defaulttype::RigidMass<3 COMMA T>, 3, "Inertia Mass Matrix", "InertialMass", typename data_type::Mat3x3, inertiaMassMatrix);

template<int N, class T>
class data_widget_container < sofa::defaulttype::RigidMass<N, T> > : public struct_data_widget_container < sofa::defaulttype::RigidMass<N, T> >
{};


////////////////////////////////////////////////////////////////
/// LaparoscopicRigid3Types support
////////////////////////////////////////////////////////////////

template<>
class struct_data_trait < sofa::defaulttype::LaparoscopicRigid3Types::Coord >
{
public:
    typedef sofa::defaulttype::LaparoscopicRigid3Types::Coord data_type;
    enum { NVAR = 2 };
    static void set( data_type& /*d*/ )
    {
    }
};
template<> STRUCT_DATA_VAR(sofa::defaulttype::LaparoscopicRigid3Types::Coord, 0, "Translation", "T", data_type::value_type, getTranslation());
template<> STRUCT_DATA_VAR(sofa::defaulttype::LaparoscopicRigid3Types::Coord, 1, "Orientation", "", sofa::defaulttype::Quat, getOrientation());

template<>
class data_widget_container < sofa::defaulttype::LaparoscopicRigid3Types::Coord > : public struct_data_widget_container < sofa::defaulttype::LaparoscopicRigid3Types::Coord >
{};

template<>
class struct_data_trait < sofa::defaulttype::LaparoscopicRigid3Types::Deriv >
{
public:
    typedef sofa::defaulttype::LaparoscopicRigid3Types::Deriv data_type;
    enum { NVAR = 2 };
    static void set( data_type& /*d*/ )
    {
    }
};
template<> STRUCT_DATA_VAR(sofa::defaulttype::LaparoscopicRigid3Types::Deriv, 0, "VTranslation", "dT", data_type::value_type, getVTranslation());
template<> STRUCT_DATA_VAR(sofa::defaulttype::LaparoscopicRigid3Types::Deriv, 1, "VOrientation", "w", sofa::defaulttype::Vector3, getVOrientation());

template<>
class data_widget_container < sofa::defaulttype::LaparoscopicRigid3Types::Deriv > : public struct_data_widget_container < sofa::defaulttype::LaparoscopicRigid3Types::Deriv >
{};

////////////////////////////////////////////////////////////////
/// sofa::component::forcefield::LinearSpring support
////////////////////////////////////////////////////////////////

#define CLASS typename sofa::component::forcefield::LinearSpring< T >

template<class T>
class struct_data_trait < CLASS >
{
public:
    typedef CLASS data_type;
    enum { NVAR = 5 };
    static void set( data_type& /*d*/)
    {
    }
};

template<class T> STRUCT_DATA_VAR(CLASS, 0, "Index 1", "Index 1", int, m1);
template<class T> STRUCT_DATA_VAR(CLASS, 1, "Index 2", "Index 2", int, m2);
template<class T> STRUCT_DATA_VAR(CLASS, 2, "Stiffness", "Ks", T, ks);
template<class T> STRUCT_DATA_VAR(CLASS, 3, "Damping", "Kd", T, kd);
template<class T> STRUCT_DATA_VAR(CLASS, 4, "Rest Length", "L", T, initpos);

template<class T>
class data_widget_container < CLASS > : public struct_data_widget_container < CLASS >
{};

#undef CLASS

////////////////////////////////////////////////////////////////
/// sofa::component::forcefield::JointSpring support
////////////////////////////////////////////////////////////////

#define CLASS typename sofa::component::forcefield::JointSpring< T >

template<class T>
class struct_data_trait < CLASS >
{
public:
    typedef CLASS data_type;
    enum { NVAR = 13 };
    static void set( data_type& /*d*/)
    {
    }
};

template<class T> STRUCT_DATA_VAR(CLASS, 0, "Index 1", "Index 1", int, m1);
template<class T> STRUCT_DATA_VAR(CLASS, 1, "Index 2", "Index 2", int, m2);
template<class T> STRUCT_DATA_VAR(CLASS, 2, "Soft Stiffness Translation", "Soft Ks Trans", typename data_type::Real, softStiffnessTrans);
template<class T> STRUCT_DATA_VAR(CLASS, 3, "Hard Stiffness Translation", "Hard Ks Trans", typename data_type::Real, hardStiffnessTrans);
template<class T> STRUCT_DATA_VAR(CLASS, 4, "Soft Stiffness Rotation", "Soft Ks Rot", typename data_type::Real, softStiffnessRot);
template<class T> STRUCT_DATA_VAR(CLASS, 5, "Hard Stiffness Rotation", "Hard Ks Rot", typename data_type::Real, hardStiffnessRot);
template<class T> STRUCT_DATA_VAR(CLASS, 6, "Damping", "Kd", typename data_type::Real, kd);
template<class T> STRUCT_DATA_VAR(CLASS, 7, "Min Angle X", "Min Angle X", typename data_type::Real, limitAngles[0]);
template<class T> STRUCT_DATA_VAR(CLASS, 8, "Max Angle Y", "Max Angle Y", typename data_type::Real, limitAngles[1]);
template<class T> STRUCT_DATA_VAR(CLASS, 9, "Min Angle Z", "Min Angle Z", typename data_type::Real, limitAngles[2]);
template<class T> STRUCT_DATA_VAR(CLASS, 10, "Max Angle X", "Max Angle X", typename data_type::Real, limitAngles[3]);
template<class T> STRUCT_DATA_VAR(CLASS, 11, "Min Angle Y", "Min Angle Y", typename data_type::Real, limitAngles[4]);
template<class T> STRUCT_DATA_VAR(CLASS, 12, "Max Angle Z", "Max Angle Z", typename data_type::Real, limitAngles[5]);

template<class T>
class data_widget_container < CLASS > : public struct_data_widget_container < CLASS >
{};

#undef CLASS

//
//////////////////////////////////////////////////////////////////
///// sofa::component::DiscreteElement support
//////////////////////////////////////////////////////////////////
//
//#define CLASS typename sofa::component::DiscreteElementModelInternalData< T >
//
//template<class T>
//class struct_data_trait < CLASS >
//{
//public:
//	typedef CLASS data_type;
//	enum { NVAR = 22 };
//	static void set( data_type& /*d*/)
//	{
//	}
//};
//
//template<class T> STRUCT_DATA_VAR(CLASS, 0, "Mass", "Mass", typename data_type::Real, mass);
//template<class T> STRUCT_DATA_VAR(CLASS, 1, "Ray", "Ray", typename data_type::Real, rayon);
//template<class T> STRUCT_DATA_VAR(CLASS, 2, "Translation Stiffness", "Ks Trans", typename data_type::VecCoord, k_tran_crit);
//template<class T> STRUCT_DATA_VAR(CLASS, 3, "Rotation Stiffness", "Ks Rot", typename data_type::VecCoord, k_rot_crit);
//template<class T> STRUCT_DATA_VAR(CLASS, 4, "Alpha", "Alpha", typename data_type::Real, alpha);
//template<class T> STRUCT_DATA_VAR(CLASS, 5, "Is armature", "Arma", bool, arma);
//template<class T> STRUCT_DATA_VAR(CLASS, 6, "Frame Direction", "Frame Dir.", typename data_type::VecCoord, dirarm);
//template<class T> STRUCT_DATA_VAR(CLASS, 7, "Utilisation Loi de Transfert du Moment", "LTM", bool, ltm);
//template<class T> STRUCT_DATA_VAR(CLASS, 8, "Utilisation Limite Plastique", "LTM Plast.", bool, ltm_plast);
//template<class T> STRUCT_DATA_VAR(CLASS, 9, "Ponderation de la raideur de flexion", "Beta", typename data_type::Real, beta);
//template<class T> STRUCT_DATA_VAR(CLASS, 10, "Limite Plastique du Moment", "LIM Plast.", typename data_type::Real, lim_plast);
//template<class T> STRUCT_DATA_VAR(CLASS, 11, "Discrete Element Type", "Type", int, type_ed);
//template<class T> STRUCT_DATA_VAR(CLASS, 12, "Masse Volumique Initiale", "Density", typename data_type::Real, density);
//template<class T> STRUCT_DATA_VAR(CLASS, 13, "Module de Young", "Young", typename data_type::Real, young_modulus);
//template<class T> STRUCT_DATA_VAR(CLASS, 14, "Coefficient de Poisson", "Poisson", typename data_type::Real, poisson_ratio);
//template<class T> STRUCT_DATA_VAR(CLASS, 15, "Limite locale a la traction", "Loc. Trac. Lim.", typename data_type::Real, local_traction_limit);
//template<class T> STRUCT_DATA_VAR(CLASS, 16, "Ecrouissage en traction", "Hard.", typename data_type::Real, hardenning);
//template<class T> STRUCT_DATA_VAR(CLASS, 17, "Deformation Max en %", "Def.", typename data_type::Real, deformation);
//template<class T> STRUCT_DATA_VAR(CLASS, 18, "Cohesion", "Cohe.", typename data_type::Real, cohesion);
//template<class T> STRUCT_DATA_VAR(CLASS, 19, "Angle de Frottement Interne", "Int. Fric. Ang.", typename data_type::Real, internal_friction_angle);
//template<class T> STRUCT_DATA_VAR(CLASS, 20, "Angle de Frottement de Contact", "Cont. Fric. Ang.", typename data_type::Real, contact_friction_angle);
//template<class T> STRUCT_DATA_VAR(CLASS, 21, "Adoucissement", "Adou.", typename data_type::Real, adoucissement);
//
//template<class T>
//class data_widget_container < CLASS > : public struct_data_widget_container < CLASS >
//{};
//
//#undef CLASS

////////////////////////////////////////////////////////////////
/// sofa::helper::io::Mesh::Material support
////////////////////////////////////////////////////////////////

template<>
class struct_data_trait < sofa::helper::io::Mesh::Material >
{
public:
    typedef sofa::helper::io::Mesh::Material data_type;
    enum { NVAR = 6 };
    static void set( data_type& /*d*/)
    {
    }
};

template<> STRUCT_DATA_VAR(sofa::helper::io::Mesh::Material, 0, "Name", "Name", std::string, name);
template<> STRUCT_DATA_VAR_CHECK(sofa::helper::io::Mesh::Material, 1, "Ambient", "Amb", sofa::defaulttype::Vec4f, ambient, useAmbient);
template<> STRUCT_DATA_VAR_CHECK(sofa::helper::io::Mesh::Material, 2, "Diffuse", "Diff", sofa::defaulttype::Vec4f, diffuse, useDiffuse);
template<> STRUCT_DATA_VAR_CHECK(sofa::helper::io::Mesh::Material, 3, "Specular", "Spec", sofa::defaulttype::Vec4f, specular, useSpecular);
template<> STRUCT_DATA_VAR_CHECK(sofa::helper::io::Mesh::Material, 4, "Emissive", "Emm", sofa::defaulttype::Vec4f, emissive, useEmissive);
template<> STRUCT_DATA_VAR_CHECK(sofa::helper::io::Mesh::Material, 5, "Shininess", "Shin", float, shininess, useShininess);

template<>
class data_widget_container < sofa::helper::io::Mesh::Material > : public struct_data_widget_container < sofa::helper::io::Mesh::Material >
{};

} // namespace qt

} // namespace gui

} // namespace sofa


#endif
