# Copyright  2016-2022 Maël Azimi <m.a@moul.re>
#
# Silkaj is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# Silkaj is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with Silkaj. If not, see <https://www.gnu.org/licenses/>.

import duniterpy.api.bma.tx as bma_tx
import pytest
from click.testing import CliRunner

from silkaj.cli import cli
from silkaj.constants import FAILURE_EXIT_STATUS, G1_SYMBOL
from silkaj.money import tools as m_tools
from silkaj.public_key import gen_pubkey_checksum
from silkaj.wot import tools
from tests.patched.test_constants import mock_ud_value
from tests.patched.wot import patched_is_member


# display_amount()
@pytest.mark.parametrize(
    "message, amount, currency_symbol", [("Total", 1000, G1_SYMBOL)]
)
def test_display_amount(message, amount, currency_symbol):
    amount_UD = round(amount / mock_ud_value, 2)
    expected = [
        [
            f"{message} (unit|relative)",
            f"{str(amount / 100)} {currency_symbol} | {str(amount_UD)} UD {currency_symbol}",
        ]
    ]
    tx = []
    m_tools.display_amount(tx, message, amount, mock_ud_value, currency_symbol)
    assert tx == expected


# display_pubkey()
@pytest.mark.parametrize(
    "message, pubkey, uid",
    [
        ("From", "CtM5RZHopnSRAAoWNgTWrUhDEmspcCAxn6fuCEWDWudp", "riri"),
        ("To", "DBM6F5ChMJzpmkUdL5zD9UXKExmZGfQ1AgPDQy4MxSBw", ""),
    ],
)
def test_display_pubkey(message, pubkey, uid, monkeypatch):
    monkeypatch.setattr(tools, "is_member", patched_is_member)

    expected = [[f"{message} (pubkey:checksum)", gen_pubkey_checksum(pubkey)]]
    if uid:
        expected.append([f"{message} (id)", uid])
    tx = []
    m_tools.display_pubkey(tx, message, pubkey)
    assert tx == expected


def test_get_sources(monkeypatch):
    """
    test that get_source() will :
      * only use simple SIG txs
      * only use blockchain sources to compute balance
      * return pending txs in first positions of the sources list
    """

    def patched_tx_sources(self, pubkey):
        return {
            "currency": "g1-test",
            "pubkey": "AhRMHUxMPXSeG7qXZrE6qCdjwK9p2bu5Eqei7xAWVEDK",
            "sources": [
                # this source will be returned in inputlist, and its amount used.
                {
                    "type": "T",
                    "noffset": 2,
                    "identifier": "0310F56D22F4CEF5E41B9D5CACB6E21F224B79D9398D53A4E754866435710242",
                    "amount": 10,
                    "base": 3,
                    "conditions": "SIG(AhRMHUxMPXSeG7qXZrE6qCdjwK9p2bu5Eqei7xAWVEDK)",
                },
                # this source will not be returned (complex unlock condition)
                {
                    "type": "T",
                    "noffset": 3,
                    "identifier": "0D6A29451E64F468C0DB19F70D0D17F65BDCC98F3A16DD55B3755BE124B3DD6C",
                    "amount": 30,
                    "base": 3,
                    "conditions": "(SIG(2VgEZnrGQ5hEgwoNrcXZnD9c8o5jL63LPBmJdvMyFhGe)\
|| (SIG(AhRMHUxMPXSeG7qXZrE6qCdjwK9p2bu5Eqei7xAWVEDK) && CSV(864)))",
                },
            ],
        }

    def patched_tx_pending(self, pubkey):
        return {
            "currency": "g1-test",
            "pubkey": "AhRMHUxMPXSeG7qXZrE6qCdjwK9p2bu5Eqei7xAWVEDK",
            "history": {
                "sending": [],
                "received": [],
                "receiving": [],
                "sent": [],
                "pending": [
                    {
                        "version": 10,
                        "locktime": 0,
                        "blockstamp": "671977-000008B6DE75715D3D83450\
A957CD75F781DA8E3E8E966D42A02F59049209533",
                        "blockstampTime": 1607363853,
                        "issuers": ["6upqFiJ66WV6N3bPc8x8y7rXT3syqKRmwnVyunCtEj7o"],
                        "inputs": [
                            "2739:3:D:6upqFiJ66WV6N3bPc8x8y7rXT3syqKRmwnVyunCtEj7o:664106"
                        ],
                        "outputs": [
                            "60:3:SIG(AhRMHUxMPXSeG7qXZrE6qCdjwK9p2bu5Eqei7xAWVEDK)",
                            "2679:3:SIG(6upqFiJ66WV6N3bPc8x8y7rXT3syqKRmwnVyunCtEj7o)",
                        ],
                        "unlocks": ["0:SIG(0)"],
                        "signatures": [
                            "lrmzr/RkecJBOczlmkp3BNCiXejBzTnHdqmNzxQJ\
yJDIx0UHON4jYkqVKeD77+nrOl8jVtonLt3ZYqd1fhi1Cw=="
                        ],
                        "comment": "DEMAIN DES L_AUBE",
                        "hash": "D5A1A1AAA43FAA242CC2B19763619DA625092BB7FD23397AD362215375A920C8",
                        "time": None,
                        "block_number": None,
                        "received": None,
                    }
                ],
            },
        }

    monkeypatch.setattr(bma_tx, "sources", patched_tx_sources)
    monkeypatch.setattr(bma_tx, "pending", patched_tx_pending)

    listinput, balance = m_tools.get_sources(
        "AhRMHUxMPXSeG7qXZrE6qCdjwK9p2bu5Eqei7xAWVEDK"
    )
    assert len(listinput) == 2
    # test SIG() only source is used
    assert balance == 10000  # 10 in unitbase 3
    assert (
        listinput[0].origin_id
        == "D5A1A1AAA43FAA242CC2B19763619DA625092BB7FD23397AD362215375A920C8"
    )


def test_balance_errors():
    """
    test balance command errors
    """

    # twice the same pubkey
    result = CliRunner().invoke(
        cli,
        [
            "money",
            "balance",
            "BFb5yv8z1fowR6Z8mBXTALy5z7gHfMU976WtXhmRsUMh",
            "BFb5yv8z1fowR6Z8mBXTALy5z7gHfMU976WtXhmRsUMh",
        ],
    )
    pubkeyCk = gen_pubkey_checksum("BFb5yv8z1fowR6Z8mBXTALy5z7gHfMU976WtXhmRsUMh")
    assert f"ERROR: pubkey {pubkeyCk} was specified many times" in result.output
    assert result.exit_code == FAILURE_EXIT_STATUS

    # wrong pubkey
    result = CliRunner().invoke(
        cli,
        [
            "money",
            "balance",
            "B",
        ],
    )
    assert "ERROR: pubkey B has a wrong format" in result.output
    assert result.exit_code == FAILURE_EXIT_STATUS

    # no pubkey
    result = CliRunner().invoke(cli, ["money", "balance"])
    assert "You should specify one or many pubkeys" in result.output
    assert result.exit_code == FAILURE_EXIT_STATUS
