#ifndef _RHEO_BASIS_ON_POINTSET_H
#define _RHEO_BASIS_ON_POINTSET_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

//! Pre-compute a piola transformation polynomial basis
//! on a lattice definied on the reference element.
//! The computation is performed one time for all
//! the first time the reference element appears.
//! Thus, when looping on the mesh for the Lagrange interpolation
//! on a Lagrange nodal basis : the lattice is the set of nodes
//! the evaluation of the fem polynomial basis.
//! The evaluation is performed only one time.

#include "rheolef/basis.h"
#include "rheolef/quadrature.h"
namespace rheolef {

// -----------------------------------------------------------------------
// basis evaluated on lattice of quadrature formulae
// -----------------------------------------------------------------------
template<class T>
class basis_on_pointset {
public:

    typedef typename std::vector<T>::size_type                     size_type;
    typedef typename std::vector<T>::const_iterator                const_iterator;
    typedef typename std::vector<point_basic<T> >::const_iterator  const_iterator_grad;

// allocators:

    basis_on_pointset ();
    basis_on_pointset (const basis_on_pointset<T>&);
    basis_on_pointset (const quadrature<T>& quad, const basis_basic<T>& b);
    basis_on_pointset (const basis_basic<T>& nb,  const basis_basic<T>& b);

// modifiers:

    void set (const quadrature<T>& quad, const basis_basic<T>& b);
    void set (const basis_basic<T>& nb,  const basis_basic<T>& b);

// accessors:

    const basis_basic<T>& get_basis() const { return _b; }
    size_type size (reference_element hat_K) const;
    void restrict_on_side (reference_element tilde_L, const side_information_type& sid);

    void evaluate      (reference_element hat_K, size_type q) const;
    void evaluate_grad (reference_element hat_K, size_type q) const;
    void evaluate      (reference_element hat_K, const point_basic<T>& hat_xq) const;
    void evaluate_grad (reference_element hat_K, const point_basic<T>& hat_xq) const;

    const_iterator      begin() const;
    const_iterator      end() const;
    const_iterator_grad begin_grad() const;
    const_iterator_grad end_grad() const;

    // new interface: random accessors
    const T& value (size_type loc_idof) const;
    const point_basic<T>& grad_value (size_type loc_idof) const;

// data:
    typedef enum {
      quad_mode     = 0,
      nodal_mode    = 1,
      max_mode      = 2
    } mode_type;
protected:
    basis_basic<T>                                      _b;
    mutable mode_type                                   _mode; 
    mutable bool                                        _specific_value;
    const quadrature<T>*                                _p_quad; // when mode: on quadrature pointset
    basis_basic<T>                                      _nb;     // when mode: on nodal basis pointset
    mutable std::array<size_type, reference_element::max_variant>                   _basis_size;
    mutable std::array<std::vector<T>,reference_element::max_variant>               _val;
    mutable std::array<std::vector<point_basic<T> >,reference_element::max_variant> _grad_val;
    mutable std::vector<bool>                           _initialized;
    mutable std::vector<bool>                           _grad_initialized;
    mutable size_type                                   _curr_K_variant;
    mutable size_type                                   _curr_q;
    void _initialize          (reference_element hat_K) const;
    void _grad_initialize     (reference_element hat_K) const;
};

}// namespace rheolef
#endif // _RHEO_BASIS_ON_POINTSET_H
