///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
#include "P1d_numbering.h"
#include "rheolef/geo.h"
namespace rheolef {
using namespace std;

template <class T, class M>
std::string
numbering_P1d<T,M>::name() const
{
  return "P1d";
}
template <class T, class M>
typename numbering_P1d<T,M>::size_type
numbering_P1d<T,M>::degree () const
{
  return 1;
}
template <class T, class M>
void
numbering_P1d<T,M>::set_degree (size_type degree) const
{
    check_macro (degree == 1, "cannot change `"<<name()<<"' degree to " << degree);
}
template <class T, class M>
bool
numbering_P1d<T,M>::is_continuous () const
{
  return false;
}
template <class T, class M>
typename numbering_P1d<T,M>::size_type
numbering_P1d<T,M>::ndof (
        const geo_size& gs, 
        size_type       map_dim) const
{
    switch (map_dim) {
      case 0: return   gs.ownership_by_variant[reference_element::p].size();
      case 1: return 2*gs.ownership_by_variant[reference_element::e].size();
      case 2: return 3*gs.ownership_by_variant[reference_element::t].size()
                   + 4*gs.ownership_by_variant[reference_element::q].size();
      case 3: return 4*gs.ownership_by_variant[reference_element::T].size()
                   + 6*gs.ownership_by_variant[reference_element::P].size()
                   + 8*gs.ownership_by_variant[reference_element::H].size();
      default : {
        error_macro ("unexpected dimension" << map_dim);
        return 0;
      }
    }
}
template <class T, class M>
typename numbering_P1d<T,M>::size_type
numbering_P1d<T,M>::dis_ndof (
        const geo_size& gs, 
        size_type       map_dim) const
{
    switch (map_dim) {
      case 0: return   gs.ownership_by_variant[reference_element::p].dis_size();
      case 1: return 2*gs.ownership_by_variant[reference_element::e].dis_size();
      case 2: return 3*gs.ownership_by_variant[reference_element::t].dis_size()
                   + 4*gs.ownership_by_variant[reference_element::q].dis_size();
      case 3: return 4*gs.ownership_by_variant[reference_element::T].dis_size()
                   + 6*gs.ownership_by_variant[reference_element::P].dis_size()
                   + 8*gs.ownership_by_variant[reference_element::H].dis_size();
      default : {
        error_macro ("unexpected dimension" << map_dim);
        return 0;
      }
    }
}
template <class T, class M>
typename numbering_P1d<T,M>::size_type
numbering_P1d<T,M>::dis_idof (
        const geo_size&       gs, 
	const geo_element&    K, 
	size_type             loc_idof) const
{
  /*
   * WARNING:  mixed mesh with t+q or T+P+H not supported too
   *           or mixtures or e+t, e+t+T etc...
   *           This requires ordered K_idx : all "e" and then all "t",
   *           and then all "q" ... => geo format version 3 ?
   */
  size_type dis_ige = K.dis_ie();
  switch (K.dimension()) {
    case 0: return   dis_ige + loc_idof;
    case 1: return 2*dis_ige + loc_idof;
    case 2: {
        if (K.variant() == reference_element::t) {
	  // triangle
	  size_type first_dis_iqua = gs.ownership_by_variant[reference_element::q].first_index();
	  assert_macro (dis_ige >= first_dis_iqua, "invalid numbering");
	  size_type dis_itri = dis_ige - first_dis_iqua;
	  size_type dis_idof = 3*dis_itri + 4*first_dis_iqua + loc_idof;
	  return dis_idof;
	} else {
	  // quadrangle
	  size_type last_dis_itri = gs.ownership_by_variant[reference_element::t].last_index();
	  assert_macro (dis_ige >= last_dis_itri, "invalid numbering");
	  size_type dis_iqua = dis_ige - last_dis_itri;
	  size_type dis_idof = 3*last_dis_itri + 4*dis_iqua + loc_idof;
	  return dis_idof;
	}
      }
    case 3:
    default : {
        if (K.variant() == reference_element::T) {
	  // tetra
	  size_type first_dis_ipri = gs.ownership_by_variant[reference_element::P].first_index();
	  size_type first_dis_ihex = gs.ownership_by_variant[reference_element::H].first_index();
	  size_type dis_itet = dis_ige - first_dis_ipri - first_dis_ihex;
	  size_type dis_idof = 4*dis_itet + 6*first_dis_ipri + 8*first_dis_ihex + loc_idof;
	  return dis_idof;
	} else if (K.variant() == reference_element::P) {
	  // prism
	  size_type  last_dis_itet = gs.ownership_by_variant[reference_element::T]. last_index();
	  size_type first_dis_ihex = gs.ownership_by_variant[reference_element::H].first_index();
	  size_type dis_ipri = dis_ige - last_dis_itet - first_dis_ihex;
	  size_type dis_idof = 4*last_dis_itet + 6*dis_ipri + 8*first_dis_ihex + loc_idof;
	  return dis_idof;
	} else {
	  // hexa
	  size_type  last_dis_itet = gs.ownership_by_variant[reference_element::T]. last_index();
	  size_type  last_dis_ipri = gs.ownership_by_variant[reference_element::P]. last_index();
	  size_type dis_ihex = dis_ige - last_dis_itet - last_dis_ipri;
	  size_type dis_idof = 4*last_dis_itet + 6*last_dis_ipri + 8*dis_ihex + loc_idof;
	  return dis_idof;
	}
    }
  }
}
template <class T, class M>
void
numbering_P1d<T,M>::dis_idof (
        const geo_size&       gs, 
	const geo_element&    K, 
	std::vector<size_type>&  dis_idof1) const
{
  dis_idof1.resize (K.size());
  for (size_type loc_idof = 0; loc_idof < K.size(); loc_idof++) {
    dis_idof1 [loc_idof] = dis_idof (gs, K, loc_idof);
  }
}
// -------------------------------------------------------------
// permut to/from ios dof numbering, for distributed environment
// -------------------------------------------------------------
template <class T, class M>
static 
void
P1d_set_ios_permutations (
    const geo_basic<T,M>& omega,
    array<distributor::size_type,M>&   idof2ios_dis_idof,
    array<distributor::size_type,M>&   ios_idof2dis_idof)
{
}
#ifdef _RHEOLEF_HAVE_MPI
template <class T>
static 
void
P1d_set_ios_permutations (
    const geo_basic<T,distributed>& omega,
    array<distributor::size_type,distributed>&   idof2ios_dis_idof,
    array<distributor::size_type,distributed>&   ios_idof2dis_idof)
{
  typedef distributor::size_type size_type;
  size_type map_dim = omega.map_dimension();
  size_type ndof, ios_ndof, dis_ndof, nvert_per_elt;
  switch (map_dim) {
	case 0: {
	  ndof     =     omega.    sizes().ownership_by_variant[reference_element::p].size();
	  ios_ndof =     omega.ios_sizes().ownership_by_variant[reference_element::p].size();
	  dis_ndof =     omega.    sizes().ownership_by_variant[reference_element::p].dis_size(); 
	  break;
        }
	case 1: {
	      ndof =   2*omega.    sizes().ownership_by_variant[reference_element::e].size();
	  ios_ndof =   2*omega.ios_sizes().ownership_by_variant[reference_element::e].size();
	  dis_ndof =   2*omega.    sizes().ownership_by_variant[reference_element::e].dis_size(); 
	  break;
        }
	case 2: {
	      ndof =   3*omega.    sizes().ownership_by_variant[reference_element::t].size()
	             + 4*omega.    sizes().ownership_by_variant[reference_element::q].size(); 
	  ios_ndof =   3*omega.ios_sizes().ownership_by_variant[reference_element::t].size() 
	             + 4*omega.ios_sizes().ownership_by_variant[reference_element::q].size();
	  dis_ndof =   3*omega.    sizes().ownership_by_variant[reference_element::t].dis_size() 
	             + 4*omega.    sizes().ownership_by_variant[reference_element::q].dis_size();
	  break;
        }
	case 3:
	default: {
	      ndof =   4*omega.    sizes().ownership_by_variant[reference_element::T].size()
	             + 6*omega.    sizes().ownership_by_variant[reference_element::P].size()
	             + 8*omega.    sizes().ownership_by_variant[reference_element::H].size();
	  ios_ndof =   4*omega.ios_sizes().ownership_by_variant[reference_element::T].size()
	             + 6*omega.ios_sizes().ownership_by_variant[reference_element::P].size()
	             + 8*omega.ios_sizes().ownership_by_variant[reference_element::H].size();
	  dis_ndof =   4*omega.    sizes().ownership_by_variant[reference_element::T].dis_size()
	             + 6*omega.    sizes().ownership_by_variant[reference_element::P].dis_size()
	             + 8*omega.    sizes().ownership_by_variant[reference_element::H].dis_size();
	  break;
        }
  }
  distributor     ownership (dis_ndof, omega.comm(),     ndof);
  distributor ios_ownership (dis_ndof, omega.comm(), ios_ndof);
  idof2ios_dis_idof.resize  (ownership);
  ios_idof2dis_idof.resize  (ios_ownership);
  switch (map_dim) {
	case 0: 
	case 1: {
	  // non-mixed meshes:
	  size_type loc_ndof_per_elt = map_dim+1;
	  // part 1: set idof2ios_dis_idof[]
          for (size_type ige = 0, nge = omega.geo_element_ownership(map_dim).size(); ige < nge; ige++) {
	    size_type ios_dis_ige = omega.ige2ios_dis_ige (map_dim, ige);
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_elt; loc_idof++) {
	      size_type idof         = loc_ndof_per_elt*ige         + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_elt*ios_dis_ige + loc_idof;
	      idof2ios_dis_idof [idof] = ios_dis_idof;
            }
          }
	  // part 2: set ios_idof2dis_idof[]
          for (size_type ios_ige = 0, ios_nge = omega.geo_element_ios_ownership(map_dim).size(); ios_ige < ios_nge; ios_ige++) {
            size_type dis_ige = omega.ios_ige2dis_ige (map_dim, ios_ige);
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_elt; loc_idof++) {
	      size_type ios_idof = loc_ndof_per_elt*ios_ige + loc_idof;
	      size_type dis_idof = loc_ndof_per_elt*dis_ige + loc_idof;
              ios_idof2dis_idof [ios_idof] = dis_idof;
            }
          }
	  break;
	}
	case 2: {
	  // possibly (t,q) mixed meshes:
	  // part 1: set idof2ios_dis_idof[] for (t,q)
	  // part 1.a: triangles
          size_type loc_ndof_per_tri = 3;
          size_type     ntri = omega.sizes().ownership_by_variant[reference_element::t].size();
          size_type dis_ntri = omega.sizes().ownership_by_variant[reference_element::t].dis_size();
          for (size_type itri = 0; itri < ntri; itri++) {
	    size_type ige = itri;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    size_type ios_dis_itri = ios_dis_ige;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_tri; loc_idof++) {
	      size_type idof         = loc_ndof_per_tri*itri + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tri*ios_dis_itri + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  // part 1.b: quadrangle
          size_type loc_ndof_per_qua = 4;
          size_type nqua = omega.sizes().ownership_by_variant[reference_element::q].size();
          for (size_type iqua = 0; iqua < nqua; iqua++) {
	    size_type ige = ntri + iqua;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    assert_macro (ios_dis_ige >= dis_ntri, "unexpected numbering for element of variant `q'");
	    size_type ios_dis_iqua = ios_dis_ige - dis_ntri;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_qua; loc_idof++) {
	      size_type idof = loc_ndof_per_tri*ntri + loc_ndof_per_qua*iqua + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tri*dis_ntri + loc_ndof_per_qua*ios_dis_iqua + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  // part 2: set ios_idof2dis_idof[] for (t,q)
	  // part 2.a: triangles
          size_type ios_ntri = omega.ios_sizes().ownership_by_variant[reference_element::t].size();
          for (size_type ios_itri = 0; ios_itri < ios_ntri; ios_itri++) {
            size_type ios_ige = ios_itri;
            size_type dis_ige = omega.ios_ige2dis_ige (map_dim, ios_ige);
            size_type owner = omega.sizes().ownership_by_dimension[map_dim].find_owner(dis_ige);
            size_type first_dis_iqua = omega.sizes().ownership_by_variant[reference_element::q].first_index(owner);
	    assert_macro (dis_ige >= first_dis_iqua, "unexpected numbering for element of variant `t'");
            size_type dis_itri = dis_ige - first_dis_iqua;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_tri; loc_idof++) {
	      size_type ios_idof = loc_ndof_per_tri*ios_itri + loc_idof;
	      size_type dis_idof = loc_ndof_per_tri*dis_itri + loc_ndof_per_qua*first_dis_iqua + loc_idof;
	      assert_macro (ios_idof < ios_idof2dis_idof.size(), "unexpected ios_idof");
	      assert_macro (dis_idof < dis_ndof, "unexpected dis_idof");
              ios_idof2dis_idof [ios_idof] = dis_idof;
	    }
	  }
	  // part 2.b: quadrangle
          size_type ios_nqua = omega.ios_sizes().ownership_by_variant[reference_element::q].size();
          for (size_type ios_iqua = 0; ios_iqua < ios_nqua; ios_iqua++) {
            size_type ios_ige = ios_ntri + ios_iqua;
            size_type dis_ige = omega.ios_ige2dis_ige (map_dim, ios_ige);
            size_type owner = omega.sizes().ownership_by_dimension[map_dim].find_owner(dis_ige);
            size_type last_dis_itri = omega.sizes().ownership_by_variant[reference_element::t].last_index(owner);
	    assert_macro (dis_ige >= last_dis_itri, "unexpected numbering for element of variant `q'");
            size_type dis_iqua = dis_ige - last_dis_itri;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_qua; loc_idof++) {
	      size_type ios_idof = loc_ndof_per_tri*ios_ntri + loc_ndof_per_qua*ios_iqua + loc_idof;
	      size_type dis_idof = loc_ndof_per_tri*last_dis_itri + loc_ndof_per_qua*dis_iqua + loc_idof;
	      assert_macro (ios_idof < ios_idof2dis_idof.size(), "ios_idof="<<ios_idof<<" out of range [0:" << ios_idof2dis_idof.size()<<"[");
	      assert_macro (dis_idof < dis_ndof, "unexpected dis_idof");
              ios_idof2dis_idof [ios_idof] = dis_idof;
	    }
	  }
	  break;
	}
	case 3:
        default: {
	  // possibly (T,P,H) mixed meshes:
	  // part 1: set idof2ios_dis_idof[] for (T,P,H)
	  // part 1.a: tetra
          size_type loc_ndof_per_tet = 4;
          size_type     ntet = omega.sizes().ownership_by_variant[reference_element::T].size();
          size_type dis_ntet = omega.sizes().ownership_by_variant[reference_element::T].dis_size();
          for (size_type itet = 0; itet < ntet; itet++) {
	    size_type ige = itet;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    size_type ios_dis_itet = ios_dis_ige;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_tet; loc_idof++) {
	      size_type idof         = loc_ndof_per_tet*itet + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tet*ios_dis_itet + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  // part 1.b: prism
          size_type loc_ndof_per_pri = 6;
          size_type     npri = omega.sizes().ownership_by_variant[reference_element::P].size();
          size_type dis_npri = omega.sizes().ownership_by_variant[reference_element::P].dis_size();
          for (size_type ipri = 0; ipri < npri; ipri++) {
	    size_type ige = ntet + ipri;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    assert_macro (ios_dis_ige >= dis_ntet, "unexpected numbering for element of variant `P'");
	    size_type ios_dis_ipri = ios_dis_ige - dis_ntet;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_pri; loc_idof++) {
	      size_type idof = loc_ndof_per_tet*ntet + loc_ndof_per_pri*ipri + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tet*dis_ntet + loc_ndof_per_pri*ios_dis_ipri + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  // part 1.c: hexa
          size_type loc_ndof_per_hex = 8;
          size_type nhex = omega.sizes().ownership_by_variant[reference_element::H].size();
          for (size_type ihex = 0; ihex < nhex; ihex++) {
	    size_type ige = ntet + npri + ihex;
	    size_type ios_dis_ige  = omega.ige2ios_dis_ige (map_dim, ige);
	    assert_macro (ios_dis_ige >= dis_ntet + dis_npri, "unexpected numbering for element of variant `H'");
	    size_type ios_dis_ihex = ios_dis_ige - dis_ntet - dis_npri;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_hex; loc_idof++) {
	      size_type idof = loc_ndof_per_tet*ntet + loc_ndof_per_pri*npri + loc_ndof_per_hex*ihex + loc_idof;
	      size_type ios_dis_idof = loc_ndof_per_tet*dis_ntet + loc_ndof_per_pri*dis_npri + loc_ndof_per_hex*ios_dis_ihex + loc_idof;
	      assert_macro (idof < idof2ios_dis_idof.size(), "unexpected idof");
	      assert_macro (ios_dis_idof < dis_ndof, "unexpected ios_dis_idof");
	      idof2ios_dis_idof [idof] = ios_dis_idof;
	    }
	  }
	  // part 2: set ios_idof2dis_idof[] for (T,P,H)
	  // part 2.a: tetra
          size_type ios_ntet = omega.ios_sizes().ownership_by_variant[reference_element::T].size();
          for (size_type ios_itet = 0; ios_itet < ios_ntet; ios_itet++) {
            size_type ios_ige = ios_itet;
            size_type dis_ige = omega.ios_ige2dis_ige (map_dim, ios_ige);
            size_type owner = omega.sizes().ownership_by_dimension[map_dim].find_owner(dis_ige);
            size_type first_dis_ipri = omega.sizes().ownership_by_variant[reference_element::P].first_index(owner);
            size_type first_dis_ihex = omega.sizes().ownership_by_variant[reference_element::H].first_index(owner);
	    assert_macro (dis_ige >= first_dis_ipri + first_dis_ihex, "unexpected numbering for element of variant `T'");
            size_type dis_itet = dis_ige - first_dis_ipri - first_dis_ihex;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_tet; loc_idof++) {
	      size_type ios_idof = loc_ndof_per_tet*ios_itet + loc_idof;
	      size_type dis_idof = loc_ndof_per_tet*dis_itet + loc_ndof_per_pri*first_dis_ipri + loc_ndof_per_hex*first_dis_ihex + loc_idof;
	      assert_macro (ios_idof < ios_idof2dis_idof.size(), "unexpected ios_idof");
	      assert_macro (dis_idof < dis_ndof, "unexpected dis_idof");
              ios_idof2dis_idof [ios_idof] = dis_idof;
	    }
	  }
	  // part 2.b: prism
          size_type ios_npri = omega.ios_sizes().ownership_by_variant[reference_element::P].size();
          for (size_type ios_ipri = 0; ios_ipri < ios_npri; ios_ipri++) {
            size_type ios_ige = ios_ntet + ios_ipri;
            size_type dis_ige = omega.ios_ige2dis_ige (map_dim, ios_ige);
            size_type owner = omega.sizes().ownership_by_dimension[map_dim].find_owner(dis_ige);
            size_type  last_dis_itet = omega.sizes().ownership_by_variant[reference_element::T].last_index(owner);
            size_type first_dis_ihex = omega.sizes().ownership_by_variant[reference_element::H].first_index(owner);
	    assert_macro (dis_ige >= last_dis_itet + first_dis_ihex, "unexpected numbering for element of variant `P'");
            size_type dis_ipri = dis_ige - last_dis_itet - first_dis_ihex;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_pri; loc_idof++) {
	      size_type ios_idof = loc_ndof_per_tet*ios_ntet + loc_ndof_per_pri*ios_ipri + loc_idof;
	      size_type dis_idof = loc_ndof_per_tet*last_dis_itet + loc_ndof_per_pri*dis_ipri + loc_ndof_per_hex*first_dis_ihex + loc_idof;
	      assert_macro (ios_idof < ios_idof2dis_idof.size(), "ios_idof="<<ios_idof<<" out of range [0:" << ios_idof2dis_idof.size()<<"[");
	      assert_macro (dis_idof < dis_ndof, "unexpected dis_idof");
              ios_idof2dis_idof [ios_idof] = dis_idof;
	    }
	  }
	  // part 2.c: hexa
          size_type ios_nhex = omega.ios_sizes().ownership_by_variant[reference_element::H].size();
          for (size_type ios_ihex = 0; ios_ihex < ios_nhex; ios_ihex++) {
            size_type ios_ige = ios_ntet + ios_npri + ios_ihex;
            size_type dis_ige = omega.ios_ige2dis_ige (map_dim, ios_ige);
            size_type owner = omega.sizes().ownership_by_dimension[map_dim].find_owner(dis_ige);
            size_type  last_dis_itet = omega.sizes().ownership_by_variant[reference_element::T].last_index(owner);
            size_type  last_dis_ipri = omega.sizes().ownership_by_variant[reference_element::P].last_index(owner);
	    assert_macro (dis_ige >= last_dis_itet + last_dis_ipri, "unexpected numbering for element of variant `H'");
            size_type dis_ihex = dis_ige - last_dis_itet - last_dis_ipri;
	    for (size_type loc_idof = 0; loc_idof < loc_ndof_per_hex; loc_idof++) {
	      size_type ios_idof = loc_ndof_per_tet*ios_ntet + loc_ndof_per_pri*ios_npri + loc_ndof_per_hex*ios_ihex + loc_idof;
	      size_type dis_idof = loc_ndof_per_tet*last_dis_itet + loc_ndof_per_pri*last_dis_ipri + loc_ndof_per_hex*dis_ihex + loc_idof;
	      assert_macro (ios_idof < ios_idof2dis_idof.size(), "ios_idof="<<ios_idof<<" out of range [0:" << ios_idof2dis_idof.size()<<"[");
	      assert_macro (dis_idof < dis_ndof, "unexpected dis_idof");
              ios_idof2dis_idof [ios_idof] = dis_idof;
	    }
	  }
	  break;
	}
  }
}
#endif // _RHEOLEF_HAVE_MPI
template <class T, class M>
void
numbering_P1d<T,M>::set_ios_permutations (
    const geo_basic<T,M>& omega,
    array<size_type,M>&   idof2ios_dis_idof,
    array<size_type,M>&   ios_idof2dis_idof) const
{
    P1d_set_ios_permutations (omega, idof2ios_dis_idof, ios_idof2dis_idof);
}
// ----------------------------------------------------------------------------
// instanciation in library
// ----------------------------------------------------------------------------

template class numbering_P1d<Float,sequential>;

#ifdef _RHEOLEF_HAVE_MPI
template class numbering_P1d<Float,distributed>;
#endif // _RHEOLEF_HAVE_MPI

} // namespace rheolef
