# Convert an irace.Rdata file into input format supported by the parameter
# importance analysis tool PyImp
# (https://github.com/automl/ParameterImportance).
#
# PyImp is developed by the AutoML-Freiburgh group (https://github.com/automl)

single_quote <- function(s) paste0("'",s,"'")
double_quote <- function(s) paste0('"',s,'"')

.irace2pyimp_header <- 
'#------------------------------------------------------------------------------
# irace2pyimp converts an irace.Rdata file into input format supported by the
# parameter importance analysis tool PyImp (https://github.com/automl/ParameterImportance)
#
# irace2pyimp was developed by Nguyen Dang <nttd@st-andrews.ac.uk>
#
# PyImp is developed by the AutoML-Freiburg-Hannover group (https://www.automl.org/)
#
# To see usage: irace2pyimp --help
# See examples in `$IRACE_HOME/inst/examples/irace2pyimp/acotsp/run.sh`
#          and in `$IRACE_HOME/inst/examples/irace2pyimp/002-TemplateDesign/run.sh`
#
# irace2pyimp can also be called as an R function instead.  To see usage of the
# function, run in R console: 
#    > library(irace)
#    > ?irace2pyimp 
#
# The generated files include:
#  * `params.pcs` : a text file containing the parameter space definition.
#  * `runhistory.json` : a JSON file containing the list of algorithm configurations
#    evaluated during the tuning and the performance data obtained.
#  * `traj_aclib2.json` : a JSON file containing the best configurations after
#     each iteration of irace.
#  * `scenario.txt` : a text file containing the definition of the tuning scenario.
#  * `instances.txt` : a text file containing the list of instances.
#  * `features.csv` : a text file containing instance features.
#------------------------------------------------------------------------------
\n'

# list of command line arguments
.irace2pyimp_args <- read.table(header=TRUE, stringsAsFactors = FALSE, row.names = "name", text='
name                      type  short       long                           default       description
.help                      x    "-h"        "--help"                       NA            "Show this help." 
normalise                  s    "-n"        "--normalise"                 "none"   	 "Normalise the cost metric values into the range of [0,1] before converting to PyImp format. Values: {none(default), instance, feature}, which correspond to no-normalisation, normalisation based on instance, and normalisation based on features, respectively."
outDir                     s    "-o"        "--out-dir"                   "./pyimp-input" "Directory where all generated data are stored. Default: ./pyimp-input"
instanceFeatureFile        s    "-f"        "--instance-feature-file"      NA            " A .csv file containing instance features (one line per instance, sorted in the same order as the list of instances input to irace). The first line contains feature names. By default, instance index is used as a feature."
filterConditions           s    "-c" "--filter-conditions"         "NA"               "Only extract configurations that satisfies the given conditions. The conditions are in R expression format. Default: no filter"
defaultConfigurationID  i    "-i"   "--default-configuration-index" 1                "Index of default configuration (starting from 1), used by ablation analysis. Default value: 1"
iraceRdataFile             s    "-r"          "--irace-data-file"  NA                     "The log .Rdata file generated by irace"
ignoreUnsupported            b    "-ignore"          "--ignore-unsupported"          0          "Forbidden configurations and repairConfiguration are not supported by the script. Set this flag to 1 to ignore them and proceed with your own risk. This may cause some unwanted behaviours, e.g., forbidden configurations may appear in ablation analysis\'s path. Values: {0,1}"                
')

load_irace_rdata <- function(filename)
{    
  load(filename)
  
  if ('iraceResults' %!in% ls()) {
    irace.error(filename, " is not a valid .Rdata file generated by irace")
  }
  
  # Some fields are not present in the old version of irace, so we assign them as new version's default values
  if ('capping' %!in% names(iraceResults$scenario))
    iraceResults$scenario$capping <- FALSE
  
  return(iraceResults)
}

filter_data <- function(rdata, conditions, defaultConfigurationID)
{
  # - remove all configurations not satistfying conditions in:
  #     + rdata$allConfigurations
  #     + rdata$iterationElites
  #     + rdata$experiments & rdata$experimentLog
  # - also re-assign default configuration if necessary
     
  conditions <- parse(text=conditions)

  # update allConfigurations
  allConfigurations <- rdata$allConfigurations
  allConfigurations <- allConfigurations[eval(conditions, allConfigurations), , drop = FALSE]
  allConfigurations <- allConfigurations[order(allConfigurations$.ID.), , drop = FALSE]

  # update defaultConfigurationID
  if (defaultConfigurationID %!in% allConfigurations$.ID.) {
    irace.warning("default configuration '", defaultConfigurationID,
        "' does not satisfy condition '", as.character(conditions),
        "', thus setting configuration ", allConfigurations$.ID.[1],
        " as default configuration instead.") 
    defaultConfigurationID <- allConfigurations$.ID.[1]
  }
  
  # update iterationElites
  removed <- setdiff(rdata$iterationElites, allConfigurations$.ID.)
  if (length(removed) > 0)
    irace.warning("elite configuration(s) ", paste0(removed, collapse=", "),
        " did not satisfy condition ", as.character(conditions),
        ' and were eliminated')

  iterationElites <- setdiff(rdata$iterationElites, removed)
    
  # update experiments
  experiments <- rdata$experiments[, allConfigurations$.ID., drop = FALSE]

  # update experimentLog
  experimentLog <- rdata$experimentLog[rdata$experimentLog[,'configuration'] %in% allConfigurations$.ID., ,drop = FALSE]
     
  rdata$allConfigurations <- allConfigurations
  rdata$experiments <- experiments
  rdata$experimentLog <- experimentLog
  rdata$iterationElites <- iterationElites
  rdata$defaultConfigurationID <- defaultConfigurationID
  return(rdata)
}

generate_feature_file <- function(file = "features.csv", instances, instanceFeatureFile)
{
  # Generate features.csv
  cat(sep="", "Generating feature file '", file, "' ...\n")
      
  #--- features.csv -----
  tFeatures <- data.frame(instance=instances)
  
  # if no instance features are provided, instance index will be the only
  # feature.
  if (is.null.or.na(instanceFeatureFile)) {
    tFeatures$id <- 1:length(instances)
    
  } else {# otherwise, add features
    if (!file.exists(instanceFeatureFile)) {
      irace.error("Instance feature file '", instanceFeatureFile,"' does not exists.")
    }
    features <- read.table(instanceFeatureFile, header=TRUE)
    if (nrow(features) != length(instances)) {
      irace.error("The number of instances (", nrow(features), ") in '", instanceFeatureFile,
                  "' does not match the instance list given to irace (", length(instances),")")
    }
    tFeatures <- cbind(tFeatures, features)
  }
  
  # write to features.csv
  write.csv(tFeatures,file = file, row.names=FALSE, quote=FALSE)
  return (tFeatures)
}

remove_fixed_parameters <- function(rdata)
{
  # remove fixed parameters, as we don't need them in the analyses
  parameters <- rdata$parameters
      
  # update parameters
  fixedParams <- names(which(parameters$isFixed))
  if (length(fixedParams) > 0) {
    cat(sep="", "Removing fixed parameters: ", paste0(fixedParams, collapse=", "), " ...\n")
    varParams <- names(which(!parameters$isFixed))
    # Go over 'names','types','switches','hierarchy', etc and keep only
    # varParams (except for elements that contain a single value as
    # $nbParameters)
    parameters <- lapply(parameters, function(x) {
      if (length(x) == 1) return(x)
      else return(x[varParams])
    })
    parameters$nbParameters <- parameters$nbParameters - length(fixedParams)
    parameters <- rdata$parameters <- parameters
    rdata$allConfigurations <- rdata$allConfigurations[, varParams]
  }
  return (rdata)
}

generate_pcs_file <- function(file, parameters, defaultConfiguration)
{
  #---- generate param file in SMAC's format ------
  cat(sep="", "Generating parameter definition file '", file, "' ...\n")
    
  # param types, ranges, and default value
  paramlines <- c()
  for (param in parameters$names) {
    type <- parameters$types[[param]]
    domain <- parameters$domain[[param]]
    isLogTransform <- FALSE
    if ('transform' %in% names(parameters) && parameters$transform[[param]]=='log')
      isLogTransform <- TRUE
    
    val <- defaultConfiguration[[param]]
    if (is.na(val)) val <- domain[1]
    
    if (type %in% c('r','i')) {
      line <- paste0(param, ' [', domain[1], ',',domain[2],'] [', val, ']',
                  if (type == 'i') 'i' else '',
                  if (isLogTransform) 'l' else '')
    } else {
      line <- paste0(param, ' {', paste(domain, collapse = ','), '} [', val, ']')
    }
    paramlines <- c(paramlines, line)
  }
  
  # param conditions
  conditionlines <- c()
  for (param in parameters$names) {
    # FIXME: This will not work if we byte-compile the conditions
    irace.assert(!is.bytecode(parameters$conditions[[param]]))
    conditions <- as.character(parameters$conditions[[param]])
    # Convert to SMAC's param format.
    if (conditions != "TRUE" && conditions != "FALSE") {
      conditions <- gsub("%in%", "in", conditions) # remove %
      conditions <- gsub("\"","", conditions) # remove double quote
      conditions <- gsub("\'","", conditions) # remove single quote

      # replace c() by {}
      conditions <- gsub("c[[:space:]]*\\(([^()]+)\\)", "{\\1}", conditions)
      
      # remove "(" and ")" at the beginning and end of the condition
      conditions <- trimws(conditions)
      conditionlines <- c(conditionlines, paste0(param," | ", conditions))
    }
  }
  lines <- c(paramlines, '\n', '#Conditions:', conditionlines)
  # FIXME, TODO: forbidden conditions?
  # nguyen: SMAC's parameter space only allows particular forbidden constraints in the format of {parameter_name_1=value_1, ..., parameter_name_N=value_N}, while irace allows more general forbidden constraints. I'm not sure how to deal with it at the moment...
  
  writeLines(lines, file)
}

process_experiments <- function(rdata, normalise, tFeatures)
{
  #---- reformat rdata$experiments and add extra information into it ---#
  cat(sep="", "Preprocessing experiment data...\n")
  experiments <- rdata$experiments
  
  # instance_id and seed
  tInstanceSeeds <- rdata$state$.irace$instancesList
  colnames(tInstanceSeeds)[colnames(tInstanceSeeds)=='instance'] <- 'instance_id'
  tInstanceSeeds <- cbind(instance_seed_id = 1:nrow(tInstanceSeeds), tInstanceSeeds)
  
  # add instance names into tInstanceSeeds
  instances <- rdata$scenario$instances
  tInstances <- data.frame(instance_id = 1:length(instances), instance=instances)
  tInstanceSeeds <- merge(tInstanceSeeds, tInstances, by = c('instance_id'))
  
  # melt experiments
  experiments <- cbind(instance_seed_id = 1:nrow(experiments), experiments)
  experiments <- na.omit(reshape(as.data.frame(experiments), direction='long',
                                 idvar='instance_seed_id',
                                 times=colnames(experiments)[2:ncol(experiments)],
                                 timevar='candidate_id',
                                 varying=list(colnames(experiments)[2:ncol(experiments)]),
                                 v.names='cost'))
  rownames(experiments) <- NULL
  
  # add instance names and instance_seed_id into experiments
  experiments <- merge(experiments, tInstanceSeeds, by='instance_seed_id')
  
  # normalise cost values if necessary
  if (normalise %!in% c('none','instance','feature'))
    irace.error("Invalid normalisation method '", normalise,"'")

  if (normalise != 'none' && rdata$scenario$capping) {
    irace.warning("normalisation for capped data is not tested and can be buggy.")
  }
  if (normalise != 'none') {
    t1 <- experiments # experiments table with minCost and maxCost for normalisation
    
    # get minCost and maxCost over each instance
    if (normalise == 'instance'){
      tMinCost <- aggregate(cost~instance_id, t1, FUN=min)
      colnames(tMinCost)[2] <- 'minCost'
      tMaxCost <- aggregate(cost~instance_id, t1, FUN=max)
      colnames(tMaxCost)[2] <- 'maxCost'
      t1 <- merge(t1, tMinCost, by='instance_id')
      t1 <- merge(t1, tMaxCost, by='instance_id')
    } else { # get minCost and maxCost over each feature vector (i.e.,  over many instances with the same feature values)
      t2 <- merge(t1, tFeatures, by='instance')
      featureNames <- colnames(tFeatures)[-c(1)]
      featureVals <- as.list(t2[, featureNames, drop=FALSE])
      tMinCost <- aggregate(t2$cost, by=featureVals, min)
      colnames(tMinCost)[colnames(tMinCost)=='x'] <- 'minCost'
      tMaxCost <- aggregate(t2$cost, by=featureVals, max)
      colnames(tMaxCost)[colnames(tMaxCost)=='x'] <- 'maxCost'
      t2 <- merge(t2, tMinCost, by=featureNames)
      t2 <- merge(t2, tMaxCost, by=featureNames)
      t1 <- t2[, names(t1) %!in% featureNames]
    }
    
    # calculate normalised cost
    # TODO: allow user-defined normalisation method
    t1$cost <- (t1$cost - t1$minCost) / (t1$maxCost - t1$minCost)
    
    # remove rows with normalisedCost == Inf
    t1 <- t1[is.finite(t1$cost), ]
    experiments <- t1[, names(t1) %!in% c('minCost','maxCost')]
  }
  
  # add extra information on iteration index & bound (when irace's capping is enabled)
  tLog <- data.frame(rdata$experimentLog)
  colnames(tLog)[colnames(tLog)=='instance'] <- 'instance_seed_id'
  colnames(tLog)[colnames(tLog)=='configuration'] <- 'candidate_id'
  tLog$candidate_id <- as.integer(tLog$candidate_id)
  log_columns <- c('instance_seed_id','candidate_id','iteration')
  if (('capping' %in% names(rdata$scenario)) && rdata$scenario$capping) {
    log_columns <- c(log_columns, 'time', 'bound')
  }
  experiments <- merge(experiments, tLog[, log_columns],
                       by=c('instance_seed_id','candidate_id'))
  return(experiments)
}

generate_runhistory <- function(file = "runhistory.json", rdata, experiments)
{
  #------- generate runhistory.json ------#
  cat(sep="", "Generating runhistory JSON file '", file, "' ...\n")
  
  # TODO: If we ever use the json R package for something else, we could use it
  # here as well, however, in runhistory.json, values are written with/without
  # quotes depending on data types. Therefore, we need to make sure they are
  # written correctly.
  
  # Configurations: The first columns is .ID., and the last column is
  # .PARENT. We are extracting parameter names only.
  ls_param_names <- setdiff(colnames(rdata$allConfigurations), c(".ID.", ".PARENT."))
  output <- paste0('{', double_quote('configs'), ': {\n')
  for (row_id in seq_len(nrow(rdata$allConfigurations))) {
    cand <- rdata$allConfigurations[row_id, ]
    cand_id <- cand$.ID.
    ls_params_vals <- sapply(ls_param_names, function(pname) {
      val <- cand[[pname]]
      if (is.na(val)) return("")
      if (rdata$parameters$types[[pname]] == 'c')
        val <- double_quote(val)
      return(paste0(double_quote(pname), ': ', val))
    })
    if (row_id > 1)
      output <- paste0(output, ',\n')
    ls_params_vals <- ls_params_vals[ls_params_vals != ""]
    output <- paste0(output, double_quote(cand_id),': {', paste0(ls_params_vals, collapse = ', '),'}')
  }
  output <- paste0(output, '}')
  
  # data
  output <- paste0(output, ',\n', double_quote('data'), ': [\n')
  experiments <- experiments[order(experiments$candidate_id,experiments$instance_seed_id),] # order rows in experiments, so that runhistory.json is always the same across different runs
  for (row_id in seq_len(nrow(experiments))) {
    row <- experiments[row_id,]
    cost <- row$cost
    
    if (('capping' %in% rdata$scenario) && rdata$scenario$capping) {
      time <- row$time
      if (row$time >= row$bound){
        if (row$bound < (rdata$scenario$boundMax - 0.01))
          status <- 'StatusType.CAPPED'
        else
          status <- 'StatusType.TIMEOUT'
      } else
        status <- 'StatusType.SUCCESS'
      
    } else {
      time <- 0.9
      status <- 'StatusType.SUCCESS'
    }
    
    if (row_id > 1)
      output <- paste0(output, ',\n')
    #rs = [[conf_id, inst, seed], [cost, time, status, {}]]
    s1 <- paste(row$candidate_id, double_quote(row$instance), row$seed, sep=', ')
    s2 <- paste(as.character(cost), as.character(time), paste('{',double_quote('__enum__'),': ', double_quote(status),'}'), '{}', sep=', ')
    output <- paste0(output,'[[', s1, '], [', s2, ']]')
  }
  output <- paste0(output,'\n]}\n')
  cat(output, file = file)
}

generate_trajectory <- function(file = 'traj_aclib2.json', rdata, experiments)
{  
  #-------------------- traj_aclib2.json ------------------------------#
  cat(sep="", "Generating '", file, "' ...\n")

  # Configurations: The first columns is .ID., and the last column is
  # .PARENT. We are extracting parameter names only.
  ls_param_names <- setdiff(colnames(rdata$allConfigurations), c(".ID.", ".PARENT."))
  output <- ""
  for (iterationId in seq_along(rdata$iterationElites)) {
    confId <- rdata$iterationElites[iterationId]
    # if this elite was eliminated due to filterConditions, ignore it
    t1 <- experiments[(experiments$candidate_id == confId) & (experiments$iteration <= iterationId), ]
    
    # cpu time
    cpu_time <- paste0(double_quote('cputime'),': ', confId)
    total_cpu_time <- paste0(double_quote('total_cpu_time'),': null')
    wallclock_time <- paste0(double_quote('wallclock_time'),': ', confId)
    # evaluations
    evaluations <- paste0(double_quote('evaluations'),': ', nrow(t1))
    # cost
    cost <- paste0(double_quote('cost'),': ', mean(t1$cost))
    # configuration string
    cand <- rdata$allConfigurations[confId, ]
    ls_params_vals <- sapply(ls_param_names, function(pname) {
      val <- cand[[pname]]
      if (is.na(val)) return("")
      if (rdata$parameters$types[[pname]] == 'c')
        val <- single_quote(val)
      # FIMXE: if type is 'c', we do single_quote(val) two times?
      # nguyen: One single_quote is indeed sufficient! I've updated the next line accordingly :)
      return(double_quote(paste0(pname, '=', val)))
    })
    ls_params_vals <- ls_params_vals[ls_params_vals != ""]
    configuration_string <- paste0(double_quote("incumbent"),': [',
                                    paste0(ls_params_vals, collapse = ', '),']')
    # combine everything
    output <- paste0(output, '{',
                     paste(cpu_time, evaluations, cost, configuration_string, total_cpu_time, wallclock_time, sep=', '),'}\n')
  }
  cat(output, file = file)
}

generate_scenario_file <- function(file = "scenario.txt", rdata)
{
  #------ create scenario file -------
  cat(sep="", "Generating scenario file '", file, "' ...\n")
  lss <- list()
  lss[['algo']] <- rdata$scenario$targetRunner
  lss[['execDir']] <- './'
  if (rdata$scenario$deterministic)
    lss[['deterministic']] <- 'true'
  else
    lss[['deterministic']] <- 'false'
  if (rdata$scenario$capping) {
    lss[['run_obj']] <- 'runtime'
    lss[['cutoff_time']] <- rdata$scenario$boundMax
    lss[['overall_obj']] <- paste0('par',rdata$scenario$boundPar)
  } else {
    lss[['run_obj']] <- 'quality'
    lss[['cutoff_time']] <- 1
  }
  lss[['tunerTimeout']] <- 999999
  lss[['overall_obj']] <- 'mean'
  lss[['paramfile']] <- 'params.pcs'
  lss[['instance_file']] <- 'instances.txt'
  lss[['feature_file']] <- 'features.csv'
  writeLines(paste0(names(lss), "=", lss), file)
}


#' Command-line interface to irace2pyimp
#'
#' This is a command-line interface for calling the [`irace2pyimp`] function,
#' which converts an `irace.Rdata` file into the input format supported by the
#' parameter importance analysis tool `PyImp`
#' (https://github.com/automl/ParameterImportance). \cr The best way to use
#' this command line interface is to run the script `irace-to-pyimp`. \cr To
#' see usage of the script, run: irace-to-pyimp --help
#' 
#' @param argv (`character()`) \cr Command-line arguments.
#' 
#' @return None.
#'
#' @examples
#'
#' irace2pyimp_cmdline("--help")
#' 
#' @seealso
#'  [`irace2pyimp`]
#' 
#' @author Nguyen Dang and Manuel López-Ibáñez
#' @export
#' @md
irace2pyimp_cmdline <- function(argv = commandArgs(trailingOnly = TRUE))
{
  parser <- CommandArgsParser$new(argv = argv, argsdef = .irace2pyimp_args)

  if (!is.null(parser$readArg(short = "-h", long = "--help"))) {
    cat(.irace2pyimp_header)
    cmdline_usage(.irace2pyimp_args)
    return(invisible(NULL))
  }
  argvalues <- lapply(rownames(.irace2pyimp_args), parser$readCmdLineParameter)
  names(argv) <- rownames(.irace2pyimp_args)

  irace2pyimp(file = argvalues$iraceRdataFile, normalise = argvalues$normalise,
              outdir = argvalues$outDir,
              instanceFeatureFile = argvalues$instanceFeatureFile,
              filterConditions = argvalues$filterConditions,
              defaultConfigurationID = argvalues$defaultConfigurationID,
              ignoreUnsupported=argvalues$ignoreUnsupported)
}

#' Convert an `irace.Rdata` file into the format supported by PyImp
#'
#' This function converts an `irace.Rdata` file generated by irace into the
#' input format supported by the parameter importance analysis tool `PyImp`
#' (https://github.com/automl/ParameterImportance).

#' 
#' The generated files include:
#'
#'  * `params.pcs` : a text file containing the parameter space definition.
#'  * `runhistory.json` : a JSON file containing the list of algorithm configurations evaluated during the tuning and the performance data obtained.
#'  * `traj_aclib2.json` : a JSON file containing the best configurations after each iteration of irace. The last configuration will be used as the target configuration in ablation analysis.
#'  * `scenario.txt` : a text file containing the definition of the tuning scenario.
#'  * `instances.txt` : a text file containing the list of instances.
#'  * `features.csv` : a text file containing instance features. If no instance features are provided, the index of each instance will be used as a feature.
#' 
#' @param file (`character(1)`) \cr Filename of the `.Rdata` file generated by
#'   irace after a tuning run is finished.
#' 
#' @param normalise (`none` | `instance` | `feature`) \cr Normalise the cost metric values into `[0, 1]` range before converting to PyImp format. Possible values are:\cr
#'     * `none` (default): no normalisation. \cr
#'     * `instance` : normalisation is done per instance. \cr
#'     * `feature` : normalisation is based on features, i.e., instances with the same feature-vector values are grouped together and the normalised cost is calculated per group.
#' 
#' @param outdir (`character(1)`) \cr Directory where all generated files are stored.
#' @param instanceFeatureFile (`character(1)`) \cr A `.csv` file containing instance features (one line per instance, 
#'   sorted in the same order as the list of instances input to irace). The first line contains feature names.
#' @param filterConditions Only extract data that satisfies the given conditions. The conditions are in R expression format.
#' @param defaultConfigurationID Index of default configuration (starting from 1), used by ablation analysis.
#' @param ignoreUnsupported Forbidden configurations and repairConfiguration are not supported by the script. Set ignoreUnsupported=1 to ignore them and proceed with your own risk. This may cause some unwanted behaviours, e.g., forbidden configurations may appear in ablation analysis's path. 
#'
#' @examples
#' \dontrun{
#' irace2pyimp(file='irace.Rdata', outdir='pyimp-run')
#' irace2pyimp(file='irace.Rdata', normalise='feature',
#'             instanceFeatureFile='feature.csv', filterConditions="algorithm!='mas'")
#' }
#' cat("See more examples in '",
#'     file.path(system.file(package="irace"), "examples/irace2pyimp/acotsp/run.sh"),
#'     "' and in '",
#'     file.path(system.file(package="irace"), "examples/irace2pyimp/002-TemplateDesign/run.sh"),
#'     "'\n")
#' 
#' @author Nguyen Dang and Manuel López-Ibáñez
#' @export
#' @md
irace2pyimp <- function(file='./irace.Rdata', normalise='none', outdir='./pyimp-input/',
                        instanceFeatureFile=NA, filterConditions=NA, defaultConfigurationID=1, ignoreUnsupported=0)
{
  if (str_sub(outdir, start=-1) != "/") outdir <- paste0(outdir, "/")
  # create output dir if it doesn't exist
  if (!file.exists(outdir)) dir.create(outdir, recursive = TRUE)
  
  rdata <- load_irace_rdata(file)

  # check forbidden constraint and repairConfiguration
  if ((ignoreUnsupported==0) && (!is.null.or.empty(rdata$scenario$forbiddenExps) || !is.null.or.empty(rdata$scenario$repairConfiguration))){
    irace.error("Forbidden constraints and/or repairConfiguration are not yet supported. To ignore them and proceed with your own risk, please set '--ignore-unsupported 1' (command line), or 'ignoreUnsupported=1' (R console) while calling irace2pyimp. This may cause some unwanted behaviours, e.g., forbidden configurations may appear in ablation analysis's path.")
  }
  # print warning about forbidden constraint and repairConfiguration before proceeding
  if (!is.null.or.empty(rdata$scenario$forbiddenExps) || !is.null.or.empty(rdata$scenario$repairConfiguration)){
    irace.warning("Forbidden constraints and repairConfiguration are ignored. Proceeding with your own risk...")
  }

  ops <- options(scipen=999) # Disable scientific notation
  on.exit(options(ops))
    
  # filter data
  if (!is.na(filterConditions) && trimws(filterConditions) != '') {
    rdata <- filter_data(rdata = rdata, conditions = filterConditions,
                         defaultConfigurationID = defaultConfigurationID)
    defaultConfigurationID <- rdata$defaultConfigurationID
  }
  
  # Generate instances.txt and features.csv
  #--- instances.txt ----
  cat(sep="", "Generating instance list file '", paste0(outdir, 'instances.txt'), "' ...\n")
  writeLines(rdata$scenario$instances, paste0(outdir, 'instances.txt'))
    
  tFeatures <- generate_feature_file(file = paste0(outdir, 'features.csv'),
                                     instances = rdata$scenario$instances,
                                     instanceFeatureFile = instanceFeatureFile)
  
  # remove fixed parameters
  rdata <- remove_fixed_parameters(rdata)
    
  # generate params.pcs
  generate_pcs_file(file = paste0(outdir, 'params.pcs'),
                    parameters = rdata$parameters,
                    defaultConfiguration = rdata$allConfigurations[defaultConfigurationID,])
  
  # generate runhistory.json and traj_aclib2.json
  experiments <- process_experiments(rdata, normalise, tFeatures)

  generate_runhistory(file = paste0(outdir, 'runhistory.json'), rdata = rdata, experiments = experiments)
  generate_trajectory(file = paste0(outdir, 'traj_aclib2.json'), rdata = rdata, experiments = experiments)
    
  # generate scenario.txt
  generate_scenario_file(file = paste0(outdir, "scenario.txt"), rdata = rdata)
}

