#ifndef JobCommand_h
#define JobCommand_h

/******************************************************************************
*
* Copyright (C) 2002 Hugo PEREIRA <mailto: hugo.pereira@free.fr>
*
* This is free software; you can redistribute it and/or modify it under the
* terms of the GNU General Public License as published by the Free Software
* Foundation; either version 2 of the License, or (at your option) any later
* version.
*
* This software is distributed in the hope that it will be useful, but WITHOUT
* Any WARRANTY; without even the implied warranty of MERCHANTABILITY or
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
* for more details.
*
* You should have received a copy of the GNU General Public License along with
* this program.  If not, see <http://www.gnu.org/licenses/>.
*
*******************************************************************************/

#include "Counter.h"
#include "Debug.h"

#include <QString>
#include <QStringList>
#include <QList>
#include <QMap>

//! handles top commands
class JobCommand: public Counter
{

    public:

    //! enumeration for command type
    enum Type
    {
        None,
        JobList,
        Signal
    };

    //! list of commands
    using List = QList<JobCommand>;

    //! constructor
    JobCommand( const Type type = None ):
        Counter( "JobCommand" ),
        id_( _counter()++ ),
        type_( type )
    {}

    //! less than operator
    bool operator < ( const JobCommand& command ) const
    { return id() < command.id(); }

    //! equal to operator
    bool operator == (const JobCommand& command ) const
    { return id() == command.id(); }

    //! command name
    QString name( void ) const
    { return name( type() ); }

    //! map Type to command name
    static QString name( const Type& type );

    //! id
    const unsigned int& id( void ) const
    { return id_; }

    //! type
    const Type& type( void ) const
    { return type_; }

    //!@name interface to arguments
    //@{

    //! add argument
    JobCommand& addArgument( const QString& argument )
    {
        arguments_ << argument;
        return *this;
    }

    //! push arguments
    JobCommand& operator << (const QString& argument )
    { return addArgument( argument ); }

    //! arguments
    const QStringList& arguments( void ) const
    { return arguments_; }

    //! clear arguments
    void clearArguments( void )
    { arguments_.clear(); }

    //@}

    private:

    //! static id counter
    static unsigned int& _counter( void );

    //! map types to command name
    using NameMap = QMap<Type,QString>;

    //! name map
    static NameMap& _names( void );

    //! unique id
    unsigned int id_;

    //! type
    Type type_;

    //! arguments
    QStringList arguments_;

    //! initialize command names
    static void _initializeNames( void );

};

#endif
