# -*- coding: utf-8 -*-


# PyMeeus: Python module implementing astronomical algorithms.
# Copyright (C) 2018  Dagoberto Salazar
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.


from math import sin, cos, tan, acos, atan2, sqrt, radians, log10

from pymeeus.Angle import Angle
from pymeeus.Epoch import Epoch, JDE2000
from pymeeus.Interpolation import Interpolation
from pymeeus.Coordinates import (
    geometric_vsop_pos, apparent_vsop_pos, orbital_elements,
    nutation_longitude, true_obliquity, ecliptical2equatorial,
    passage_nodes_elliptic
)
from pymeeus.Earth import Earth
from pymeeus.Sun import Sun


"""
.. module:: Venus
   :synopsis: Class to model Venus planet
   :license: GNU Lesser General Public License v3 (LGPLv3)

.. moduleauthor:: Dagoberto Salazar
"""


VSOP87_L = [
    # L0
    [
        [317614666.774, 0.00000000000, 0.00000000000],
        [1353968.419, 5.59313319619, 10213.28554621100],
        [89891.645, 5.30650048468, 20426.57109242200],
        [5477.201, 4.41630652531, 7860.41939243920],
        [3455.732, 2.69964470778, 11790.62908865880],
        [2372.061, 2.99377539568, 3930.20969621960],
        [1317.108, 5.18668219093, 26.29831979980],
        [1664.069, 4.25018935030, 1577.34354244780],
        [1438.322, 4.15745043958, 9683.59458111640],
        [1200.521, 6.15357115319, 30639.85663863300],
        [761.380, 1.95014702120, 529.69096509460],
        [707.676, 1.06466707214, 775.52261132400],
        [584.836, 3.99839884762, 191.44826611160],
        [769.314, 0.81629615911, 9437.76293488700],
        [499.915, 4.12340210074, 15720.83878487840],
        [326.221, 4.59056473097, 10404.73381232260],
        [429.498, 3.58642859752, 19367.18916223280],
        [326.967, 5.67736583705, 5507.55323866740],
        [231.937, 3.16251057072, 9153.90361602180],
        [179.695, 4.65337915578, 1109.37855209340],
        [128.263, 4.22604493736, 20.77539549240],
        [155.464, 5.57043888948, 19651.04848109800],
        [127.907, 0.96209822685, 5661.33204915220],
        [105.547, 1.53721191253, 801.82093112380],
        [85.722, 0.35589249966, 3154.68708489560],
        [99.121, 0.83288185132, 213.29909543800],
        [98.804, 5.39389655503, 13367.97263110660],
        [82.094, 3.21596990826, 18837.49819713819],
        [88.031, 3.88868860307, 9999.98645077300],
        [71.577, 0.11145739345, 11015.10647733480],
        [56.122, 4.24039855475, 7.11354700080],
        [70.239, 0.67458813282, 23581.25817731760],
        [50.796, 0.24531603049, 11322.66409830440],
        [46.111, 5.31576465717, 18073.70493865020],
        [44.574, 6.06282201966, 40853.14218484400],
        [42.594, 5.32873337210, 2352.86615377180],
        [42.635, 1.79955421680, 7084.89678111520],
        [41.177, 0.36240972161, 382.89653222320],
        [35.749, 2.70448479296, 10206.17199921020],
        [33.893, 2.02347322198, 6283.07584999140],
        [29.138, 3.59230925768, 22003.91463486980],
        [28.479, 2.22375414002, 1059.38193018920],
        [29.850, 4.02176977477, 10239.58386601080],
        [33.252, 2.10025596509, 27511.46787353720],
        [30.172, 4.94191919890, 13745.34623902240],
        [29.252, 3.51392387787, 283.85931886520],
        [24.424, 2.70177493852, 8624.21265092720],
        [20.274, 3.79493637509, 14143.49524243060],
        [24.322, 4.27814493315, 5.52292430740],
        [26.260, 0.54067587552, 17298.18232732620],
        [20.492, 0.58547075036, 38.02767263580],
        [18.988, 4.13811500642, 4551.95349705880],
        [23.739, 4.82870797552, 6872.67311951120],
        [15.885, 1.50067222283, 8635.94200376320],
        [19.069, 6.12025580313, 29050.78374334920],
        [18.269, 3.04740408477, 19999.97290154599],
        [13.656, 4.41336292334, 3532.06069281140],
        [17.094, 3.52161526430, 31441.67756975680],
        [10.955, 2.84562790076, 18307.80723204360],
        [11.048, 2.58361219075, 9786.68735533500],
        [9.904, 1.08737710389, 7064.12138562280],
        [10.576, 0.85419784436, 10596.18207843420],
        [9.231, 5.52471655579, 12566.15169998280],
        [11.599, 5.81007422699, 19896.88012732740],
        [11.807, 1.91250672543, 21228.39202354580],
        [10.105, 2.34270786693, 10742.97651130560],
        [8.154, 1.92331359797, 15.25247118500],
        [8.893, 1.97291388515, 10186.98722641120],
        [9.352, 4.94508904764, 35371.88726597640],
        [6.821, 4.39733188968, 8662.24032356300],
        [6.688, 1.55310437864, 14945.31617355440],
        [6.413, 2.17677652923, 10988.80815753500],
        [5.802, 1.93462125906, 3340.61242669980],
        [5.950, 2.96578175391, 4732.03062734340],
        [5.275, 5.01877102496, 28286.99048486120],
        [7.047, 1.00111452053, 632.78373931320],
        [5.048, 4.27886209626, 29580.47470844380],
        [6.305, 0.35506331180, 103.09277421860],
        [5.959, 5.04792949464, 245.83164622940],
        [4.651, 0.85218058876, 6770.71060124560],
        [5.580, 0.48723384809, 522.57741809380],
        [5.327, 3.03115417024, 10021.83728009940],
        [5.010, 5.77375166500, 28521.09277825460],
        [4.608, 1.93302108394, 4705.73230754360],
        [5.526, 3.36797048901, 25158.60171976540],
        [3.863, 4.89351531412, 25934.12433108940],
        [5.303, 0.08161426841, 39302.09696219600],
        [4.254, 5.36046113295, 21535.94964451540],
        [3.763, 1.05304597315, 19.66976089979],
        [4.407, 4.02575374517, 74.78159856730],
        [4.145, 1.14356412295, 9676.48103411560],
        [4.318, 4.38289970585, 316.39186965660],
        [3.642, 6.11733529325, 3128.38876509580],
        [3.238, 5.39551036769, 419.48464387520],
        [3.909, 4.05263635330, 9690.70812811720],
        [3.152, 0.72553551731, 16496.36139620240],
        [3.496, 0.72414615705, 3723.50895892300],
        [3.755, 3.80208713127, 19786.67380610799],
        [2.891, 3.33782737770, 32217.20018108080],
        [3.016, 1.57249112496, 17277.40693183380],
        [3.825, 0.19612312903, 426.59819087600],
        [3.798, 0.45524571743, 10316.37832042960],
        [2.570, 1.20813474107, 13936.79450513400],
        [2.796, 3.65128969074, 206.18554843720],
        [2.466, 3.61988676373, 1551.04522264800],
        [3.108, 1.50325806664, 43232.30665841560],
        [2.976, 4.79415001304, 29088.81141598500],
        [2.217, 3.59623681714, 24356.78078864160],
        [2.227, 4.96059221940, 536.80451209540],
        [2.397, 3.45249688427, 19374.30270923360],
        [2.462, 0.53295178258, 19360.07561523199],
        [2.205, 2.70399309963, 12592.45001978260],
        [2.230, 3.01413465913, 18875.52586977400],
        [1.858, 4.06129152783, 2379.16447357160],
        [1.807, 3.15086214479, 9573.38825989700],
        [2.238, 5.52216925076, 10138.50394764370],
        [2.195, 2.32046770554, 8094.52168583260],
        [2.101, 2.90421302975, 9967.45389998160],
        [1.916, 4.56513949099, 2218.75710418680],
        [1.467, 2.42640162465, 10234.06094170340],
        [1.726, 5.59790693845, 20452.86941222180],
        [1.455, 2.44757248737, 1589.07289528380],
        [1.991, 4.04623390359, 31749.23519072640],
        [1.406, 2.71736996917, 16983.99614745660],
        [1.658, 0.11252373292, 153.77881048480],
        [1.851, 2.92898027939, 47162.51635463520],
        [1.492, 1.07513892753, 9103.90699411760],
        [1.247, 2.48433565896, 17778.11626694899],
        [1.549, 4.20553654300, 3442.57494496540],
        [1.243, 3.95452438599, 170.67287061920],
        [1.694, 6.20694480406, 33019.02111220460],
        [1.221, 4.77931820602, 30110.16567353840],
        [1.206, 0.30531303095, 29864.33402730900],
        [1.238, 5.05581820608, 20213.27199698400],
        [1.152, 3.26229919481, 11.72935283600],
        [1.179, 1.69491074791, 20400.27277262220],
        [1.165, 2.88995128147, 574.34479833480],
        [1.306, 0.14519588607, 9146.79006902100],
        [1.113, 1.52598846804, 10426.58464164900],
        [1.104, 2.58791423813, 18849.22754997420],
        [1.045, 0.57539216420, 15874.61759536320],
        [1.360, 2.41976595457, 38734.37832446560],
        [0.981, 4.37930727798, 110.20632121940],
        [1.095, 0.49492867814, 51066.42773105500],
        [1.146, 4.54241454215, 10220.39909321180],
        [0.981, 1.65915064733, 10103.07922499160],
        [1.270, 4.69374306132, 9050.81084180320],
        [1.065, 4.41645258887, 22805.73556599360],
        [0.854, 2.34437926957, 6681.22485339960],
        [1.104, 0.49781459714, 1.48447270830],
        [1.075, 1.09857593161, 377.37360791580],
        [1.114, 4.35024775806, 51092.72605085480],
        [0.829, 5.41196274578, 27991.40181316000],
        [0.900, 2.74195379617, 41962.52073693740],
        [1.010, 2.96092073452, 135.62532501000],
        [0.768, 3.98260860494, 18844.61174413899],
        [1.018, 1.36891050752, 36949.23080842420],
        [0.726, 1.67728773965, 21202.09370374600],
        [0.727, 0.89048212541, 467.96499035440],
        [0.869, 2.93767679827, 10192.51015071860],
        [0.696, 5.35698039414, 10063.72234907640],
        [0.920, 4.17128923588, 18734.40542291960],
        [0.691, 1.50594097883, 27197.28169366760],
        [0.835, 0.48050621092, 20618.01935853360],
        [0.711, 0.19750098222, 18830.38465013739],
        [0.811, 0.16685071959, 12432.04265039780],
        [0.756, 3.79022623226, 9161.01716302260],
        [0.622, 5.33659507738, 9411.46461508720],
        [0.862, 5.72705356405, 10175.15251057320],
        [0.853, 0.10404188453, 2107.03450754240],
        [0.742, 3.96365892051, 813.55028395980],
        [0.705, 0.71229660616, 220.41264243880],
        [0.584, 1.71900692700, 36.02786667740],
        [0.612, 0.36418385449, 949.17560896980],
        [0.587, 1.58648949290, 6.62855890001],
        [0.581, 5.49288908804, 6309.37416979120],
        [0.581, 4.80353237853, 24150.08005134500],
        [0.516, 6.07328802561, 38.13303563780],
        [0.627, 5.47281424954, 9580.50180689780],
        [0.601, 1.40500080774, 1162.47470440780],
        [0.620, 4.00681042667, 9992.87290377220],
        [0.611, 3.62010998629, 7255.56965173440],
        [0.697, 2.22359630727, 348.92442044800],
        [0.693, 5.77432072851, 55022.93574707440],
        [0.494, 0.29761886866, 7058.59846131540],
        [0.563, 0.24172140474, 37410.56723987860],
        [0.487, 5.86917216517, 10137.01947493540],
        [0.493, 2.04534833854, 735.87651353180],
        [0.636, 2.79707392326, 40879.44050464380],
        [0.519, 4.13945657630, 16522.65971600220],
        [0.535, 4.60569597820, 19573.37471066999],
        [0.555, 5.88120421263, 26735.94526221320],
        [0.440, 5.61490649795, 23958.63178523340],
        [0.541, 0.62494922735, 10007.09999777380],
        [0.427, 4.02335620501, 14.22709400160],
        [0.434, 0.29028429049, 9264.10993724120],
        [0.451, 1.66320363626, 26087.90314157420],
        [0.422, 3.38413582674, 10787.63034454580],
        [0.569, 5.14001758731, 27490.69247804480],
        [0.421, 4.23407313457, 39793.76025465480],
        [0.458, 5.28786368820, 49.75702547180],
        [0.418, 5.69097790790, 14765.23904326980],
        [0.475, 0.97544690438, 1052.26838318840],
        [0.387, 4.41665162999, 21.85082932640],
        [0.523, 2.90512426870, 20235.12282631040],
        [0.506, 5.26999314618, 29999.95935231900],
        [0.438, 2.49457071132, 20956.26205751660],
        [0.530, 0.71368442157, 33794.54372352860],
        [0.382, 1.92119365480, 3.93215326310],
        [0.365, 3.81715328784, 20419.45754542119],
        [0.426, 2.06384083608, 38204.68735937100],
        [0.496, 0.44077356179, 9835.91193829520],
        [0.410, 4.93346326003, 19264.09638801420],
        [0.380, 3.79573358631, 8521.11987670860],
        [0.334, 5.51158557799, 10251.31321884680],
        [0.412, 2.56129670728, 77.67377042800],
        [0.418, 2.37865963521, 32.53255079140],
        [0.325, 6.03020523465, 18947.70451835760],
        [0.400, 0.91999360201, 227.47613278900],
        [0.437, 0.91420135162, 58953.14544329400],
        [0.360, 0.82477639126, 22.77520145080],
        [0.413, 4.22381905655, 44809.65020086340],
        [0.375, 3.15657291896, 19992.85935454519],
        [0.371, 6.05370874275, 20007.08644854680],
        [0.361, 5.44371227904, 19470.28193645139],
        [0.386, 5.28120540405, 47623.85278608960],
        [0.389, 0.73216672240, 19050.79729257620],
        [0.320, 2.84811591194, 10199.05845220940],
        [0.386, 3.88754165531, 1975.49254585600],
        [0.276, 4.33979180814, 20809.46762464520],
        [0.276, 0.50647429773, 9830.38901398780],
        [0.309, 3.79299100668, 18204.71445782499],
        [0.377, 0.73768791281, 11506.76976979360],
        [0.322, 0.96138442100, 30666.15495843280],
        [0.363, 1.30472406690, 9367.20271145980],
        [0.366, 2.79972786028, 11272.66747640020],
        [0.271, 4.66141338193, 846.08283475120],
        [0.259, 0.42031175750, 39264.06928956020],
        [0.285, 0.40546033634, 30.91412563500],
        [0.247, 4.80676426942, 36147.40987730040],
        [0.264, 2.71608177583, 11.04570026390],
        [0.233, 2.76423842887, 187.92514776260],
        [0.248, 1.60765612338, 10497.14486507620],
        [0.271, 0.82348919630, 19793.78735310880],
        [0.225, 3.80080957016, 8631.32619792800],
        [0.263, 1.92311689852, 37724.75341974820],
        [0.214, 5.01663795092, 639.89728631400],
        [0.289, 0.12342601246, 20277.00789528740],
        [0.210, 0.12771800254, 29.82143814880],
        [0.227, 4.18036609801, 17468.85519794540],
        [0.274, 2.34929343000, 62883.35513951360],
        [0.260, 5.65254501655, 48739.85989708300],
        [0.271, 4.95325404028, 4214.06901508480],
        [0.219, 2.08775228014, 194.97138446060],
        [0.191, 2.49267248333, 568.82187402740],
        [0.250, 1.52909737354, 6037.24420376200],
        [0.231, 5.23674429498, 491.66329245880],
        [0.182, 4.98046042571, 18418.01355326299],
        [0.188, 2.82273639603, 1385.89527633620],
        [0.204, 4.09939796199, 14919.01785375460],
        [0.176, 3.82400982460, 9360.08916445900],
        [0.198, 2.76491873243, 10217.21769947410],
        [0.168, 5.19268384202, 1066.49547719000],
        [0.199, 1.95301487982, 7564.83072073800],
        [0.171, 2.59623459612, 20405.79569692960],
        [0.172, 5.29332132623, 11764.33076885900],
        [0.165, 2.88557908025, 10207.76262190360],
        [0.164, 3.25435371801, 3914.95722503460],
        [0.200, 3.82443218090, 18314.92077904440],
        [0.169, 1.78341902878, 31022.75317085620],
        [0.179, 0.90840065587, 7880.08915333899],
        [0.163, 2.79665037814, 41.55079098480],
        [0.154, 3.90796293476, 30213.25844775700],
        [0.153, 0.07463240782, 28528.20632525540],
        [0.194, 5.95838706838, 8617.09910392640],
        [0.171, 4.58206324409, 20447.34648791440],
        [0.150, 2.11647586229, 17248.42530185440],
        [0.149, 2.17259986320, 9929.42622734580],
        [0.191, 0.82310353823, 52670.06959330260],
        [0.148, 2.94315921485, 41654.96311596780],
        [0.149, 4.49798039726, 30831.30490474460],
        [0.184, 2.46923348701, 34596.36465465240],
        [0.146, 2.69452930300, 43071.89928903080],
        [0.159, 2.11137713570, 19317.19254032860],
        [0.154, 2.76536164654, 28513.97923125379],
        [0.140, 4.94595038686, 9256.99639024040],
        [0.141, 2.57248458154, 13553.89797291080],
        [0.137, 1.66482327575, 2636.72547263700],
        [0.140, 5.23039605990, 22645.32819660879],
        [0.132, 5.35690599728, 19624.75016129820],
        [0.140, 2.90637712614, 48947.66387067660],
        [0.129, 3.95303623681, 32858.61374281979],
        [0.156, 6.01143316387, 29057.89729034999],
        [0.134, 5.75241675118, 68050.42387851159],
        [0.154, 3.66827363753, 276.74577186440],
        [0.176, 3.77298381177, 66813.56483573320],
        [0.126, 5.00217740223, 27461.71084806540],
        [0.135, 1.34807013920, 53285.18483524180],
        [0.150, 0.25029475344, 290.97286586600],
        [0.152, 3.13035670092, 29043.67019634839],
        [0.169, 5.04348109430, 73.29712585900],
        [0.166, 5.39219948035, 41236.03871706720],
        [0.163, 5.59796070987, 7576.56007357400],
        [0.126, 0.77391784606, 49.99662190420],
        [0.163, 0.44241846674, 20350.30502114640],
        [0.136, 3.09066368912, 418.92439890060],
        [0.154, 0.47086190960, 28418.00000403600],
        [0.120, 0.88536981986, 29573.36116144300],
        [0.132, 1.48009769040, 17085.95866572220],
        [0.126, 1.39497760964, 966.97087743560],
        [0.143, 3.84026797958, 14128.24277124560],
        [0.147, 2.11627427804, 34363.36559755600],
        [0.106, 2.04696932293, 37674.99639427640],
        [0.106, 1.43873202489, 27682.14074415640],
        [0.149, 0.09286508794, 8144.27871130440],
        [0.103, 0.01992041470, 18300.69368504279],
        [0.121, 3.57602835443, 45.14121963660],
        [0.125, 0.11630302078, 149.56319713460],
        [0.102, 4.17947097730, 2333.19639287200],
        [0.099, 1.51324741657, 10419.47109464820],
        [0.133, 3.02183293676, 76251.32777062019],
        [0.136, 4.17517197268, 3646.35037735440],
        [0.123, 0.44045588682, 515.46387109300],
        [0.113, 5.69261397718, 10110.19277199240],
        [0.098, 6.23797900467, 202.25339517410],
        [0.099, 3.75627530197, 59728.66805461800],
        [0.101, 4.62832557536, 65236.22129328540],
        [0.111, 1.25947267588, 10846.06928552420],
        [0.110, 5.87455577536, 38500.27603107220],
        [0.128, 6.01024562160, 90394.82301305079],
        [0.091, 1.77665981007, 1539.31586981200],
        [0.092, 0.99804571578, 95.97922721780],
        [0.120, 3.93060866244, 38526.57435087200],
        [0.117, 2.24143299549, 56600.27928952220],
        [0.118, 6.09121325940, 29786.66025688100],
        [0.098, 4.60938156207, 11787.10597030980],
        [0.097, 3.92727733144, 11794.15220700780],
        [0.093, 5.23395435043, 14169.79356223040],
        [0.096, 5.27525709038, 8734.41897214660],
        [0.094, 0.18166654805, 67589.08744705719],
        [0.110, 4.96279287076, 48417.97290558199],
        [0.086, 0.39533409505, 3956.50801601940],
        [0.085, 5.69642646462, 37703.97802425580],
        [0.114, 5.19676285428, 70743.77453195279],
        [0.081, 5.51324815184, 412.37109687440],
        [0.089, 2.13409771828, 44768.09940987860],
        [0.084, 6.02475904578, 10632.77019008620],
        [0.085, 4.60912614442, 45585.17281218740],
        [0.078, 4.47358603432, 114.43928868521],
        [0.097, 4.02223363535, 10218.80847051840],
        [0.081, 1.03870237004, 9793.80090233580],
        [0.092, 0.80301220092, 24383.07910844140],
        [0.087, 2.15124790938, 28313.28880466100],
        [0.075, 5.17868679355, 63658.87775083760],
        [0.078, 5.81927313665, 567.71863773040],
        [0.075, 1.72618192481, 19580.48825767080],
        [0.071, 0.10259261764, 90695.75207512038],
        [0.077, 6.16012067704, 1573.82042409880],
        [0.076, 5.12884307551, 49515.38250840700],
        [0.069, 0.29569499484, 10175.25787357520],
        [0.061, 4.80385549281, 19889.76658032659],
        [0.060, 4.56685040226, 30426.55754319500],
        [0.062, 4.16222812699, 42430.48572729180],
        [0.050, 6.17899839001, 22779.43724619380],
        [0.048, 1.52546758016, 20639.87018786000],
        [0.046, 4.41738494249, 34570.06633485260],
        [0.037, 4.69675087759, 44007.82926973960],
    ],
    # L1
    [
        [1021352943052.898, 0.00000000000, 0.00000000000],
        [95707.712, 2.46424448979, 10213.28554621100],
        [14444.977, 0.51624564679, 20426.57109242200],
        [213.374, 1.79547929368, 30639.85663863300],
        [151.669, 6.10635282369, 1577.34354244780],
        [173.904, 2.65535879443, 26.29831979980],
        [82.233, 5.70234133730, 191.44826611160],
        [69.734, 2.68136034979, 9437.76293488700],
        [52.408, 3.60013087656, 775.52261132400],
        [38.318, 1.03379038025, 529.69096509460],
        [29.633, 1.25056322354, 5507.55323866740],
        [25.056, 6.10664792855, 10404.73381232260],
        [17.772, 6.19369798901, 1109.37855209340],
        [16.510, 2.64330452640, 7.11354700080],
        [14.230, 5.45138233941, 9153.90361602180],
        [12.607, 1.24464400689, 40853.14218484400],
        [11.627, 4.97604495371, 213.29909543800],
        [12.563, 1.88122199199, 382.89653222320],
        [8.869, 0.95282732248, 13367.97263110660],
        [7.374, 4.39476760580, 10206.17199921020],
        [6.552, 2.28168808058, 2352.86615377180],
        [6.255, 4.08056644034, 3154.68708489560],
        [6.697, 5.05673427795, 801.82093112380],
        [4.084, 4.12103826030, 18837.49819713819],
        [4.882, 3.44515199115, 11015.10647733480],
        [3.549, 6.19934345402, 5.52292430740],
        [3.448, 1.77405651704, 11322.66409830440],
        [4.290, 0.08154809210, 6283.07584999140],
        [3.694, 2.48453945256, 5661.33204915220],
        [3.555, 1.48036949420, 1059.38193018920],
        [3.023, 2.24092938317, 18073.70493865020],
        [3.000, 0.39169917698, 15.25247118500],
        [2.563, 0.35147506973, 22003.91463486980],
        [2.774, 1.45683830639, 10239.58386601080],
        [2.951, 5.34618097429, 7084.89678111520],
        [2.344, 2.36652432105, 17298.18232732620],
        [2.405, 2.36085282088, 10596.18207843420],
        [1.720, 4.72129626061, 10186.98722641120],
        [2.209, 2.07730338665, 8635.94200376320],
        [2.122, 4.47091605309, 8624.21265092720],
        [1.527, 0.67146857292, 14143.49524243060],
        [1.473, 2.59350470099, 7064.12138562280],
        [1.311, 0.90408820221, 12566.15169998280],
        [1.474, 5.92236241437, 9786.68735533500],
        [1.237, 2.59740787132, 4551.95349705880],
        [1.219, 2.83617320088, 9676.48103411560],
        [1.116, 3.83715584719, 21228.39202354580],
        [1.006, 4.26200749078, 426.59819087600],
        [1.150, 2.35531987378, 9690.70812811720],
        [1.219, 2.27324315182, 522.57741809380],
        [1.150, 0.81088598778, 10742.97651130560],
        [1.101, 3.74248783564, 18307.80723204360],
        [1.031, 2.03889374176, 38.02767263580],
        [0.971, 6.10590045414, 3532.06069281140],
        [0.844, 4.75124127613, 10988.80815753500],
        [0.908, 1.06613723738, 10021.83728009940],
        [0.824, 0.23090829723, 28286.99048486120],
        [0.821, 2.60456032773, 19.66976089979],
        [0.728, 0.10716917942, 4705.73230754360],
        [0.744, 3.33129778857, 536.80451209540],
        [0.816, 1.27303930175, 19896.88012732740],
        [0.929, 1.08024621325, 11790.62908865880],
        [0.797, 2.23891816523, 3723.50895892300],
        [0.704, 5.95307260017, 20.77539549240],
        [0.665, 0.21346689192, 7860.41939243920],
        [0.733, 2.22147883292, 19360.07561523199],
        [0.702, 1.76206343944, 19374.30270923360],
        [0.575, 2.38792087791, 6770.71060124560],
        [0.538, 1.52023264138, 25934.12433108940],
        [0.690, 4.01873754171, 19651.04848109800],
        [0.532, 4.41576130890, 574.34479833480],
        [0.540, 2.15936134728, 16496.36139620240],
        [0.576, 5.41170044566, 206.18554843720],
        [0.482, 0.40815793538, 3340.61242669980],
        [0.501, 3.08578363577, 245.83164622940],
        [0.488, 5.22311611589, 25158.60171976540],
        [0.450, 0.21279844600, 11.72935283600],
        [0.432, 1.32004964493, 103.09277421860],
        [0.434, 5.91094755233, 19786.67380610799],
        [0.564, 0.38776462529, 19367.18916223280],
        [0.421, 2.71057839701, 13936.79450513400],
        [0.549, 6.08792865644, 3930.20969621960],
        [0.478, 4.70234715828, 14945.31617355440],
        [0.408, 4.80890663927, 32217.20018108080],
        [0.404, 2.85003595942, 29864.33402730900],
        [0.407, 2.94002049006, 10220.39909321180],
        [0.359, 0.72354778897, 419.48464387520],
        [0.449, 1.44520508753, 8662.24032356300],
        [0.353, 2.22195492336, 51066.42773105500],
        [0.324, 1.40308439067, 29580.47470844380],
        [0.443, 1.93864353398, 9146.79006902100],
        [0.314, 0.96837035284, 20618.01935853360],
        [0.324, 5.10759068171, 24356.78078864160],
        [0.324, 1.80146948625, 18830.38465013739],
        [0.370, 6.16895004656, 2218.75710418680],
        [0.278, 2.20429108375, 18844.61174413899],
        [0.286, 3.08459438435, 17277.40693183380],
        [0.383, 0.13890934755, 4732.03062734340],
        [0.292, 0.43528982259, 29088.81141598500],
        [0.273, 5.84415407168, 9573.38825989700],
        [0.324, 2.14144542781, 9999.98645077300],
        [0.264, 5.20407029554, 220.41264243880],
        [0.254, 0.34411959301, 28521.09277825460],
        [0.300, 3.76014360906, 8094.52168583260],
        [0.301, 3.64457981649, 20400.27277262220],
        [0.287, 1.84003536598, 1589.07289528380],
        [0.206, 0.97167234723, 10234.06094170340],
        [0.212, 0.24173677600, 36.02786667740],
        [0.216, 5.88618923030, 18875.52586977400],
        [0.198, 1.89506914939, 20452.86941222180],
        [0.258, 6.27611355094, 1551.04522264800],
        [0.197, 2.09222675324, 9683.59458111640],
        [0.217, 5.79472589364, 9103.90699411760],
        [0.188, 0.39123199129, 19573.37471066999],
        [0.195, 6.23142464829, 30110.16567353840],
        [0.187, 5.49670351645, 170.67287061920],
        [0.178, 4.90042854659, 10787.63034454580],
        [0.188, 1.62614804098, 9161.01716302260],
        [0.211, 2.71884568392, 15720.83878487840],
        [0.177, 1.88170417337, 33019.02111220460],
        [0.209, 2.66033422116, 3442.57494496540],
        [0.164, 4.92240093026, 10426.58464164900],
        [0.186, 5.13678812068, 7255.56965173440],
        [0.177, 5.70206821967, 9992.87290377220],
        [0.214, 2.70027196648, 3128.38876509580],
        [0.208, 3.38876526854, 17778.11626694899],
        [0.147, 4.25008782855, 16983.99614745660],
        [0.148, 3.46404418130, 21202.09370374600],
        [0.189, 1.43553862242, 2379.16447357160],
        [0.139, 2.99154379541, 110.20632121940],
        [0.159, 5.23851679605, 10007.09999777380],
        [0.136, 0.88942869764, 22805.73556599360],
        [0.155, 5.90500835975, 12592.45001978260],
        [0.151, 0.03422618975, 27991.40181316000],
        [0.153, 4.01743770323, 33794.54372352860],
        [0.121, 0.51392111799, 21535.94964451540],
        [0.109, 2.25388616761, 26735.94526221320],
        [0.109, 0.78612823474, 6681.22485339960],
        [0.122, 4.84805105466, 19992.85935454519],
        [0.112, 3.31796669604, 36949.23080842420],
        [0.106, 3.34507236765, 10103.07922499160],
        [0.114, 4.36384000196, 20007.08644854680],
        [0.098, 5.07711736751, 135.62532501000],
        [0.120, 5.41870615047, 37724.75341974820],
        [0.103, 2.62610244425, 20213.27199698400],
        [0.085, 5.04808202087, 9830.38901398780],
        [0.103, 2.01549383816, 45585.17281218740],
        [0.088, 2.62613816931, 21.85082932640],
        [0.084, 3.50355880173, 639.89728631400],
        [0.099, 0.61079620895, 41654.96311596780],
        [0.088, 3.63836700262, 49515.38250840700],
        [0.098, 2.42401801881, 23581.25817731760],
        [0.081, 0.46468679835, 77.67377042800],
        [0.092, 4.82530051729, 29043.67019634839],
        [0.102, 4.27051236894, 15874.61759536320],
        [0.090, 4.34075776744, 29057.89729034999],
        [0.081, 0.01896422336, 24150.08005134500],
        [0.093, 1.79250830018, 12432.04265039780],
        [0.087, 5.25157021446, 14128.24277124560],
        [0.089, 5.65756996753, 377.37360791580],
        [0.097, 5.67942873241, 227.47613278900],
        [0.076, 2.93363913259, 38204.68735937100],
        [0.091, 2.60544242067, 1052.26838318840],
        [0.087, 3.82284200928, 27511.46787353720],
        [0.073, 4.75280755154, 40879.44050464380],
        [0.067, 3.54815262526, 30666.15495843280],
        [0.067, 5.81350818057, 20809.46762464520],
        [0.064, 4.24772678145, 153.77881048480],
        [0.064, 2.99454749109, 27197.28169366760],
        [0.070, 4.03868009742, 56600.27928952220],
        [0.071, 4.33628806850, 39264.06928956020],
        [0.069, 1.73648747605, 37410.56723987860],
        [0.065, 1.08206062736, 68050.42387851159],
        [0.062, 4.77698454650, 3914.95722503460],
        [0.061, 4.96121014691, 34596.36465465240],
        [0.063, 5.04865067599, 53445.59220462660],
        [0.058, 3.74010494151, 1066.49547719000],
        [0.057, 5.39355890141, 20419.45754542119],
        [0.057, 3.59399518494, 735.87651353180],
        [0.065, 2.10322000074, 74.78159856730],
        [0.073, 1.31083648835, 11272.66747640020],
        [0.055, 1.33161298098, 18300.69368504279],
        [0.065, 4.21150522641, 49.75702547180],
        [0.061, 5.66161679402, 17468.85519794540],
        [0.053, 4.30231233835, 18849.22754997420],
        [0.055, 2.63906959481, 52670.06959330260],
        [0.050, 5.69803054279, 39793.76025465480],
        [0.049, 0.77345264124, 35371.88726597640],
        [0.048, 6.00565977593, 283.85931886520],
        [0.047, 2.63299859494, 51868.24866217880],
        [0.046, 0.05105081843, 38526.57435087200],
        [0.050, 4.37549274002, 28513.97923125379],
        [0.046, 2.93422086586, 27682.14074415640],
        [0.051, 5.45979584751, 60530.48898574180],
        [0.045, 5.59492908223, 467.96499035440],
        [0.045, 2.34680401001, 9411.46461508720],
        [0.045, 0.02999265111, 44809.65020086340],
        [0.043, 5.62725673544, 14.22709400160],
        [0.047, 3.73567275749, 64460.69868196140],
        [0.046, 0.12586526756, 57375.80190084620],
        [0.044, 2.03114426076, 18314.92077904440],
        [0.039, 0.99375127466, 94138.32702008578],
        [0.053, 0.41974404621, 30831.30490474460],
        [0.055, 1.38351566741, 38500.27603107220],
        [0.041, 4.47012768909, 40077.61957352000],
        [0.041, 0.36665992484, 19999.97290154599],
        [0.040, 3.06358586355, 813.55028395980],
        [0.040, 2.16802870803, 59728.66805461800],
        [0.037, 1.08739100421, 17085.95866572220],
        [0.039, 1.31040309875, 48739.85989708300],
        [0.036, 1.43280677914, 42456.78404709160],
        [0.037, 0.14190533464, 29050.78374334920],
        [0.037, 3.66792179278, 20956.26205751660],
        [0.025, 3.38876180652, 7058.59846131540],
        [0.031, 6.16829805337, 10192.51015071860],
    ],
    # L2
    [
        [54127.076, 0.00000000000, 0.00000000000],
        [3891.460, 0.34514360047, 10213.28554621100],
        [1337.880, 2.02011286082, 20426.57109242200],
        [23.836, 2.04592119012, 26.29831979980],
        [19.331, 3.53527371458, 30639.85663863300],
        [9.984, 3.97130221102, 775.52261132400],
        [7.046, 1.51962593409, 1577.34354244780],
        [6.014, 0.99926757893, 191.44826611160],
        [3.163, 4.36095475762, 9437.76293488700],
        [2.125, 2.65810625752, 40853.14218484400],
        [1.934, 3.39287946981, 382.89653222320],
        [1.460, 6.04899046273, 529.69096509460],
        [1.346, 2.94633106219, 5507.55323866740],
        [1.025, 1.40598904981, 10404.73381232260],
        [1.221, 3.73339139385, 3154.68708489560],
        [1.033, 3.52858472904, 11015.10647733480],
        [0.955, 5.11133878923, 801.82093112380],
        [0.742, 1.49198584483, 1109.37855209340],
        [0.525, 3.32087042103, 213.29909543800],
        [0.578, 0.92614279843, 10239.58386601080],
        [0.602, 5.19220099775, 7084.89678111520],
        [0.431, 2.67159914364, 13367.97263110660],
        [0.389, 4.14116341739, 8635.94200376320],
        [0.355, 1.12061570874, 9153.90361602180],
        [0.301, 3.90047984197, 10596.18207843420],
        [0.212, 5.32697688872, 18837.49819713819],
        [0.260, 0.22761369281, 2352.86615377180],
        [0.243, 4.70747902991, 6283.07584999140],
        [0.196, 4.10467294392, 11790.62908865880],
        [0.194, 6.01197759470, 7860.41939243920],
        [0.140, 4.97015671653, 14143.49524243060],
        [0.134, 4.10529011674, 17298.18232732620],
        [0.119, 3.39375528828, 11322.66409830440],
        [0.126, 0.09854516140, 18073.70493865020],
        [0.122, 5.92478855457, 574.34479833480],
        [0.107, 0.35660030184, 1059.38193018920],
        [0.108, 2.25352052666, 12566.15169998280],
        [0.093, 5.48716819776, 10021.83728009940],
        [0.084, 4.89744332968, 18307.80723204360],
        [0.074, 2.35354025573, 426.59819087600],
        [0.093, 4.99316908815, 14945.31617355440],
        [0.069, 3.86409065860, 51066.42773105500],
        [0.082, 5.36280178643, 10186.98722641120],
        [0.077, 3.75728548372, 3723.50895892300],
        [0.063, 5.39882267787, 21228.39202354580],
        [0.056, 4.11564786973, 7064.12138562280],
        [0.056, 6.26920407723, 32217.20018108080],
        [0.060, 5.02186497542, 19367.18916223280],
        [0.058, 5.13263709670, 20400.27277262220],
        [0.051, 4.52870390511, 22003.91463486980],
        [0.041, 3.83822107919, 16496.36139620240],
        [0.041, 3.36020411807, 4705.73230754360],
        [0.043, 5.98371820588, 15720.83878487840],
        [0.047, 0.18498155367, 18875.52586977400],
        [0.038, 0.52232581277, 1551.04522264800],
        [0.039, 5.05391675878, 10742.97651130560],
        [0.036, 3.16242472203, 20452.86941222180],
        [0.035, 5.17462577483, 29088.81141598500],
        [0.035, 3.47325394141, 24356.78078864160],
        [0.031, 4.74511706231, 28521.09277825460],
        [0.029, 0.19383091192, 19896.88012732740],
        [0.033, 1.80059867635, 20618.01935853360],
        [0.024, 0.14022912457, 21202.09370374600],
        [0.022, 4.73565067573, 10988.80815753500],
        [0.018, 0.74039763161, 25158.60171976540],
        [0.019, 1.53770232218, 28286.99048486120],
        [0.014, 1.49084059765, 30110.16567353840],
        [0.013, 4.72171283479, 29864.33402730900],
        [0.013, 5.79700427846, 29580.47470844380],
        [0.014, 3.69205225010, 27511.46787353720],
    ],
    # L3
    [
        [135.742, 4.80389020993, 10213.28554621100],
        [77.846, 3.66876371591, 20426.57109242200],
        [26.023, 0.00000000000, 0.00000000000],
        [1.214, 5.31970006917, 30639.85663863300],
        [0.254, 4.15021671822, 40853.14218484400],
        [0.008, 5.55523563261, 51066.42773105500],
        [0.008, 1.40501229148, 1577.34354244780],
        [0.006, 1.27791479726, 10404.73381232260],
        [0.006, 5.76447068962, 10239.58386601080],
    ],
    # L4
    [
        [114.016, 3.14159265359, 0.00000000000],
        [3.209, 5.20514170164, 20426.57109242200],
        [1.714, 2.51099591706, 10213.28554621100],
        [0.050, 0.71356059861, 30639.85663863300],
        [0.023, 5.68127607034, 40853.14218484400],
    ],
    # L5
    [
        [0.874, 3.14159265359, 0.00000000000],
        [0.117, 0.54643013000, 20426.57109242200],
        [0.118, 1.90548541922, 10213.28554621100],
        [0.002, 1.07734277826, 40853.14218484400],
        [0.002, 1.89460223529, 30639.85663863300],
    ],
]
"""This table contains Venus's periodic terms (all of them) from the planetary
theory VSOP87 for the heliocentric longitude at the equinox of date (taken from
the 'D' solution). In Meeus' book a shortened version can be found in
pages 418-420."""


VSOP87_B = [
    # B0
    [
        [5923638.472, 0.26702775813, 10213.28554621100],
        [40107.978, 1.14737178106, 20426.57109242200],
        [32814.918, 3.14159265359, 0.00000000000],
        [1011.392, 1.08946123021, 30639.85663863300],
        [149.458, 6.25390296069, 18073.70493865020],
        [137.788, 0.86020146523, 1577.34354244780],
        [129.973, 3.67152483651, 9437.76293488700],
        [119.507, 3.70468812804, 2352.86615377180],
        [107.971, 4.53903677647, 22003.91463486980],
        [92.029, 1.53954562706, 9153.90361602180],
        [52.982, 2.28138172277, 5507.55323866740],
        [45.617, 0.72319641722, 10239.58386601080],
        [38.855, 2.93437865147, 10186.98722641120],
        [43.491, 6.14015776699, 11790.62908865880],
        [41.700, 5.99126845246, 19896.88012732740],
        [39.644, 3.86842095901, 8635.94200376320],
        [39.175, 3.94960351174, 529.69096509460],
        [33.320, 4.83194909595, 14143.49524243060],
        [23.711, 2.90646621218, 10988.80815753500],
        [23.500, 2.00770618322, 13367.97263110660],
        [21.809, 2.69701424951, 19651.04848109800],
        [20.653, 0.98666685459, 775.52261132400],
        [16.976, 4.13711782135, 10021.83728009940],
        [17.835, 5.96268643102, 25934.12433108940],
        [14.949, 5.61075168206, 10404.73381232260],
        [18.579, 1.80529277514, 40853.14218484400],
        [15.407, 3.29563855296, 11015.10647733480],
        [12.936, 5.42651448496, 29580.47470844380],
        [11.962, 3.57604253827, 10742.97651130560],
        [11.827, 1.19070919600, 8624.21265092720],
        [11.466, 5.12780364967, 6283.07584999140],
        [9.484, 2.75167834335, 191.44826611160],
        [13.129, 5.70735942511, 9683.59458111640],
        [8.583, 0.43182249199, 9786.68735533500],
        [9.763, 0.14614896296, 20618.01935853360],
        [8.148, 1.30548515603, 15720.83878487840],
        [6.050, 6.26541665966, 11322.66409830440],
        [5.955, 4.92235372433, 1059.38193018920],
        [6.983, 3.44920932146, 17298.18232732620],
        [6.228, 1.13312070908, 29864.33402730900],
        [6.186, 4.92498052443, 19367.18916223280],
        [6.155, 2.42413946900, 4705.73230754360],
        [5.204, 3.42528906628, 9103.90699411760],
        [6.000, 3.57639095526, 3154.68708489560],
        [4.796, 3.86676184909, 7860.41939243920],
        [5.289, 4.99182712443, 7084.89678111520],
        [4.070, 5.58798144740, 12566.15169998280],
        [3.942, 5.68758787835, 10206.17199921020],
        [3.797, 3.89520601076, 10192.51015071860],
        [3.798, 6.06410995916, 10234.06094170340],
        [3.579, 0.73789669235, 4551.95349705880],
        [3.641, 2.61501257205, 15874.61759536320],
        [3.266, 0.97517223854, 23581.25817731760],
        [2.813, 0.29951755546, 9411.46461508720],
        [3.048, 2.51085146990, 33794.54372352860],
        [2.559, 4.58043833032, 801.82093112380],
        [2.462, 5.05790874754, 29050.78374334920],
        [2.593, 5.73113176751, 20213.27199698400],
        [2.625, 4.24272906574, 213.29909543800],
        [2.246, 0.82112963936, 28286.99048486120],
        [2.229, 2.22457598233, 10426.58464164900],
        [1.742, 1.48394229233, 7058.59846131540],
        [1.660, 5.42775825275, 32217.20018108080],
        [1.491, 4.64883377941, 1109.37855209340],
        [2.010, 0.75702888128, 9999.98645077300],
        [1.562, 3.93962080463, 37724.75341974820],
        [1.538, 2.17309577080, 21535.94964451540],
        [1.546, 4.70759186462, 14945.31617355440],
        [1.200, 1.48282382657, 9830.38901398780],
        [1.224, 5.55090394449, 5661.33204915220],
        [1.111, 1.20276209213, 9573.38825989700],
        [1.064, 1.98891375536, 26.29831979980],
        [1.041, 5.38535116069, 7.11354700080],
        [1.036, 1.16719443387, 8662.24032356300],
        [1.143, 3.20596958337, 3532.06069281140],
        [1.201, 0.81913312536, 8094.52168583260],
        [1.005, 2.38429892132, 27511.46787353720],
        [1.047, 4.56525030769, 20419.45754542119],
        [0.968, 6.18496721871, 25158.60171976540],
        [1.044, 1.98055689074, 10596.18207843420],
        [0.962, 0.48573513747, 23958.63178523340],
        [0.846, 0.01566400887, 3128.38876509580],
        [0.792, 5.39686899735, 24356.78078864160],
        [0.858, 5.34692750735, 41654.96311596780],
        [0.757, 6.25904553773, 20452.86941222180],
        [0.801, 4.62406152514, 9929.42622734580],
        [0.802, 5.37234892520, 10497.14486507620],
        [0.750, 3.85219782842, 21228.39202354580],
        [0.700, 1.98097957188, 3930.20969621960],
        [0.719, 6.11596800207, 10218.80847051840],
        [0.672, 6.23429601219, 14765.23904326980],
        [0.639, 5.37566437358, 1589.07289528380],
        [0.605, 2.42330391120, 10251.31321884680],
        [0.726, 6.16683781802, 18875.52586977400],
        [0.613, 5.99731180690, 4732.03062734340],
        [0.720, 3.84286345199, 10207.76262190360],
        [0.637, 6.17053891156, 10220.39909321180],
        [0.515, 1.03001478293, 22779.43724619380],
        [0.574, 0.43813688572, 17085.95866572220],
        [0.510, 1.41065159851, 9161.01716302260],
        [0.569, 3.34601425125, 3340.61242669980],
        [0.608, 1.25236241968, 10175.25787357520],
        [0.524, 2.39794248670, 26087.90314157420],
        [0.542, 1.34665646732, 29088.81141598500],
        [0.527, 4.01994270827, 18849.22754997420],
        [0.569, 1.65498800378, 39264.06928956020],
        [0.518, 4.96996115446, 30213.25844775700],
        [0.514, 5.78413007838, 12592.45001978260],
        [0.538, 4.56198493922, 10063.72234907640],
        [0.484, 4.18538027381, 14919.01785375460],
        [0.493, 4.79939382739, 9146.79006902100],
        [0.427, 3.76876868949, 11272.66747640020],
        [0.495, 0.49175293655, 45585.17281218740],
        [0.494, 3.74345863918, 31441.67756975680],
        [0.524, 0.97991794166, 30110.16567353840],
        [0.483, 1.87898057316, 51066.42773105500],
        [0.505, 3.70047474212, 20400.27277262220],
        [0.351, 4.34026574490, 10137.01947493540],
        [0.355, 5.56672554631, 18837.49819713819],
        [0.328, 3.78427378910, 6681.22485339960],
        [0.349, 4.20550749672, 20956.26205751660],
        [0.333, 4.44969281739, 28521.09277825460],
        [0.296, 2.83205515646, 17277.40693183380],
        [0.311, 2.57334132897, 20809.46762464520],
        [0.294, 0.75089224483, 3149.16416058820],
        [0.377, 3.98143308775, 21202.09370374600],
        [0.272, 5.56183082489, 16496.36139620240],
        [0.314, 0.02584607093, 13745.34623902240],
        [0.263, 0.55328410985, 36147.40987730040],
        [0.286, 5.16408902215, 426.59819087600],
        [0.279, 4.29871615943, 19999.97290154599],
        [0.280, 1.92925047377, 49515.38250840700],
        [0.265, 4.81168402147, 20235.12282631040],
        [0.273, 5.12740051559, 35371.88726597640],
        [0.306, 5.28903794869, 382.89653222320],
        [0.223, 2.50591724074, 26709.64694241340],
        [0.235, 5.96522395118, 10198.03307502600],
        [0.234, 3.52866583267, 10228.53801739600],
        [0.224, 6.24561979789, 7064.12138562280],
        [0.251, 2.84739274245, 33019.02111220460],
        [0.196, 1.50610393790, 31749.23519072640],
        [0.192, 1.69321442572, 13341.67431130680],
        [0.180, 6.19353087076, 39793.76025465480],
        [0.199, 1.16433321880, 22805.73556599360],
        [0.180, 3.72646417141, 1551.04522264800],
        [0.173, 3.35235705827, 53445.59220462660],
        [0.195, 1.51901264131, 43232.30665841560],
        [0.174, 2.84049662693, 9967.45389998160],
        [0.163, 4.29160537719, 36949.23080842420],
        [0.169, 0.37000676558, 10459.11719244040],
        [0.137, 5.61149803116, 10529.67741586760],
        [0.139, 0.87847805052, 16522.65971600220],
        [0.139, 4.12576475427, 36301.18868778519],
        [0.127, 5.14447758616, 5481.25491886760],
        [0.131, 3.11317801589, 9896.89367655440],
        [0.131, 0.89697384735, 3442.57494496540],
        [0.121, 1.32802112907, 38734.37832446560],
        [0.122, 1.59017183044, 10110.19277199240],
        [0.123, 2.33714216061, 10316.37832042960],
        [0.133, 2.90682399304, 9793.80090233580],
        [0.111, 2.52077634760, 13936.79450513400],
        [0.120, 0.36076947165, 536.80451209540],
        [0.115, 2.53355582059, 26735.94526221320],
        [0.108, 2.65839634325, 10232.95530711079],
        [0.108, 0.55230439694, 10193.61578531121],
        [0.138, 1.06919239240, 65236.22129328540],
        [0.101, 3.17012502017, 19317.19254032860],
        [0.127, 5.63110477712, 10288.06714477830],
        [0.127, 3.86278127025, 10138.50394764370],
        [0.137, 2.93350659460, 47162.51635463520],
        [0.095, 5.03917884334, 52175.80628314840],
        [0.094, 0.71308489207, 38500.27603107220],
        [0.092, 5.46204624886, 11764.33076885900],
        [0.096, 1.52914774412, 9690.70812811720],
        [0.101, 0.83318284426, 6489.77658728800],
        [0.115, 3.76443612245, 522.57741809380],
        [0.089, 2.53312656681, 10735.86296430480],
        [0.082, 0.85628515615, 2379.16447357160],
        [0.103, 5.22683237620, 103.09277421860],
        [0.090, 2.12423586627, 28313.28880466100],
        [0.090, 0.39668501735, 9580.50180689780],
        [0.074, 6.02680095550, 3723.50895892300],
        [0.081, 5.25045057985, 10419.47109464820],
        [0.080, 4.23724598221, 10007.09999777380],
        [0.091, 2.48874147947, 10846.06928552420],
        [0.085, 3.82784790321, 51868.24866217880],
        [0.081, 2.26235214191, 3903.91137641980],
        [0.097, 0.77295091600, 18307.80723204360],
        [0.094, 0.17063414792, 6872.67311951120],
        [0.080, 5.62254102739, 29999.95935231900],
        [0.068, 2.71762936670, 16983.99614745660],
        [0.066, 0.76731351736, 20.77539549240],
        [0.075, 0.36155638007, 39302.09696219600],
        [0.075, 2.27327165974, 8521.11987670860],
        [0.058, 2.14482855875, 8631.32619792800],
        [0.064, 5.83569051301, 2118.76386037840],
        [0.058, 2.98524209824, 19889.76658032659],
        [0.054, 1.78260668333, 40077.61957352000],
        [0.055, 4.70485939861, 639.89728631400],
        [0.060, 5.89661892920, 41962.52073693740],
        [0.066, 2.24746237999, 74.78159856730],
        [0.061, 3.40726181591, 27490.69247804480],
        [0.051, 3.07811180039, 24150.08005134500],
        [0.057, 2.30081371235, 20529.66386664059],
        [0.052, 2.37192464233, 29573.36116144300],
        [0.052, 4.76610409132, 57375.80190084620],
        [0.047, 1.61630288856, 30831.30490474460],
        [0.054, 5.89684197257, 19903.99367432819],
        [0.040, 5.32101847424, 42430.48572729180],
        [0.051, 5.29186795569, 29587.58825544460],
    ],
    # B1
    [
        [513347.602, 1.80364310797, 10213.28554621100],
        [4380.100, 3.38615711591, 20426.57109242200],
        [196.586, 2.53001197486, 30639.85663863300],
        [199.162, 0.00000000000, 0.00000000000],
        [14.031, 2.27087044687, 9437.76293488700],
        [12.958, 1.50735622957, 18073.70493865020],
        [11.941, 5.60462450426, 1577.34354244780],
        [10.324, 5.24224313355, 2352.86615377180],
        [9.294, 6.07545631303, 22003.91463486980],
        [7.441, 1.50257909439, 11790.62908865880],
        [8.031, 0.29371105198, 9153.90361602180],
        [7.514, 5.08081885990, 10186.98722641120],
        [4.669, 3.87801635015, 10239.58386601080],
        [4.399, 3.58872736593, 40853.14218484400],
        [3.975, 1.28397121206, 10404.73381232260],
        [4.657, 0.75073886819, 5507.55323866740],
        [3.783, 4.33004753984, 19651.04848109800],
        [3.390, 4.88976070903, 10988.80815753500],
        [3.555, 1.25927550356, 19896.88012732740],
        [3.479, 5.50797002160, 529.69096509460],
        [2.884, 0.08549582037, 14143.49524243060],
        [1.786, 0.37134513186, 13367.97263110660],
        [1.600, 1.68378002982, 20618.01935853360],
        [1.539, 1.21683853657, 25934.12433108940],
        [1.341, 2.90077139758, 15720.83878487840],
        [0.993, 1.74681248965, 11322.66409830440],
        [1.165, 6.13437155401, 7860.41939243920],
        [1.115, 0.66743690380, 29580.47470844380],
        [0.923, 2.25384969096, 10021.83728009940],
        [0.965, 1.36425494833, 9683.59458111640],
        [0.973, 0.39071758442, 6283.07584999140],
        [0.805, 0.53331923557, 8624.21265092720],
        [0.913, 0.76046003719, 8635.94200376320],
        [0.991, 0.55319879330, 19367.18916223280],
        [0.609, 2.62364470139, 23581.25817731760],
        [0.532, 5.10925676528, 9786.68735533500],
        [0.476, 6.17672999981, 11015.10647733480],
        [0.472, 1.69672629200, 17298.18232732620],
        [0.503, 2.65840772485, 29864.33402730900],
        [0.456, 5.01205315518, 10742.97651130560],
        [0.478, 3.94100005156, 775.52261132400],
        [0.477, 3.71554345922, 10596.18207843420],
        [0.347, 2.34551062680, 9411.46461508720],
        [0.458, 2.31894399069, 9999.98645077300],
        [0.374, 3.76878356974, 21228.39202354580],
        [0.440, 4.33400244581, 15874.61759536320],
        [0.349, 1.31468836511, 10234.06094170340],
        [0.310, 5.45422332781, 10192.51015071860],
        [0.346, 0.94242286364, 1059.38193018920],
        [0.308, 4.90145899142, 3930.20969621960],
        [0.331, 4.89498986674, 10206.17199921020],
        [0.269, 2.39650266204, 801.82093112380],
        [0.269, 0.00589873499, 9830.38901398780],
        [0.261, 3.48196147279, 7058.59846131540],
        [0.290, 0.10953964861, 29050.78374334920],
        [0.283, 6.12133736787, 20419.45754542119],
        [0.232, 3.07845850030, 12566.15169998280],
        [0.265, 4.02431894973, 33794.54372352860],
        [0.220, 2.37315851889, 4551.95349705880],
        [0.247, 3.07626728158, 28286.99048486120],
        [0.202, 3.56872121409, 21535.94964451540],
        [0.225, 5.76888896320, 213.29909543800],
        [0.217, 0.88382111135, 20213.27199698400],
        [0.172, 6.12653050186, 9161.01716302260],
        [0.195, 5.47240855400, 37724.75341974820],
        [0.153, 4.07656151671, 27511.46787353720],
        [0.174, 1.33676849359, 32217.20018108080],
        [0.157, 5.98474214437, 26.29831979980],
        [0.163, 5.45519134760, 10426.58464164900],
        [0.129, 2.08748660996, 3128.38876509580],
        [0.131, 1.51959002513, 10218.80847051840],
        [0.139, 4.42330401713, 10220.39909321180],
        [0.126, 2.62296638037, 22779.43724619380],
        [0.146, 4.69869606856, 25158.60171976540],
        [0.172, 6.13435208788, 18837.49819713819],
        [0.157, 5.44507403858, 4705.73230754360],
        [0.117, 6.18296175153, 20400.27277262220],
        [0.164, 3.30849473132, 51066.42773105500],
        [0.113, 3.64412860654, 7.11354700080],
        [0.109, 5.21220660788, 8662.24032356300],
        [0.133, 1.78047296245, 191.44826611160],
        [0.117, 0.14681677884, 9146.79006902100],
        [0.116, 0.61940521198, 41654.96311596780],
        [0.096, 1.49631428731, 7084.89678111520],
        [0.096, 1.21744230443, 10198.03307502600],
        [0.082, 1.45863866349, 10207.76262190360],
        [0.085, 6.04057728058, 21202.09370374600],
        [0.083, 0.19985600927, 14919.01785375460],
        [0.077, 5.50132310610, 5661.33204915220],
        [0.077, 2.00173927326, 10228.53801739600],
        [0.093, 1.85466268819, 45585.17281218740],
        [0.066, 3.25826124156, 1109.37855209340],
        [0.089, 0.64100435648, 3154.68708489560],
        [0.061, 3.80043027736, 11272.66747640020],
        [0.077, 1.85516358950, 3532.06069281140],
        [0.062, 0.81341290651, 382.89653222320],
        [0.072, 2.35312965005, 9103.90699411760],
        [0.053, 3.21969389511, 20452.86941222180],
        [0.067, 1.42090542131, 24356.78078864160],
        [0.056, 2.97733070198, 30110.16567353840],
        [0.051, 4.22406663447, 20809.46762464520],
        [0.058, 6.20761936031, 29088.81141598500],
        [0.061, 3.27309494322, 49515.38250840700],
        [0.046, 5.49443476235, 31441.67756975680],
        [0.050, 4.16651052942, 13341.67431130680],
        [0.047, 1.25473247769, 33019.02111220460],
        [0.047, 2.03402044389, 23958.63178523340],
        [0.036, 5.24409311105, 3149.16416058820],
        [0.038, 4.15337829669, 18849.22754997420],
        [0.042, 0.43005959574, 1589.07289528380],
        [0.041, 1.21289342964, 12592.45001978260],
        [0.038, 5.91928287144, 28521.09277825460],
        [0.033, 3.98241699279, 4732.03062734340],
        [0.035, 2.24417218267, 16496.36139620240],
        [0.040, 6.13293942728, 26087.90314157420],
        [0.044, 1.78123294860, 426.59819087600],
        [0.041, 3.16744909855, 39264.06928956020],
        [0.033, 4.96183427323, 536.80451209540],
        [0.034, 0.12963030501, 30213.25844775700],
        [0.036, 5.41167321573, 522.57741809380],
        [0.027, 4.44250239485, 17277.40693183380],
        [0.034, 5.94541303751, 9929.42622734580],
        [0.033, 0.40689057274, 10497.14486507620],
        [0.023, 2.59067946967, 10175.25787357520],
        [0.022, 0.69625017371, 19999.97290154599],
        [0.023, 3.76162101633, 10251.31321884680],
        [0.023, 0.62711494266, 35371.88726597640],
        [0.022, 4.64142978776, 19889.76658032659],
        [0.020, 4.01315480107, 26709.64694241340],
        [0.020, 4.03344400680, 29573.36116144300],
        [0.023, 0.90416640595, 8094.52168583260],
        [0.022, 1.92092469688, 17085.95866572220],
        [0.019, 5.04938942644, 6681.22485339960],
    ],
    # B2
    [
        [22377.665, 3.38509143877, 10213.28554621100],
        [281.739, 0.00000000000, 0.00000000000],
        [173.164, 5.25563766915, 20426.57109242200],
        [26.945, 3.87040891568, 30639.85663863300],
        [1.174, 0.09768632072, 10186.98722641120],
        [0.685, 3.19139067811, 11790.62908865880],
        [0.788, 4.36515965295, 10239.58386601080],
        [0.592, 5.22270440328, 40853.14218484400],
        [0.515, 6.12821215207, 10988.80815753500],
        [0.538, 0.57550272342, 2352.86615377180],
        [0.540, 3.11657836329, 18073.70493865020],
        [0.454, 2.79306867629, 10404.73381232260],
        [0.374, 6.10468482446, 9437.76293488700],
        [0.431, 4.00778431184, 1577.34354244780],
        [0.360, 6.01747842320, 19651.04848109800],
        [0.375, 1.31319959789, 22003.91463486980],
        [0.354, 5.12509281266, 9153.90361602180],
        [0.150, 4.58623687118, 15720.83878487840],
        [0.164, 5.41790158607, 5507.55323866740],
        [0.159, 2.78191550878, 19896.88012732740],
        [0.157, 0.65774905071, 529.69096509460],
        [0.155, 2.54824315372, 9683.59458111640],
        [0.109, 2.01866665583, 14143.49524243060],
        [0.106, 2.28289033017, 6283.07584999140],
        [0.115, 3.23636374193, 20618.01935853360],
        [0.128, 5.32400510939, 13367.97263110660],
        [0.087, 3.28265082435, 11322.66409830440],
        [0.090, 5.23585072275, 10596.18207843420],
        [0.055, 4.82369879741, 7058.59846131540],
        [0.044, 0.58444963462, 10206.17199921020],
        [0.044, 2.34401612969, 19367.18916223280],
        [0.038, 4.55053233088, 9999.98645077300],
        [0.039, 5.84340580032, 10220.39909321180],
        [0.036, 4.41006216127, 51066.42773105500],
        [0.039, 3.14348236386, 9411.46461508720],
        [0.033, 4.55748660340, 10742.97651130560],
        [0.037, 2.79630938717, 25934.12433108940],
        [0.034, 0.55287110072, 11015.10647733480],
        [0.034, 2.25809144959, 29580.47470844380],
        [0.038, 1.88638747393, 801.82093112380],
        [0.034, 1.22706917271, 10021.83728009940],
        [0.027, 4.83867137637, 9830.38901398780],
        [0.027, 4.31140179350, 23581.25817731760],
        [0.027, 2.17187621336, 8635.94200376320],
        [0.020, 5.66581696952, 21228.39202354580],
        [0.024, 2.17208107850, 18849.22754997420],
        [0.020, 5.29318634138, 775.52261132400],
        [0.019, 2.73486845601, 3128.38876509580],
        [0.013, 3.40362915274, 1059.38193018920],
        [0.014, 0.05074160195, 7860.41939243920],
        [0.014, 5.43035907265, 26.29831979980],
        [0.012, 3.24834347355, 9103.90699411760],
        [0.013, 5.04826725887, 7.11354700080],
        [0.015, 1.42027402522, 29050.78374334920],
        [0.010, 4.98138067490, 10426.58464164900],
        [0.011, 0.85773045784, 17298.18232732620],
        [0.011, 4.23048200054, 29864.33402730900],
        [0.010, 0.26447399758, 3930.20969621960],
        [0.011, 1.46728576671, 20419.45754542119],
    ],
    # B3
    [
        [646.671, 4.99166565277, 10213.28554621100],
        [19.952, 3.14159265359, 0.00000000000],
        [5.540, 0.77376923951, 20426.57109242200],
        [2.526, 5.44493763020, 30639.85663863300],
        [0.079, 1.51447613604, 10186.98722641120],
        [0.056, 0.63647808442, 40853.14218484400],
        [0.058, 5.70731176550, 10239.58386601080],
        [0.031, 4.72523061067, 11790.62908865880],
        [0.026, 1.02068113372, 10988.80815753500],
        [0.025, 5.60599130442, 9437.76293488700],
        [0.017, 2.05293621864, 2352.86615377180],
        [0.011, 4.33056892256, 10404.73381232260],
        [0.009, 1.36283915068, 19651.04848109800],
        [0.007, 4.69592781899, 18073.70493865020],
        [0.006, 2.97926526705, 22003.91463486980],
    ],
    # B4
    [
        [14.102, 0.31537190181, 10213.28554621100],
        [0.190, 2.35466404492, 20426.57109242200],
        [0.164, 0.74476215141, 30639.85663863300],
        [0.214, 3.14159265359, 0.00000000000],
        [0.004, 2.34190883009, 40853.14218484400],
    ],
    # B5
    [
        [0.239, 2.05201727566, 10213.28554621100],
        [0.039, 0.00000000000, 0.00000000000],
        [0.011, 3.82500275251, 20426.57109242200],
        [0.009, 2.32953116868, 30639.85663863300],
    ],
]
"""This table contains Venus's periodic terms (all of them) from the planetary
theory VSOP87 for the heliocentric latitude at the equinox of date (taken from
the 'D' solution). In Meeus' book a shortened version can be found in
page 420."""


VSOP87_R = [
    # R0
    [
        [72334820.905, 0.00000000000, 0.00000000000],
        [489824.185, 4.02151832268, 10213.28554621100],
        [1658.058, 4.90206728012, 20426.57109242200],
        [1632.093, 2.84548851892, 7860.41939243920],
        [1378.048, 1.12846590600, 11790.62908865880],
        [498.399, 2.58682187717, 9683.59458111640],
        [373.958, 1.42314837063, 3930.20969621960],
        [263.616, 5.52938185920, 9437.76293488700],
        [237.455, 2.55135903978, 15720.83878487840],
        [221.983, 2.01346776772, 19367.18916223280],
        [119.467, 3.01975365264, 10404.73381232260],
        [125.896, 2.72769833559, 1577.34354244780],
        [76.178, 1.59577224486, 9153.90361602180],
        [85.336, 3.98607953754, 19651.04848109800],
        [74.347, 4.11957854039, 5507.55323866740],
        [41.904, 1.64273363458, 18837.49819713819],
        [42.493, 3.81864530735, 13367.97263110660],
        [39.430, 5.39019422358, 23581.25817731760],
        [29.042, 5.67739528728, 5661.33204915220],
        [27.555, 5.72392407794, 775.52261132400],
        [27.283, 4.82151812709, 11015.10647733480],
        [31.274, 2.31806719544, 9999.98645077300],
        [19.700, 4.96157560245, 11322.66409830440],
        [19.809, 0.53189326492, 27511.46787353720],
        [13.567, 3.75530870628, 18073.70493865020],
        [12.921, 1.13381083556, 10206.17199921020],
        [16.215, 0.56453834290, 529.69096509460],
        [11.821, 5.09025877427, 3154.68708489560],
        [11.728, 0.23432298744, 7084.89678111520],
        [13.079, 5.24353197586, 17298.18232732620],
        [13.180, 3.37207825651, 13745.34623902240],
        [9.097, 3.07004895769, 1109.37855209340],
        [10.818, 2.45024712908, 10239.58386601080],
        [11.438, 4.56838894696, 29050.78374334920],
        [8.377, 5.78327612352, 30639.85663863300],
        [8.193, 1.95023111860, 22003.91463486980],
        [9.308, 1.61615909286, 2352.86615377180],
        [10.652, 1.95528396140, 31441.67756975680],
        [10.357, 1.20234990061, 15874.61759536320],
        [9.585, 1.46639856228, 19999.97290154599],
        [6.506, 2.17390732263, 14143.49524243060],
        [7.562, 1.13789564977, 8624.21265092720],
        [6.434, 0.84419623033, 6283.07584999140],
        [5.898, 0.01093731110, 8635.94200376320],
        [5.632, 3.94956548631, 12566.15169998280],
        [5.523, 1.27394296557, 18307.80723204360],
        [4.488, 2.47835729057, 191.44826611160],
        [4.529, 4.73027770400, 19896.88012732740],
        [6.193, 3.25881250939, 6872.67311951120],
        [6.070, 0.35337419942, 21228.39202354580],
        [4.315, 2.59737099519, 4551.95349705880],
        [6.005, 3.37874723475, 35371.88726597640],
        [3.852, 1.01162850357, 9786.68735533500],
        [4.033, 0.00050855580, 801.82093112380],
        [3.920, 5.56542869407, 10596.18207843420],
        [2.709, 5.80195530112, 7064.12138562280],
        [3.216, 0.39767254848, 10186.98722641120],
        [3.089, 6.26174762876, 14945.31617355440],
        [2.982, 4.21196716354, 28521.09277825460],
        [3.284, 0.70709821006, 10742.97651130560],
        [3.484, 4.79878191875, 39302.09696219600],
        [3.172, 1.80518954174, 25158.60171976540],
        [2.463, 0.68708153678, 10988.80815753500],
        [2.374, 3.77948685343, 21535.94964451540],
        [2.198, 2.82996372521, 8662.24032356300],
        [1.958, 5.41763804167, 16496.36139620240],
        [1.876, 2.63426768393, 29580.47470844380],
        [1.902, 2.85782199133, 3532.06069281140],
        [1.706, 3.67573010379, 26.29831979980],
        [1.817, 0.41611036449, 4705.73230754360],
        [1.858, 1.50368318296, 10021.83728009940],
        [2.087, 6.22112874639, 43232.30665841560],
        [1.950, 2.21447019683, 19786.67380610799],
        [1.497, 0.00134773824, 17277.40693183380],
        [1.819, 3.23144993268, 29088.81141598500],
        [1.423, 5.85979618707, 9676.48103411560],
        [1.223, 5.55818994329, 6770.71060124560],
        [1.140, 5.92088900094, 13936.79450513400],
        [1.484, 2.47665429253, 31749.23519072640],
        [1.185, 1.42087628351, 4732.03062734340],
        [1.323, 2.48821075422, 9690.70812811720],
        [1.249, 1.88323673734, 19374.30270923360],
        [1.270, 5.24647873116, 19360.07561523199],
        [1.402, 5.17536780118, 10316.37832042960],
        [1.042, 3.05454698508, 25934.12433108940],
        [1.174, 1.42913732999, 18875.52586977400],
        [1.278, 1.35747287297, 47162.51635463520],
        [0.917, 6.26337648765, 20618.01935853360],
        [0.905, 1.12740203561, 12592.45001978260],
        [1.093, 4.64451720605, 33019.02111220460],
        [1.014, 1.09259406433, 1059.38193018920],
        [0.783, 2.02118183873, 24356.78078864160],
        [0.779, 0.41585274010, 3340.61242669980],
        [0.700, 1.14936815714, 16983.99614745660],
        [0.878, 0.87852464964, 38734.37832446560],
        [0.623, 0.89976912165, 17778.11626694899],
        [0.608, 1.58476225197, 9573.38825989700],
        [0.800, 3.94213003073, 10138.50394764370],
        [0.760, 1.31851313748, 9967.45389998160],
        [0.802, 2.78173370208, 51092.72605085480],
        [0.664, 4.45864682400, 3128.38876509580],
        [0.674, 5.11214939998, 382.89653222320],
        [0.530, 0.85392938403, 10234.06094170340],
        [0.509, 3.56809374595, 28286.99048486120],
        [0.600, 4.25927726907, 41962.52073693740],
        [0.601, 5.78144137895, 213.29909543800],
        [0.595, 2.83045104588, 22805.73556599360],
        [0.673, 6.06079908421, 36949.23080842420],
        [0.535, 5.85422519711, 9103.90699411760],
        [0.544, 5.44806074800, 3723.50895892300],
        [0.492, 3.83802404893, 27991.40181316000],
        [0.635, 0.76494024849, 8094.52168583260],
        [0.434, 6.22214487735, 27197.28169366760],
        [0.459, 3.55062885479, 20213.27199698400],
        [0.398, 6.16269975784, 10426.58464164900],
        [0.378, 2.41665947591, 18844.61174413899],
        [0.421, 4.86552697954, 9146.79006902100],
        [0.500, 4.20351458644, 55022.93574707440],
        [0.404, 4.95834410782, 37410.56723987860],
        [0.402, 2.97963246945, 10220.39909321180],
        [0.464, 2.59869499733, 18734.40542291960],
        [0.352, 0.08963076359, 10103.07922499160],
        [0.348, 4.90260339364, 18830.38465013739],
        [0.338, 3.22520096478, 24150.08005134500],
        [0.375, 6.17532088136, 26087.90314157420],
        [0.425, 1.20052578280, 40879.44050464380],
        [0.408, 3.12833060705, 9050.81084180320],
        [0.385, 1.94284690176, 283.85931886520],
        [0.337, 4.87838699272, 12432.04265039780],
        [0.326, 4.27369741426, 26735.94526221320],
        [0.309, 0.50597475053, 38204.68735937100],
        [0.329, 3.88430599153, 29864.33402730900],
        [0.313, 1.36138752543, 10192.51015071860],
        [0.347, 3.58439807209, 27490.69247804480],
        [0.251, 3.78618457047, 10063.72234907640],
        [0.244, 3.83523342668, 9411.46461508720],
        [0.281, 4.50895206233, 32217.20018108080],
        [0.237, 0.87748812245, 6681.22485339960],
        [0.315, 5.62657778233, 58953.14544329400],
        [0.311, 4.15626121491, 10175.15251057320],
        [0.247, 2.53637594113, 16522.65971600220],
        [0.219, 5.08729383251, 7058.59846131540],
        [0.291, 3.72567217056, 29999.95935231900],
        [0.267, 2.97685503991, 19573.37471066999],
        [0.280, 3.70200084294, 47623.85278608960],
        [0.239, 3.94545782067, 9580.50180689780],
        [0.246, 2.18244883930, 9161.01716302260],
        [0.253, 2.69506547016, 3442.57494496540],
        [0.265, 2.62811801237, 44809.65020086340],
        [0.194, 4.78926136175, 33794.54372352860],
        [0.187, 3.65620881095, 20452.86941222180],
        [0.224, 2.43601863127, 9992.87290377220],
        [0.193, 2.55112161845, 2379.16447357160],
        [0.201, 1.90356905733, 1551.04522264800],
        [0.176, 4.29837616553, 10137.01947493540],
        [0.184, 6.16061560223, 36147.40987730040],
        [0.175, 2.71984797040, 20809.46762464520],
        [0.186, 2.55098927966, 14919.01785375460],
        [0.161, 4.13272567123, 23958.63178523340],
        [0.221, 4.83552377614, 20277.00789528740],
        [0.160, 1.81472642729, 10787.63034454580],
        [0.199, 5.74259798330, 30666.15495843280],
        [0.160, 4.46270605493, 18947.70451835760],
        [0.187, 2.98688597588, 2218.75710418680],
        [0.189, 5.34607810282, 10007.09999777380],
        [0.198, 0.77846666692, 62883.35513951360],
        [0.144, 5.00261963924, 9264.10993724120],
        [0.171, 2.05212624568, 7255.56965173440],
        [0.188, 4.08173534559, 48739.85989708300],
        [0.146, 3.94191715702, 6309.37416979120],
        [0.146, 5.06313558118, 39264.06928956020],
        [0.135, 5.93689169614, 37724.75341974820],
        [0.139, 2.81266025896, 20.77539549240],
        [0.177, 5.16224804657, 9835.91193829520],
        [0.119, 1.37254262864, 40077.61957352000],
        [0.120, 0.21443767468, 31022.75317085620],
        [0.128, 2.92458887798, 7.11354700080],
        [0.150, 5.73646272556, 632.78373931320],
        [0.106, 0.62224833817, 11272.66747640020],
        [0.114, 2.63301326520, 17468.85519794540],
        [0.123, 6.22518843711, 53285.18483524180],
        [0.107, 1.17258978900, 43071.89928903080],
        [0.103, 1.09613781581, 41654.96311596780],
        [0.109, 2.01412667085, 20419.45754542119],
        [0.102, 4.23406964348, 10251.31321884680],
        [0.116, 1.27731728606, 10199.05845220940],
        [0.103, 5.25887538465, 9830.38901398780],
        [0.112, 2.24436894064, 18204.71445782499],
        [0.111, 2.23547857955, 8521.11987670860],
        [0.118, 0.23754207200, 10497.14486507620],
        [0.123, 0.88054816668, 34596.36465465240],
        [0.102, 4.39438646620, 18300.69368504279],
        [0.131, 6.01711652115, 9367.20271145980],
        [0.100, 5.00532389609, 10175.25787357520],
        [0.107, 0.41270197502, 40853.14218484400],
        [0.132, 5.45008342761, 11506.76976979360],
        [0.098, 1.07722950958, 13553.89797291080],
        [0.094, 2.91720097590, 44007.82926973960],
        [0.097, 1.04004223634, 68050.42387851159],
        [0.127, 2.20215372683, 66813.56483573320],
        [0.111, 1.57823839032, 29043.67019634839],
        [0.118, 2.33268176890, 18314.92077904440],
        [0.090, 2.42353056125, 32858.61374281979],
        [0.109, 3.82796787296, 19470.28193645139],
        [0.111, 4.47666957576, 29057.89729034999],
        [0.101, 3.41528493660, 19264.09638801420],
        [0.092, 3.66289799512, 22645.32819660879],
        [0.094, 6.07530805791, 10846.06928552420],
        [0.114, 4.02718653431, 7576.56007357400],
        [0.087, 6.01842459303, 17085.95866572220],
        [0.109, 5.46886607309, 52670.06959330260],
        [0.107, 0.54805946713, 34363.36559755600],
        [0.108, 5.44460610707, 19050.79729257620],
        [0.076, 6.15177368654, 27682.14074415640],
        [0.107, 4.80525404063, 8144.27871130440],
        [0.073, 1.60549217847, 20956.26205751660],
        [0.097, 5.13542051130, 22779.43724619380],
        [0.068, 2.31300447144, 8631.32619792800],
        [0.091, 4.28652743953, 10110.19277199240],
        [0.093, 5.27290609264, 522.57741809380],
        [0.071, 3.65565961690, 11764.33076885900],
        [0.089, 1.79712963206, 45585.17281218740],
        [0.067, 2.25900071584, 9360.08916445900],
        [0.085, 0.67062144972, 56600.27928952220],
        [0.080, 1.58278081077, 19992.85935454519],
        [0.065, 6.23472325597, 10419.47109464820],
        [0.064, 0.53356325917, 17248.42530185440],
        [0.085, 4.52011215904, 29786.66025688100],
        [0.068, 4.48235266554, 10632.77019008620],
        [0.064, 4.33495700921, 47938.03896595920],
        [0.071, 3.03858484137, 11787.10597030980],
        [0.087, 4.81823063172, 2107.03450754240],
        [0.070, 2.35648061034, 11794.15220700780],
        [0.080, 2.33248094128, 38526.57435087200],
        [0.070, 3.70454061100, 8734.41897214660],
        [0.077, 4.49569185467, 20007.08644854680],
        [0.072, 1.19410424468, 10217.21769947410],
        [0.068, 2.01841060183, 14128.24277124560],
        [0.064, 5.39293951654, 7880.08915333899],
        [0.066, 3.20467071127, 14765.23904326980],
        [0.080, 3.41620457770, 48417.97290558199],
        [0.080, 3.39651161571, 245.83164622940],
        [0.066, 5.85414440204, 9793.80090233580],
        [0.082, 3.62592908644, 70743.77453195279],
        [0.058, 4.95174942212, 30110.16567353840],
        [0.079, 6.24161471033, 6037.24420376200],
        [0.069, 5.50183658445, 19793.78735310880],
        [0.056, 1.24148350566, 10207.76262190360],
        [0.070, 2.45123308846, 10218.80847051840],
        [0.064, 5.53983104501, 10735.86296430480],
        [0.054, 3.62259713240, 27461.71084806540],
        [0.073, 1.75882480924, 1589.07289528380],
        [0.075, 3.38244819846, 4214.06901508480],
        [0.054, 0.64971567468, 9929.42622734580],
        [0.054, 3.40959637230, 18418.01355326299],
        [0.056, 3.65815006538, 14169.79356223040],
        [0.056, 0.71243223808, 9896.89367655440],
        [0.052, 1.33348131940, 20400.27277262220],
        [0.067, 3.12806595400, 5481.25491886760],
        [0.058, 0.54482893546, 28313.28880466100],
        [0.054, 0.15603935681, 19580.48825767080],
        [0.051, 3.37515473510, 9256.99639024040],
        [0.063, 3.38848970950, 49515.38250840700],
        [0.069, 4.90917651401, 63498.47038145279],
        [0.057, 5.07437742030, 18521.10632748160],
        [0.050, 1.59156823654, 18631.31264870099],
        [0.054, 6.25816208666, 37674.99639427640],
        [0.057, 5.48065460919, 24383.07910844140],
        [0.045, 1.10466490660, 10408.25693067160],
        [0.051, 3.61196470313, 426.59819087600],
        [0.057, 2.09567711267, 60530.48898574180],
        [0.060, 5.94659889997, 13897.66359620120],
        [0.051, 5.47238517720, 57837.13833230060],
        [0.051, 2.32438478428, 19779.56025910719],
        [0.052, 3.23766328818, 18940.59097135679],
        [0.043, 5.74921510909, 51868.24866217880],
        [0.048, 1.12206254877, 9779.57380833420],
        [0.058, 3.08646083897, 12074.48840752400],
        [0.046, 4.07536026888, 7863.94251078820],
        [0.045, 4.75746520642, 7856.89627409019],
        [0.054, 4.43528236634, 8617.09910392640],
        [0.050, 3.70569982975, 42456.78404709160],
        [0.044, 1.29248911155, 69166.43098950500],
        [0.046, 0.41229872114, 7564.83072073800],
        [0.044, 6.17937388307, 13341.67431130680],
        [0.053, 4.71388531889, 53445.59220462660],
        [0.041, 3.48003037828, 37895.42629036740],
        [0.040, 1.23305546260, 10228.53801739600],
        [0.053, 5.04979874661, 74673.98422817240],
        [0.039, 1.36646013032, 21202.09370374600],
        [0.039, 2.15376025201, 8947.71806758460],
        [0.041, 6.17532984460, 65236.22129328540],
        [0.052, 1.29052331493, 90394.82301305079],
        [0.039, 0.70253732683, 18093.37469954999],
        [0.052, 1.18164377451, 10211.80107350270],
        [0.047, 1.78672260794, 10401.21069397360],
        [0.040, 3.66961416802, 10198.03307502600],
        [0.051, 2.71698589018, 94325.03270927040],
        [0.036, 1.25091711620, 10323.49186743040],
        [0.049, 1.21335959420, 9721.62225375220],
        [0.042, 6.05968230173, 105460.99111839019],
        [0.046, 5.06978748275, 20350.30502114640],
        [0.040, 1.97645050921, 32243.49850088060],
        [0.036, 4.96702216961, 36301.18868778519],
        [0.037, 5.29642935562, 38.02767263580],
        [0.039, 0.52064327313, 26709.64694241340],
        [0.035, 2.34112124655, 58946.51688439399],
        [0.034, 1.82989750626, 17675.02349273040],
        [0.034, 0.76493664110, 55798.45835839840],
        [0.035, 1.09353675147, 69159.80243060499],
        [0.031, 5.59148330297, 10639.88373708700],
        [0.032, 3.32960781870, 71519.29714327680],
        [0.031, 5.98191446392, 24341.52831745660],
        [0.031, 0.68615213145, 10202.23984594710],
        [0.030, 4.42039942947, 10459.11719244040],
        [0.029, 1.30367701539, 20103.06567576459],
        [0.031, 4.51793347997, 2957.71589447660],
        [0.035, 4.05634321290, 19903.99367432819],
        [0.030, 1.32113757427, 574.34479833480],
        [0.029, 3.36506645849, 10288.06714477830],
        [0.029, 1.40019042576, 9988.94075050910],
        [0.032, 0.21932095318, 24978.52458948080],
        [0.034, 5.22945947227, 8673.96967639900],
        [0.039, 4.50883171158, 16004.69810374360],
        [0.028, 2.32945945641, 11392.48008525060],
        [0.034, 3.92498967835, 536.80451209540],
        [0.032, 5.46972716255, 64607.84893354619],
        [0.028, 2.38858990128, 20235.12282631040],
        [0.030, 3.34585843979, 39793.76025465480],
        [0.026, 5.36096904409, 1478.86657406440],
    ],
    # R1
    [
        [34551.039, 0.89198710598, 10213.28554621100],
        [234.203, 1.77224942714, 20426.57109242200],
        [233.998, 3.14159265359, 0.00000000000],
        [23.864, 1.11274502648, 9437.76293488700],
        [10.568, 4.59168210921, 1577.34354244780],
        [9.124, 4.53540907003, 10404.73381232260],
        [6.599, 5.97703999838, 5507.55323866740],
        [4.667, 3.87683960551, 9153.90361602180],
        [3.840, 5.66196924375, 13367.97263110660],
        [2.666, 2.82413291285, 10206.17199921020],
        [2.194, 2.05314419626, 775.52261132400],
        [2.094, 2.55137285015, 18837.49819713819],
        [1.782, 2.64808558644, 30639.85663863300],
        [1.845, 1.87612936641, 11015.10647733480],
        [1.303, 0.20613045603, 11322.66409830440],
        [1.169, 0.79431893441, 17298.18232732620],
        [1.001, 6.16555101536, 10239.58386601080],
        [0.915, 4.59854496966, 1109.37855209340],
        [0.884, 0.66706834422, 18073.70493865020],
        [0.849, 5.58641571940, 12566.15169998280],
        [1.071, 4.94792017474, 6283.07584999140],
        [0.887, 2.47785193216, 3154.68708489560],
        [0.904, 0.81413053841, 10596.18207843420],
        [0.818, 0.90016838097, 5661.33204915220],
        [0.845, 5.48504338112, 529.69096509460],
        [0.824, 3.74837629121, 7084.89678111520],
        [0.652, 5.07444932607, 22003.91463486980],
        [0.847, 0.44119876869, 8635.94200376320],
        [0.638, 4.10125791268, 191.44826611160],
        [0.615, 3.14417599741, 10186.98722641120],
        [0.527, 5.86792949279, 2352.86615377180],
        [0.520, 5.33201358267, 14143.49524243060],
        [0.576, 2.25212731258, 21228.39202354580],
        [0.662, 2.86880467345, 8624.21265092720],
        [0.554, 2.17186191243, 18307.80723204360],
        [0.515, 4.34331395104, 9786.68735533500],
        [0.501, 5.56479589366, 10742.97651130560],
        [0.426, 1.02161443120, 7064.12138562280],
        [0.418, 1.26803034691, 9676.48103411560],
        [0.391, 0.78974645621, 9690.70812811720],
        [0.334, 3.18175822557, 10988.80815753500],
        [0.375, 0.66142254036, 19360.07561523199],
        [0.364, 0.19369831864, 19374.30270923360],
        [0.313, 1.09734397626, 4551.95349705880],
        [0.330, 0.58817502306, 16496.36139620240],
        [0.339, 5.76768761396, 10021.83728009940],
        [0.291, 3.65846764668, 25158.60171976540],
        [0.223, 4.33581625553, 19786.67380610799],
        [0.266, 3.57408827667, 801.82093112380],
        [0.274, 5.73346687248, 11790.62908865880],
        [0.275, 5.65814317085, 19896.88012732740],
        [0.212, 4.27038489878, 4705.73230754360],
        [0.230, 6.13406345590, 1059.38193018920],
        [0.204, 4.87348390351, 7860.41939243920],
        [0.241, 1.13551531894, 26.29831979980],
        [0.206, 0.31907973682, 382.89653222320],
        [0.216, 2.54741101724, 19651.04848109800],
        [0.212, 3.15264941106, 14945.31617355440],
        [0.163, 1.13604744392, 13936.79450513400],
        [0.151, 5.11341268743, 28521.09277825460],
        [0.151, 0.81278755582, 6770.71060124560],
        [0.150, 5.02227334847, 29088.81141598500],
        [0.146, 1.37568138685, 10220.39909321180],
        [0.127, 4.49298610074, 3532.06069281140],
        [0.121, 6.26589208179, 29580.47470844380],
        [0.147, 6.16092774714, 8662.24032356300],
        [0.114, 0.00114012635, 25934.12433108940],
        [0.115, 3.56897715344, 24356.78078864160],
        [0.124, 0.67547060274, 3723.50895892300],
        [0.145, 0.36415036222, 9146.79006902100],
        [0.104, 4.27865011376, 9573.38825989700],
        [0.136, 5.09581116181, 19367.18916223280],
        [0.102, 1.53637788668, 17277.40693183380],
        [0.117, 0.57543238496, 9999.98645077300],
        [0.092, 0.22936081655, 18830.38465013739],
        [0.112, 4.04771058036, 9103.90699411760],
        [0.098, 3.78447692407, 213.29909543800],
        [0.085, 5.84471458481, 10234.06094170340],
        [0.079, 0.64440357793, 18844.61174413899],
        [0.084, 0.56950139213, 9683.59458111640],
        [0.107, 1.77067111589, 17778.11626694899],
        [0.081, 6.19048382717, 20618.01935853360],
        [0.087, 0.15771136594, 33019.02111220460],
        [0.082, 4.80683817059, 3930.20969621960],
        [0.086, 2.21505615071, 8094.52168583260],
        [0.064, 2.69215119482, 16983.99614745660],
        [0.069, 0.83385751986, 3128.38876509580],
        [0.081, 4.88025042367, 4732.03062734340],
        [0.059, 3.34348033725, 10787.63034454580],
        [0.061, 0.04044699966, 9161.01716302260],
        [0.064, 4.13127333938, 9992.87290377220],
        [0.060, 6.24603986632, 32217.20018108080],
        [0.054, 3.38449893196, 10426.58464164900],
        [0.054, 5.15939119644, 28286.99048486120],
        [0.063, 4.32339245083, 12592.45001978260],
        [0.060, 4.48753846170, 18875.52586977400],
        [0.057, 3.64912085313, 10007.09999777380],
        [0.049, 5.10267262491, 19573.37471066999],
        [0.047, 5.79444960738, 68050.42387851159],
        [0.052, 3.56658420552, 7255.56965173440],
        [0.050, 1.61783309819, 36949.23080842420],
        [0.053, 2.64370544855, 15874.61759536320],
        [0.040, 3.93466530964, 20419.45754542119],
        [0.051, 0.79154899901, 23581.25817731760],
        [0.038, 1.77428239418, 10103.07922499160],
        [0.049, 1.12423644455, 3442.57494496540],
        [0.040, 5.22874487975, 21535.94964451540],
        [0.038, 1.12473430132, 7.11354700080],
        [0.038, 0.11510547453, 11272.66747640020],
        [0.036, 2.02476324983, 7058.59846131540],
        [0.047, 0.05589432390, 12432.04265039780],
        [0.034, 3.45481114998, 9830.38901398780],
        [0.045, 4.59817214088, 10192.51015071860],
        [0.037, 4.93959675364, 3340.61242669980],
        [0.044, 0.70533027806, 20213.27199698400],
        [0.034, 2.16487642765, 64460.69868196140],
        [0.031, 1.57612397319, 36147.40987730040],
        [0.028, 2.56454760402, 94138.32702008578],
        [0.033, 1.08907268562, 29864.33402730900],
        [0.029, 0.59718407064, 59728.66805461800],
        [0.031, 3.04423979263, 40879.44050464380],
        [0.035, 0.32247158762, 1589.07289528380],
        [0.031, 3.27727318906, 19992.85935454519],
        [0.027, 5.83705748551, 17085.95866572220],
        [0.032, 2.64260788260, 41962.52073693740],
        [0.028, 4.90613317287, 29050.78374334920],
        [0.025, 4.55050389739, 14919.01785375460],
        [0.028, 3.58851614957, 40853.14218484400],
        [0.029, 2.79705093386, 20007.08644854680],
        [0.033, 0.93862065616, 15720.83878487840],
        [0.024, 2.74970637101, 18947.70451835760],
        [0.024, 4.38966861409, 46386.99374331120],
        [0.024, 0.73361964525, 9411.46461508720],
        [0.028, 4.19559784013, 37674.99639427640],
        [0.023, 1.00023735538, 22779.43724619380],
        [0.026, 0.46990555736, 13745.34623902240],
        [0.028, 4.65181292126, 1551.04522264800],
        [0.025, 4.18690270765, 44007.82926973960],
        [0.022, 0.98102807789, 426.59819087600],
        [0.030, 1.24986033487, 27461.71084806540],
        [0.027, 3.94986823486, 17468.85519794540],
        [0.021, 6.09897508157, 18300.69368504279],
        [0.025, 4.75875623888, 27991.40181316000],
        [0.022, 2.95281481673, 40077.61957352000],
        [0.028, 6.12038264955, 38500.27603107220],
        [0.022, 4.11184201321, 19779.56025910719],
        [0.027, 3.72446446080, 19793.78735310880],
        [0.020, 4.27086627368, 31441.67756975680],
        [0.022, 4.99040169444, 31022.75317085620],
        [0.023, 1.33505132122, 65236.22129328540],
        [0.021, 4.46897353468, 53285.18483524180],
        [0.020, 4.15140915983, 2218.75710418680],
        [0.025, 2.18447182965, 27511.46787353720],
        [0.019, 1.43653410349, 27197.28169366760],
        [0.027, 1.22555218015, 42430.48572729180],
        [0.019, 3.65054338893, 49515.38250840700],
        [0.022, 5.88380811711, 10218.80847051840],
        [0.018, 2.29853355765, 19264.09638801420],
        [0.017, 5.44429906531, 6681.22485339960],
        [0.020, 3.68116637773, 14128.24277124560],
        [0.021, 4.30316190532, 44809.65020086340],
        [0.020, 2.48583613985, 33794.54372352860],
        [0.017, 3.02735393984, 28528.20632525540],
        [0.019, 5.92656850674, 22805.73556599360],
        [0.022, 5.30827572791, 10207.76262190360],
        [0.020, 0.75829381378, 18314.92077904440],
        [0.017, 5.63315744126, 16522.65971600220],
        [0.016, 1.71021408448, 536.80451209540],
        [0.015, 5.27016880041, 53445.59220462660],
        [0.017, 5.61443395877, 47938.03896595920],
        [0.015, 5.81110284451, 43071.89928903080],
        [0.015, 4.96237667003, 19999.97290154599],
        [0.018, 0.55618686515, 14765.23904326980],
        [0.014, 3.48144272414, 29786.66025688100],
        [0.015, 5.84132627836, 10228.53801739600],
        [0.016, 1.05720065324, 26735.94526221320],
        [0.014, 6.08462030302, 35371.88726597640],
        [0.014, 2.84532871890, 574.34479833480],
        [0.015, 5.34517715140, 10198.03307502600],
        [0.013, 0.45004137509, 20452.86941222180],
    ],
    # R2
    [
        [1406.587, 5.06366395190, 10213.28554621100],
        [15.529, 5.47321687981, 20426.57109242200],
        [13.059, 0.00000000000, 0.00000000000],
        [1.099, 2.78883988292, 9437.76293488700],
        [0.488, 6.27806914496, 1577.34354244780],
        [0.361, 6.11914188253, 10404.73381232260],
        [0.310, 1.38984998403, 5507.55323866740],
        [0.389, 1.95017779915, 11015.10647733480],
        [0.372, 2.33222828423, 775.52261132400],
        [0.207, 5.63406721595, 10239.58386601080],
        [0.168, 1.10765197296, 13367.97263110660],
        [0.175, 6.16674652950, 30639.85663863300],
        [0.168, 3.64495311632, 7084.89678111520],
        [0.120, 5.85815843789, 9153.90361602180],
        [0.160, 2.21564938463, 3154.68708489560],
        [0.118, 2.62358866565, 8635.94200376320],
        [0.112, 2.36235956804, 10596.18207843420],
        [0.092, 0.72664449269, 12566.15169998280],
        [0.067, 3.76089669118, 18837.49819713819],
        [0.065, 2.47983709990, 11790.62908865880],
        [0.048, 4.26620187144, 2352.86615377180],
        [0.048, 5.50898189550, 191.44826611160],
        [0.048, 2.54730918293, 17298.18232732620],
        [0.046, 3.40293459332, 14143.49524243060],
        [0.041, 1.83997113019, 11322.66409830440],
        [0.037, 6.17871126027, 1109.37855209340],
        [0.039, 4.77190210316, 18073.70493865020],
        [0.035, 3.10133256432, 4705.73230754360],
        [0.046, 3.30090415967, 6283.07584999140],
        [0.034, 3.91721765773, 10021.83728009940],
        [0.034, 3.24663787383, 22003.91463486980],
        [0.042, 3.39360926939, 14945.31617355440],
        [0.044, 4.42979374073, 7860.41939243920],
        [0.034, 2.16381407025, 16496.36139620240],
        [0.031, 0.45714618479, 26.29831979980],
        [0.035, 3.62868651241, 801.82093112380],
        [0.032, 1.84138997078, 382.89653222320],
        [0.025, 3.32908650295, 18307.80723204360],
        [0.026, 3.64313769818, 29088.81141598500],
        [0.029, 3.82967178810, 10186.98722641120],
        [0.022, 3.17741520378, 28521.09277825460],
        [0.021, 2.52643834111, 529.69096509460],
        [0.025, 5.71401244457, 21202.09370374600],
        [0.021, 3.77813434325, 21228.39202354580],
        [0.019, 5.24505118517, 19896.88012732740],
        [0.018, 4.62463651925, 19651.04848109800],
        [0.016, 3.35893297896, 28286.99048486120],
        [0.015, 5.05571633205, 33019.02111220460],
        [0.014, 2.83786355803, 19786.67380610799],
        [0.014, 1.79922718553, 9830.38901398780],
        [0.014, 3.14801263138, 19367.18916223280],
        [0.014, 3.57896195191, 10988.80815753500],
        [0.013, 3.06303088617, 10742.97651130560],
        [0.013, 5.43981998532, 25158.60171976540],
        [0.015, 4.83166312889, 18875.52586977400],
        [0.012, 2.54141086214, 7064.12138562280],
        [0.012, 4.45255110769, 15720.83878487840],
        [0.010, 1.87933121728, 24356.78078864160],
        [0.011, 2.58708635685, 9103.90699411760],
        [0.010, 2.17901309900, 3723.50895892300],
        [0.008, 3.63520673832, 1059.38193018920],
        [0.008, 4.67523115598, 25934.12433108940],
        [0.009, 5.97856553283, 9683.59458111640],
    ],
    # R3
    [
        [49.582, 3.22263554520, 10213.28554621100],
        [0.831, 3.21219077104, 20426.57109242200],
        [0.112, 3.14159265359, 0.00000000000],
        [0.013, 3.77448689585, 30639.85663863300],
        [0.009, 4.19802043629, 10239.58386601080],
        [0.006, 0.20714935358, 10186.98722641120],
        [0.005, 0.68781956122, 8635.94200376320],
    ],
    # R4
    [
        [0.573, 0.92229697820, 10213.28554621100],
        [0.040, 0.95468912157, 20426.57109242200],
        [0.006, 3.14159265359, 0.00000000000],
    ],
    # R5
    [
        [0.045, 0.30032866722, 10213.28554621100],
        [0.002, 5.29627718483, 20426.57109242200],
    ],
]
"""This table contains Venus's periodic terms (all of them) from the planetary
theory VSOP87 for the radius vector at the equinox of date (taken from the 'D'
solution). In Meeus' book a shortened version can be found in pages 420-421."""


ORBITAL_ELEM = [
    [181.979801, 58519.2130302, 0.00031014, 0.000000015],       # L
    [0.72332982, 0.0, 0.0, 0.0],                                # a
    [0.00677192, -0.000047765, 0.0000000981, 0.00000000046],    # e
    [3.394662, 0.0010037, -0.00000088, -0.000000007],           # i
    [76.67992, 0.9011206, 0.00040618, -0.000000093],            # Omega
    [131.563703, 1.4022288, -0.00107618, -0.000005678]          # pie
]
"""This table contains the parameters to compute Venus' orbital elements for
the mean equinox of date. Based in Table 31.A, page 212"""


ORBITAL_ELEM_J2000 = [
    [181.979801, 58517.815676, 0.00000165, -0.000000002],   # L
    [3.394662, -0.0008568, -0.00003244, 0.000000009],       # i
    [76.67992, -0.2780134, -0.00014257, -0.000000164],      # Omega
    [131.563703, 0.0048746, -0.00138467, -0.000005695]      # pie
]
"""This table contains the parameters to compute Venus' orbital elements for
the standard equinox J2000.0. Based on Table 31.B, page 214"""


class Venus(object):
    """
    Class Venus models that planet.
    """

    @staticmethod
    def geometric_heliocentric_position(epoch, tofk5=True):
        """This method computes the geometric heliocentric position of planet
        Venus for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Venus position, as an Epoch object
        :type epoch: :py:class:`Epoch`
        :param tofk5: Whether or not the small correction to convert to the FK5
            system will be applied or not
        :type tofk5: bool

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input value is of wrong type.

        >>> epoch = Epoch(1992, 12, 20.0)
        >>> l, b, r = Venus.geometric_heliocentric_position(epoch, tofk5=False)
        >>> print(round(l, 5))
        26.11412
        >>> print(round(b, 4))
        -2.6206
        >>> print(round(r, 6))
        0.724602
        """

        return geometric_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R, tofk5)

    @staticmethod
    def apparent_heliocentric_position(epoch):
        """This method computes the apparent heliocentric position of planet
        Venus for a given epoch, using the VSOP87 theory.

        :param epoch: Epoch to compute Venus position, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple with the heliocentric longitude and latitude (as
            :py:class:`Angle` objects), and the radius vector (as a float,
            in astronomical units), in that order
        :rtype: tuple
        :raises: TypeError if input value is of wrong type.
        """

        return apparent_vsop_pos(epoch, VSOP87_L, VSOP87_B, VSOP87_R)

    @staticmethod
    def orbital_elements_mean_equinox(epoch):
        """This method computes the orbital elements of Venus for the mean
        equinox of the date for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input value is of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Venus.orbital_elements_mean_equinox(epoch)
        >>> print(round(l, 6))
        338.646306
        >>> print(round(a, 8))
        0.72332982
        >>> print(round(e, 7))
        0.0067407
        >>> print(round(i, 6))
        3.395319
        >>> print(round(ome, 5))
        77.27012
        >>> print(round(arg, 6))
        55.211257
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM)

    @staticmethod
    def orbital_elements_j2000(epoch):
        """This method computes the orbital elements of Venus for the standard
        equinox J2000.0 for a given epoch.

        :param epoch: Epoch to compute orbital elements, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the following six orbital elements:
            - Mean longitude of the planet (Angle)
            - Semimajor axis of the orbit (float, astronomical units)
            - eccentricity of the orbit (float)
            - inclination on the plane of the ecliptic (Angle)
            - longitude of the ascending node (Angle)
            - argument of the perihelion (Angle)
        :rtype: tuple
        :raises: TypeError if input value is of wrong type.

        >>> epoch = Epoch(2065, 6, 24.0)
        >>> l, a, e, i, ome, arg = Venus.orbital_elements_j2000(epoch)
        >>> print(round(l, 6))
        337.731227
        >>> print(round(a, 8))
        0.72332982
        >>> print(round(e, 7))
        0.0067407
        >>> print(round(i, 6))
        3.394087
        >>> print(round(ome, 5))
        76.49782
        >>> print(round(arg, 6))
        55.068476
        """

        return orbital_elements(epoch, ORBITAL_ELEM, ORBITAL_ELEM_J2000)

    @staticmethod
    def geocentric_position(epoch):
        """This method computes the geocentric position of Venus (right
        ascension and declination) for the given epoch, as well as the
        elongation angle.

        :param epoch: Epoch to compute geocentric position, as an Epoch object
        :type epoch: :py:class:`Epoch`

        :returns: A tuple containing the right ascension, the declination and
            the elongation angle as Angle objects
        :rtype: tuple
        :raises: TypeError if input value is of wrong type.

        >>> epoch = Epoch(1992, 12, 20.0)
        >>> ra, dec, elon = Venus.geocentric_position(epoch)
        >>> print(ra.ra_str(n_dec=1))
        21h 4' 41.5''
        >>> print(dec.dms_str(n_dec=1))
        -18d 53' 16.8''
        >>> print(elon.dms_str(n_dec=1))
        44d 46' 8.9''
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Compute the heliocentric position of Venus
        l, b, r = Venus.geometric_heliocentric_position(epoch, tofk5=False)
        # Compute the heliocentric position of the Earth
        l0, b0, r0 = Earth.geometric_heliocentric_position(epoch, tofk5=False)
        # Convert to radians
        lr = l.rad()
        br = b.rad()
        l0r = l0.rad()
        b0r = b0.rad()
        # Compute first iteration
        x = r * cos(br) * cos(lr) - r0 * cos(b0r) * cos(l0r)
        y = r * cos(br) * sin(lr) - r0 * cos(b0r) * sin(l0r)
        z = r * sin(br) - r0 * sin(b0r)
        delta = sqrt(x * x + y * y + z * z)
        tau = 0.0057755183 * delta
        # Adjust the epoch for light-time
        epoch -= tau
        # Compute again Venus coordinates with this correction
        l, b, r = Venus.geometric_heliocentric_position(epoch, tofk5=False)
        # Compute second iteration
        lr = l.rad()
        br = b.rad()
        x = r * cos(br) * cos(lr) - r0 * cos(b0r) * cos(l0r)
        y = r * cos(br) * sin(lr) - r0 * cos(b0r) * sin(l0r)
        z = r * sin(br) - r0 * sin(b0r)
        # Compute longitude and latitude
        lamb = atan2(y, x)
        beta = atan2(z, sqrt(x * x + y * y))
        # Now, let's compute the aberration effect
        t = (epoch - JDE2000) / 36525
        e = 0.016708634 + t * (-0.000042037 - t * 0.0000001267)
        pie = 102.93735 + t * (1.71946 + t * 0.00046)
        pie = radians(pie)
        lon = l0 + 180.0
        lon = lon.rad()
        k = 20.49552    # The constant of aberration
        deltal1 = k * (-cos(lon - lamb) + e * cos(pie - lamb)) / cos(beta)
        deltab1 = -k * sin(beta) * (sin(lon - lamb) - e * sin(pie - lamb))
        deltal1 = Angle(0, 0, deltal1)
        deltab1 = Angle(0, 0, deltab1)
        # Correction to FK5 system
        lamb = Angle(lamb, radians=True)
        lamb = lamb.to_positive()
        beta = Angle(beta, radians=True)
        l_prime = lamb - t * (1.397 + t * 0.00031)
        deltal2 = Angle(0, 0, -0.09033)
        a = 0.03916 * (cos(l_prime.rad()) + sin(l_prime.rad()))
        a = a * tan(b.rad())
        deltal2 += Angle(0, 0, a)
        deltab2 = 0.03916 * (cos(l_prime.rad()) - sin(l_prime.rad()))
        deltab2 = Angle(0, 0, deltab2)
        # Apply the corrections
        lamb = lamb + deltal1 + deltal2
        beta = beta + deltab1 + deltab2
        # Correction for nutation
        dpsi = nutation_longitude(epoch)
        lamb += dpsi
        e = true_obliquity(epoch)
        ra, dec = ecliptical2equatorial(lamb, beta, e)
        # Let's compute the elongation angle
        lons, lats, rs = Sun.apparent_geocentric_position(epoch)
        lambr = lamb.rad()
        lsr = lons.rad()
        betar = beta.rad()
        elon = acos(cos(betar) * cos(lambr - lsr))
        elon = Angle(elon, radians=True)
        return ra, dec, elon

    @staticmethod
    def inferior_conjunction(epoch):
        """This method computes the time of the inferior conjunction closest to
        the given epoch.

        :param epoch: Epoch close to the desired inferior conjunction
        :type epoch: :py:class:`Epoch`

        :returns: The time when the inferior conjunction happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(1882, 12, 1.0)
        >>> conjunction = Venus.inferior_conjunction(epoch)
        >>> y, m, d = conjunction.get_date()
        >>> print(y)
        1882
        >>> print(m)
        12
        >>> print(round(d, 1))
        6.7
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Venus' inferior conjunction
        a = 2451996.706
        b = 583.921361
        m0 = 82.7311
        m1 = 215.513058
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        corr = (-0.0096 + t * (0.0002 - t * 0.00001) +
                sin(m) * (2.0009 + t * (-0.0033 - t * 0.00001)) +
                cos(m) * (0.598 + t * (-0.0104 + t * 0.00001)) +
                sin(2.0 * m) * (0.0967 + t * (-0.0018 - t * 0.00003)) +
                cos(2.0 * m) * (0.0913 + t * (0.0009 - t * 0.00002)) +
                sin(3.0 * m) * (0.0046 - t * 0.0002) +
                cos(3.0 * m) * (0.0079 + t * 0.0001))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def superior_conjunction(epoch):
        """This method computes the time of the superior conjunction closest to
        the given epoch.

        :param epoch: Epoch close to the desired superior conjunction
        :type epoch: :py:class:`Epoch`

        :returns: The time when the superior conjunction happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(1993, 10, 1.0)
        >>> conjunction = Venus.superior_conjunction(epoch)
        >>> y, m, d = conjunction.get_date()
        >>> print(y)
        1994
        >>> print(m)
        1
        >>> print(round(d, 2))
        17.05
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Venus' superior conjunction
        a = 2451704.746
        b = 583.921361
        m0 = 154.9745
        m1 = 215.513058
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        corr = (0.0099 + t * (-0.0002 - t * 0.00001) +
                sin(m) * (4.1991 + t * (-0.0121 - t * 0.00003)) +
                cos(m) * (-0.6095 + t * (0.0102 - t * 0.00002)) +
                sin(2.0 * m) * (0.25 + t * (-0.0028 - t * 0.00003)) +
                cos(2.0 * m) * (0.0063 + t * (0.0025 - t * 0.00002)) +
                sin(3.0 * m) * (0.0232 + t * (-0.0005 - t * 0.00001)) +
                cos(3.0 * m) * (0.0031 + t * 0.0004))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def western_elongation(epoch):
        """This method computes the time of the western elongation closest to
        the given epoch, as well as the corresponding maximum elongation angle.

        :param epoch: Epoch close to the desired western elongation
        :type epoch: :py:class:`Epoch`

        :returns: A tuple with the time when the western elongation happens, as
            an Epoch, and the maximum elongation angle, as an Angle
        :rtype: tuple
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(2019, 1, 1.0)
        >>> time, elongation = Venus.western_elongation(epoch)
        >>> y, m, d = time.get_date()
        >>> print(y)
        2019
        >>> print(m)
        1
        >>> print(round(d, 4))
        6.1895
        >>> print(round(elongation, 4))
        46.9571
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Venus' inferior conjunction
        a = 2451996.706
        b = 583.921361
        m0 = 82.7311
        m1 = 215.513058
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        corr = (70.7462 - t * t * 0.00001 +
                sin(m) * (1.1218 + t * (-0.0025 - t * 0.00001)) +
                cos(m) * (0.4538 - t * 0.0066) +
                sin(2.0 * m) * (0.132 + t * (0.002 - t * 0.00003)) +
                cos(2.0 * m) * (-0.0702 + t * (0.0022 + t * 0.00004)) +
                sin(3.0 * m) * (0.0062 - t * 0.0001) +
                cos(3.0 * m) * (0.0015 - t * t * 0.00001))
        elon = (46.3245 +
                sin(m) * (-0.5366 + t * (-0.0003 + t * 0.00001)) +
                cos(m) * (0.3097 + t * (0.0016 - t * 0.00001)) +
                sin(2.0 * m) * (-0.0163) +
                cos(2.0 * m) * (-0.0075 + t * 0.0001))
        elon = Angle(elon).to_positive()
        to_return = jde0 + corr
        return Epoch(to_return), elon

    @staticmethod
    def eastern_elongation(epoch):
        """This method computes the time of the eastern elongation closest to
        the given epoch, as well as the corresponding maximum elongation angle.

        :param epoch: Epoch close to the desired eastern elongation
        :type epoch: :py:class:`Epoch`

        :returns: A tuple with the time when the eastern elongation happens, as
            an Epoch, and the maximum elongation angle, as an Angle
        :rtype: tuple
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(2019, 10, 1.0)
        >>> time, elongation = Venus.eastern_elongation(epoch)
        >>> y, m, d = time.get_date()
        >>> print(y)
        2020
        >>> print(m)
        3
        >>> print(round(d, 4))
        24.9179
        >>> print(round(elongation, 4))
        46.078
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Venus' inferior conjunction
        a = 2451996.706
        b = 583.921361
        m0 = 82.7311
        m1 = 215.513058
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        corr = (-70.76 + t * (0.0002 - t * 0.00001) +
                sin(m) * (1.0282 + t * (-0.001 - t * 0.00001)) +
                cos(m) * (0.2761 - t * 0.006) +
                sin(2.0 * m) * (-0.0438 + t * (-0.0023 + t * 0.00002)) +
                cos(2.0 * m) * (0.166 + t * (-0.0037 - t * 0.00004)) +
                sin(3.0 * m) * (0.0036 + t * 0.0001) +
                cos(3.0 * m) * (-0.0011 + t * t * 0.00001))
        elon = (46.3173 + t * 0.0001 +
                sin(m) * (0.6916 - t * 0.0024) +
                cos(m) * (0.6676 - t * 0.0045) +
                sin(2.0 * m) * (0.0309 - t * 0.0002) +
                cos(2.0 * m) * (0.0036 - t * 0.0001))
        elon = Angle(elon).to_positive()
        to_return = jde0 + corr
        return Epoch(to_return), elon

    @staticmethod
    def station_longitude_1(epoch):
        """This method computes the time of the 1st station in longitude
        (i.e. when the planet is stationary and begins to move westward -
        retrograde - among the starts) closest to the given epoch.

        :param epoch: Epoch close to the desired inferior conjunction
        :type epoch: :py:class:`Epoch`

        :returns: Time when the 1st station in longitude happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(2018, 12, 1.0)
        >>> sta1 = Venus.station_longitude_1(epoch)
        >>> y, m, d = sta1.get_date()
        >>> print(y)
        2018
        >>> print(m)
        10
        >>> print(round(d, 4))
        5.7908
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Venus' inferior conjunction
        a = 2451996.706
        b = 583.921361
        m0 = 82.7311
        m1 = 215.513058
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        corr = (-21.0672 + t * (0.0002 - t * 0.00001) +
                sin(m) * (1.9396 + t * (-0.0029 - t * 0.00001)) +
                cos(m) * (1.0727 - t * 0.0102) +
                sin(2.0 * m) * (0.0404 + t * (-0.0023 - t * 0.00001)) +
                cos(2.0 * m) * (0.1305 + t * (-0.0004 - t * 0.00003)) +
                sin(3.0 * m) * (-0.0007 - t * 0.0002) +
                cos(3.0 * m) * (0.0098))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def station_longitude_2(epoch):
        """This method computes the time of the 1st station in longitude
        (i.e. when the planet is stationary and begins to move eastward -
        prograde - among the starts) closest to the given epoch.

        :param epoch: Epoch close to the desired inferior conjunction
        :type epoch: :py:class:`Epoch`

        :returns: Time when the 2nd station in longitude happens, as an Epoch
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input value is of wrong type.
        :raises: ValueError if input epoch outside the -2000/4000 range.

        >>> epoch = Epoch(2018, 12, 1.0)
        >>> sta2 = Venus.station_longitude_2(epoch)
        >>> y, m, d = sta2.get_date()
        >>> print(y)
        2018
        >>> print(m)
        11
        >>> print(round(d, 4))
        16.439
        """

        # First check that input value is of correct types
        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input type")
        # Check that the input epoch is within valid range
        y = epoch.year()
        if y < -2000.0 or y > 4000.0:
            raise ValueError("Epoch outside the -2000/4000 range")
        # Set some specific constants for Venus' inferior conjunction
        a = 2451996.706
        b = 583.921361
        m0 = 82.7311
        m1 = 215.513058
        k = round((365.2425 * y + 1721060.0 - a) / b)
        jde0 = a + k * b
        m = m0 + k * m1
        m = Angle(m).to_positive()
        m = m.rad()
        t = (jde0 - 2451545.0) / 36525.0
        corr = (21.0623 - t * t * 0.00001 +
                sin(m) * (1.9913 + t * (-0.004 - t * 0.00001)) +
                cos(m) * (-0.0407 - t * 0.0077) +
                sin(2.0 * m) * (0.1351 + t * (-0.0009 - t * 0.00004)) +
                cos(2.0 * m) * (0.0303 + t * 0.0019) +
                sin(3.0 * m) * (0.0089 - t * 0.0002) +
                cos(3.0 * m) * (0.0043 + t * 0.0001))
        to_return = jde0 + corr
        return Epoch(to_return)

    @staticmethod
    def perihelion_aphelion(epoch, perihelion=True):
        """This method computes the time of Perihelion (or Aphelion) closer to
        a given epoch.

        :param epoch: Epoch close to the desired Perihelion (or Aphelion)
        :type epoch: :py:class:`Epoch`
        :param peihelion: If True, the epoch of the closest Perihelion is
            computed, if False, the epoch of the closest Aphelion is found.
        :type bool:

        :returns: The epoch of the desired Perihelion (or Aphelion)
        :rtype: :py:class:`Epoch`
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(1978, 10, 15.0)
        >>> e = Venus.perihelion_aphelion(epoch)
        >>> y, m, d, h, mi, s = e.get_full_date()
        >>> print(y)
        1978
        >>> print(m)
        12
        >>> print(d)
        31
        >>> print(h)
        4
        >>> epoch = Epoch(1979, 2, 1.0)
        >>> e = Venus.perihelion_aphelion(epoch, perihelion=False)
        >>> y, m, d, h, mi, s = e.get_full_date()
        >>> print(y)
        1979
        >>> print(m)
        4
        >>> print(d)
        22
        >>> print(h)
        12
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input value")
        # First approximation
        k = 1.62549 * (epoch.year() - 2000.53)
        if perihelion:
            k = round(k)
        else:
            k = round(k + 0.5) - 0.5
        jde = 2451738.233 + k * (224.7008188 - k * 0.0000000327)
        # Compute the epochs half a day before and after
        jde_before = jde - 0.5
        jde_after = jde + 0.5
        # Compute the Sun-Venus distance for each epoch
        l, b, r_b = Venus.geometric_heliocentric_position(Epoch(jde_before))
        l, b, r = Venus.geometric_heliocentric_position(Epoch(jde))
        l, b, r_a = Venus.geometric_heliocentric_position(Epoch(jde_after))
        # Call an interpolation object
        m = Interpolation([jde_before, jde, jde_after], [r_b, r, r_a])
        sol = m.minmax()
        return Epoch(sol)

    @staticmethod
    def passage_nodes(epoch, ascending=True):
        """This function computes the time of passage by the nodes (ascending
        or descending) of Venus, nearest to the given epoch.

        :param epoch: Epoch closest to the node passage
        :type epoch: :py:class:`Epoch`
        :param ascending: Whether the time of passage by the ascending (True)
            or descending (False) node will be computed
        :type ascending: bool

        :returns: Tuple containing:
            - Time of passage through the node (:py:class:`Epoch`)
            - Radius vector when passing through the node (in AU, float)
        :rtype: tuple
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(1979, 1, 1)
        >>> time, r = Venus.passage_nodes(epoch)
        >>> year, month, day = time.get_date()
        >>> print(year)
        1978
        >>> print(month)
        11
        >>> print(round(day, 1))
        27.4
        >>> print(round(r, 4))
        0.7205
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input types")
        # Get the orbital parameters
        l, a, e, i, ome, arg = Venus.orbital_elements_mean_equinox(epoch)
        # Compute the time of passage through perihelion
        t = Venus.perihelion_aphelion(epoch)
        # Get the time of passage through the node
        time, r = passage_nodes_elliptic(arg, e, a, t, ascending)
        return time, r

    @staticmethod
    def illuminated_fraction(epoch):
        """This function computes an approximation of the illuminated fraction
        of Venus disk, as seen from Earth.

        :param epoch: Epoch to compute the illuminated fraction
        :type epoch: :py:class:`Epoch`

        :returns: Illuminated fraction of Venus disk
        :rtype: float
        :raises: TypeError if input values are of wrong type.

        >>> epoch = Epoch(1992, 12, 20)
        >>> k = Venus.illuminated_fraction(epoch)
        >>> print(round(k, 2))
        0.64
        """

        if not isinstance(epoch, Epoch):
            raise TypeError("Invalid input types")
        t = (epoch.jde() - 2451545.0) / 36525.0
        v = Angle(261.51 + 22518.443 * t)
        m = Angle(177.53 + 35999.050 * t)
        n = Angle(50.42 + 58517.811 * t)
        w = Angle(v + 1.91 * sin(m.rad()) + 0.78 * sin(n.rad()))
        delta2 = abs(1.52321 + 1.44666 * cos(w.rad()))
        delta = sqrt(delta2)
        k = ((0.72333 + delta) * (0.72333 + delta) - 1.0) / (2.89332 * delta)
        return k

    @staticmethod
    def magnitude(sun_dist, earth_dist, phase_angle):
        """This function computes the approximate magnitude of Venus.

        :param sun_dist: Distance from Venus to the Sun, in Astronomical Units
        :type sun_dist: float
        :param earth_dist: Distance from Venus to Earth, in Astronomical Units
        :type earth_dist: float
        :param phase_angle: Venus phase angle
        :type phase_angle: float, :py:class:`Angle`

        :returns: Venus' magnitude
        :rtype: float
        :raises: TypeError if input values are of wrong type.

        >>> sun_dist = 0.724604
        >>> earth_dist = 0.910947
        >>> phase_angle = Angle(72.96)
        >>> m = Venus.magnitude(sun_dist, earth_dist, phase_angle)
        >>> print(m)
        -3.8
        """

        if not (isinstance(sun_dist, float) and isinstance(earth_dist, float)
                and isinstance(phase_angle, (float, Angle))):
            raise TypeError("Invalid input types")
        i = float(phase_angle)
        m = (-4.0 + 5.0 * log10(sun_dist * earth_dist) + 0.01322 * i
             + 0.0000004247 * i * i * i)
        return round(m, 1)


def main():

    # Let's define a small helper function
    def print_me(msg, val):
        print("{}: {}".format(msg, val))

    # Let's show some uses of Venus class
    print("\n" + 35 * "*")
    print("*** Use of Venus class")
    print(35 * "*" + "\n")

    # Let's now compute the heliocentric position for a given epoch
    epoch = Epoch(1992, 12, 20.0)
    lon, lat, r = Venus.geometric_heliocentric_position(epoch)
    print_me("Geometric Heliocentric Longitude", lon.to_positive())
    print_me("Geometric Heliocentric Latitude", lat)
    print_me("Radius vector", r)

    print("")

    # Compute the geocentric position for 1992/12/20:
    epoch = Epoch(1992, 12, 20.0)
    ra, dec, elon = Venus.geocentric_position(epoch)
    print_me("Right ascension", ra.ra_str(n_dec=1))
    print_me("Declination", dec.dms_str(n_dec=1))
    print_me("Elongation", elon.dms_str(n_dec=1))

    print("")

    # Print mean orbital elements for Venus at 2065.6.24
    epoch = Epoch(2065, 6, 24.0)
    l, a, e, i, ome, arg = Venus.orbital_elements_mean_equinox(epoch)
    print_me("Mean longitude of the planet", round(l, 6))       # 338.646306
    print_me("Semimajor axis of the orbit (UA)", round(a, 8))   # 0.72332982
    print_me("Eccentricity of the orbit", round(e, 7))          # 0.0067407
    print_me("Inclination on plane of the ecliptic", round(i, 6))   # 3.395319
    print_me("Longitude of the ascending node", round(ome, 5))  # 77.27012
    print_me("Argument of the perihelion", round(arg, 6))       # 55.211257

    print("")

    # Compute the time of the inferior conjunction close to 1882/12/1.0
    epoch = Epoch(1882, 12, 1.0)
    conjunction = Venus.inferior_conjunction(epoch)
    y, m, d = conjunction.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Inferior conjunction date", date)

    # Compute the time of the superior conjunction close to 1993/10/1
    epoch = Epoch(1993, 10, 1.0)
    conjunction = Venus.superior_conjunction(epoch)
    y, m, d = conjunction.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Superior conjunction date", date)

    print("")

    # Compute the time and angle of the western elongation close to 2019/1/1
    epoch = Epoch(2019, 1, 1.0)
    time, elongation = Venus.western_elongation(epoch)
    y, m, d = time.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Western elongation date", date)
    elong = round(elongation, 4)
    print_me("Maximum western elongation angle", elong)

    print("")

    # Compute the time and angle of the eastern elongation close to 2019/10/1
    epoch = Epoch(2019, 10, 1.0)
    time, elongation = Venus.eastern_elongation(epoch)
    y, m, d = time.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Eastern elongation date", date)
    elong = round(elongation, 4)
    print_me("Maximum eastern elongation angle", elong)

    print("")

    # Compute the time of the station in longitude #1 close to 2018/12/1
    epoch = Epoch(2018, 12, 1.0)
    sta1 = Venus.station_longitude_1(epoch)
    y, m, d = sta1.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Date of station in longitude #1", date)

    # Compute the time of the station in longitude #2 close to 2018/12/1
    epoch = Epoch(2018, 12, 1.0)
    sta2 = Venus.station_longitude_2(epoch)
    y, m, d = sta2.get_date()
    d = round(d, 4)
    date = "{}/{}/{}".format(y, m, d)
    print_me("Date of station in longitude #2", date)

    print("")

    # Find the epoch of the Perihelion closer to 1978/10/15
    epoch = Epoch(1978, 10, 15.0)
    e = Venus.perihelion_aphelion(epoch)
    y, m, d, h, mi, s = e.get_full_date()
    peri = str(y) + '/' + str(m) + '/' + str(d) + ' at ' + str(h) + ' hours'
    print_me("The Perihelion closest to 1978/10/15 happened on", peri)

    print("")

    # Compute the time of passage through an ascending node
    epoch = Epoch(1979, 1, 1)
    time, r = Venus.passage_nodes(epoch)
    y, m, d = time.get_date()
    d = round(d, 1)
    print("Time of passage through ascending node: {}/{}/{}".format(y, m, d))
    # 1978/11/27.4
    print("Radius vector at ascending node: {}".format(round(r, 4)))  # 0.7205

    print("")

    # Compute the (approximate) illuminated fraction of Venus disk for an Epoch
    epoch = Epoch(1992, 12, 20)
    k = Venus.illuminated_fraction(epoch)
    print_me("Approximate illuminated fraction of Venus", round(k, 2))  # 0.64

    # Compute the magnitude of Venus
    sun_dist = 0.724604
    earth_dist = 0.910947
    phase_angle = Angle(72.96)
    m = Venus.magnitude(sun_dist, earth_dist, phase_angle)
    print_me("Venus' magnitude", round(m, 1))                           # -3.8


if __name__ == "__main__":

    main()
