/*
 *@BEGIN LICENSE
 *
 * PSI4: an ab initio quantum chemistry software package
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 *@END LICENSE
 */

/*! \file
    \ingroup CIS
    \brief Enter brief description of file here 
*/
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>
#include <libciomr/libciomr.h>
#include <libdpd/dpd.h>
#include <libchkpt/chkpt.h>
#include <libqt/qt.h>
#include <physconst.h>
#include <psifiles.h>
#include "Params.h"
#include "MOInfo.h"
#include "Local.h"
#include "globals.h"

namespace psi { namespace cis {

void init_io(int argc, char *argv[]);
void title(void);
void exit_io(void);
int **cacheprep_uhf(int level, int *cachefiles);
int **cacheprep_rhf(int level, int *cachefiles);
void cachedone_rhf(int **cachelist);
void cachedone_uhf(int **cachelist);
void get_moinfo(void);
void get_params(Options & options);
void cleanup(void);
void build_A(void);
void d_corr(void);
void local_init(Options & options);
void local_done(void);
void amp_write_T1(dpdfile2 *T1, int length, std::string OutFileRMR);
void diag(void);

PsiReturnType cis(Options & options, int argc, char *argv[])
{
  char lbl[32];
  int **cachelist, *cachefiles;
  int h, i, j, jj, k, nroot, a, max_j, max_a;
  double *weakp, value, d_value, **B_PAO;
  double *Bt, max_val, test_val, sum_val;
  int *spin, count, ivalue;
  dpdfile2 B;

  init_io(argc, argv);
  title();

  get_moinfo();
  get_params(options);

  cachefiles = init_int_array(PSIO_MAXUNIT);

  if(params.ref == 0 || params.ref == 1) { /** RHF/ROHF **/
    cachelist = cacheprep_rhf(0, cachefiles);

    dpd_init(0, moinfo.nirreps, params.memory, 0, cachefiles, 
        cachelist, NULL, 2, moinfo.occpi, moinfo.occ_sym, 
        moinfo.virtpi, moinfo.vir_sym);
  }
  else if(params.ref == 2) { /** UHF **/
    cachelist = cacheprep_uhf(0, cachefiles);

    dpd_init(0, moinfo.nirreps, params.memory, 0, cachefiles, 
        cachelist, NULL, 4, moinfo.aoccpi, moinfo.aocc_sym, moinfo.avirtpi,
        moinfo.avir_sym, moinfo.boccpi, moinfo.bocc_sym, moinfo.bvirtpi, moinfo.bvir_sym);
  }

  /* build components of CIS response matrix, A */
  build_A();

  /* diagonalize A within each irrep */
  diag();

  /* moved up with impunity??? */
  if(params.local) local_init(options);

  /* print eigenvalues and largest components of eigenvectors in ascending order */
  if(params.ref == 0) {

    outfile->Printf( "\tRHF-CIS Singlet Excitation Energies:\n");
    outfile->Printf( "\t------------------------------------\n\n");
    outfile->Printf( "\t  Root Irrep       Hartree          eV          cm-1  \n");
    outfile->Printf( "\t  ---- ----- ------------------  ---------  ----------\n");
    for(h=0; h < moinfo.nirreps; h++) {
      for(i=0; i < params.rpi[h]; i++) {
        value = moinfo.singlet_evals[h][i];

        outfile->Printf(   "CIS State %4d   %3s %18.14f  %9.5f  %10.2f\n", i, moinfo.labels[h],
            value, value*pc_hartree2ev, value*pc_hartree2wavenumbers);

        outfile->Printf( "\nLargest components of singlet excited wave function #%d/#%d:\n",
            h, i);
        sprintf(lbl, "BIA(%d)[%d] singlet", i, h);
        global_dpd_->file2_init(&B, PSIF_CC_OEI, h, 0, 1, lbl);
        amp_write_T1(&B, 5, outfile);
        global_dpd_->file2_close(&B);

      }
    }
    outfile->Printf( "\n");

    outfile->Printf( "\tRHF-CIS Triplet Excitation Energies:\n");
    outfile->Printf( "\t------------------------------------\n\n");
    outfile->Printf( "\t  Root Irrep       Hartree          eV          cm-1  \n");
    outfile->Printf( "\t  ---- ----- ------------------  ---------  ----------\n");
    for(h=0; h < moinfo.nirreps; h++) {
      for(i=0; i < params.rpi[h]; i++) {
        value = moinfo.triplet_evals[h][i];

        outfile->Printf(   "CIS State %4d   %3s %18.14f  %9.5f  %10.2f\n", i, moinfo.labels[h],
            value, value*pc_hartree2ev, value*pc_hartree2wavenumbers);

        outfile->Printf( "\nLargest components of triplet excited wave function #%d/#%d:\n", h, i);
        sprintf(lbl, "BIA(%d)[%d] triplet", i, h);
        global_dpd_->file2_init(&B, PSIF_CC_OEI, h, 0, 1, lbl);
        amp_write_T1(&B, 5, outfile);
        global_dpd_->file2_close(&B);
        outfile->Printf( "\n");

      }
    }
    outfile->Printf( "\n");

  }

  

  /* compute the (D) correction to each CIS singlet excitation energy */
  d_corr();

  outfile->Printf( "\n");

  if(params.local) local_done();

  /* print corrected eigenvalues in ascending order */
  if(params.ref == 0) {

    outfile->Printf( "\tRHF-CIS(D) Singlet Corrections:\n");
    outfile->Printf( "\t-------------------------------\n");
    outfile->Printf( "\t  Root Irrep       Hartree          eV          cm-1  \n");
    outfile->Printf( "\t  ---- ----- ------------------  ---------  ----------\n");
    for(h=0; h < moinfo.nirreps; h++) {
      for(i=0; i < params.rpi[h]; i++) {
        value = moinfo.singlet_d[h][i];

        outfile->Printf(   "CIS State %4d   %3s %18.14f  %9.5f  %10.2f\n", i, moinfo.labels[h],
            value, value*pc_hartree2ev, value*pc_hartree2wavenumbers);
      }
    }
    outfile->Printf( "\n");

    outfile->Printf( "\tRHF-CIS(D) Singlet Excitation Energies:\n");
    outfile->Printf( "\t---------------------------------------\n\n");
    outfile->Printf( "\t  Root Irrep       Hartree          eV          cm-1  \n");
    outfile->Printf( "\t  ---- ----- ------------------  ---------  ----------\n");
    for(h=0; h < moinfo.nirreps; h++) {
      for(i=0; i < params.rpi[h]; i++) {
        value = moinfo.singlet_evals[h][i];
        value += moinfo.singlet_d[h][i];

        outfile->Printf(   "CIS State %4d   %3s %18.14f  %9.5f  %10.2f\n", i, moinfo.labels[h],
            value, value*pc_hartree2ev, value*pc_hartree2wavenumbers);
      }
    }
    outfile->Printf( "\n");

    if(params.local) {
      outfile->Printf( "\tRHF-CIS(D) Weak Pair Corrections:\n");
      outfile->Printf( "\t---------------------------------\n");
      outfile->Printf( "\t  Root Irrep       Hartree          eV          cm-1  \n");
      outfile->Printf( "\t  ---- ----- ------------------  ---------  ----------\n");
      for(h=0; h < moinfo.nirreps; h++) {
        for(i=0; i < params.rpi[h]; i++) {
          value = moinfo.singlet_weakp[h][i];

          outfile->Printf(   "CIS State %4d   %3s %18.14f  %9.5f  %10.2f\n", i, moinfo.labels[h],
              value, value*pc_hartree2ev, value*pc_hartree2wavenumbers);
        }
      }
      outfile->Printf( "\n");
    }
  }

  else if(params.ref == 2) {

    outfile->Printf( "\tUHF-CIS Excitation Energies:\n");
    outfile->Printf( "\t----------------------------\n\n");
    outfile->Printf( "\t  Root Irrep       Hartree          eV          cm-1\n");
    outfile->Printf( "\t  ---- ----- ------------------  ---------  -----------\n");
    for(h=0; h < moinfo.nirreps; h++) {
      for(i=0; i < params.rpi[h]; i++) {
        value = moinfo.uhf_evals[h][i];

        outfile->Printf(   "CIS State %4d   %3s %18.14f  %9.5f  %10.2f\n", i, moinfo.labels[h],
            value, value*pc_hartree2ev, value*pc_hartree2wavenumbers);
      }
    }
    outfile->Printf( "\n");

    outfile->Printf( "\tUHF-CIS(D) Corrections:\n");
    outfile->Printf( "\t-----------------------\n\n");

    outfile->Printf( "\t  Root Irrep       Hartree          eV         cm-1\n");
    outfile->Printf( "\t  ---- ----- ------------------  ---------  ----------\n");
    for(h=0; h < moinfo.nirreps; h++) {
      for(i=0; i < params.rpi[h]; i++) {
        value = moinfo.uhf_d[h][i];

        outfile->Printf(   "CIS State %4d   %3s %18.14f  %9.5f  %10.2f\n", i, moinfo.labels[h],
            value, value*pc_hartree2ev, value*pc_hartree2wavenumbers);
      }
    }
    outfile->Printf( "\n");

    outfile->Printf( "\tUHF-CIS(D) Excitation Energies:\n");
    outfile->Printf( "\t-------------------------------\n\n");

    outfile->Printf( "\t  Root Irrep       Hartree          eV         cm-1\n");
    outfile->Printf( "\t  ---- ----- ------------------  ---------  ----------\n");
    for(h=0; h < moinfo.nirreps; h++) {
      for(i=0; i < params.rpi[h]; i++) {
        value = moinfo.uhf_evals[h][i];
        value += moinfo.uhf_d[h][i];

        outfile->Printf(   "CIS State %4d   %3s %18.14f  %9.5f  %10.2f\n", i, moinfo.labels[h],
            value, value*pc_hartree2ev, value*pc_hartree2wavenumbers);
      }
    }
    outfile->Printf( "\n");

  }

  dpd_close(0);
  if(params.ref == 0 || params.ref == 1) cachedone_rhf(cachelist);
  else if(params.ref == 2) cachedone_uhf(cachelist);
  free(cachefiles);

  cleanup();

  exit_io();
  return(Success);
}

void init_io(int argc, char *argv[])
{
  tstart();

  psio_open(PSIF_CC_INFO, 1);
  psio_open(PSIF_CC_OEI, 1);
  psio_open(PSIF_CC_CINTS, 1);
  psio_open(PSIF_CC_DINTS, 1);
  psio_open(PSIF_CC_EINTS, 1);
  psio_open(PSIF_CC_FINTS, 1);
  psio_open(PSIF_CC_DENOM, 1);
  psio_open(PSIF_CC_MISC, 0);
  psio_open(PSIF_CC_TMP0, 0);
  psio_open(PSIF_CC_TMP1, 0);
}

void title(void)
{
  outfile->Printf( "\t\t\t*************************\n");
  outfile->Printf( "\t\t\t*                       *\n");
  outfile->Printf( "\t\t\t*          CIS          *\n");
  outfile->Printf( "\t\t\t*                       *\n");
  outfile->Printf( "\t\t\t*************************\n");
}

void exit_io(void)
{
  psio_close(PSIF_CC_INFO, 1);
  psio_close(PSIF_CC_OEI, 1);
  psio_close(PSIF_CC_CINTS, 1);
  psio_close(PSIF_CC_DINTS, 1);
  psio_close(PSIF_CC_EINTS, 1);
  psio_close(PSIF_CC_FINTS, 1);
  psio_close(PSIF_CC_DENOM, 1);
  psio_close(PSIF_CC_MISC, 1);
  psio_close(PSIF_CC_TMP0, 0);
  psio_close(PSIF_CC_TMP1, 0);

  tstop();
}

}} // namespace psi::cis
