//
// Cipher.cpp
//
// $Id: //poco/1.3/Crypto/src/Cipher.cpp#3 $
//
// Library: Crypto
// Package: Cipher
// Module:  Cipher
//
// Copyright (c) 2008, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#include "Poco/Crypto/Cipher.h"
#include "Poco/Crypto/CryptoStream.h"
#include "Poco/Crypto/CryptoTransform.h"
#include "Poco/Base64Encoder.h"
#include "Poco/Base64Decoder.h"
#include "Poco/HexBinaryEncoder.h"
#include "Poco/HexBinaryDecoder.h"
#include "Poco/StreamCopier.h"
#include "Poco/Exception.h"
#include <sstream>
#include <memory>


namespace Poco {
namespace Crypto {


Cipher::Cipher()
{
}


Cipher::~Cipher()
{
}


std::string Cipher::encryptString(const std::string& str, Encoding encoding)
{
	std::istringstream source(str);
	std::ostringstream sink;

	encrypt(source, sink, encoding);

	return sink.str();
}


std::string Cipher::decryptString(const std::string& str, Encoding encoding)
{
	std::istringstream source(str);
	std::ostringstream sink;

	decrypt(source, sink, encoding);
	return sink.str();
}


void Cipher::encrypt(std::istream& source, std::ostream& sink, Encoding encoding)
{
	CryptoInputStream encryptor(source, createEncryptor());

	switch (encoding)
	{
	case ENC_NONE:
		StreamCopier::copyStream(encryptor, sink);
		break;

	case ENC_BASE64:
		{
			Poco::Base64Encoder encoder(sink);
			StreamCopier::copyStream(encryptor, encoder);
			encoder.close();
		}
		break;

	case ENC_BINHEX:
		{
			Poco::HexBinaryEncoder encoder(sink);
			StreamCopier::copyStream(encryptor, encoder);
			encoder.close();
		}
		break;

	default:
		throw Poco::InvalidArgumentException("Invalid argument", "encoding");
	}
}


void Cipher::decrypt(std::istream& source, std::ostream& sink, Encoding encoding)
{
	CryptoOutputStream decryptor(sink, createDecryptor());

	switch (encoding)
	{
	case ENC_NONE:
		StreamCopier::copyStream(source, decryptor);
		decryptor.close();
		break;

	case ENC_BASE64:
		{
			Poco::Base64Decoder decoder(source);
			StreamCopier::copyStream(decoder, decryptor);
			decryptor.close();
		}
		break;

	case ENC_BINHEX:
		{
			Poco::HexBinaryDecoder decoder(source);
			StreamCopier::copyStream(decoder, decryptor);
			decryptor.close();
		}
		break;

	default:
		throw Poco::InvalidArgumentException("Invalid argument", "encoding");
	}
}


} } // namespace Poco::Crypto
