/*	Stream_Viewer

PIRL CVS ID: Stream_Viewer.java,v 1.8 2012/04/16 06:22:59 castalia Exp

Copyright (C) 2008-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package	PIRL.Viewers;

import	PIRL.Viewers.Stream_Monitor;
import	PIRL.Conductor.Stream_Logger;
import	PIRL.Utilities.Streams;
import	PIRL.Configuration.*;

import	javax.swing.JFrame;
import	javax.swing.JMenuBar;
import	javax.swing.JMenu;
import	javax.swing.JMenuItem;
import	javax.swing.JPanel;
import	javax.swing.KeyStroke;
import	java.awt.Dimension;
import	java.awt.GridBagLayout;
import	java.awt.GridBagConstraints;
import	java.awt.Event;
import	java.awt.event.ActionListener;
import	java.awt.event.ActionEvent;
import	java.io.InputStream;
import	java.io.OutputStreamWriter;


/**	A <i>Stream_Viewer</i> provides a viewer application attached to stdin.
<p>
	This is a demonstration application for the use of a Stream_Monitor
	with a Stream_Logger.
<p>
	@author	Bradford Castalia - UA/PIRL
	@version	1.8
	@see		Stream_Monitor
*/
public class Stream_Viewer
	extends JFrame
{
/**	Class name and version identification.
*/
public static final String
	ID = "PIRL.Viewers.Stream_Viewer (1.8 2012/04/16 06:22:59)";


public static final String
	DEFAULT_CONFIGURATION_PATHNAME	= "Stream_Viewer.conf";


private Stream_Monitor
	Monitor = null;


/**	Exit status on success.
*/
public static final int
	EXIT_SUCCESS				= 0;

/**	Exit status when a command line syntax problem occurs.
*/
public static final int
	EXIT_COMMAND_LINE_SYNTAX	= 1;

/**	Exit status when a configuration problem occurs.
*/
public static final int
	EXIT_CONFIGURATION_PROBLEM	= 1;

/*==============================================================================
	Constructors
*/
/**	Constructs a Stream_Viewer.
<p>
	@param	title	The title String for the window. If null,
		the default "Stream Viewer" title will be used.
	@param	source	An InputStream used as the source of text to
		be viewed. If null System.in will be used.
	@param	configuration	A Configuration (not currently used).
*/
public Stream_Viewer
	(
	String			title,
	InputStream		source,
	Configuration	configuration
	)
{
super ((title == null) ? "Stream Viewer" : title);

Configure (configuration);

Monitor = new Stream_Monitor ();

setJMenuBar (Menus ());
setContentPane (Panels ());
pack ();
setDefaultCloseOperation (JFrame.EXIT_ON_CLOSE);
setVisible (true);

Monitor
	.Write
		(ID + "\n"
		+Stream_Monitor.ID + "\n\n", Monitor.NOTICE_STYLE)
	.Auto_Style (true);

if (source == null)
	source = System.in;
Stream_Logger
	stream_logger = new Stream_Logger
		(null, source, Monitor.Writer ());
stream_logger.start ();
}

/**	Constructs a default Stream_Viewer.
<p>
	@see	#Stream_Viewer(String, InputStream, Configuration)
*/
public Stream_Viewer ()
{this (null, null, null);}

/*==============================================================================
	Configure
*/
private void Configure
	(
	Configuration	configuration
	)
{
//	A configuration file might provide auto-styling definitions.
}

/*==============================================================================
	Menus
*/
private JMenuBar Menus ()
{
JMenuBar
	menu_bar = new JMenuBar ();
JMenu
	menu = Monitor.File_Menu ();
JMenuItem
	menu_item;

for (int
		index = 0;
		index < menu.getItemCount ();
		index++)
	{
	menu_item = menu.getItem (index);
	if (menu_item != null)
		{
		String
			text = menu_item.getText ();
		if (text == null)
			continue;
		if (text.startsWith ("Save As"))
			{
			menu_item.setMnemonic ('A');
			menu_item.setAccelerator
				(KeyStroke.getKeyStroke ('A', Event.CTRL_MASK));
			}
		else
		if (text.startsWith ("Save"))
			{
			menu_item.setMnemonic ('S');
			menu_item.setAccelerator
				(KeyStroke.getKeyStroke ('S', Event.CTRL_MASK));
			}
		}
	}

menu.addSeparator ();

menu_item = new JMenuItem ("Exit"); 
menu_item.setMnemonic ('X');
menu_item.setAccelerator (KeyStroke.getKeyStroke ('Q', Event.CTRL_MASK));
menu_item.addActionListener (new ActionListener ()
	{public void actionPerformed (ActionEvent event)
	{System.exit (EXIT_SUCCESS);}});
menu.add (menu_item);

menu_bar.add (menu);

return menu_bar;
}

/*==============================================================================
	Panels
*/
private JPanel Panels ()
{
JPanel
	content = new JPanel (new GridBagLayout ());
GridBagConstraints
	location = new GridBagConstraints ();
location.fill		= GridBagConstraints.BOTH;
location.weightx	= 1.0;
location.weighty	= 1.0;
location.gridwidth	= GridBagConstraints.REMAINDER;
content.add (Monitor, location);

content.setPreferredSize (new Dimension (640, 480));
return content;
}

/*==============================================================================
	Application main
*/
/**	Start a Stream_Viewer.
<p>
	@param	args	The application command line arguments.
	@see	#Usage()
*/
public static void main
	(
	String[]    args
	)
{
String
	title = null,
	source_name = null,
	configuration_source = null;
InputStream
	source = null;
Configuration
	configuration = null;

for (int
		count = 0;
		count < args.length;
		count++)
	{
	if (args[count].length () == 0)
		continue;
	if (args[count].charAt (0) == '-' &&
		args[count].length () > 1)
		{
		switch (args[count].charAt (1))
			{
			case 'C':	//	Configuration source
			case 'c':
				if (++count == args.length ||
					args[count].charAt (0) == '-')
					{
					System.out.println
						("Missing configuration source.");
					Usage ();
					}
				if (configuration_source != null)
					{
					System.out.println
						("Multiple configuration sources specified -\n"
						+"  Old: " + configuration_source + '\n'
						+"  New: " + args[count]);
					System.exit (EXIT_COMMAND_LINE_SYNTAX);
					}
				configuration_source = args[count];
				break;

			case 'T':	//	Title name
			case 't':
				if (++count == args.length ||
					args[count].charAt (0) == '-')
					{
					System.out.println
						("Missing title name.");
					Usage ();
					}
				if (title != null)
					{
					System.out.println
						("Multiple titles specified -\n"
						+"  Old: " + title + '\n'
						+"  New: " + args[count]);
					System.exit (EXIT_COMMAND_LINE_SYNTAX);
					}
				title = args[count];
				break;

			default:
				System.out.println ("Unknown option: " + args[count]);

			case 'H':	//	Help
			case 'h':
				Usage ();
			}
		}
	else
		{
		if (source_name != null)
			{
			System.out.println
				("Multiple sources specified -\n"
				+ source_name + '\n'
				+"and\n"
				+ args[count]);
			Usage ();
			}
		source_name = args[count];
		}
	}

Configuration.Default_Source (DEFAULT_CONFIGURATION_PATHNAME);
try {configuration = new Configuration (configuration_source);}
catch (IllegalArgumentException exception)
	{
	if (configuration_source != null)
		{
		System.out.println (ID + '\n'
			+"Unable to find the configuration source: "
				+ configuration_source + '\n'
			+ exception.getMessage ());
		System.exit (EXIT_CONFIGURATION_PROBLEM);
		}
	//	Ignore failure to find default config file.
	}
catch (Configuration_Exception exception)
	{
	System.out.println (ID + '\n'
		+"Unable to load the configuration source.\n"
		+ exception.getMessage ());
	System.exit (EXIT_CONFIGURATION_PROBLEM);
	}

if (source_name == null)
	{
	System.out.println
		("No input source specified.");
	Usage ();
	}
if (source_name.equals ("-"))
	{
	source_name = "stdin";
	source = System.in;
	}
else
if ((source = Streams.Get_Stream (source_name)) == null)
	{
	System.out.println
		("An input stream could not be obtained for " + args[0]);
	Usage ();
	}

Stream_Viewer
	viewer = new Stream_Viewer (title, source, configuration);
}

/**	Prints the command line usage syntax.
<p>
<blockquote><pre>
Usage: <b>Stream_Viewer</b> &lt;<i>Options</i>&gt;
&nbsp;&nbsp;Options -
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>C</u>onfiguration</b> &lt;<i>source</i>&gt;]
&nbsp;&nbsp;&nbsp;&nbsp;[<b>-<u>H</u>elp</b>]
</pre></blockquote>
<p>
	<b>N.B.</b>This method always results in a System.exit with a status
	of {@link #EXIT_COMMAND_LINE_SYNTAX}.

*/
public static void Usage ()
{
System.out.println
	(
	"Usage: Stream_Viewer <source>\n" +
	"  The source may be a filename, URL or '-' for stdin."
	);
System.exit (EXIT_COMMAND_LINE_SYNTAX);
}

}
