/*
 * Copyright (C) 2025 The Phosh Developers
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 *
 * Author: Guido Günther <agx@sigxcpu.org>
 */

#define G_LOG_DOMAIN "cbd-modem-3gpp"

#include "cbd-modem-3gpp.h"

#include "gmobile.h"

/**
 * CbdModem3gpp:
 *
 * Object tracking ModemManagers's 3gpp interface
 */

enum {
  PROP_0,
  PROP_MODEM_3GPP,
  PROP_COUNTRY,
  PROP_LAST_PROP
};
static GParamSpec *props[PROP_LAST_PROP];

struct _CbdModem3gpp {
  GObject      parent;

  MMModem3gpp *modem_3gpp;
  char        *operator_code;
  char        *country;
};
G_DEFINE_TYPE (CbdModem3gpp, cbd_modem_3gpp, G_TYPE_OBJECT)


static void
on_operator_code_changed (CbdModem3gpp *self)
{
  g_autoptr (GError) err = NULL;
  const char *operator_code, *country;

  g_assert (CBD_IS_MODEM_3GPP (self));
  operator_code = mm_modem_3gpp_get_operator_code (self->modem_3gpp);

  if (!g_strcmp0 (self->operator_code, operator_code))
    return;

  g_debug ("Operator is '%s'", operator_code);
  g_set_str (&self->operator_code, operator_code);

  if (!self->operator_code)
    return;

  country = gm_mcc_to_iso (self->operator_code, &err);
  if (!country) {
    g_warning ("Failed to determine country from '%s'", self->operator_code);
    return;
  }

  if (!g_strcmp0 (self->country, country))
    return;

  g_set_str (&self->country, country);

  g_object_notify_by_pspec (G_OBJECT (self), props[PROP_COUNTRY]);
}


static void
cbd_modem_3gpp_set_modem (CbdModem3gpp *self, MMModem3gpp *modem)
{
  /* Construct only */
  g_assert (self->modem_3gpp == NULL);

  self->modem_3gpp = g_object_ref (modem);

  g_signal_connect_swapped (self->modem_3gpp,
                            "notify::operator-code",
                            G_CALLBACK (on_operator_code_changed),
                            self);
  on_operator_code_changed (self);
}


static void
cbd_modem_3gpp_set_property (GObject      *object,
                             guint         property_id,
                             const GValue *value,
                             GParamSpec   *pspec)
{
  CbdModem3gpp *self = CBD_MODEM_3GPP (object);

  switch (property_id) {
  case PROP_MODEM_3GPP:
    cbd_modem_3gpp_set_modem (self, g_value_get_object (value));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    break;
  }
}


static void
cbd_modem_3gpp_get_property (GObject    *object,
                             guint       property_id,
                             GValue     *value,
                             GParamSpec *pspec)
{
  CbdModem3gpp *self = CBD_MODEM_3GPP (object);

  switch (property_id) {
  case PROP_MODEM_3GPP:
    g_value_set_object (value, self->modem_3gpp);
    break;
  case PROP_COUNTRY:
    g_value_set_string (value, cbd_modem_3gpp_get_country (self));
    break;
  default:
    G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    break;
  }
}



static void
cbd_modem_3gpp_dispose (GObject *object)
{
  CbdModem3gpp *self = CBD_MODEM_3GPP (object);

  g_signal_handlers_disconnect_by_data (self->modem_3gpp, self);

  g_clear_object (&self->modem_3gpp);

  G_OBJECT_CLASS (cbd_modem_3gpp_parent_class)->dispose (object);
}


static void
cbd_modem_3gpp_finalize (GObject *object)
{
  CbdModem3gpp *self = CBD_MODEM_3GPP (object);

  g_clear_pointer (&self->operator_code, g_free);
  g_clear_pointer (&self->country, g_free);

  G_OBJECT_CLASS (cbd_modem_3gpp_parent_class)->finalize (object);
}


static void
cbd_modem_3gpp_class_init (CbdModem3gppClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  object_class->get_property = cbd_modem_3gpp_get_property;
  object_class->set_property = cbd_modem_3gpp_set_property;
  object_class->dispose = cbd_modem_3gpp_dispose;
  object_class->finalize = cbd_modem_3gpp_finalize;

  props[PROP_MODEM_3GPP] =
    g_param_spec_object ("modem-3gpp", "", "",
                         MM_TYPE_MODEM_3GPP,
                         G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_STRINGS);

  props[PROP_COUNTRY] =
    g_param_spec_string ("country", "", "",
                         NULL,
                         G_PARAM_READABLE | G_PARAM_EXPLICIT_NOTIFY | G_PARAM_STATIC_STRINGS);

  g_object_class_install_properties (object_class, PROP_LAST_PROP, props);
}


static void
cbd_modem_3gpp_init (CbdModem3gpp *self)
{
}


CbdModem3gpp *
cbd_modem_3gpp_new (MMModem3gpp *modem)
{
  return g_object_new (CBD_TYPE_MODEM_3GPP, "modem-3gpp", modem, NULL);
}


gboolean
cbd_modem_3gpp_match (CbdModem3gpp *self, MMModem3gpp *modem_3gpp)
{
  g_assert (CBD_IS_MODEM_3GPP (self));

  return self->modem_3gpp == modem_3gpp;
}


const char *
cbd_modem_3gpp_get_country (CbdModem3gpp *self)
{
  g_assert (CBD_IS_MODEM_3GPP (self));

  return self->country;
}


const char *
cbd_modem_3gpp_get_operator_code (CbdModem3gpp *self)
{
  g_assert (CBD_IS_MODEM_3GPP (self));

  return self->operator_code;
}
