"""
This module offers functions to returns the lists of
polynomial strings of some well known examples.
The test solves all systems and tests on the number of solutions.
Running the test can take some time.
"""

def binomials():
    """
    A binomial system is a polynomial system where every equation
    has exactly two monomials with nonzero coefficient.
    A pure dimensional binomial system can be solved quickly.
    The example below has negative exponents.
    """
    pol1 = "x1*x2 - x3^-1*x4;"
    pol2 = "x1^3*x4^-3 + 2*x3^-1*x2;"
    pol3 = "x2^2*x3^-2 - 3*x1*x4^2;"
    pol4 = "x1*x2*x3*x4 - 1;"
    return [pol1, pol2, pol3, pol4]

def cyclic7():
    """
    The cyclic 7-roots problem is a standard benchmark problem
    for polynomial system solvers.
    Reference: Backelin, J. and Froeberg, R.:
    How we proved that there are exactly 924 cyclic 7-roots,
    Proceedings of ISSAC-91, pp 103-111, ACM, New York, 1991.
    """
    pol1 = "z0 + z1 + z2 + z3 + z4 + z5 + z6;"
    pol2 = "z0*z1 + z1*z2 + z2*z3 + z3*z4 + z4*z5 + z5*z6 + z6*z0;"
    pol3 = "z0*z1*z2 + z1*z2*z3 + z2*z3*z4 + z3*z4*z5 + z4*z5*z6" \
         + " + z5*z6*z0 + z6*z0*z1;"
    pol4 = "z0*z1*z2*z3 + z1*z2*z3*z4 + z2*z3*z4*z5 + z3*z4*z5*z6" \
         + "+ z4*z5*z6*z0 + z5*z6*z0*z1 + z6*z0*z1*z2;"
    pol5 = " z0*z1*z2*z3*z4 + z1*z2*z3*z4*z5 + z2*z3*z4*z5*z6" \
         + " + z3*z4*z5*z6*z0 + z4*z5*z6*z0*z1 + z5*z6*z0*z1*z2" \
         + " + z6*z0*z1*z2*z3;"
    pol6 = "z0*z1*z2*z3*z4*z5 + z1*z2*z3*z4*z5*z6 + z2*z3*z4*z5*z6*z0" \
         + " + z3*z4*z5*z6*z0*z1 + z4*z5*z6*z0*z1*z2" \
         + " + z5*z6*z0*z1*z2*z3 + z6*z0*z1*z2*z3*z4;"
    pol7 = "z0*z1*z2*z3*z4*z5*z6 - 1;"
    return [pol1, pol2, pol3, pol4, pol5, pol6, pol7]

def sysd1():
    """
    A sparse system, know as benchmark D1 of the paper by
    H. Hong and V. Stahl:
    Safe Starting Regions by Fixed Points and Tightening,
    Computing 53(3-4): 322-335, 1995.
    """
    pol1 = "x1**2  + x2**2 - 1;"
    pol2 = "x3**2  + x4**2 - 1;"
    pol3 = "x5**2  + x6**2 - 1;"
    pol4 = "x7**2  + x8**2 - 1;"
    pol5 = "x9**2  + x10**2 - 1;"
    pol6 = "x11**2 + x12**2 - 1;"
    pol7 = "3*x3 + 2*x5 + x7 - 3.9701;"
    pol8 = "3*x1*x4 + 2*x1*x6 + x1*x8 - 1.7172;"
    pol9 = "3*x2*x4 + 2*x2*x6 + x2*x8 - 4.0616;"
    pol10 = "x3*x9 + x5*x9 + x7*x9 - 1.9791;"
    pol11 = "x2*x4*x9 + x2*x6*x9 + x2*x8*x9 + x1*x10 - 1.9115;"
    pol12 = "- x3*x10*x11 - x5*x10*x11 - x7*x10*x11 + x4*x12" \
        + " + x6*x12 + x8*x12 - 0.4077;"
    return [pol1, pol2, pol3, pol4, pol5, pol6, pol7, pol8, pol9, \
    pol10, pol11, pol12]

def fbrfive4():
    """
    Four-bar linkage through five points, 4-dimensional version.
    Reference: Charles W. Wampler:
    Isotropic coordinates, circularity and Bezout numbers:
    planar kinematics from a new perspective.
    Publication R&D-8188, General Motors Corporation, Research
    and Development Center.
    Proceedings of the 1996 ASME Design Engineering Technical Conference,
    Irvine, California August 18-22, 1996. CD-ROM edited by McCarthy, J.M.,
    American society of mechanical engineers.
    """
    pol1 = "0.141050018728*i*bx**2*by*y - 0.00097847866344*bx**2*by*y" \
         + " + 0.0782395368483*i*bx**2*y**2 + 0.00123918464673*i*bx**2*y" \
         + " + 0.00969424108804*bx**2*y**2 + 0.00125705255662*bx**2*y" \
         + " + 0.0328315845569*i*bx*by**2*x - 0.0725922104353*bx*by**2*x" \
         + " + 0.000160324879058*i*bx*by*x - 0.14642522425*i*bx*by*y" \
         + " - 0.2391158765*bx*by*x*y - 0.00109445130977*bx*by*x" \
         + " + 0.168022431284*bx*by*y - 0.0328315845569*i*bx*x*y**2" \
         + " - 0.000160324879058*i*bx*x*y - 0.0471003874787*i*bx*y**2" \
         + " - 0.00105590259776*i*bx*y - 0.0725922104353*bx*x*y**2" \
         + " - 0.00109445130977*bx*x*y - 0.0000135684269291*bx*x" \
         + " + 0.0654830371929*bx*y**2 - 0.000300187113597*bx*y" \
         + " - 0.0782395368483*i*by**2*x**2 + 0.0471003874787*i*by**2*x" \
         + " + 0.00969424108804*by**2*x**2 + 0.0654830371929*by**2*x" \
         + " - 0.141050018728*i*by*x**2*y - 0.00123918464673*i*by*x**2" \
         + " + 0.14642522425*i*by*x*y + 0.00105590259776*i*by*x" \
         + " - 0.00097847866344*by*x**2*y + 0.00125705255662*by*x**2" \
         + " + 0.168022431284*by*x*y - 0.000300187113597*by*x" \
         + " - 0.0888122554974*by*y;"
    pol2 = " - 0.0249469889303*i*bx**2*by*y + 0.00931583747381*bx**2*by*y" \
         + " + 0.0294053156051*i*bx**2*y**2 - 0.00105533016484*i*bx**2*y" \
         + " + 0.00786359271692*bx**2*y**2 - 0.0177007048352*bx**2*y" \
         + " + 0.0595383104696*i*bx*by**2*x + 0.079236107727*bx*by**2*x" \
         + " - 0.108418415833*i*bx*by*x + 0.00451693895279*i*bx*by*y" \
         + " - 0.205718481117*bx*by*x*y + 0.0366433756262*bx*by*x" \
         + " - 0.00437803776486*bx*by*y - 0.0595383104696*i*bx*x*y**2" \
         + " + 0.108418415833*i*bx*x*y - 0.0129569311769*i*bx*y**2" \
         + " + 0.00353939806669*i*bx*y + 0.079236107727*bx*x*y**2" \
         + " + 0.0366433756262*bx*x*y - 0.0322818097234*bx*x" \
         + " + 0.00301410665727*bx*y**2 + 0.00546021583064*bx*y" \
         + " - 0.0294053156051*i*by**2*x**2 + 0.0129569311769*i*by**2*x"  \
         + " + 0.00786359271692*by**2*x**2 + 0.00301410665727*by**2*x" \
         + " + 0.0249469889303*i*by*x**2*y + 0.00105533016484*i*by*x**2" \
         + " - 0.00451693895279*i*by*x*y - 0.00353939806669*i*by*x" \
         + " + 0.00931583747381*by*x**2*y - 0.0177007048352*by*x**2" \
         + " - 0.00437803776486*by*x*y + 0.00546021583064*by*x" \
         + " - 0.0013116146819*by*y;"
    pol3 = " - 0.0218107478514*i*bx**2*by*y - 0.0227503225264*bx**2*by*y" \
         + " + 0.0145785542176*i*bx**2*y**2 + 0.0133382023419*i*bx**2*y" \
         + " + 0.0289746279868*bx**2*y**2 - 0.0104815931774*bx**2*y" \
         + " + 0.0537999863131*i*bx*by**2*x + 0.112859448991*bx*by**2*x" \
         + " - 0.124451937985*i*bx*by*x + 0.00929216513796*i*bx*by*y" \
         + " - 0.249135161545*bx*by*x*y + 0.0288876047041*bx*by*x" \
         + " + 0.00595806356335*bx*by*y - 0.0537999863131*i*bx*x*y**2" \
         + " + 0.124451937985*i*bx*x*y - 0.00917830603125*i*bx*y**2" \
         + " - 0.00230902546847*i*bx*y + 0.112859448991*bx*x*y**2" \
         + " + 0.0288876047041*bx*x*y - 0.0330774656043*bx*x" \
         + " - 0.00830926429131*bx*y**2 + 0.00563037140451*bx*y" \
         + " - 0.0145785542176*i*by**2*x**2 + 0.00917830603125*i*by**2*x" \
         + " + 0.0289746279868*by**2*x**2 - 0.00830926429131*by**2*x" \
         + " + 0.0218107478514*i*by*x**2*y - 0.0133382023419*i*by*x**2" \
         + " - 0.00929216513796*i*by*x*y + 0.00230902546847*i*by*x" \
         + " - 0.0227503225264*by*x**2*y - 0.0104815931774*by*x**2" \
         + " + 0.00595806356335*by*x*y + 0.00563037140451*by*x" \
         + " - 0.0011195743111*by*y;"
    pol4 = " - 0.0114868064801*i*bx**2*by*y - 0.0119413549909*bx**2*by*y" \
         + " - 0.0139828726508*i*bx**2*y**2 + 0.000914652140862*i*bx**2*y" \
         + " - 0.0209721546687*bx**2*y**2 + 0.00721574419444*bx**2*y" \
         + " - 0.00930458610211*i*bx*by**2*x + 0.0140687638574*bx*by**2*x" \
         + " + 0.00726787677736*i*bx*by*x + 0.00908990049543*i*bx*by*y" \
         + " + 0.0363929678175*bx*by*x*y - 0.00210685119583*bx*by*x" \
         + " - 0.0189904701902*bx*by*y + 0.00930458610211*i*bx*x*y**2" \
         + " - 0.00726787677736*i*bx*x*y + 0.0207832691669*i*bx*y**2" \
         + " + 0.000392165732469*i*bx*y + 0.0140687638574*bx*x*y**2" \
         + " - 0.00210685119583*bx*x*y - 0.000223340726321*bx*x" \
         + " - 0.00631749512119*bx*y**2 + 0.000524903641698*bx*y" \
         + " + 0.0139828726508*i*by**2*x**2 - 0.0207832691669*i*by**2*x" \
         + " - 0.0209721546687*by**2*x**2 - 0.00631749512119*by**2*x" \
         + " + 0.0114868064801*i*by*x**2*y - 0.000914652140862*i*by*x**2" \
         + " - 0.00908990049543*i*by*x*y - 0.000392165732469*i*by*x" \
         + " - 0.0119413549909*by*x**2*y + 0.00721574419444*by*x**2" \
         + " - 0.0189904701902*by*x*y + 0.000524903641698*by*x" \
         + " - 0.00192225485187*by*y;"
    return [pol1, pol2, pol3, pol4]

def game4two():
    """
    Totally mixed Nash equilibria for 4 players with two pure strategies.
    See the paper by Andrew McLennan on The maximal generic number of pure
    Nash equilibria, Journal of Economic Theory, Vol 72, pages 408-410, 1997.
    Another reference is the paper by Richard D. McKelvey and Andrew McLennan:
    on The maximal number of regular totally mixed Nash equilibria,
    Journal of Economic Theory, Volume 72, pages 411-425, 1997.
    For more on equilibria, see David M. Kreps and Robert Wilson:
    Sequential Equibria, Econometrica, Vol 50, No 4, pages 863-894,1982.
    """
    pol1 = "- 1.122525605*p3 - 0.1202590019*p3*p2 + 1.078803537*p2" \
         + " + 0.7696590455*p4 + 0.1628390188*p4*p3*p2 - 0.1163105815*p4*p3" \
         + " - 0.5333928492*p4*p2 - 0.6823118878;"
    pol2 = "- 0.4266067331*p3 + 0.9756757996*p4 + 0.646954510E-01*p4*p3" \
         + "- 0.2142068753*p3*p1 + 2.814522341*p1 + 0.3980853635*p4*p3*p1" \
         + "- 2.546266040*p4*p1 - 1.157918209;"
    pol3 = "2.023260100*p2 - 0.1342402078*p4 - 0.3904810713*p4*p2" \
         + " + 0.4274147938*p1 + 0.5269506245*p4*p1 - 3.069473137*p2*p1" \
         + " + 0.3284270487*p4*p2*p1 - 0.9128973443;"
    pol4 = "- 0.6256820158*p3 - 0.1614530476*p3*p2 + 1.080721123*p2" \
         + " + 1.366746822*p3*p1 - 2.347725424*p1 + 0.5941402017*p2*p1" \
         + " - 2.233884240*p3*p2*p1 + 1.026687422;"
    return [pol1, pol2, pol3, pol4]

def katsura6():
    """
    A problem of magnetism in physics, see for examples the paper
    by S. Katsura, in New Trends in Magnetism,
    edited by M.D. Coutinho-Filho and S.M. Resende, World Scientific, 1990.
    It became part of the PoSSo test suite,
    Shigotoshi Katsura: "Users posing problems to PoSSo",
    in the PoSSo Newsletter, no. 2, July 1994,
    edited by L. Gonzalez-Vega and T. Recio.
    See also the paper by W. Boege, R. Gebauer, and H. Kredel:
    Some examples for solving systems of algebraic equations by
    calculating Groebner bases, J. Symbolic Computation, 2:83-98, 1986.
    The system can be written for any dimension.
    """
    pol1 = "1*x1+2*x2+2*x3+2*x4+2*x5+2*x6+2*x7-1;"
    pol2 = "2*x4*x3+2*x5*x2+2*x6*x1+2*x7*x2-1*x6;"
    pol3 = "1*x3^2+2*x4*x2+2*x5*x1+2*x6*x2+2*x7*x3-1*x5;"
    pol4 = "2*x3*x2+2*x4*x1+2*x5*x2+2*x6*x3+2*x7*x4-1*x4;"
    pol5 = "1*x2^2+2*x3*x1+2*x4*x2+2*x5*x3+2*x6*x4+2*x7*x5-1*x3;"
    pol6 = "2*x2*x1+2*x3*x2+2*x4*x3+2*x5*x4+2*x6*x5+2*x7*x6-1*x2;"
    pol7 = "1*x1^2+2*x2^2+2*x3^2+2*x4^2+2*x5^2+2*x6^2+2*x7^2-1*x1;"
    return [pol1, pol2, pol3, pol4, pol5, pol6, pol7]

def noon3():
    """
    A neural network modeled by an adaptive Lotka-Volterra system.
    See the paper by Karin Gatermann on Symbolic solution of polynomial
    equation systems with symmetry, Proceedings of ISSAC-90, pages 112-119,
    ACM New York, 1990.
    The system belongs to a family of problems of V.W. Noonburg:
    A neural network modeled by an adaptive Lotka-Volterra system,
    SIAM J. Appl. Math., Vol. 49, No. 6, 1779-1792, 1989.
    """
    pol1 = "x1*x2^2 + x1*x3^2 - 1.1*x1 + 1;"
    pol2 = "x2*x1^2 + x2*x3^2 - 1.1*x2 + 1;"
    pol3 = "x3*x1^2 + x3*x2^2 - 1.1*x3 + 1;"
    return [pol1, pol2, pol3]

def rps10():
    """
    RPS Serial Chains for 10 positions on a circular hyperboloid.
    From the paper of Hai-Jun Su and J. Michael McCarthy:
    Kinematics Synthesis of RPS Serial Chains,
    In Proceedings of the ASME Design Engineering Technical Conferences
    (CDROM).  Paper DETC03/DAC-48813.  Chicago, IL, Sept.02-06, 2003.
    """
    pol1 = "-0.1279703687075118*g1**2 - 0.48596123125526264*g1*g2" \
         + " + 0.30699556370717496*g2**2 + 0.3778977698527674*g1*g3" \
         + " - 0.23404544076569642*g2*g3 + 0.01563626178508072*g3**2" \
         + " + 0.327228678790004*g1**2*p1 + 0.8426829275672494*g1*g2*p1" \
         + " + 0.6075645757034159*g2**2*p1 - 1.1371405598667543*g1*g3*p1" \
         + " + 0.229293271620915*g2*g3*p1 - 0.21948911177437957*g3**2*p1" \
         + " - 0.2075154964282774*g1**2*p1**2" \
         + " - 0.37702968479068544*g1*g2*p1**2" \
         + " - 0.16688906819159421*g2**2*p1**2" \
         + " + 0.7986954318323025*g1*g3*p1**2" \
         + " + 0.866826144775651*g2*g3*p1**2" \
         + " + 0.37440456461987165*g3**2*p1**2" \
         + " + 1.5614616440131446*g1**2*p2 - 1.7388380675822595*g1*g2*p2" \
         + " + 0.06790915713070725*g2**2*p2 - 0.4309121044684771*g1*g3*p2" \
         + " + 0.9086272006283425*g2*g3*p2 - 0.2764931751394603*g3**2*p2" \
         + " - 1.8163349832174116*g1**2*p1*p2" \
         + " - 0.9167144057621401*g1*g2*p1*p2" \
         + " + 1.0203368504488892*g2**2*p1*p2" \
         + " - 0.23194646823111892*g1*g3*p1*p2" \
         + " + 0.539670777307627*g2*g3*p1*p2" \
         + " + 0.7959981327685224*g3**2*p1*p2" \
         + " + 0.08717268867521591*g1**2*p2**2" \
         + " + 0.9504154644263471*g1*g2*p2**2" \
         + " - 0.48206756571420756*g2**2*p2**2" \
         + " - 1.065062423127697*g1*g3*p2**2" \
         + " + 0.1209952909274163*g2*g3*p2**2" \
         + " + 0.3948948770389917*g3**2*p2**2" \
         + " + 0.289766299873838*g1**2*p3 - 1.2778927965251532*g1*g2*p3" \
         + " + 0.9087896778886251*g2**2*p3 - 0.5812612591154215*g1*g3*p3" \
         + " - 0.7595904624983555*g2*g3*p3 + 0.5084892760496751*g3**2*p3" \
         + " - 0.3268802641947883*g1**2*p1*p3" \
         + " + 0.657630238424344*g1*g2*p1*p3" \
         + " + 1.1093919363972093*g2**2*p1*p3" \
         + " + 0.4551393419480071*g1*g3*p1*p3" \
         + " + 1.8553852513069364*g2*g3*p1*p3" \
         + " - 0.7825116722024211*g3**2*p1*p3" \
         + " + 0.5810469298461638*g1**2*p2*p3" \
         + " - 1.1557382363783264*g1*g2*p2*p3" \
         + " - 0.11367961187637783*g2**2*p2*p3" \
         + " + 1.7077140933509898*g1*g3*p2*p3" \
         + " - 0.36547942767108677*g2*g3*p2*p3" \
         + " - 0.4673673179697859*g3**2*p2*p3" \
         + " + 0.12034280775306151*g1**2*p3**2" \
         + " - 0.5733857796356615*g1*g2*p3**2" \
         + " + 0.6489566339058018*g2**2*p3**2" \
         + " + 0.2663669912953945*g1*g3*p3**2" \
         + " - 0.9878214357030672*g2*g3*p3**2" \
         + " - 0.7692994416588633*g3**2*p3**2" \
         + " - 0.19466145678474384*q0 - 0.7153041427190404*p1*q0" \
         + " - 1.3528776260043915*p2*q0 - 1.7070452538121381*p3*q0" \
         + " - 1.0516635822669562*q1 + 1.2244185478631853*p1*q1" \
         + " - 0.05844567698552443*p2*q1 - 0.37706149953585283*p3*q1" \
         + " + 0.580102254517945*q2 + 1.2898860704586343*p1*q2" \
         + " - 0.6655948497180294*p2*q2 + 0.697758704890495*p3*q2" \
         + " - 0.042921436747585445*q3 + 0.5172073855756967*p1*q3" \
         + " + 0.6917094054122289*p2*q3 - 1.4579672250860476*p3*q3;"
    pol2 = "0.16011034303688113*g1**2 - 0.9005468824403076*g1*g2" \
         + " - 0.3519015838689263*g2**2 + 0.5202586158306898*g1*g3" \
         + " + 0.908682123022068*g2*g3 - 0.4464562170645777*g3**2" \
         + " - 0.13844524415679324*g1**2*p1 + 1.5568085644333742*g1*g2*p1" \
         + " + 1.6863862382239232*g2**2*p1 - 1.7409458121154344*g1*g3*p1" \
         + " - 0.13872356093602894*g2*g3*p1 - 0.5159047084859331*g3**2*p1" \
         + " - 0.2741643484200128*g1**2*p1**2" \
         + " - 0.34212012775550327*g1*g2*p1**2" \
         + " - 0.07542436599114127*g2**2*p1**2" \
         + " + 0.37458987278720324*g1*g3*p1**2" \
         + " + 0.4782561996467687*g2*g3*p1**2" \
         + " + 0.3495887144111541*g3**2*p1**2" \
         + " + 0.41377445473869573*g1**2*p2 - 1.5789383736211624*g1*g2*p2" \
         + " + 1.268319517294935*g2**2*p2 + 0.6163793667190677*g1*g3*p2" \
         + " - 0.43374574206406646*g2*g3*p2 - 0.2061458017243186*g3**2*p2" \
         + " + 0.14555549639831628*g1**2*p1*p2" \
         + " - 1.1674745895517964*g1*g2*p1*p2" \
         + " - 0.9428064489876502*g2**2*p1*p2" \
         + " + 0.0024916775818734295*g1*g3*p1*p2" \
         + " + 0.5291621555283466*g2*g3*p1*p2" \
         + " + 0.7972509525893339*g3**2*p1*p2" \
         + " + 0.1807885464109201*g1**2*p2**2" \
         + " + 0.9404541869824675*g1*g2*p2**2" \
         + " - 0.5780030515551372*g2**2*p2**2" \
         + " - 1.0257418447585547*g1*g3*p2**2" \
         + " + 0.09251778173989315*g2*g3*p2**2" \
         + " + 0.39721450514421713*g3**2*p2**2" \
         + " + 0.40272988912109214*g1**2*p3 - 0.8272484673958682*g1*g2*p3" \
         + " + 1.057139636924469*g2**2*p3 - 0.12353226665002319*g1*g3*p3" \
         + " - 2.5741855761862396*g2*g3*p3 + 1.560474007685759*g3**2*p3" \
         + " - 0.6150996832616941*g1**2*p1*p3" \
         + " + 0.09937192239106099*g1*g2*p1*p3" \
         + " + 0.8226042775491553*g2**2*p1*p3" \
         + " + 0.4732438203631739*g1*g3*p1*p3" \
         + " + 1.6946050580334677*g2*g3*p1*p3" \
         + " - 0.20750459428746135*g3**2*p1*p3" \
         + " + 0.7556396990592089*g1**2*p2*p3" \
         + " - 1.412614951501404*g1*g2*p2*p3" \
         + " - 0.09676545515565128*g2**2*p2*p3" \
         + " - 0.9781576342585658*g1*g3*p2*p3" \
         + " + 2.5006617995144724*g2*g3*p2*p3" \
         + " - 0.6588742439035575*g3**2*p2*p3" \
         + " + 0.09337580200909272*g1**2*p3**2" \
         + " - 0.5983340592269643*g1*g2*p3**2" \
         + " + 0.6534274175462785*g2**2*p3**2" \
         + " + 0.6511519719713513*g1*g3*p3**2" \
         + " - 0.5707739813866619*g2*g3*p3**2" \
         + " - 0.7468032195553712*g3**2*p3**2" \
         + " + 0.6382474578966228*q0 - 1.032036285581197*p1*q0" \
         + " - 1.4759481703093122*p2*q0 - 3.0203435337313205*p3*q0" \
         + " - 0.22812438675350769*q1 - 0.2157590670168509*p1*q1" \
         + " - 0.1270558344695696*p2*q1 - 0.5148593639524484*p3*q1" \
         + " + 1.3667793800860086*q2 - 0.06171123442924746*p1*q2" \
         + " - 0.7314954155886625*p2*q2 + 0.7189348075213543*p3*q2" \
         + " - 0.902118536026858*q3 + 0.43214823742186254*p1*q3" \
         + " + 0.6677624868260497*p2*q3 + 0.5162571144422815*p3*q3;"
    pol3 = "0.20816475809219404*g1**2 - 0.44624795696445435*g1*g2" \
         + " + 0.1573457781818856*g2**2 + 0.2432511536576595*g1*g3" \
         + " - 0.4587424991969163*g2*g3 - 0.048748564896809544*g3**2" \
         + " - 0.542424299098038*g1**2*p1 + 0.412606879197033*g1*g2*p1" \
         + " + 0.026228493490255755*g2**2*p1 - 0.7135454436169615*g1*g3*p1" \
         + " + 0.38566725023570736*g2*g3*p1 + 0.06641935708182738*g3**2*p1" \
         + " + 0.1946156279601214*g1**2*p1**2" \
         + " + 1.0726514255671113*g1*g2*p1**2" \
         + " - 0.29746151974577967*g2**2*p1**2" \
         + " + 0.3474150051655493*g1*g3*p1**2" \
         + " + 1.5662794253637933*g2*g3*p1**2" \
         + " + 0.10284589178565828*g3**2*p1**2" \
         + " + 0.8655281158446179*g1**2*p2 - 1.4227007533612923*g1*g2*p2" \
         + " - 0.3561608986253729*g2**2*p2 - 0.7002053827479838*g1*g3*p2" \
         + " + 0.1451263721376322*g2*g3*p2 - 0.4446427929457582*g3**2*p2" \
         + " - 0.35089234105147404*g1**2*p1*p2" \
         + " - 1.8637325747105546*g1*g2*p1*p2" \
         + " - 0.7643910878410862*g2**2*p1*p2" \
         + " + 0.7421389633104346*g1*g3*p1*p2" \
         + " + 0.8043890896223826*g2*g3*p1*p2" \
         + " + 1.1152834288925604*g3**2*p1*p2" \
         + " - 0.05058692105297476*g1**2*p2**2" \
         + " - 0.8545531093164939*g1*g2*p2**2" \
         + " - 0.25045809562785276*g2**2*p2**2" \
         + " - 1.482438556873845*g1*g3*p2**2" \
         + " - 0.2760311985894717*g2*g3*p2**2" \
         + " + 0.30104501668082756*g3**2*p2**2" \
         + " + 0.41615153726461007*g1**2*p3 - 1.6031132124173149*g1*g2*p3" \
         + " + 1.1652768530802575*g2**2*p3 + 0.1236694347662175*g1*g3*p3 " \
         + " - 0.033510271732486586*g2*g3*p3 + 0.6625023868605743*g3**2*p3" \
         + " - 0.06941899872446193*g1**2*p1*p3" \
         + " - 0.5612725019588681*g1*g2*p1*p3" \
         + " + 1.4835363108262836*g2**2*p1*p3" \
         + " - 0.8310204341509994*g1*g3*p1*p3" \
         + " + 1.3650887611787323*g2*g3*p1*p3" \
         + " - 1.4141173121018216*g3**2*p1*p3" \
         + " - 0.2915853970368523*g1**2*p2*p3" \
         + " - 1.2521117933146961*g1*g2*p2*p3" \
         + " + 0.38706376702247*g2**2*p2*p3" \
         + " + 1.2309129178715645*g1*g3*p2*p3" \
         + " + 2.001338697637118*g2*g3*p2*p3" \
         + " - 0.09547836998561768*g3**2*p2*p3" \
         + " - 0.14402870690714664*g1**2*p3**2" \
         + " - 0.2180983162506176*g1*g2*p3**2" \
         + " + 0.5479196153736324*g2**2*p3**2" \
         + " + 1.1350235517082958*g1*g3*p3**2" \
         + " - 1.2902482267743214*g2*g3*p3**2" \
         + " - 0.40389090846648584*g3**2*p3**2" \
         + " - 0.31676197137727014*q0 + 0.44977644852595483*p1*q0" \
         + " - 0.06472442427348668*p2*q0 - 2.2439307772054415*p3*q0" \
         + " - 0.5547165223690258*q1 + 0.23831878651082344*p1*q1" \
         + " + 0.031977776730485255*p2*q1 + 0.16687455406564522*p3*q1" \
         + " + 0.9423377906275198*q2 + 1.376589178886685*p1*q2" \
         + " + 0.5306523901876015*p2*q2 + 0.4754891181933043*p3*q2" \
         + " + 0.09673230093655334*q3 + 0.0892904130224598*p1*q3" \
         + " + 0.943551163213123*p2*q3 - 1.2527250130712726*p3*q3;"
    pol4 = "-0.04095049824628835*g1**2 + 0.043116025511842154*g1*g2" \
         + " + 0.003940499198786224*g2**2 + 0.7629770334036455*g1*g3" \
         + " - 0.8492350760146794*g2*g3 - 0.2784174783424625*g3**2" \
         + " + 0.4281444292173086*g1**2*p1 - 0.8828960936117035*g1*g2*p1" \
         + " + 0.5676682886279524*g2**2*p1 - 1.3924527881029736*g1*g3*p1" \
         + " - 0.08671339002537767*g2*g3*p1 - 0.7256104095984146*g3**2*p1" \
         + " - 0.5422313181564682*g1**2*p1**2" \
         + " + 0.4871946471731439*g1*g2*p1**2" \
         + " + 0.33193585698170985*g2**2*p1**2" \
         + " - 0.10273772900088107*g1*g3*p1**2" \
         + " + 0.6624874115365778*g2*g3*p1**2" \
         + " + 0.21029546117475836*g3**2*p1**2" \
         + " + 1.1356911567255628*g1**2*p2 - 1.222100685178249*g1*g2*p2" \
         + " - 0.09334002143332033*g2**2*p2 - 1.6524959396527132*g1*g3*p2" \
         + " + 1.5569725124184146*g2*g3*p2 + 0.7669386068453008*g3**2*p2" \
         + " - 1.5587675264538823*g1**2*p1*p2" \
         + " - 1.1317581527003464*g1*g2*p1*p2" \
         + " + 0.01829080736739283*g2**2*p1*p2" \
         + " + 1.0020439814840232*g1*g3*p1*p2" \
         + " + 0.7984049760283556*g2*g3*p1*p2" \
         + " + 1.5404767190864894*g3**2*p1*p2" \
         + " + 0.4331904808414006*g1**2*p2**2" \
         + " - 0.1389285705830233*g1*g2*p2**2" \
         + " - 0.012183710127155694*g2**2*p2**2" \
         + " - 0.23769385136664706*g1*g3*p2**2" \
         + " + 0.24382837068218804*g2*g3*p2**2" \
         + " - 0.4210067707142449*g3**2*p2**2" \
         + " + 0.21198555697618326*g1**2*p3 - 0.01647709154129892*g1*g2*p3" \
         + " - 0.18945218115272705*g2**2*p3 + 0.77528301920843*g1*g3*p3" \
         + " - 2.0514046696465*g2*g3*p3 + 0.10432028494512646*g3**2*p3" \
         + " - 0.1676311045685404*g1**2*p1*p3" \
         + " - 1.325546381572095*g1*g2*p1*p3" \
         + " + 0.8078478514339609*g2**2*p1*p3" \
         + " - 0.9669176278885212*g1*g3*p1*p3" \
         + " - 0.36872926793739896*g2*g3*p1*p3" \
         + " - 0.6402167468654205*g3**2*p1*p3" \
         + " + 0.6297207100844667*g1**2*p2*p3" \
         + " - 2.067771321161895*g1*g2*p2*p3 " \
         + " - 0.17862819697751522*g2**2*p2*p3" \
         + " + 0.020391323549034297*g1*g3*p2*p3" \
         + " + 2.777563669744398*g2*g3*p2*p3" \
         + " - 0.45109251310695153*g3**2*p2*p3" \
         + " + 0.10904083731506761*g1**2*p3**2" \
         + " - 0.3482660765901206*g1*g2*p3**2" \
         + " - 0.31975214685455416*g2**2*p3**2" \
         + " + 0.34043158036752813*g1*g3*p3**2" \
         + " - 0.9063157822187657*g2*g3*p3**2" \
         + " + 0.21071130953948652*g3**2*p3**2" \
         + " + 0.31542747738996463*q0 - 0.27020230824684643*p1*q0" \
         + " - 1.8092897421375431*p2*q0 - 0.12685366076858268*p3*q0" \
         + " - 0.9721054060313574*q1 + 1.4332583965298273*p1*q1" \
         + " - 0.3658292969614953*p2*q1 - 0.458292808629767*p3*q1" \
         + " + 1.1057480001700448*q2 + 0.649216154064302*p1*q2" \
         + " + 0.1435470147844548*p2*q2 + 1.8049686045262234*p3*q2" \
         + " + 0.3619641675513017*q3 + 1.0386298649000567*p1*q3" \
         + " - 0.2739870731830222*p2*q3 - 0.38992289294835114*p3*q3;"
    pol5 = "-0.41615764608945516*g1**2 - 1.2331171001793817*g1*g2" \
         + " + 0.10423594498637195*g2**2 + 0.4451741240918564*g1*g3" \
         + " - 0.0807794759847403*g2*g3 + 0.015584822151867354*g3**2" \
         + " + 0.5169791211840113*g1**2*p1 - 0.3281633186673521*g1*g2*p1" \
         + " + 0.10768082059655043*g2**2*p1 - 1.78387184821123*g1*g3*p1" \
         + " + 0.1962385955438586*g2*g3*p1 - 0.0932755727182936*g3**2*p1" \
         + " - 0.12307658314371513*g1**2*p1**2" \
         + " + 1.3649915585405705*g1*g2*p1**2" \
         + " - 0.01641144275933561*g2**2*p1**2" \
         + " + 1.282522294958988*g1*g3*p1**2" \
         + " + 0.13118389677242223*g2*g3*p1**2" \
         + " + 0.13948802590305073*g3**2*p1**2" \
         + " - 0.4784260518169776*g1**2*p2 - 2.092134198423298*g1*g2*p2" \
         + " - 0.2652478875380973*g2**2*p2 - 0.9322070346912057*g1*g3*p2" \
         + " + 0.28229645793462466*g2*g3*p2 - 0.07438003692790207*g3**2*p2" \
         + " + 0.1376969425780227*g1**2*p1*p2" \
         + " - 1.3462021315216954*g1*g2*p1*p2" \
         + " - 0.35831006800801096*g2**2*p1*p2" \
         + " - 0.3236078908735904*g1*g3*p1*p2" \
         + " - 0.23312973570099904*g2*g3*p1*p2" \
         + " + 0.2206131254299883*g3**2*p1*p2" \
         + " + 0.03212463900566726*g1**2*p2**2" \
         + " - 0.543299069419884*g1*g2*p2**2" \
         + " - 0.1081313237618179*g2**2*p2**2" \
         + " - 1.8348908578280814*g1*g3*p2**2" \
         + " - 0.3379484782876818*g2*g3*p2**2" \
         + " + 0.07600668475615065*g3**2*p2**2" \
         + " + 0.6002572265406737*g1**2*p3 + 0.1802055521069689*g1*g2*p3" \
         + " + 1.5644331744196656*g2**2*p3 - 0.5267676534246675*g1*g3*p3" \
         + " - 1.676157338774887*g2*g3*p3 + 0.4008307913692461*g3**2*p3" \
         + " - 0.5875157312206984*g1**2*p1*p3" \
         + " - 0.26794777918572443*g1*g2*p1*p3" \
         + " + 1.8161766879761405*g2**2*p1*p3" \
         + " + 0.08217873131336825*g1*g3*p1*p3" \
         + " + 0.41625228622759664*g2*g3*p1*p3" \
         + " - 1.228660956755442*g3**2*p1*p3" \
         + " + 0.801736157469905*g1**2*p2*p3" \
         + " - 0.06119905069237429*g1*g2*p2*p3" \
         + " + 0.5320140756032581*g2**2*p2*p3" \
         + " - 0.5120321678148483*g1*g3*p2*p3" \
         + " + 1.3994959245799465*g2*g3*p2*p3" \
         + " - 1.3337502330731632*g3**2*p2*p3" \
         + " + 0.09095194413804787*g1**2*p3**2" \
         + " - 0.8216924891206866*g1*g2*p3**2" \
         + " + 0.12454276652115351*g2**2*p3**2" \
         + " + 0.5523685628690934*g1*g3*p3**2" \
         + " + 0.20676458151525956*g2*g3*p3**2" \
         + " - 0.2154947106592014*g3**2*p3**2" \
         + " + 0.29633687895121585*q0 - 0.5313843690622682*p1*q0" \
         + " + 0.8180539762829769*p2*q0 - 2.5655211923295855*p3*q0" \
         + " - 1.2602089223582702*q1 + 1.1609512634985952*p1*q1" \
         + " - 1.5289512288758575*p2*q1 - 0.17424561505966216*p3*q1" \
         + " + 0.10364901876603111*q2 - 0.013973408764994696*p1*q2" \
         + " - 0.30080374272031296*p2*q2 + 1.5473286276087392*p3*q2" \
         + " - 0.04276145686338927*q3 + 0.12497482938060817*p1*q3" \
         + " + 0.06337974500071619*p2*q3 - 1.1223229062458282*p3*q3;"
    pol6 = "0.0995239208560676*g1**2 - 0.4069835646127759*g1*g2" \
         + " + 0.2822661867004282*g2**2 + 0.8451107786883508*g1*g3" \
         + " - 0.5316708978452792*g2*g3 - 0.8798108963881374*g3**2" \
         + " - 0.08841286657967666*g1**2*p1 - 0.5437440767827946*g1*g2*p1" \
         + " + 0.8310187547943032*g2**2*p1 - 2.300828301805621*g1*g3*p1" \
         + " + 1.7576212044508612*g2*g3*p1 + 0.8123567734335738*g3**2*p1" \
         + " - 0.06820768665581543*g1**2*p1**2" \
         + " + 1.3043060864108098*g1*g2*p1**2" \
         + " - 0.009054065062096384*g2**2*p1**2" \
         + " + 1.3077574118576432*g1*g3*p1**2" \
         + " + 0.06768474224643237*g2*g3*p1**2" \
         + " + 0.07726175171791182*g3**2*p1**2" \
         + " + 0.21964796928975452*g1**2*p2 - 0.7213869971382741*g1*g2*p2" \
         + " + 0.06428880062023586*g2**2*p2 + 0.6769693907722261*g1*g3*p2" \
         + " + 1.0938889186830445*g2*g3*p2 + 1.033778176369894*g3**2*p2" \
         + " + 0.04579106142198317*g1**2*p1*p2" \
         + " - 1.6136177673887118*g1*g2*p1*p2" \
         + " - 0.08313705215573315*g2**2*p1*p2" \
         + " - 0.3180870028357637*g1*g3*p1*p2" \
         + " - 0.1323253404673372*g2*g3*p1*p2" \
         + " + 0.03734599073374998*g3**2*p1*p2" \
         + " + 0.06270407236710761*g1**2*p2**2" \
         + " - 0.19162792586107563*g1*g2*p2**2" \
         + " - 0.009068232422623026*g2**2*p2**2" \
         + " - 1.9648502282295741*g1*g3*p2**2" \
         + " - 0.09696983966017489*g2*g3*p2**2" \
         + " - 0.053635839944484585*g3**2*p2**2" \
         + " + 0.09136896607569936*g1**2*p3 - 1.5831271499881143*g1*g2*p3" \
         + " + 1.1586203238882382*g2**2*p3 + 1.561944037084699*g1*g3*p3" \
         + " + 0.5373765622781439*g2*g3*p3 - 0.7510078085771675*g3**2*p3" \
         + " + 0.03313601431114031*g1**2*p1*p3" \
         + " + 0.999575794109116*g1*g2*p1*p3" \
         + " + 0.0005673645755714211*g2**2*p1*p3" \
         + " - 1.8768450822200746*g1*g3*p1*p3" \
         + " + 0.15574381848783894*g2*g3*p1*p3" \
         + " - 0.03370337888671173*g3**2*p1*p3" \
         + " + 0.040671941002624216*g1**2*p2*p3" \
         + " - 2.5271478321380254*g1*g2*p2*p3" \
         + " - 0.11221443152128503*g2**2*p2*p3" \
         + " + 0.5426249873354888*g1*g3*p2*p3" \
         + " + 0.01844664972668835*g2*g3*p2*p3" \
         + " + 0.07154249051866082*g3**2*p2*p3" \
         + " + 0.005503614288707821*g1**2*p3**2" \
         + " - 1.1126781605497342*g1*g2*p3**2" \
         + " + 0.01812229748471941*g2**2*p3**2" \
         + " + 0.657092816371931*g1*g3*p3**2" \
         + " + 0.029285097413742515*g2*g3*p3**2" \
         + " - 0.02362591177342723*g3**2*p3**2" \
         + " + 0.49802078883164164*q0 - 1.5549626616482004*p1*q0" \
         + " - 1.3177149462798845*p2*q0 - 0.49898148138677*p3*q0" \
         + " - 0.12338824573791841*q1 - 0.06111510706623446*p1*q1" \
         + " - 0.04157014428073322*p2*q1 - 0.007996832327295606*p3*q1" \
         + " + 0.3953545453627241*q2 + 1.168253359907005*p1*q2" \
         + " + 0.1289788318383644*p2*q2 + 1.6166074402033737*p3*q2" \
         + " - 1.1634376524391858*q3 + 1.1753697910765462*p1*q3" \
         + " + 1.303746265623441*p2*q3 - 0.9550529463247988*p3*q3;"
    pol7 = "-0.43946259392041137*g1**2 - 0.2794194400312886*g1*g2" \
         + " + 0.28633513817241923*g2**2 + 0.4837842932289669*g1*g3" \
         + " - 0.2789447622513483*g2*g3 + 0.00840391192868436*g3**2" \
         + " - 1.5744202005495247*g1**2*p1 - 0.2861663506179279*g1*g2*p1" \
         + " + 0.7386941869051702*g2**2*p1 + 0.5331769017373268*g1*g3*p1" \
         + " + 0.3754313238644043*g2*g3*p1 - 0.49557272342979514*g3**2*p1" \
         + " - 0.1697121445535322*g1**2*p1**2" \
         + " + 0.17649121044520624*g1*g2*p1**2" \
         + " + 0.230772384999563*g2**2*p1**2" \
         + " + 1.5487222154552227*g1*g3*p1**2" \
         + " + 1.1549190062177301*g2*g3*p1**2" \
         + " - 0.061060240446030825*g3**2*p1**2" \
         + " + 0.3428887153466959*g1**2*p2 - 2.0520172954900926*g1*g2*p2" \
         + " + 0.2548434377395737*g2**2*p2 - 1.1661156986017325*g1*g3*p2" \
         + " + 1.1418016311188108*g2*g3*p2 - 0.03717326023724154*g3**2*p2" \
         + " - 1.1298456541690676*g1**2*p1*p2" \
         + " - 2.0395244570443807*g1*g2*p1*p2" \
         + " - 0.46158802071478294*g2**2*p1*p2" \
         + " - 0.7937227851031279*g1*g3*p1*p2" \
         + " + 1.3925863980743391*g2*g3*p1*p2" \
         + " + 1.5914336748838505*g3**2*p1*p2" \
         + " + 0.5461657368466324*g1**2*p2**2" \
         + " - 0.40076456254394*g1*g2*p2**2" \
         + " - 0.5789448963236655*g2**2*p2**2" \
         + " - 1.2606913391501393*g1*g3*p2**2" \
         + " - 0.8839870841626388*g2*g3*p2**2" \
         + " + 0.03277915947703309*g3**2*p2**2" \
         + " - 0.8186969313136523*g1**2*p3 - 0.02925076747851844*g1*g2*p3" \
         + " + 0.7392135200738452*g2**2*p3 + 0.12848993732517067*g1*g3*p3" \
         + " - 0.9246721961161062*g2*g3*p3 + 0.34711806557393554*g3**2*p3" \
         + " - 1.3114980542204153*g1**2*p1*p3" \
         + " + 0.7458693987541652*g1*g2*p1*p3" \
         + " + 1.309443056027262*g2**2*p1*p3" \
         + " - 0.5609050703954811*g1*g3*p1*p3" \
         + " - 0.560863866588139*g2*g3*p1*p3" \
         + " + 0.0020549981931533984*g3**2*p1*p3" \
         + " + 0.21040044489655793*g1**2*p2*p3" \
         + " - 2.094674294769259*g1*g2*p2*p3" \
         + " + 0.9014935747766074*g2**2*p2*p3" \
         + " - 0.7741828342315165*g1*g3*p2*p3" \
         + " + 1.0179705141740856*g2*g3*p2*p3" \
         + " - 1.1118940196731655*g3**2*p2*p3" \
         + " - 0.37645359229310016*g1**2*p3**2" \
         + " + 0.22427335209873375*g1*g2*p3**2" \
         + " + 0.34817251132410243*g2**2*p3**2" \
         + " - 0.2880308763050832*g1*g3*p3**2" \
         + " - 0.27093192205509137*g2*g3*p3**2" \
         + " + 0.028281080968997733*g3**2*p3**2" \
         + " + 0.14472354381930777*q0 + 1.3312987370741496*p1*q0" \
         + " - 0.560558892849028*p2*q0 - 0.26763465433412853*p3*q0" \
         + " + 0.34474968158768826*q1 + 1.1967487089081013*p1*q1" \
         + " - 0.5416788729536276*p2*q1 + 0.35732161373948407*p3*q1" \
         + " + 0.4086834090152563*q2 + 0.9057045158335053*p1*q2" \
         + " + 0.790785041402257*p2*q2 + 0.25330644272284614*p3*q2" \
         + " - 0.3854162586341408*q3 - 0.04738889206888738*p1*q3" \
         + " + 1.2360880142692436*p2*q3 - 0.03146357181747231*p3*q3;"
    pol8 = "-0.3364985136694329*g1**2 - 0.6220713377668979*g1*g2" \
         + " + 0.30253663989085705*g2**2 + 0.09691241616903694*g1*g3" \
         + " - 0.23475287081412524*g2*g3 + 0.03759204880218851*g3**2" \
         + " - 1.0655428995245861*g1**2*p1 + 1.2015525036653503*g1*g2*p1" \
         + " + 0.7601103825805371*g2**2*p1 - 0.19738187400246698*g1*g3*p1" \
         + " + 0.6151569823342218*g2*g3*p1 - 0.19896444557080828*g3**2*p1" \
         + " + 0.12867336914719074*g1**2*p1**2" \
         + " + 1.2422164684570598*g1*g2*p1**2" \
         + " - 0.32698199621342966*g2**2*p1**2" \
         + " + 0.988903111633841*g1*g3*p1**2" \
         + " + 0.015998020944011523*g2*g3*p1**2" \
         + " + 0.19830862706623895*g3**2*p1**2" \
         + " - 0.36506543203523656*g1**2*p2 - 1.8191248166962*g1*g2*p2" \
         + " - 0.04702353062492582*g2**2*p2 + 0.22650992191881464*g1*g3*p2" \
         + " + 0.6757394159618821*g2*g3*p2 - 0.20784006514510822*g3**2*p2" \
         + " - 1.3823450011564475*g1**2*p1*p2" \
         + " - 0.10921823662847378*g1*g2*p1*p2" \
         + " + 0.9411204746791397*g2**2*p1*p2" \
         + " + 0.6416176172245012*g1*g3*p1*p2" \
         + " + 0.6050178988297725*g2*g3*p1*p2" \
         + " + 0.4412245264773078*g3**2*p1*p2" \
         + " + 0.06735301456610396*g1**2*p2**2" \
         + " - 0.8585408362609629*g1*g2*p2**2" \
         + " - 0.3090762258515019*g2**2*p2**2" \
         + " - 0.4001418483728859*g1*g3*p2**2" \
         + " + 0.4779675484647562*g2*g3*p2**2" \
         + " + 0.24172321128539795*g3**2*p2**2" \
         + " - 0.5137007261636957*g1**2*p3 - 0.9820327320175907*g1*g2*p3" \
         + " + 0.926095293449879*g2**2*p3 - 0.7003487284409724*g1*g3*p3" \
         + " - 0.9430374895060032*g2*g3*p3 + 0.08868217302851986*g3**2*p3" \
         + " - 0.8028419556001375*g1**2*p1*p3" \
         + " + 1.4979362188623169*g1*g2*p1*p3" \
         + " + 0.6798639272820932*g2**2*p1*p3" \
         + " - 1.0625669164194511*g1*g3*p1*p3" \
         + " + 1.6499737530190743*g2*g3*p1*p3" \
         + " + 0.12297802831804432*g3**2*p1*p3" \
         + " - 0.2829951083769471*g1**2*p2*p3" \
         + " - 1.67180387717421*g1*g2*p2*p3" \
         + " + 0.22862840829725906*g2**2*p2*p3" \
         + " - 0.45573353517584536*g1*g3*p2*p3" \
         + " - 0.030399560249614144*g2*g3*p2*p3" \
         + " + 0.05436670007968804*g3**2*p2*p3" \
         + " - 0.1960263837132947*g1**2*p3**2" \
         + " - 0.3836756321960968*g1*g2*p3**2" \
         + " + 0.6360582220649316*g2**2*p3**2" \
         + " - 0.5887612632609552*g1*g3*p3**2" \
         + " - 0.49396556940876774*g2*g3*p3**2" \
         + " - 0.4400318383516369*g3**2*p3**2" \
         + " - 0.0036301750236126602*q0 + 0.5043969625148573*p1*q0" \
         + " + 0.6199290278052706*p2*q0 - 0.5010767403147031*p3*q0" \
         + " + 0.27845755063810307*q1 + 0.9141858220671707*p1*q1" \
         + " - 0.04474642022970836*p2*q1 + 0.21510107355697375*p3*q1" \
         + " + 0.6261152163591358*q2 - 0.23441023070199246*p1*q2" \
         + " + 0.5860551333365724*p2*q2 + 0.6514610848268766*p3*q2" \
         + " - 0.1414398428437503*q3 + 0.18837597417330545*p1*q3" \
         + " + 0.23529746996939166*p2*q3 + 0.34502969365002717*p3*q3;"
    pol9 = "-0.4997153826800627*g1**2 + 1.1379854833548109*g1*g2" \
         + " - 0.6474309248194395*g2**2 + 0.5441177673182162*g1*g3 " \
         + " - 0.6356209362624222*g2*g3 - 0.00249167421336129*g3**2" \
         + " - 1.5544041627556424*g1**2*p1 + 2.3739950333676623*g1*g2*p1" \
         + " - 0.626498203472872*g2**2*p1 - 1.251167893581656*g1*g3*p1" \
         + " + 1.014528145879352*g2*g3*p1 - 0.07360314669929015*g3**2*p1" \
         + " - 0.11700211538167525*g1**2*p1**2" \
         + " + 1.4292168664824174*g1*g2*p1**2" \
         + " - 0.08561887506760898*g2**2*p1**2" \
         + " + 0.11756149787203274*g1*g3*p1**2" \
         + " + 1.2753331275027957*g2*g3*p1**2" \
         + " + 0.20262099044928422*g3**2*p1**2" \
         + " - 0.5322121916934887*g1**2*p2 - 0.5705836762208184*g1*g2*p2" \
         + " + 1.338552997747994*g2**2*p2 + 0.1626686039427204*g1*g3*p2" \
         + " + 0.8255938648528275*g2*g3*p2 - 0.2877403739032339*g3**2*p2" \
         + " - 1.7484788356484942*g1**2*p1*p2" \
         + " - 0.7462669895235802*g1*g2*p1*p2" \
         + " + 0.9024930142580471*g2**2*p1*p2" \
         + " - 0.836428661723292*g1*g3*p1*p2" \
         + " + 0.10315211869079574*g2*g3*p1*p2" \
         + " + 0.8459858213904471*g3**2*p1*p2" \
         + " + 0.0524174914208927*g1**2*p2**2" \
         + " - 1.0349080883999526*g1*g2*p2**2" \
         + " - 0.4456138068634661*g2**2*p2**2" \
         + " - 0.8702119648626987*g1*g3*p2**2" \
         + " + 0.1347633798985455*g2*g3*p2**2" \
         + " + 0.3931963154425734*g3**2*p2**2" \
         + " + 0.7263228121576023*g1**2*p3 - 2.2210658961930227*g1*g2*p3" \
         + " + 1.6108892968930464*g2**2*p3 - 1.0504867050871953*g1*g3*p3" \
         + " + 1.0659099359741329*g2*g3*p3 + 0.5672833993063003*g3**2*p3" \
         + " - 0.06811413086649659*g1**2*p1*p3" \
         + " - 0.31001429706461026*g1*g2*p1*p3" \
         + " + 1.4153449120254278*g2**2*p1*p3" \
         + " - 1.8526453839326555*g1*g3*p1*p3" \
         + " + 1.3321822393355465*g2*g3*p1*p3" \
         + " - 1.3472307811589312*g3**2*p1*p3" \
         + " + 0.8918357940981352*g1**2*p2*p3" \
         + " - 1.500420306224534*g1*g2*p2*p3" \
         + " - 0.43633002867072346*g2**2*p2*p3" \
         + " - 1.087845994748532*g1*g3*p2*p3" \
         + " - 0.15102803926606495*g2*g3*p2*p3" \
         + " - 0.4555057654274117*g3**2*p2*p3" \
         + " + 0.06458462396078254*g1**2*p3**2" \
         + " - 0.39430877808246473*g1*g2*p3**2" \
         + " + 0.5312326819310751*g2**2*p3**2" \
         + " + 0.752650466990666*g1*g3*p3**2" \
         + " - 1.4100965074013412*g2*g3*p3**2" \
         + " - 0.5958173058918577*g3**2*p3**2" \
         + " + 1.1496379817128635*q0 + 2.2545055129278047*p1*q0" \
         + " - 0.5186004321512713*p2*q0 - 2.904495508356949*p3*q0" \
         + " + 0.3811371457709*q1 + 1.1561384774363503*p1*q1" \
         + " - 0.03459051968276095*p2*q1 - 0.5860104174434722*p3*q1" \
         + " - 0.42262305475993184*q2 - 0.06960283700014054*p1*q2" \
         + " + 0.6975228406191866*p2*q2 + 1.1761445768437886*p3*q2" \
         + " - 0.4132504788764938*q3 + 1.0477095469315294*p1*q3" \
         + " + 0.5581768684334514*p2*q3 + 0.4360576481532085*p3*q3;"
    pol10 = "-1. + 0.9336143308746049*g1 + 1.1781580271766483*g2" \
          + " + 0.551650235747964*g3;"
    return [pol1, pol2, pol3, pol4, pol5, pol6, pol7, pol8, pol9, pol10]

def stewgou40():
    """
    Stewart-Gough platform with 40 real postures.
    From the paper of Peter Dietmaier:
    The Stewart-Gough platform of general
    geometry can have 40 real postures, pages 1-10, in
    Advances in Robot Kinematics: Analysis and Control,
    edited by Jadran Lenarcic and Manfred L. Husty,
    Kluwer Academic Publishers, Dordrecht, 1998.
    """
    pol1 = "n1^2 + n2^2 + n3^2 - 1;"
    pol2 = "a11^2 + a12^2 + a13^2 - 1;"
    pol3 = "a21^2 + a22^2 + a23^2 - 1;"
    pol4 = "a11*a21 + a12*a22 + a13*a23;"
    pol5 = " n1^2 + 1.085610*n1*a11 - 2.215830*n1 + 0.2946372680*a11^2" \
         + " - 1.202763603*a11 + 0.8110958214+n2^2 + 1.085610*n2*a12" \
         + " + 0.2946372680*a12^2 + n3^2 + 1.085610*n3*a13" \
         + " + 0.2946372680*a13^2;"
    pol6 = "- 1.050876963*a11 - 1.446982100*a12 + 0.5808481060*a21" \
         + " + 0.7997861232*a22 - 1.098188*n1 - 1.512126*n2" \
         + " - 1.012257626*a12*a22 + 0.9156939726*a11^2 + 0.9156939726*a12^2" \
         + " + 0.2797510772*a23^2 + 0.9156939726*a13^2 + n2^2 + n3^2" \
         + " - 1.012257626*a11*a21 + n1^2 - 1.012257626*a13*a23" \
         + " + 0.2797510772*a22^2 + 0.2797510772*a21^2 - 1.057830*n3*a23" \
         + " - 0.3068774482 + 1.913838*n1*a11 + 1.913838*n2*a12" \
         + " + 1.913838*n3*a13 - 1.057830*n1*a21 - 1.057830*n2*a22;"
    pol7 = "2.805076*n1*a12*a23 - 0.9789534962*a11 + 0.2982299150*a12" \
         + " - 0.7004990340*a13 + 0.5196729762*a21 - 0.1583139833*a22" \
         + " + 0.3718567014*a23 - 1.470154*n1 + 0.447870*n2 - 1.051982*n3" \
         + " + 2.805076*n3*a11*a22 - 0.4707567232*a12*a22" \
         + " - 2.061946850*a12*a23 - 0.6281546940*a11*a23" \
         + " - 1.475444730*a11*a22 + 1.475444730*a12*a21" \
         + " - 2.805076*n1*a13*a22 + 2.805076*n2*a13*a21" \
         + " - 2.805076*n2*a11*a23" \
         + " + 0.4434028332*a11^2 + 0.4434028332*a12^2 + 0.1249495243*a23^2" \
         + " + 0.4434028332*a13^2 + 0.6281546940*a13*a21" \
         + " + 2.061946850*a13*a22" \
         + " + n2^2 + n3^2 - 2.805076*n3*a12*a21 - 0.4707567232*a11*a21" \
         + " + n1^2" \
         + " - 0.4707567232*a13*a23 - 1.393177215 + 0.1249495243*a22^2" \
         + " + 0.1249495243*a21^2 - 3.934225682*a11*a22*a12*a21" \
         + " + 1.967112841*a11^2*a22^2 + 1.967112841*a12^2*a21^2" \
         + " - 0.706964*n3*a23 + 1.967112841*a12^2*a23^2" \
         + " + 1.967112841*a13^2*a22^2 + 1.331770*n1*a11 + 1.331770*n2*a12" \
         + " + 1.331770*n3*a13 - 3.934225682*a12*a23*a13*a22" \
         + " + 1.967112841*a13^2*a21^2 + 1.967112841*a11^2*a23^2" \
         + " - 3.934225682*a13*a21*a11*a23 - 0.706964*n1*a21" \
         + " - 0.706964*n2*a22;"
    pol8 = "0.215344*n1*a12*a23 - 0.4919329150*a11 + 0.5032939412*a12" \
         + " + 0.3524721322*a13 - 1.191622463*a21 + 1.219142585*a22" \
         + " + 0.8538028204*a23 - 1.028376*n1 + 1.052126*n2 + 0.736836*n3" \
         + " + 0.215344*n3*a11*a22 + 1.108589329*a12*a22" \
         + " - 0.1107273007*a12*a23" \
         + " - 0.1132845107*a11*a23 + 0.7933660580E-1*a11*a22" \
         + " - 0.7933660580E-1*a12*a21 - 0.215344*n1*a13*a22" \
         + " + 0.215344*n2*a13*a21 - 0.215344*n2*a11*a23 + 0.2288273329*a11^2" \
         + " + 0.2288273329*a12^2 + 1.342683023*a23^2 + 0.2288273329*a13^2" \
         + " + 0.1132845107*a13*a21 + 0.1107273007*a13*a22 + n2^2 + n3^2" \
         + " - 0.215344*n3*a12*a21 + 1.108589329*a11*a21 + n1^2" \
         + " + 1.108589329*a13*a23 + 1.342683023*a22^2 + 1.342683023*a21^2" \
         + " - 0.2318651916E-1*a11*a22*a12*a21 + 0.1159325958E-1*a11^2*a22^2" \
         + " + 0.1159325958E-1*a12^2*a21^2 + 2.317484*n3*a23" \
         + " + 0.1159325958E-1*a12^2*a23^2 + 0.1159325958E-1*a13^2*a22^2" \
         + " + 0.956718*n1*a11 + 0.956718*n2*a12 + 0.956718*n3*a13" \
         + " - 0.2318651916E-1*a12*a23*a13*a22 + 0.1159325958E-1*a13^2*a21^2" \
         + " + 0.1159325958E-1*a11^2*a23^2 - 0.2318651916E-1*a13*a21*a11*a23" \
         + " + 2.317484*n1*a21 + 2.317484*n2*a22 - 0.9664888140;"
    pol9 = "0.707826*n1*a12*a23 + 0.1618923443*a11 + 0.2597332554E-1*a12" \
         + " - 0.5621060514E-1*a13 + 0.2776652044*a21 + 0.4454743538E-1*a22" \
         + " - 0.9640807436E-1*a23 - 1.180946*n1 - 0.189466*n2 + 0.410036*n3" \
         + " + 0.707826*n3*a11*a22 + 0.6446406506E-1*a12*a22" \
         + " - 0.4179521416*a12*a23 + 0.6705448046E-1*a11*a23" \
         + " + 0.1451170709*a11*a22 - 0.1451170709*a12*a21" \
         + " - 0.707826*n1*a13*a22 + 0.707826*n2*a13*a21" \
         + " - 0.707826*n2*a11*a23" \
         + " + 0.1879284557E-1*a11^2 + 0.1879284557E-1*a12^2" \
         + " + 0.5528188464E-1*a23^2 + 0.1879284557E-1*a13^2" \
         + " - 0.6705448046E-1*a13*a21 + 0.4179521416*a13*a22 + n2^2 + n3^2" \
         + " - 0.707826*n3*a12*a21 + 0.6446406506E-1*a11*a21 + n1^2" \
         + " + 0.6446406506E-1*a13*a23 + 0.5528188464E-1*a22^2" \
         + " + 0.5528188464E-1*a21^2 - 0.2505088232*a11*a22*a12*a21" \
         + " + 0.1252544116*a11^2*a22^2 + 0.1252544116*a12^2*a21^2" \
         + " - 0.470242*n3*a23 + 0.1252544116*a12^2*a23^2" \
         + " + 0.1252544116*a13^2*a22^2 - 0.274174*n1*a11 - 0.274174*n2*a12" \
         + " - 0.274174*n3*a13 - 0.2505088232*a12*a23*a13*a22" \
         + " + 0.1252544116*a13^2*a21^2 + 0.1252544116*a11^2*a23^2" \
         + " - 0.2505088232*a13*a21*a11*a23 - 0.470242*n1*a21" \
         + " - 0.470242*n2*a22 - 0.1948854017;"
    return [pol1, pol2, pol3, pol4, pol5, pol6, pol7, pol8, pol9]

def tangents():
    """
    Three lines tangent to four given spheres, each with multiplicity 4.
    Originally formulated as a polynomial system by Cassiano Durand.
    Thorsten Teobald came up with the positioning of the centers of
    the spheres, each with radius 0.5 at the vertices of a tethrahedron.
    See the paper by Frank Sottile: From Enumerative Geometry to
    Solving Systems of Polynomial Equations".
    In "Computations in Algebraic Geometry with Macaulay 2",
    edited by David Eisenbud, Daniel R. Grayson, Michael Stillman,
    and Bernd Sturmfels, Volume 8 of Algorithms and Computation
    in Mathematics, pages 101-129, Springer-Verlag, 2002.
    Another reference is the paper by Frank Sottile and Thorsten Theobald:
    Lines tangents to 2n - 2 spheres in R^n.
    Trans. Amer. Math. Soc. 354:4815-4829, 2002.
    """
    pol1 = "x0**2 + x1**2 + x2**2 - 1;"
    pol2 = "x0*x3 + x1*x4 + x2*x5;"
    pol3 = "x3**2 + x4**2 + x5**2 - 0.25;"
    pol4 = "x3**2 + x4**2 - 2*x2*x4 + x2**2 + x5**2 + 2*x1*x5 + x1**2 - 0.25;"
    pol5 = "x3**2 + 1.73205080756888*x2*x3 + 0.75*x2**2 + x4**2 - x2*x4" \
         + " + 0.25*x2**2 + x5**2 - 1.73205080756888*x0*x5 + x1*x5" \
         + " + 0.75*x0**2 - 0.86602540378444*x0*x1 + 0.25*x1**2 - 0.25;"
    pol6 = "x3**2 - 1.63299316185545*x1*x3 + 0.57735026918963*x2*x3" \
         + " + 0.66666666666667*x1**2 - 0.47140452079103*x1*x2" \
         + " + 0.08333333333333*x2**2 + x4**2 + 1.63299316185545*x0*x4" \
         + " - x2*x4 + 0.66666666666667*x0**2 - 0.81649658092773*x0*x2" \
         + " + 0.25*x2**2 + x5**2 - 0.57735026918963*x0*x5 + x1*x5" \
         + " + 0.08333333333333*x0**2 - 0.28867513459481*x0*x1" \
         + " + 0.25*x1**2 - 0.25;"
    return [pol1, pol2, pol3, pol4, pol5, pol6]

def sevenbar():
    """
    Returns the list of strings which represent Laurent polynomials for a
    special sevenbar mechanism that has isolated solutions and a cubic curve.
    A reference for the general case is the paper by Carlo Innocenti:
    Polynomial solution to the position analysis of the 7-line Assur kinematic
    chain with one quaternary link, in Mech. Mach. Theory, Vol. 30, No. 8,
    pages 1295-1303, 1995.  The special case was introduced in the paper
    with title: Numerical decomposition of the solution sets of polynomial
    systems into irreducible components, SIAM J. Numer. Anal. 38(6):2022-2046,
    2001, by Andrew Sommese, Jan Verschelde, and Charles Wampler.
    """
    pol1 = "0.710358341606049*t1 + 0.46*t2 - 0.41*t3 + 0.240761300555115" \
         + " + 1.07248215701824*I;"
    pol2 = "t2*(-0.11 + 0.49*I) + 0.41*t3 - 0.502195181179589*t4 + 0.41*t5;"
    pol3 = "0.502195181179589*t4 + t5*(-0.0980434782608696 " \
         + " + 0.436739130434783*I) - 0.775518556663656*t6 - 1.2;"
    pol4 = "0.710358341606049*t1**(-1) + 0.46*t2**(-1) - 0.41*t3**(-1)" \
         + " + 0.240761300555115 - 1.07248215701824*I;"
    pol5 = "t2**(-1)*(-0.11 - 0.49*I) + 0.41*t3**(-1) " \
         + " - 0.502195181179589*t4**(-1) + 0.41*t5**(-1);"
    pol6 = "0.502195181179589*t4**(-1) + t5**(-1)*(-0.0980434782608696 " \
         + " - 0.436739130434783*I) - 0.775518556663656*t6**(-1) - 1.2;"
    return [pol1, pol2, pol3, pol4, pol5, pol6]

def solve_binomials():
    """
    Runs the test on solving the binomials example.
    Asserts that the number of solutions equals 20.
    """
    from phcpy.solver import solve
    print('\nsolving a random binomial system...')
    pols = binomials()
    sols = solve(pols)
    assert len(sols) == 20
    print('test on binomial system passed')

def solve_cyclic7():
    """
    Runs the test on solving the cyclic 7-roots problem.
    Asserts that the number of solutions equals 924.
    """
    from phcpy.solver import solve
    print('\nsolving the cyclic 7-roots problem...')
    pols = cyclic7()
    sols = solve(pols)
    assert len(sols) == 924
    print('test on cyclic 7-roots passed')

def solve_sysd1():
    """
    Runs the test on solving the benchmark problem D1.
    Asserts that the number of solutions equals 48.
    """
    from phcpy.solver import solve
    print('\nsolving the benchmark problem D1...')
    pols = sysd1()
    sols = solve(pols)
    assert len(sols) == 48
    print('test on benchmark problem D1 passed')

def solve_fbrfive4():
    """
    Runs the test on solving a generic 4-bar problem.
    Asserts that the number of solutions equals 36.
    """
    from phcpy.solver import solve
    print('\nsolving a generic 5-point 4-bar design problem...')
    pols = fbrfive4()
    sols = solve(pols)
    assert len(sols) == 36
    print('test on 4-bar system passed')

def solve_game4two():
    """
    Runs the test on solving the Nash equilibrium problem.
    Asserts that the number of solutions equals 9.
    """
    from phcpy.solver import solve
    print('\ncomputing all Nash equilibria...')
    pols = game4two()
    sols = solve(pols)
    assert len(sols) == 9
    print('test on Nash equilibria for 4 players passed')

def solve_katsura6():
    """
    Runs the test on solving the katsura6 problem.
    Asserts that the number of solutions equals 64.
    """
    from phcpy.solver import solve
    print('\nsolving a problem in magnetism...')
    pols = katsura6()
    sols = solve(pols)
    assert len(sols) == 64
    print('test on a problem in magnetism passed')

def solve_noon3():
    """
    Test on solving the noon3 system.
    Asserts that the number of solutions equals 21.
    """
    from phcpy.solver import solve
    print('\nsolving a neural network model...')
    pols = noon3()
    sols = solve(pols)
    assert len(sols) == 21
    print('test on a neural network model passed')

def solve_rps10():
    """
    Test on solving a mechanical design problem.
    Asserts that the number of solutions equals 1024.
    """
    from phcpy.solver import solve
    print('\nsolving a mechanical design problem...')
    pols = rps10()
    sols = solve(pols)
    assert len(sols) == 1024
    print('test on RPS serial chains problem passed')

def solve_stewgou40():
    """
    Test on solving a fully reall Stewart-Gough platform.
    Asserts that the number of solutions equals 40.
    """
    from phcpy.solver import solve
    print('\nsolving a fully real Stewart-Gough platform...')
    pols = stewgou40()
    sols = solve(pols)
    assert len(sols) == 40
    print('test on real Stewart-Gough platform passed')

def solve_tangents():
    """
    Test on solving the tangents to 4 spheres problem.
    Asserts that the number of solutions equals 6.
    """
    from phcpy.solver import solve
    print('\ncomputing all tangent lines to 4 spheres...')
    pols = tangents()
    sols = solve(pols)
    assert len(sols) == 6
    print('test on multiple tangent lines to spheres passed')

def solve_sevenbar():
    """
    Test on solving a special 7-bar problem.
    Asserts that the number of isolated solutions equals 6
    and that there are three generic points at a curve.
    """
    from phcpy.factor import solve
    print('\nsolving a special 7-bar problem...')
    pols = sevenbar()
    sols = solve(6, 1, pols, islaurent=True, verbose=False)
    assert len(sols[1][1]) == 3
    assert len(sols[0][1]) == 6
    print('test on solving special 7-bar problem passed')

def test():
    """
    Solves the systems and tests on their number of solutions.
    """
    from phcpy.phcpy2c3 import py2c_set_seed
    py2c_set_seed(834798278)
    solve_binomials()
    solve_cyclic7()
    solve_sysd1()
    solve_fbrfive4()
    solve_game4two()
    solve_katsura6()
    solve_noon3()
    solve_rps10()
    solve_stewgou40()
    solve_tangents()
    solve_sevenbar()

if __name__ == "__main__":
    test()
