///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/actions/ActionManager.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/viewport/input/XFormManager.h>
#include <core/viewport/ViewportPanel.h>
#include <core/viewport/snapping/SnappingManager.h>
#include <core/plugins/PluginManager.h>
#include <core/data/DataSetManager.h>
#include <core/data/units/ParameterUnit.h>
#include <core/plugins/branding/Branding.h>
#include <core/gui/ApplicationManager.h>
#include <core/utilities/PathManager.h>
#include <core/undo/UndoManager.h>
#include <core/scene/animation/controller/Controller.h>
#include <core/scene/animation/AnimManager.h>
#include <core/data/importexport/ImportExportManager.h>

namespace Core {

QtMsgHandler ApplicationManager::defaultQtMessageHandler = NULL;

/******************************************************************************
* This creates the application object and is called from the program's main() method.
******************************************************************************/
int ApplicationManager::main(int argc, char** argv)
{
	// Check command line arguments for the console mode switch.
	bool noGUI = false;
	for(int i=0; i<argc; i++) {
		if(qstrcmp(argv[i], "--nogui") == 0) {
			noGUI = true;
			break;
		}
		// Stop parsing the command line after the --script switch because these parameters are passed to the script.
		if(qstrcmp(argv[i], "--script") == 0) break;
	}

	// Initialize application manager.
	ApplicationManager app(argc, argv, noGUI);
	if(!app.initialize())
		return 1;

	// Run event loop.
	int result = app.runApplication();

	// Shutdown application.
	app.shutdown();

	return result;
}

/******************************************************************************
* Handler method for Qt error messages.
* This can be used to set a debugger breakpoint for the OVITO_ASSERT macros.
******************************************************************************/
void ApplicationManager::qtMessageOutput(QtMsgType type, const char *msg)
{
	if(type == QtFatalMsg) {
		// This code line can be used to set a breakpoint to debug OVITO_ASSERT() macro exceptions.
		int a = 1;
	}
	// Pass message on to default handler.
	if(defaultQtMessageHandler) defaultQtMessageHandler(type, msg);
	else std::cerr << msg << std::endl;
}

/******************************************************************************
* This is called on program startup.
******************************************************************************/
bool ApplicationManager::initialize()
{
	MainFrame* frame = NULL;

	// Install custom Qt error message handler to catch fatal errors in debug mode.
	defaultQtMessageHandler = qInstallMsgHandler(qtMessageOutput);

	// Install global exception handler.
	// The GUI exception handler shows a message box with the error message.
	// The console mode exception handlers just prints the error message to stderr.
	if(guiMode())
		Exception::setExceptionHandler(guiExceptionHandler);
	else
		Exception::setExceptionHandler(consoleExceptionHandler);

	try {
		// Parse command line arguments.
		if(!parseCommandLine())
			return false;
	}
	catch(const Exception& ex) {
		ex.showError();
		return false;
	}

	try {
		// Initialize all manager classes.
		PathManager::initialize();
		UndoManager::initialize();
		PluginManager::initialize();
		// Run auto start objects.
		loadAutoStartObjects();
		callAutoStartObjects(AutoStart::PluginManagerInitialized);
		UnitsManager::initialize();
		ViewportManager::initialize();
		callAutoStartObjects(AutoStart::ViewportManagerInitialized);
		DataSetManager::initialize();
		callAutoStartObjects(AutoStart::DataSetManagerInitialized);
		ViewportInputManager::initialize();
		callAutoStartObjects(AutoStart::ViewportInputManagerInitialized);
		XFormManager::initialize();
		ControllerManager::initialize();
		AnimManager::initialize();
		callAutoStartObjects(AutoStart::AnimManagerInitialized);
		ActionManager::initialize();
		callAutoStartObjects(AutoStart::ActionManagerInitialized);
		SnappingManager::initialize();
		ImportExportManager::initialize();

		// Load event filter plugins.
		loadGUIEventFilters();

		// Set the application name provided by the active branding class.
		setApplicationName(BrandingManager::primaryBranding()->applicationName());
		setOrganizationName(BrandingManager::primaryBranding()->organizationName());
		setApplicationVersion(BrandingManager::primaryBranding()->productVersion());

		// Print banner and copyright message.
		if(!arguments().contains("--nobanner")) {
			cout << BrandingManager::primaryBranding()->productName().toStdString() << endl;
			cout << BrandingManager::primaryBranding()->productVersion().toStdString() << endl;
			cout << BrandingManager::primaryBranding()->copyrightString().toStdString() << endl;
		}

		// Call the brandings.
		Q_FOREACH(const Branding::SmartPtr& branding, BrandingManager::brandings())
			branding->preConfigureMainWindow();

		// Create the main application window.
		if(guiMode()) {
			// Set the application icon.
			QIcon mainWindowIcon;
			mainWindowIcon.addFile(":/core/main/window_icon_256.png");
			mainWindowIcon.addFile(":/core/main/window_icon_128.png");
			mainWindowIcon.addFile(":/core/main/window_icon_48.png");
			mainWindowIcon.addFile(":/core/main/window_icon_32.png");
			mainWindowIcon.addFile(":/core/main/window_icon_16.png");
			setWindowIcon(mainWindowIcon);

			// Check active Qt style.
			// The Oxygen style produces several problems.
			if(qstrcmp(QApplication::style()->metaObject()->className(), "OxygenStyle") == 0)
				QMessageBox::warning(NULL, BrandingManager::primaryBranding()->productName(), tr("This application is set to use the Oxygen style for its user interface. It is recommended to use another style instead. The active UI style can be changed using the \"qtconfig\" program or the \"-style cleanlooks\" command line switch."));

			frame = new MainFrame(BrandingManager::primaryBranding()->productName());

			// Inform the GUI event filters about the newly created main window.
			Q_FOREACH(const UIEventFilter::SmartPtr& filter, guiEventFilters)
				filter->mainWindowCreated(frame);

			callAutoStartObjects(AutoStart::MainWindowCreated);
		}

		// Call the brandings again.
		Q_FOREACH(const Branding::SmartPtr& branding, BrandingManager::brandings())
			branding->postConfigureMainWindow();

		if(!startupSceneFile().isEmpty()) {
			// Load scene file specified on the command line.
			QFileInfo startupFile(startupSceneFile());
			if(!DATASET_MANAGER.fileLoad(startupFile.absoluteFilePath()))
				DATASET_MANAGER.fileReset();
		}
		else {
			// Create an empty data set.
			DATASET_MANAGER.fileReset();
		}
		callAutoStartObjects(AutoStart::InitialSceneLoaded);

		// Enable the viewports now. Viewport updates are suspended by default.
		VIEWPORT_MANAGER.resumeViewportUpdates();

		// Show the main window.
		if(guiMode()) {
			frame->showMaximized();
			connect(frame, SIGNAL(destroyed(QObject*)), this, SLOT(quit()));
			frame->viewportPanel()->layoutViewports();
			callAutoStartObjects(AutoStart::MainWindowShown);
		}
		VIEWPORT_MANAGER.updateViewports(true);

		// This will make the scripting plugin execute the startup script.
		callAutoStartObjects(AutoStart::ApplicationRunning);

		return true;
	}
	catch(const Exception& ex) {
		ex.showError();
		shutdown();
		return false;
	}
}

/******************************************************************************
* Enters the main event loop.
******************************************************************************/
int ApplicationManager::runApplication()
{
	if(guiMode()) {
		// Enter the main event loop.
		return exec();
	}
	else {
		// No event processing needed in console mode.
		// Just exit the application.
		return _exitCode;
	}
}

/******************************************************************************
* This is called on program shutdown.
******************************************************************************/
void ApplicationManager::shutdown()
{
	VerboseLogger() << "Application is shutting down." << endl;

	// Release object arrays.
	guiEventFilters.clear();
	autostartObjects.clear();

	// Shutdown all manager classes in reverse order.
	ImportExportManager::shutdown();
	BrandingManager::shutdown();
	SnappingManager::shutdown();
	ActionManager::shutdown();
	DataSetManager::shutdown();
	AnimManager::shutdown();
	ControllerManager::shutdown();
	XFormManager::shutdown();
	ViewportInputManager::shutdown();
	ViewportManager::shutdown();
	UnitsManager::shutdown();
	UndoManager::shutdown();
	PluginManager::shutdown();
	PathManager::shutdown();
}

/******************************************************************************
* Parses the command line parameters.
******************************************************************************/
bool ApplicationManager::parseCommandLine()
{
	QStringList params = QCoreApplication::arguments();

	// Parse parameters one by one.
	for(int i=1; i<params.size(); i++) {
		const QString& p = params[i];
		_arguments.push_back(p);

		if(p == "--nogui") {
			// The console mode flag has already been interpreted by the main() method.
			OVITO_ASSERT(_consoleMode == true);
		}
		else if(p == "--verbose") {
			_verboseMode = true;
			VerboseLogger().setEnabled(true);
		}
		else if(p == "--experimental") {
			_experimentalMode = true;
		}
		else if(p == "--help") {
			MsgLogger() << "Usage: ovito [options]" << endl << endl;
			MsgLogger() << "Options:" << endl;
			MsgLogger() << "  --help                   Show help about options" << endl;
			MsgLogger() << "  --verbose                Enable additional diagnostic console output" << endl;
			MsgLogger() << "  --nogui                  Run in console mode without graphical user interface" << endl;
			MsgLogger() << "  --script <file>          Execute the given script file on startup" << endl;
			MsgLogger() << "  --nobanner               Suppress the license banner shown in the console" << endl;
			MsgLogger() << "  --experimental           Enable experimental program features" << endl;
			return false;
		}
		else if(p == "--script") {
			i++;
			if(i == params.size())
				throw Exception(tr("Script filename has not been specified."));
			_startupScriptFile = params[i];
			// All parameters that follow --script are passed to the script file
			// and will be therefore ignored here.
			break;
		}
		else if(p == "--nobanner") { /* This is handled in main(). */ }
		else {
			if(p.startsWith("-")) {
				MsgLogger() << "ERROR: Unknown command line parameter: " << p << endl;
				return false;
			}
			else {
				_startupSceneFile = p;
			}
		}
	}
	return true;
}

/******************************************************************************
* Handles events from the event queue.
******************************************************************************/
bool ApplicationManager::notify(QObject* receiver, QEvent* event)
{
	// Do normal event processing.
	bool result = QApplication::notify(receiver, event);

	// Process any pending update requests for the viewports that might have been
	// generated during the event processing.
	if(event->type() != QEvent::Paint)
		Window3D::postWindowUpdates();

	return result;
}

/******************************************************************************
* Loads all installed auto start objects on application startup.
******************************************************************************/
void ApplicationManager::loadAutoStartObjects()
{
	Q_FOREACH(PluginClassDescriptor* clazz, PLUGIN_MANAGER.listClasses(PLUGINCLASSINFO(AutoStart))) {
		VerboseLogger() << logdate << "Invoking auto start object:" << clazz->name() << endl;
		autostartObjects.push_back(static_object_cast<AutoStart>(clazz->createInstance()));
	}
}

/******************************************************************************
* Calls the AutoStart::startEvent() method on all installed auto start objects.
******************************************************************************/
void ApplicationManager::callAutoStartObjects(AutoStart::InitializationStage stage)
{
	for(QVector<AutoStart::SmartPtr>::const_iterator obj = autostartObjects.constBegin(); obj != autostartObjects.constEnd(); ++obj) {
		(*obj)->startEvent(stage);
	}
}

/******************************************************************************
* Loads all installed GUI event filters.
******************************************************************************/
void ApplicationManager::loadGUIEventFilters()
{
	Q_FOREACH(PluginClassDescriptor* clazz, PLUGIN_MANAGER.listClasses(PLUGINCLASSINFO(UIEventFilter))) {
		VerboseLogger() << logdate << "Installing GUI event filter:" << clazz->name() << endl;
		guiEventFilters.push_back(static_object_cast<UIEventFilter>(clazz->createInstance()));
	}
}

#if defined(Q_WS_X11)

/******************************************************************************
* Handles X11 GUI events.
******************************************************************************/
bool ApplicationManager::x11EventFilter(XEvent* event)
{
	// The event is passed on to all installed event filter plugins.
	for(QVector<UIEventFilter::SmartPtr>::const_iterator handler = guiEventFilters.constBegin(); handler != guiEventFilters.constEnd(); ++handler) {
		if((*handler)->x11EventFilter(event)) return true;
	}
	return false;
}

#elif defined(Q_WS_WIN)

/******************************************************************************
* Handles Windows GUI events.
******************************************************************************/
bool ApplicationManager::winEventFilter(MSG * msg, long * result)
{
	// The event is passed on to all installed event filter plugins.
	for(QVector<UIEventFilter::SmartPtr>::const_iterator handler = guiEventFilters.constBegin(); handler != guiEventFilters.constEnd(); ++handler) {
		if((*handler)->winEventFilter(msg, result)) return true;
	}
	return false;
}

#elif defined(Q_WS_MAC)

/******************************************************************************
* Handles Carbon GUI events.
******************************************************************************/
bool ApplicationManager::macEventFilter(EventHandlerCallRef caller, EventRef event)
{
	// The event is passed on to all installed event filter plugins.
	for(QVector<UIEventFilter::SmartPtr>::const_iterator handler = guiEventFilters.constBegin(); handler != guiEventFilters.constEnd(); ++handler) {
		if((*handler)->macEventFilter(caller, event)) return true;
	}
	return false;
}

#endif


/******************************************************************************
* Handler function for exceptions used in GUI mode.
******************************************************************************/
void ApplicationManager::guiExceptionHandler(const Exception& exception)
{
	OVITO_ASSERT(APPLICATION_MANAGER.guiMode());

	exception.logError();
	QMessageBox msgbox;
	msgbox.setWindowTitle(tr("Error - %1").arg(QCoreApplication::applicationName()));
	msgbox.setStandardButtons(QMessageBox::Ok);
	msgbox.setText(exception.message());
	msgbox.setIcon(QMessageBox::Critical);
	if(exception.messages().size() > 1) {
		QString detailText;
		for(int i=1; i<exception.messages().size(); i++)
			detailText += exception.messages()[i] + "\n";
		msgbox.setDetailedText(detailText);
	}
	msgbox.exec();
}

/******************************************************************************
* Handler function for exceptions used in console mode.
******************************************************************************/
void ApplicationManager::consoleExceptionHandler(const Exception& exception)
{
	for(int i=exception.messages().size()-1; i>=0; i--) {
		std::cerr << "ERROR: ";
		std::cerr << exception.messages()[i].toAscii().constData() << std::endl;
	}
	std::cerr << std::flush;
}

};
