///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program. If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <base/Base.h>
#include <base/linalg/Scaling.h>
#include <base/linalg/AffineTransformation.h>
#include <base/linalg/AffineDecomposition.h>

namespace Base {

/// This dummy instance should be passed to the Scaling class constructor to initialize it to the identity.
BASE_DLLEXPORT IdentityScaling IDENTITY_SCALING;

/******************************************************************************
* Performs the multiplication of two scaling structures.
******************************************************************************/
Scaling Scaling::operator*(const Scaling& s2) const
{
	if(Q == s2.Q) {
		return Scaling(Vector3(S.X * s2.S.X, S.Y * s2.S.Y, S.Z * s2.S.Z), Q);
	}
	else {
		AffineDecomposition decomp(AffineTransformation::scaling(*this) * AffineTransformation::scaling(s2));
		return decomp.scaling;
	}
}

/******************************************************************************
* Calculates a linear interpolation between two scaling structures.
******************************************************************************/
Scaling Scaling::interpolate(const Scaling& s1, const Scaling& s2, FloatType t)
{
	return Scaling(t * s2.S + ((FloatType)1.0 - t) * s1.S, Quaternion::interpolate(s1.Q, s2.Q, t));
}

/******************************************************************************
* Computes a quadratic interpolation between two scaling structures.
******************************************************************************/
Scaling Scaling::interpolateQuad(const Scaling& s1, const Scaling& s2, const Scaling& out, const Scaling& in, FloatType t)
{
	FloatType Ti = 1.0 - t;
	FloatType U2 = square(t), T2 = square(Ti);
	FloatType U3 = U2 * t, T3 = T2 * Ti;
	Vector3 s = s1.S * T3 + out.S * (3.0 * t * T2) + in.S * (3.0 * U2 * Ti) + s2.S * U3;
	return Scaling(s, Quaternion::interpolateQuad(s1.Q, s2.Q, in.Q, out.Q, t));
}

};	// End of namespace Base
