//                                               -*- C++ -*-
/**
 *  @file  KrigingResult.cxx
 *  @brief The result of a chaos expansion
 *
 *  Copyright 2005-2015 Airbus-EDF-IMACS-Phimeca
 *
 *  This library is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  along with this library.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  @author dutka
 *  @date   2008-05-21 11:21:38 +0200 (Wed, 21 May 2008)
 */
#include "KrigingResult.hxx"
#include "OSS.hxx"
#include "PersistentObjectFactory.hxx"

BEGIN_NAMESPACE_OPENTURNS

CLASSNAMEINIT(KrigingResult);
static Factory<KrigingResult> RegisteredFactory("KrigingResult");

/* Default constructor */
KrigingResult::KrigingResult()
  : MetaModelResult()
{
  // Nothing to do
}


/* Default constructor */
KrigingResult::KrigingResult(const NumericalSample & inputSample,
                             const NumericalSample & outputSample,
                             const NumericalMathFunction & metaModel,
                             const NumericalPoint & residuals,
                             const NumericalPoint & relativeErrors,
                             const Basis & basis,
                             const NumericalSample & trendCoefficients,
                             const CovarianceModelCollection & covarianceModels,
                             const NumericalSample & covarianceCoefficients)
  : MetaModelResult(NumericalMathFunction(inputSample, outputSample), metaModel, residuals, relativeErrors)
  , basis_(basis)
  , trendCoefficients_(trendCoefficients)
  , covarianceModels_(covarianceModels)
  , covarianceCoefficients_(covarianceCoefficients)
{
  // Nothing to do
}


/* Virtual constructor */
KrigingResult * KrigingResult::clone() const
{
  return new KrigingResult(*this);
}


/* String converter */
String KrigingResult::__repr__() const
{
  return OSS(true) << "class=" << getClassName()
         << ", basis=" << basis_
         << ", trend coefficients=" << trendCoefficients_
         << ", covariance models=" << covarianceModels_
         << ", covariance coefficients=" << covarianceCoefficients_;
}

String KrigingResult::__str__(const String & offset) const
{
  OSS oss(false);
  oss << getClassName() << "(";
  oss << "covariance models=" << covarianceModels_;
  oss << ", covariance coefficients=" << covarianceCoefficients_;
  oss << ", basis=" << basis_;
  oss << ", trend coefficients=" << trendCoefficients_ << ")";
  return oss;
}

/* Basis accessor */
Basis KrigingResult::getBasis() const
{
  return basis_;
}

/* Trend coefficients accessor */
NumericalSample KrigingResult::getTrendCoefficients() const
{
  return trendCoefficients_;
}

/* Covariance models accessor */
KrigingResult::CovarianceModelCollection KrigingResult::getCovarianceModels() const
{
  return covarianceModels_;
}

/* Covariance coefficients accessor */
NumericalSample KrigingResult::getCovarianceCoefficients() const
{
  return covarianceCoefficients_;
}

/* Method save() stores the object through the StorageManager */
void KrigingResult::save(Advocate & adv) const
{
  MetaModelResult::save(adv);
  adv.saveAttribute( "basis_", basis_ );
  adv.saveAttribute( "trendCoefficients_", trendCoefficients_ );
  adv.saveAttribute( "covarianceModels_", covarianceModels_ );
  adv.saveAttribute( "covarianceCoefficients_", covarianceCoefficients_ );
}


/* Method load() reloads the object from the StorageManager */
void KrigingResult::load(Advocate & adv)
{
  MetaModelResult::load(adv);
  adv.loadAttribute( "basis_", basis_ );
  adv.loadAttribute( "trendCoefficients_", trendCoefficients_ );
  adv.loadAttribute( "covarianceModels_", covarianceModels_ );
  adv.loadAttribute( "covarianceCoefficients_", covarianceCoefficients_ );
}



END_NAMESPACE_OPENTURNS
