/*
 *  Copyright (c) 2008 Cyrille Berger <cberger@cberger.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version of the License.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include "Tree.h"

#include <llvm/Constant.h>
#include <llvm/GlobalVariable.h>
#include <llvm/Type.h>

#include "FunctionDeclaration.h"

#include "GTLCore/Debug.h"
#include "GTLCore/LLVMBackend/CodeGenerator_p.h"
#include "GTLCore/LLVMBackend/ExpressionGenerationContext_p.h"
#include "GTLCore/LLVMBackend/ExpressionResult_p.h"
#include "GTLCore/Type.h"
#include "GTLCore/Type_p.h"
#include "GTLCore/VariableNG_p.h"
#include "GTLCore/Utils_p.h"
#include "GTLCore/ModuleData_p.h"
#include "GTLCore/LLVMBackend/Visitor_p.h"

#include "Expression.h"
#include <llvm/Module.h>
#include <llvm/DerivedTypes.h>

using namespace GTLCore::AST;

GlobalConstantDeclaration::GlobalConstantDeclaration( const GTLCore::ScopedName& _name, const GTLCore::Type* _type, Expression* _initialiser, bool _dependant, bool _external) :
    m_name( _name), m_initialiser(_initialiser), m_variable( new GTLCore::VariableNG( _type, true, _dependant ) ), m_type(_type), m_constant(not _dependant), m_external(_external)
{
  GTL_ASSERT( _initialiser );
}

GlobalConstantDeclaration::GlobalConstantDeclaration( const GTLCore::ScopedName& _name, const GTLCore::Type* _type, const std::list< int >& sizes, bool _dependant, bool _external) :
    m_name( _name), m_initialiser(0), m_variable( new GTLCore::VariableNG( _type, true, _dependant ) ), m_type(_type), m_sizes(sizes), m_constant(not _dependant), m_external(_external)
{
}

GlobalConstantDeclaration::~GlobalConstantDeclaration()
{
  delete m_initialiser;
  delete m_variable;
}


void GlobalConstantDeclaration::generate( GTLCore::ModuleData* _module, llvm::Module* _llvmModule, LLVMBackend::CodeGenerator* _codeGenerator, llvm::LLVMContext* _llvmContext)
{
  LLVMBackend::GenerationContext gc( _codeGenerator, _llvmContext, 0, 0, _module, _llvmModule);
  llvm::Value* pointer = 0;
  
  if(m_external)
  {
    pointer = _llvmModule->getOrInsertGlobal((const std::string&)nameToSymbol(), m_type->d->type(*_llvmContext));
  } else {
    llvm::Constant* value = 0;
    if( m_initialiser )
    {
      GTL_DEBUG(*m_initialiser->type());
      LLVMBackend::ExpressionGenerationContext egc(0);
      value = _codeGenerator->convertConstantTo( m_initialiser->generateValue( gc, egc ).constant(), m_initialiser->type() , m_type );
      GTL_ASSERT( value );
    } else {
      value = LLVMBackend::Visitor::getVisitorFor( m_type )->createStaticVariable( _llvmModule, m_type, m_sizes );
    }
    pointer = new llvm::GlobalVariable( *_llvmModule, value ? value->getType() : m_type->d->type(*_llvmContext) , m_constant, llvm::GlobalValue::ExternalLinkage, value, (const std::string&)nameToSymbol() );
  }
  GTL_ASSERT(pointer);
  m_variable->initialise( gc, 0, pointer);
}

GTLCore::String GlobalConstantDeclaration::nameToSymbol()
{
  return nameToSymbol(m_name);
}

GTLCore::String GlobalConstantDeclaration::nameToSymbol(const GTLCore::ScopedName& name)
{
  if(name.nameSpace().isEmpty())
  {
    return name.name();
  } else {
    return name.nameSpace() + "_" + name.name();
  }
}

Tree::Tree()
{
}

Tree::~Tree()
{
  deleteAll( m_functions );
  deleteAll( m_globalConstants );
}

const std::list<FunctionDeclaration*>& Tree::functionsDeclarations() const
{
  return m_functions;
}

void Tree::append(FunctionDeclaration* fd)
{
  m_functions.push_back( fd );
}
const std::list<GlobalConstantDeclaration*>& Tree::globalConstantDeclarations() const
{
  return m_globalConstants;
}

void Tree::append(GlobalConstantDeclaration* constant)
{
  m_globalConstants.push_back( constant );
}

bool Tree::containsGlobalConstant( const GTLCore::ScopedName& _name ) const
{
  for( std::list<GlobalConstantDeclaration*>::const_iterator it = m_globalConstants.begin();
       it != m_globalConstants.end(); ++it)
  {
    if( (*it)->name() == _name ) return true;
  }
  return false;
}


void Tree::generate( GTLCore::ModuleData* _module, llvm::Module* _llvmModule, LLVMBackend::CodeGenerator* _codeGenerator, llvm::LLVMContext* _llvmContext)
{
  // Generate constants
  for(std::list<GlobalConstantDeclaration*>::iterator it = m_globalConstants.begin();
      it != m_globalConstants.end(); ++it)
  {
    (*it)->generate( _module, _llvmModule, _codeGenerator, _llvmContext);
  }
  // Generate functions
  for(std::list<FunctionDeclaration*>::iterator it = m_functions.begin();
      it != m_functions.end(); ++it)
  {
    (*it)->generate( _module, _llvmModule, _codeGenerator, _llvmContext);
  }
}
