# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2013 Canonical
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.

"""Tests for Notes app"""

from __future__ import absolute_import

from testtools.matchers import Equals
from autopilot.matchers import Eventually

from notes_app.tests import NotesAppTestCase

import sqlite3


class TestExpandCollapse(NotesAppTestCase):
    """Tests deleting notes"""

    def setUp(self):
        # Setup the database before starting the app
        self.setup_db()
        super(TestExpandCollapse, self).setUp()

    def setup_db(self):
        notes = [
            """{ "elements" : [
                {"content":"This is a note.","type":"text"}
            ]}""",
            """{ "elements" : [
                {"content":"This is another note.","type":"text"}
            ]}"""
        ]
        path = self.ensure_db()
        conn = sqlite3.connect(path)
        cursor = conn.cursor()
        cursor.execute("DELETE FROM notes")
        for note in notes:
            cursor.execute("INSERT INTO notes (date, note) "
                           "VALUES ('2013-04-07', '" + note + "')")
        conn.commit()
        conn.close()

    def test_expand_and_collapse_many(self):
        notes = self.main_window.get_notes()
        first = notes[0]
        second = notes[1]

        self.assert_note_eventually_collapsed(first)
        self.assert_note_eventually_collapsed(second)

        self.pointing_device.click_object(first)
        self.assert_note_eventually_expanded(first)

        self.pointing_device.click_object(second)
        self.assert_note_eventually_collapsed(first)
        self.assert_note_eventually_expanded(second)

        self.pointing_device.click_object(first)
        self.assert_note_eventually_expanded(first)
        self.assert_note_eventually_collapsed(second)

    def test_collapse_header(self):
        # Skipping the test for now because in order to release qt5.2 we want
        # to match the results as much as possible to what we get in the
        # current image. As the notes app is going to be deprecated, and this
        # bug has a low impact on the user experience, it's not critical and
        # won't be fixed before the qt5.2 release. -- elopio - 2014-03-14
        self.skipTest('The test is failing because of bug #1288876')
        first = self.main_window.get_notes()[0]
        header = self.main_window.get_header()

        self.pointing_device.click_object(first)
        self.assert_note_eventually_expanded(first)

        self.pointing_device.click_object(header)
        self.assert_note_eventually_collapsed(first)

    def test_collapse_bottom(self):
        notes = self.main_window.get_notes()
        first = notes[0]
        second = notes[1]

        self.pointing_device.click_object(first)
        self.assert_note_eventually_expanded(first)

        # click in the empty space after all the notes
        note_x, note_y, note_w, note_h = second.globalRect
        self.pointing_device.move(note_x + note_w / 2, note_y + note_h + 20)
        self.pointing_device.click()

        self.assert_note_eventually_collapsed(first)

    def test_collapse_between(self):
        first = self.main_window.get_notes()[0]

        self.pointing_device.click_object(first)
        self.assert_note_eventually_expanded(first)

        # click in the empty space between notes
        note_x, note_y, note_w, note_h = first.globalRect
        self.pointing_device.move(note_x + note_w / 2, note_y + note_h + 2)
        self.pointing_device.click()

        self.assert_note_eventually_collapsed(first)
