/*
* @license Apache-2.0
*
* Copyright (c) 2021 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

// TypeScript Version: 2.0

/* tslint:disable:max-line-length */
/* tslint:disable:max-file-line-count */

import cdf = require( './../../../../../base/dists/logistic/cdf' );
import Logistic = require( './../../../../../base/dists/logistic/ctor' );
import entropy = require( './../../../../../base/dists/logistic/entropy' );
import kurtosis = require( './../../../../../base/dists/logistic/kurtosis' );
import logcdf = require( './../../../../../base/dists/logistic/logcdf' );
import logpdf = require( './../../../../../base/dists/logistic/logpdf' );
import mean = require( './../../../../../base/dists/logistic/mean' );
import median = require( './../../../../../base/dists/logistic/median' );
import mgf = require( './../../../../../base/dists/logistic/mgf' );
import mode = require( './../../../../../base/dists/logistic/mode' );
import pdf = require( './../../../../../base/dists/logistic/pdf' );
import quantile = require( './../../../../../base/dists/logistic/quantile' );
import skewness = require( './../../../../../base/dists/logistic/skewness' );
import stdev = require( './../../../../../base/dists/logistic/stdev' );
import variance = require( './../../../../../base/dists/logistic/variance' );

/**
* Interface describing the `logistic` namespace.
*/
interface Namespace {
	/**
	* Logistic distribution cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns evaluated CDF
	*
	* @example
	* var y = ns.cdf( 2.0, 0.0, 1.0 );
	* // returns ~0.881
	*
	* var mycdf = ns.cdf.factory( 3.0, 1.5 );
	*
	* y = mycdf( 1.0 );
	* // returns ~0.209
	*/
	cdf: typeof cdf;

	/**
	* Logistic distribution.
	*/
	Logistic: typeof Logistic;

	/**
	* Returns the differential entropy for a logistic distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns entropy
	*
	* @example
	* var y = ns.entropy( 0.0, 1.0 );
	* // returns 2.0
	*
	* @example
	* var y = ns.entropy( 5.0, 2.0 );
	* // returns ~2.693
	*
	* @example
	* var y = ns.entropy( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.entropy( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.entropy( 0.0, 0.0 );
	* // returns NaN
	*/
	entropy: typeof entropy;

	/**
	* Returns the excess kurtosis for a logistic distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns excess kurtosis
	*
	* @example
	* var y = ns.kurtosis( 0.0, 1.0 );
	* // returns 1.2
	*
	* @example
	* var y = ns.kurtosis( 5.0, 2.0 );
	* // returns 1.2
	*
	* @example
	* var y = ns.kurtosis( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.kurtosis( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.kurtosis( 0.0, 0.0 );
	* // returns NaN
	*/
	kurtosis: typeof kurtosis;

	/**
	* Logistic distribution logarithm of cumulative distribution function (CDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns evaluated logCDF
	*
	* @example
	* var y = ns.logcdf( 10.0, 0.0, 3.0 );
	* // returns ~-0.036
	*
	* y = ns.logcdf( 0.0, 0.0, 3.0 );
	* // returns ~-1
	*
	* var mylogcdf = ns.logcdf.factory( 2.0, 3.0 );
	* y = mylogcdf( 10.0 );
	* // returns ~-0.069
	*
	* y = mylogcdf( 2.0 );
	* // returns ~-1
	*/
	logcdf: typeof logcdf;

	/**
	* Logistic distribution natural logarithm of probability density function (logPDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns evaluated logPDF
	*
	* @example
	* var y = ns.logpdf( 2.0, 0.0, 1.0 );
	* // returns ~-2.254
	*
	* var mylogpdf = ns.logpdf.factory( 10.0, 2.0 );
	* y = mylogpdf( 10.0 );
	* // returns -2.079
	*/
	logpdf: typeof logpdf;

	/**
	* Returns the expected value for a logistic distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns expected value
	*
	* @example
	* var y = ns.mean( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.mean( 5.0, 2.0 );
	* // returns 5.0
	*
	* @example
	* var y = ns.mean( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mean( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.mean( 0.0, 0.0 );
	* // returns NaN
	*/
	mean: typeof mean;

	/**
	* Returns the median for a logistic distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns median
	*
	* @example
	* var y = ns.median( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.median( 5.0, 2.0 );
	* // returns 5.0
	*
	* @example
	* var y = ns.median( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.median( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.median( 0.0, 0.0 );
	* // returns NaN
	*/
	median: typeof median;

	/**
	* Gamma distribution moment-generating function (MGF).
	*
	* @param t - input value
	* @param mu - mean
	* @param s - scale parameter
	* @returns evaluated MGF
	*
	* @example
	* var y = ns.mgf( 0.9, 0.0, 1.0 );
	* // returns ~9.15
	*
	* y = ns.mgf( 0.1, 4.0, 4.0 );
	* // returns ~1.971
	*
	* y = ns.mgf( -0.2, 4.0, 4.0 );
	* // returns ~1.921
	*
	* var mymgf = ns.mgf.factory( 10.0, 0.5 );
	*
	* y = mymgf( 0.5 );
	* // returns ~164.846
	*
	* y = mymgf( 2.0 );
	* // returns Infinity
	*/
	mgf: typeof mgf;

	/**
	* Returns the mode for a logistic distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns mode
	*
	* @example
	* var y = ns.mode( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.mode( 5.0, 2.0 );
	* // returns 5.0
	*
	* @example
	* var y = ns.mode( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.mode( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.mode( 0.0, 0.0 );
	* // returns NaN
	*/
	mode: typeof mode;

	/**
	* Logistic distribution probability density function (PDF).
	*
	* @param x - input value
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns evaluated PDF
	*
	* @example
	* var y = ns.pdf( 2.0, 0.0, 1.0 );
	* // returns ~0.105
	*
	* var myPDF = ns.pdf.factory( 10.0, 2.0 );
	* y = myPDF( 10.0 );
	* // returns 0.125
	*/
	pdf: typeof pdf;

	/**
	* Logistic distribution quantile function.
	*
	* @param p - input value
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns evaluated quantile function
	*
	* @example
	* var y = ns.quantile( 0.8, 0.0, 1.0 );
	* // returns ~1.386
	*
	* var myQuantile = ns.quantile.factory( 10.0, 2.0 );
	* y = myQuantile( 0.5 );
	* // returns 10.0
	*/
	quantile: typeof quantile;

	/**
	* Returns the skewness for a logistic distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns skewness
	*
	* @example
	* var y = ns.skewness( 0.0, 1.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.skewness( 5.0, 2.0 );
	* // returns 0.0
	*
	* @example
	* var y = ns.skewness( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.skewness( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.skewness( 0.0, 0.0 );
	* // returns NaN
	*/
	skewness: typeof skewness;

	/**
	* Returns the standard deviation for a logistic distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns standard deviation
	*
	* @example
	* var y = ns.stdev( 0.0, 1.0 );
	* // returns ~1.814
	*
	* @example
	* var y = ns.stdev( 5.0, 2.0 );
	* // returns ~3.628
	*
	* @example
	* var y = ns.stdev( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.stdev( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.stdev( 0.0, 0.0 );
	* // returns NaN
	*/
	stdev: typeof stdev;

	/**
	* Returns the variance for a logistic distribution with location `mu` and scale `s`.
	*
	* ## Notes
	*
	* -   If provided `s <= 0`, the function returns `NaN`.
	*
	* @param mu - location parameter
	* @param s - scale parameter
	* @returns variance
	*
	* @example
	* var y = ns.variance( 0.0, 1.0 );
	* // returns ~3.29
	*
	* @example
	* var y = ns.variance( 5.0, 2.0 );
	* // returns ~13.159
	*
	* @example
	* var y = ns.variance( NaN, 1.0 );
	* // returns NaN
	*
	* @example
	* var y = ns.variance( 0.0, NaN );
	* // returns NaN
	*
	* @example
	* var y = ns.variance( 0.0, 0.0 );
	* // returns NaN
	*/
	variance: typeof variance;
}

/**
* Logistic distribution.
*/
declare var ns: Namespace;


// EXPORTS //

export = ns;
