/**
* @license Apache-2.0
*
* Copyright (c) 2018 The Stdlib Authors.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*    http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

'use strict';

// MODULES //

var tape = require( 'tape' );
var isnan = require( './../../../../base/assert/is-nan' );
var abs = require( './../../../../base/special/abs' );
var PINF = require( '@stdlib/constants/float64/pinf' );
var EPS = require( '@stdlib/constants/float64/eps' );
var fallingFactorial = require( './../lib' );


// FIXTURES //

var negative = require( './fixtures/cpp/negative.json' );
var large = require( './fixtures/cpp/large.json' );
var small = require( './fixtures/cpp/small.json' );


// TESTS //

tape( 'main export is a function', function test( t ) {
	t.ok( true, __filename );
	t.strictEqual( typeof fallingFactorial, 'function', 'main export is a function' );
	t.end();
});

tape( 'the function returns `NaN` if provided a negative `n`', function test( t ) {
	var val = fallingFactorial( 0.5, -1.0 );
	t.strictEqual( isnan( val ), true, 'returns NaN' );

	val = fallingFactorial( 0.5, -2.0 );
	t.strictEqual( isnan( val ), true, 'returns NaN' );
	t.end();
});

tape( 'the function returns `NaN` if provided a non-integer `n`', function test( t ) {
	var val = fallingFactorial( 0.5, 1.5 );
	t.strictEqual( isnan( val ), true, 'returns NaN' );

	val = fallingFactorial( 0.5, 0.3 );
	t.strictEqual( isnan( val ), true, 'returns NaN' );

	val = fallingFactorial( 2.0, -0.3 );
	t.strictEqual( isnan( val ), true, 'returns NaN' );
	t.end();
});

tape( 'the function returns `0` for `x = 0`', function test( t ) {
	var val = fallingFactorial( 0.0, 2 );
	t.strictEqual( val, 0, 'returns 0' );

	val = fallingFactorial( 0.0, 1 );
	t.strictEqual( val, 0, 'returns 0' );
	t.end();
});

tape( 'the function returns `1` for `n = 0`', function test( t ) {
	var val = fallingFactorial( 2.0, 0 );
	t.strictEqual( val, 1, 'returns 1' );

	val = fallingFactorial( -1.7, 0 );
	t.strictEqual( val, 1, 'returns 1' );
	t.end();
});

tape( 'the function returns `+infinity` for small `x` and large `n`', function test( t ) {
	t.equal( fallingFactorial( 0.2, 200 ), PINF, 'equals +infinity' );
	t.equal( fallingFactorial( 0.4, 175 ), PINF, 'equals +infinity' );
	t.equal( fallingFactorial( 0.1, 250 ), PINF, 'equals +infinity' );
	t.end();
});

tape( 'the function evaluates the falling factorial for large `x`', function test( t ) {
	var expected;
	var delta;
	var tol;
	var i;
	var n;
	var x;
	var y;

	expected = large.expected;
	x = large.x;
	n = large.n;
	for ( i = 0; i < x.length; i++ ) {
		y = fallingFactorial( x[i], n[i] );
		if ( y === expected[i] ) {
			t.strictEqual( y, expected[i], 'x: '+x[i]+'. n: '+n[i]+', y: '+y+'. expected: '+expected[i] );
		} else {
			delta = abs( y - expected[ i ] );
			tol = 80.0 * EPS * abs( expected[ i ] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. n: '+n[i]+'. y: '+y+'. E: '+expected[i]+'. Δ: '+delta+'. tol: '+tol );
		}
	}
	t.end();
});

tape( 'the function evaluates the falling factorial for small `x`', function test( t ) {
	var expected;
	var delta;
	var tol;
	var i;
	var n;
	var x;
	var y;

	expected = small.expected;
	x = small.x;
	n = small.n;
	for ( i = 0; i < x.length; i++ ) {
		y = fallingFactorial( x[i], n[i] );
		if ( y === expected[i] ) {
			t.strictEqual( y, expected[i], 'x: '+x[i]+'. n: '+n[i]+', y: '+y+'. expected: '+expected[i] );
		} else {
			delta = abs( y - expected[ i ] );
			tol = 80.0 * EPS * abs( expected[ i ] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. n: '+n[i]+'. y: '+y+'. E: '+expected[i]+'. Δ: '+delta+'. tol: '+tol );
		}
	}
	t.end();
});

tape( 'the function evaluates the falling factorial for negative `x`', function test( t ) {
	var expected;
	var delta;
	var tol;
	var i;
	var n;
	var x;
	var y;

	expected = negative.expected;
	x = negative.x;
	n = negative.n;
	for ( i = 0; i < x.length; i++ ) {
		y = fallingFactorial( x[i], n[i] );
		if ( y === expected[i] ) {
			t.strictEqual( y, expected[i], 'x: '+x[i]+'. n: '+n[i]+', y: '+y+'. expected: '+expected[i] );
		} else {
			delta = abs( y - expected[ i ] );
			tol = 100.0 * EPS * abs( expected[ i ] );
			t.ok( delta <= tol, 'within tolerance. x: '+x[i]+'. n: '+n[i]+'. y: '+y+'. E: '+expected[i]+'. Δ: '+delta+'. tol: '+tol );
		}
	}
	t.end();
});
