/* send_expect.c
  
   The "send_receive" Tcl command is implemented here.

   Note that our "send_receive" command has nothing to do with Scapy's
   send-and-receive family of functions - our "send_expect" command is
   equivalent to Scapy's send-and-receive family of functions, and
   "send_receive" is just a simple thoughput tester that injects stimuli
   and then counts the number of received packets.

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

#include "nexp_speakers.h"
#include "util-tcl.h"

/*
 * List of listeners we'll monitor for answers to injected stimulus. The
 * maximum number of listeners in this list is just an arbitrary number.
 * I reckon most people will only use a handful of listeners at the most,
 * and dynamically managing the size of this list is not worth the trouble.
 */
static struct nexp_listener *listener;

static struct timeval timeout, receive_start_time, curr_time;

static int count;
static pcb_t pcb;

static int nreceived;

static void
usage(void)
{
    fprintf(stderr, "\
usage: send_receive [-speaker <speaker ID>] [-listener <listener ID>]\n\
          [-count <count>] [-delay <secs>] [-rate <pps>] [-timeout <secs>]\n\
          <PDU definition>");
}

static int
process_options(Tcl_Interp *interp, int argc, Tcl_Obj * const *objv)
{
    int i, index;
    char *pdudef = NULL; 
    double delay, t;
    char errbuf[PDU_ERRBUF_SIZE];
    static const char *options[] = {
	"-o", "-i", "-delay", "-rate", "-timeout", "-count", NULL
    };
    enum options {
	OPT_SPEAKER, OPT_LISTENER, OPT_DELAY, OPT_RATE, OPT_TIMEOUT,
	OPT_COUNT
    };

    /*
     * Parse command-line arguments.
     */
    for (i = 1; i < argc && *Tcl_GetString(objv[i]) == '-'; i++) {
	if (Tcl_GetIndexFromObj(interp, objv[i], options, "option", 0, &index)
	    != TCL_OK)
	    return -1;

	switch ( (enum options) index) {
	case OPT_SPEAKER:
	    if (++i >= argc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-o speaker");
		goto error;
	    }

	    pcb.speaker = lookup_speaker(Tcl_GetString(objv[i]) );
	    if (!pcb.speaker) {
		nexp_error(interp, "No speaker named \"%s\". Use "
			   "\"spawn_network -info\" to find out existing "
			   "speakers.", Tcl_GetString(objv[i]) );
		goto error;
	    }
	    break;
	case OPT_LISTENER:
	    if (++i >= argc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-o listener");
		goto error;
	    }

	    listener = lookup_listener(Tcl_GetString(objv[i]) );
	    if (!listener) {
		nexp_error(interp, "No listener named \"%s\". Use "
			   "\"spawn_network -info\" to find out existing "
			   "listeners.", Tcl_GetString(objv[i]) );
		goto error;
	    }

	    if (listener->type != LISTENER_LIVE) {
		nexp_error(interp, "The send_expect command only supports "
				   "live listeners.");
		goto error;
	    }
	    break;
	case OPT_DELAY:
	    if (++i >= argc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-d seconds");
		goto error;
	    }

	    if (Tcl_GetDoubleFromObj(interp, objv[i], &delay) != TCL_OK)
		goto error;

	    pcb.delay.tv_sec = delay;
	    pcb.delay.tv_usec = (delay - pcb.delay.tv_sec)*1000000UL;
	    break;
	case OPT_RATE:
	    if (++i >= argc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-r PPS");
		goto error;
	    }

	    /* Convert a packets-per-second rate to a usecs delay */
	    if (Tcl_GetDoubleFromObj(interp, objv[i], &delay) != TCL_OK)
		goto error;

	    if (delay == 0.0) {
		nexp_error(interp, "Rate can't be 0 packets per second.");
		goto error;
	    }

	    delay = 1/delay;

	    pcb.delay.tv_sec = delay;
	    pcb.delay.tv_usec = (delay - pcb.delay.tv_sec)*1000000UL;
	    break;
	case OPT_TIMEOUT:
	    if (++i >= argc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-timeout seconds");
		goto error;
	    }

	    t = strtod(Tcl_GetString(objv[i]), NULL);
	    if (t < 0.0) {
		nexp_error(interp, "Timeout can't be negative");
		goto error;
	    }

	    timeout.tv_sec = t;
	    timeout.tv_usec = (t - timeout.tv_sec)*1000000UL;
	    break;
	case OPT_COUNT:
	    if (++i >= argc) {
		Tcl_WrongNumArgs(interp, 1, objv, "-c npackets");
		goto error;
	    }

	    if (Tcl_GetIntFromObj(interp, objv[i], &count) != TCL_OK)
		goto error;
	    break;
	}
    }

    /*
     * We treat whatever is left on the command line, i.e. anything that
     * is not an option (anything that doesn't start with '-'), as a PDU
     * definition.
     */
    pdudef = copy_objv(argc - i, &objv[i]);
    if (!pdudef) {
	usage();
	return -1;
    }

#ifdef DEBUG
    printf("PDU definition = %s\n", pdudef);
#endif

    if ( (pcb.pdu = pb_parsedef(pdudef, errbuf) ) == NULL) {
	nexp_error(interp, "%s", errbuf);
	goto error;
    }

    pcb.def = pdudef;

    return 0;

error:
    if (pdudef)
	free(pdudef);

    return -1;
}

static int
send_packets(void)
{
    int i, j;
    int packet_count, packets_sent;
    uint8_t *packet;
    size_t packet_size;

    packet_count = count ? count : pb_permutation_count(pcb.pdu);
    packet = xmalloc(pb_len(pcb.pdu) );

    for (packets_sent = i = 0; i < packet_count; i++) {
	packet_size = pb_build(pcb.pdu, packet, NULL);

	for (j = 0; j < pcb.repeat_count + 1; j++) {
	    /*
	     * Handle inter-packet sending delay.
	     */
	    if (timerisset(&pcb.delay)
		&& timerisset(&pcb.speaker->ts) ) {
		struct timeval now, later;

		timeradd(&pcb.speaker->ts, &pcb.delay, &later);

		gettimeofday(&now, NULL);
		/*
		 * Some versions of Solaris (seen in Solaris 8) don't
		 * like using timercmp() with "<=" or ">=". Rearranging
		 * the operands and just using '>' or '<' works around
		 * this limitation.
		 */
		while (timercmp(&later, &now, >) )
		    gettimeofday(&now, NULL);
	    }

	    nexp_pdu_output(&pcb, packet, packet_size);

	    if (vflag)
		putchar('S');

	    packets_sent++;
	}
    }

    if (vflag)
	putchar('\n');

    free(packet);

    return packets_sent;
}

/*
 * Process a received packet where "process" just means "count".
 */
static void
process_pkt(u_char *user _U_, const struct pcap_pkthdr *h,
	    const u_char *bytes _U_)
{
    curr_time = h->ts;

    if (!timerisset(&receive_start_time) )
	receive_start_time = curr_time;

    nreceived++;
}

/*
 * Handles the SIGUSR1 signal sent by the forked process that injects packets.
 * This happens when the forked process is finished injecting packets. This
 * handler needs to set an interval timer that will let us know, when it
 * expires, that packet reception needs to stop.
 */
static void
sigusr1_handler(int signum _U_)
{
    struct itimerval timer;
#ifdef DEBUG
    printf("Caught SIGUSR1 (%d); packet injection has finished\n", signum);
#endif

    timer.it_interval = timer.it_value = timeout;

    if (setitimer(ITIMER_REAL, &timer, NULL) == -1)
	fprintf(stderr, "setitimer(): %s (errno = %d)\n", strerror(errno),
		errno);
}

static void
sigalrm_handler(int signum _U_)
{
    struct itimerval timer;
#ifdef DEBUG
    printf("Caught SIGALRM (%d); packet reception has finished\n", signum);
#endif

    /* Stop timer */
    timerclear(&timer.it_interval);
    timerclear(&timer.it_value);

    if (setitimer(ITIMER_REAL, &timer, NULL) == -1)
	fprintf(stderr, "setitimer(): %s (errno = %d)\n", strerror(errno),
		errno);

    pcap_breakloop(listener->_live.pd);
}

static int
NExp_SendReceiveObjCmd(ClientData clientData _U_, Tcl_Interp *interp, int argc,
		       Tcl_Obj * const *objv)
{
    int retval;
    struct timeval time_delta;
    Tcl_Obj *obj;
    double receive_time;
    int pfds[2]; /* Pipe file descriptors */
    int pid;
    int nsent;
    struct sigaction sigusr1_action, sigalrm_action;

    listener = NULL;
    timeout.tv_sec = 1; timeout.tv_usec = 0;
    nreceived = 0;

    memset(&pcb, 0, sizeof(pcb) );

    retval = process_options(interp, argc, objv);
    if (retval == -1)
	goto error;

    /*
     * Make sure we have a listener we can use to listen for answers.
     */
    if (!listener) {
	listener = lookup_listener(nexp_get_var(interp,
						LISTENER_SPAWN_ID_VARNAME) );
	if (!listener) {
	    nexp_error(interp, "Can't find a suitable listener! Use "
			       "spawn_network to create one.");
	    goto error;
	}
    }

    /*
     * Make sure the PCB has an assigned speaker: if the user has not
     * explicitely specified a speaker, then we use the default speaker,
     * which is referred to by name via the Tcl variable "speaker_id"
     * (SPEAKER_SPAWN_ID_VARNAME).
     */
    if (!pcb.speaker) {
	pcb.speaker = lookup_speaker(nexp_get_var(interp,
					SPEAKER_SPAWN_ID_VARNAME) );
	if (!pcb.speaker) {
	    nexp_error(interp, "Can't find a suitable speaker! Use "
			       "spawn_network to create one.");
	    goto error;
	}
    }

    if (pipe(pfds) == -1) {
	nexp_error(interp, "Couldn't open pipe: %s", strerror(errno) );
	goto error;
    }

    /* Set up handler for SIGUSR1 (indicates end of injection) */
    memset(&sigusr1_action, 0, sizeof(sigusr1_action) );
    sigusr1_action.sa_handler = sigusr1_handler;
    sigusr1_action.sa_flags = SA_RESETHAND;
    sigemptyset(&sigusr1_action.sa_mask);
    sigaction(SIGUSR1, &sigusr1_action, NULL);

    /* Set up handler for SIGALRM (indicates end of reception) */
    memset(&sigalrm_action, 0, sizeof(sigalrm_action) );
    sigalrm_action.sa_handler = sigalrm_handler;
    sigalrm_action.sa_flags = SA_RESETHAND;
    sigemptyset(&sigalrm_action.sa_mask);
    sigaction(SIGALRM, &sigalrm_action, NULL);

    /*
     * Stimulus injection is done by a forked (child) process. Receiving
     * packets is done by the parent process.
     */

    pid = fork();
    if (pid == 0) {
	/*
	 * Child.
	 */

	close(pfds[0]); /* Child doesn't read from parent */

	nsent = send_packets();

	/* Let parent know how many packets we sent. */
	write(pfds[1], &nsent, sizeof(nsent) );

	/* Let parent know that packet injection has finished. */
	kill(getppid(), SIGUSR1);

	close(pfds[1]);
	_exit(EXIT_SUCCESS);
    } else if (pid < 0) {
	nexp_error(interp, "fork(): %s", strerror(errno) );
	goto error;
    }

    /*
     * Parent.
     */

    /* All the work is done by the pcap_handler callback function. */
    retval = pcap_loop(listener->_live.pd, -1, process_pkt, NULL);
    if (retval == -1) {
	nexp_error(interp, "pcap_loop(): %s", pcap_geterr(listener->_live.pd) );
	goto error;
    }

    if (vflag)
	putchar('\n');

    wait(NULL);

    /*
     * Done sending and receiving packets.
     */

    /*
     * Read from the pipe between parent and child the number of packets
     * sent by the child process.
     */
    retval = read(pfds[0], &nsent, sizeof(nsent) );
    if (retval == 0) {
	/* EOF */
	nexp_error(interp, "read(): got EOF. This should not happen!");
	goto error;
    } else if (retval == -1) {
	/* read() error */
	nexp_error(interp, "read(): %s", strerror(errno) );
	goto error;
    } else if (retval != sizeof(nsent) ) {
	/* We got less than what we asked for */
	nexp_error(interp, "read() read less than requested. This "
			   "should not happen!");
	goto error;
    }

    close(pfds[0]);

    if (vflag)
	printf("Sent %d packets; received %d\n", nsent, nreceived);

    obj = Tcl_NewIntObj(nsent);
    Tcl_SetVar2Ex(interp, "_", "sent", obj, 0);

    obj = Tcl_NewIntObj(nreceived);
    Tcl_SetVar2Ex(interp, "_", "received", obj, 0);

    time_delta = curr_time;
    timersub(&time_delta, &receive_start_time, &time_delta);

    receive_time = time_delta.tv_sec + time_delta.tv_usec/1e6;

    obj = Tcl_NewDoubleObj(receive_time);
    Tcl_SetVar2Ex(interp, "_", "time", obj, 0);

    obj = Tcl_NewDoubleObj( (nreceived - 1)/receive_time);
    Tcl_SetVar2Ex(interp, "_", "pps", obj, 0);

    /*
     * Clean up.
     */

    pcb_destroy(&pcb);

    return TCL_OK;

error:
    return TCL_ERROR;
}

static struct nexp_cmd_data cmd_data[] = {
    {"send_receive", NExp_SendReceiveObjCmd, NULL, 0, 0},

    {NULL, NULL, NULL, 0, 0}
};

void
nexp_init_send_receive_cmd(Tcl_Interp *interp)
{
    nexp_create_commands(interp, cmd_data);
}
