/* pdu-bgp.c
  
   PDU builder for BGP messages

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"
#include "pdu-bgp.h"

/*********************************************************************/
/************************* BGP OPEN message **************************/
/*********************************************************************/

static void
pdu_bgpopen_builder(const GNode *pdu, void *dest)
{
    struct bgphdr *bgp;
    uint8_t *bgp_open;
    void *field;
    size_t opts_len;

    bgp = dest;

    /**************** BGP header ****************/
    memset(bgp->marker, 0xff, sizeof(bgp->marker) );
    SSVAL(&bgp->len, 0, sizeof(struct bgphdr) + 10);
    bgp->type = BGP_MSG_OPEN;

    bgp_open = (uint8_t *) bgp + sizeof(struct bgphdr);

    *bgp_open++ = (field = _pb_pdata(pdu, "version") )
		  ? num_next(field) : BGP_VERSION_4;

    SSVAL(bgp_open, 0, htons(num_next(_pb_pdata(pdu, "asn") ) ) );
    bgp_open += sizeof(uint16_t);

    SSVAL(bgp_open, 0, htons(num_next(_pb_pdata(pdu, "holdtime" ) ) ) );
    bgp_open += sizeof(uint16_t);

    SIVAL(bgp_open, 0, ip_next(_pb_pdata(pdu, "id") ) );
    bgp_open += sizeof(struct in_addr);

    *bgp_open++ = num_next(_pb_pdata(pdu, "parmlen") );

    /* Finally, store the length of the entire BGP message */
    if ( (field = _pb_pdata(pdu, "length") ) )
	bgp->len = htons(num_next(field) );
    else {
	opts_len = ( (struct node_data *) pdu->data)->_data_pdu.opts_len;
	bgp->len = htons( (void *) bgp_open - dest + opts_len);
    }
}

#if 0
void
pdu_bgpopen_dumper(pdu_t *p, const char *prefix)
{
    struct bgpopenhdr_options *hdr_data;
    char addr[INET_ADDRSTRLEN];

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    Version: %s\n", prefix, num_info(hdr_data->version) );
    printf("%s    ASN: %d\n", prefix, hdr_data->asn);
    printf("%s    Holdtime: %s\n", prefix, num_info(hdr_data->holdtime) );
    printf("%s    ID: %s\n", prefix,
	   inet_ntop(AF_INET, &hdr_data->id, addr, INET_ADDRSTRLEN) );
    printf("%s    Optional parms. length: %s\n", prefix, num_info(hdr_data->parmlen) );
    printf("%s    Message length: %s\n", prefix,
	   hdr_data->length ? num_info(hdr_data->length) : "automatic");
}
#endif

/*********************************************************************/
/************************ BGP UPDATE message *************************/
/*********************************************************************/

static void
pdu_bgpupdate_builder(const GNode *pdu, void *dest)
{
    int i;
    struct in_addr network;
    struct bgphdr *bgp;
    uint8_t *bgp_payload;
    uint8_t *unfeasible_routes_len_ptr;
    uint8_t *tot_path_attr_len_ptr;
    uint8_t *path_attr_ptr;
    uint8_t *net_reachability_ptr;
    uint8_t prefix;
    void *field;

    bgp = dest;

    /**************** BGP header ****************/
    memset(bgp->marker, 0xff, sizeof(bgp->marker) );
    SSVAL(&bgp->len, 0, 51);
    bgp->type = BGP_MSG_UPDATE;

    bgp_payload = (unsigned char *) bgp + sizeof(struct bgphdr);

    /**************** Unfeasible routes ****************/
    unfeasible_routes_len_ptr = bgp_payload;
    SSVAL(unfeasible_routes_len_ptr, 0, 0); /* We're not withdrawing any
					       routes */

    /**************** Path attributes ****************/
    tot_path_attr_len_ptr =  bgp_payload
			     + sizeof(uint16_t)
			     + ntohs(SVAL(unfeasible_routes_len_ptr, 0) );

    path_attr_ptr = tot_path_attr_len_ptr + sizeof(uint16_t);

    /* Attribute #1: ORIGIN */
    *path_attr_ptr++ = BGP_ATTR_FLAG_TRANS; /* Attribute flags */
    *path_attr_ptr++ = BGP_ATTR_ORIGIN; /* Attribute type */
    *path_attr_ptr++ = 1; /* Length of this attribute */
    *path_attr_ptr++ = BGP_ORIGIN_IGP; /* Origin */

    /* Attribute #2: AS_PATH */
    *path_attr_ptr++ = BGP_ATTR_FLAG_TRANS; /* Attribute flags */
    *path_attr_ptr++ = BGP_ATTR_AS_PATH; /* Attribute type */
    *path_attr_ptr++ = 4; /* Length of this attribute */
    *path_attr_ptr++ = AS_SEQUENCE; /* Path segment type */
    *path_attr_ptr++ = 1; /* Path segment length = 1 AS */
    SSVAL(path_attr_ptr, 0, htons(num_next(_pb_pdata(pdu, "asn") ) ) ); /* Path segment value */
    path_attr_ptr+= sizeof(uint16_t);

    /* Attribute #3: NEXT_HOP */
    *path_attr_ptr++ = BGP_ATTR_FLAG_TRANS; /* Attribute flags */
    *path_attr_ptr++ = BGP_ATTR_NEXT_HOP; /* Attribute type */
    *path_attr_ptr++ = 4; /* Length of this attribute */

    SIVAL(path_attr_ptr, 0, ip_next(_pb_pdata(pdu, "next-hop") ) ); /* Yay! */
    path_attr_ptr += sizeof(uint32_t);

    /* Attribute #4: MULTI_EXIT_DISC */
    *path_attr_ptr++ = BGP_ATTR_FLAG_OPTIONAL; /* Attribute flags */
    *path_attr_ptr++ = BGP_ATTR_MULTI_EXIT_DISC; /* Attribute type */
    *path_attr_ptr++ = 4; /* Length of this attribute */

    SIVAL(path_attr_ptr, 0, 0);
    path_attr_ptr += sizeof(uint32_t);

    /* Finally, compute the length of all the attributes */
    SSVAL(tot_path_attr_len_ptr, 0, htons(path_attr_ptr
					  - tot_path_attr_len_ptr
					  - sizeof(uint16_t) ) );

    /**************** Network layer reachability info.  ****************/
    net_reachability_ptr = path_attr_ptr;

    prefix = *net_reachability_ptr++ = num_next(_pb_pdata(pdu, "prefix") ); /* Length in bits of the IP address prefix */

    network.s_addr = htonl(ip_next(_pb_pdata(pdu, "dest-net") ) );
    for (i = 0; i < prefix/8; i++) {
	*net_reachability_ptr++ = (network.s_addr & 0xff000000) >> 24;
	network.s_addr <<= 8;
    }

    /* Finally, store the length of the entire BGP message */
    if ( (field = _pb_pdata(pdu, "length") ) )
	bgp->len = htons(num_next(field) );
    else
	bgp->len = htons(net_reachability_ptr - (uint8_t *) bgp);
}

#if 0
void
pdu_bgpupdate_dumper(pdu_t *p, const char *prefix)
{
    struct bgpupdatehdr_options *hdr_data;
    char addr[INET_ADDRSTRLEN];

    hdr_data = p->header_data;

    printf("%s  Parameters:\n", prefix);
    printf("%s    ASN: %d\n", prefix, hdr_data->asn);
    printf("%s    Next hop: %s\n", prefix,
	   inet_ntop(AF_INET, &hdr_data->next_hop, addr, INET_ADDRSTRLEN) );
    printf("%s    Destination network: %s\n", prefix,
	   inet_ntop(AF_INET, &hdr_data->dest_net, addr, INET_ADDRSTRLEN) );
    printf("%s    Prefix: %d\n", prefix, hdr_data->prefix);
    printf("%s    Message length: %s\n", prefix,
	   hdr_data->length ? num_info(hdr_data->length) : "automatic");
}
#endif

static const pdu_t pdu_bgpopen = {
    .name = "bgp-open",
    .description = "BGP OPEN message",
    .documented_in = "RFC 1771",
    .len = sizeof(struct bgphdr) + 10, /* FIXME */
    .fields = (field_t []) {
	{.name = "version", .type = PDU_FTYPE_NUMTYPE},
	{.name = "asn", .type = PDU_FTYPE_NUMTYPE},
	{.name = "holdtime", .type = PDU_FTYPE_NUMTYPE},
	{.name = "id", .type = PDU_FTYPE_IP},
	{.name = "parmlen", .type = PDU_FTYPE_NUMTYPE},
	{.name = "length", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .build = &pdu_bgpopen_builder
};

static const pdu_t pdu_bgpupdate = {
    .name = "bgp-update",
    .description = "BGP UPDATE message",
    .documented_in = "RFC 1771",
    .len = 51, /* Ask me later how I came up with this number. peloy.- */
    .fields = (field_t []) {
	{.name = "prefix", .type = PDU_FTYPE_NUMTYPE},
	{.name = "asn", .type = PDU_FTYPE_NUMTYPE},
	{.name = "next-hop", .type = PDU_FTYPE_IP},
	{.name = "dest-net", .type = PDU_FTYPE_IP},
	{.name = "length", .type = PDU_FTYPE_NUMTYPE},
	{.name = NULL}
    },
    .build = &pdu_bgpupdate_builder
};

void
_pb_register_bgp(void)
{
    _pb_register_protocol(&pdu_bgpopen);
    _pb_register_protocol(&pdu_bgpupdate);
}
