"use strict"

var o = require("../../ospec/ospec")
var parseURL = require("../../test-utils/parseURL")

o.spec("parseURL", function() {
	var root = {protocol: "http:", hostname: "localhost", port: "", pathname: "/"}

	o.spec("full URL", function() {
		o("parses full URL", function() {
			var data = parseURL("http://www.google.com:80/test?a=b#c")
			o(data.protocol).equals("http:")
			o(data.hostname).equals("www.google.com")
			o(data.port).equals("80")
			o(data.pathname).equals("/test")
			o(data.search).equals("?a=b")
			o(data.hash).equals("#c")
		})
		o("parses full URL omitting optionals", function() {
			var data = parseURL("http://www.google.com")
			o(data.protocol).equals("http:")
			o(data.hostname).equals("www.google.com")
			o(data.port).equals("")
			o(data.pathname).equals("/")
			o(data.search).equals("")
			o(data.hash).equals("")
		})
	})
	o.spec("absolute path", function() {
		o("parses absolute path", function() {
			var data = parseURL("/test?a=b#c", root)
			o(data.protocol).equals(root.protocol)
			o(data.hostname).equals(root.hostname)
			o(data.port).equals(root.port)
			o(data.pathname).equals("/test")
			o(data.search).equals("?a=b")
			o(data.hash).equals("#c")
		})
		o("parses absolute path omitting optionals", function() {
			var data = parseURL("/test?a=b#c", root)
			o(data.protocol).equals(root.protocol)
			o(data.hostname).equals(root.hostname)
			o(data.port).equals(root.port)
			o(data.pathname).equals("/test")
			o(data.search).equals("?a=b")
			o(data.hash).equals("#c")
		})
	})
	o.spec("relative path", function() {
		o("parses relative URL", function() {
			var data = parseURL("test?a=b#c", root)
			o(data.protocol).equals(root.protocol)
			o(data.hostname).equals(root.hostname)
			o(data.port).equals(root.port)
			o(data.pathname).equals("/test")
			o(data.search).equals("?a=b")
			o(data.hash).equals("#c")
		})
		o("parses relative URL omitting optionals", function() {
			var data = parseURL("test", root)
			o(data.protocol).equals(root.protocol)
			o(data.hostname).equals(root.hostname)
			o(data.port).equals(root.port)
			o(data.pathname).equals("/test")
			o(data.search).equals("")
			o(data.hash).equals("")
		})
		o("parses relative URL with dot", function() {
			var data = parseURL("././test?a=b#c", root)
			o(data.protocol).equals(root.protocol)
			o(data.hostname).equals(root.hostname)
			o(data.port).equals(root.port)
			o(data.pathname).equals("/test")
			o(data.search).equals("?a=b")
			o(data.hash).equals("#c")
		})
		o("parses relative URL with dotdot", function() {
			var data = parseURL("foo/bar/../../test?a=b#c", root)
			o(data.protocol).equals(root.protocol)
			o(data.hostname).equals(root.hostname)
			o(data.port).equals(root.port)
			o(data.pathname).equals("/test")
			o(data.search).equals("?a=b")
			o(data.hash).equals("#c")
		})
		o("clamps invalid dotdot", function() {
			var data = parseURL("../../test?a=b#c", root)
			o(data.protocol).equals(root.protocol)
			o(data.hostname).equals(root.hostname)
			o(data.port).equals(root.port)
			o(data.pathname).equals("/test")
			o(data.search).equals("?a=b")
			o(data.hash).equals("#c")
		})
		o("clamps invalid dotdot after dot", function() {
			var data = parseURL("./../../test?a=b#c", root)
			o(data.protocol).equals(root.protocol)
			o(data.hostname).equals(root.hostname)
			o(data.port).equals(root.port)
			o(data.pathname).equals("/test")
			o(data.search).equals("?a=b")
			o(data.hash).equals("#c")
		})
		o("clamps invalid dotdot after valid path", function() {
			var data = parseURL("a/../../test?a=b#c", root)
			o(data.protocol).equals(root.protocol)
			o(data.hostname).equals(root.hostname)
			o(data.port).equals(root.port)
			o(data.pathname).equals("/test")
			o(data.search).equals("?a=b")
			o(data.hash).equals("#c")
		})
	})
	o.spec("edge cases", function() {
		o("handles hash w/ question mark", function() {
			var data = parseURL("http://www.google.com/test#a?c")
			o(data.pathname).equals("/test")
			o(data.search).equals("")
			o(data.hash).equals("#a?c")
		})
		o("handles hash w/ slash", function() {
			var data = parseURL("http://www.google.com/test#a/c")
			o(data.pathname).equals("/test")
			o(data.search).equals("")
			o(data.hash).equals("#a/c")
		})
		o("handles hash w/ colon", function() {
			var data = parseURL("http://www.google.com/test#a:c")
			o(data.pathname).equals("/test")
			o(data.search).equals("")
			o(data.hash).equals("#a:c")
		})
		o("handles search w/ slash", function() {
			var data = parseURL("http://www.google.com/test?a/c")
			o(data.pathname).equals("/test")
			o(data.search).equals("?a/c")
			o(data.hash).equals("")
		})
		o("handles search w/ colon", function() {
			var data = parseURL("http://www.google.com/test?a:c")
			o(data.pathname).equals("/test")
			o(data.search).equals("?a:c")
			o(data.hash).equals("")
		})
		o("handles pathname w/ colon", function() {
			var data = parseURL("http://www.google.com/a:b")
			o(data.pathname).equals("/a:b")
		})
	})
})
