/* BEGIN software license
 *
 * MsXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright(C) 2009,...,2018 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 *
 * This file is part of the MsXpertSuite project.
 *
 * The MsXpertSuite project is the successor of the massXpert project. This
 * project now includes various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// Stdlib includes


/////////////////////// Qt includes
#include <QString>
#include <QFileInfo>
#include <QDir>


/////////////////////// Local includes
#include "MsXpS/libXpertMassCore/PolChemDefSpec.hpp"


namespace MsXpS
{
namespace libXpertMassCore
{


/*!
\class MsXpS::libXpertMassCore::PolChemDefSpec
\inmodule libXpertMassCore
\ingroup PolChemDef
\inheaderfile PolChemDefSpec.hpp

\brief The PolChemDefSpec class provides metadata for accessing a given polymer
chemistry definition's data on disk.

\sa PolChemDef
*/

/*!
\variable MsXpS::libXpertMassCore::PolChemDefSpec::m_name

\brief The name of the polymer chemistry definition, like \e{protein-1-letter}
or \e{nucac}, for example.

This name is typically identical to both the name of the directory where all
the data defining this \c PolChemDef is stored and the name of the XML file
that contains the definition itself.
*/


/*!
\variable MsXpS::libXpertMassCore::PolChemDefSpec::m_filePath

\brief The path to the polymer chemistry defintion.

The path is relative to the polymer chemistry definitions directory in the data
directory, as found in the catalogue files, like
protein-1-letter/protein-1-letter.xml.
*/

/*!
\brief Constructs a polymer chemistry definition specification instance using \a name as the PolChemDef name.
*/
PolChemDefSpec::PolChemDefSpec(const QString &name) : m_name(name)
{
}


/*!
\brief Destroys the polymer chemistry definition specification instance.
*/
PolChemDefSpec::~PolChemDefSpec()
{
}

/*!
\brief Sets the \a name of the polymer chemistry definition.
*/
void
PolChemDefSpec::setName(const QString &name)
{
  m_name = name;
}

/*!
\brief Returns the name of the polymer chemistry definition.
*/
const QString &
PolChemDefSpec::getName() const
{
  return m_name;
}

/*!
\brief Sets the path to the polymer chemistry definition file to \a file_path.
*/
void
PolChemDefSpec::setFilePath(const QString &file_path)
{
  m_filePath = file_path;
}

/*!
\brief Returns the path to the polymer chemistry definition file.
*/
QString
PolChemDefSpec::getFilePath() const
{
  return m_filePath;
}

// GCOV_EXCL_START
/*!
\brief Returns the absolute path to the polymer chemistry definition file.

If m_filePath is "protein-1-letter/protein-1-letter.xml",
returns "<abspathtodir>/protein-1-letter"
*/
QString
PolChemDefSpec::dirPath()
{
  QFileInfo fileInfo(m_filePath);
  QDir dir(fileInfo.dir());
  return dir.absolutePath();
}
// GCOV_EXCL_STOP

} // namespace libXpertMassCore

} // namespace MsXpS
