#ifndef _OPENING_BOOK_H
#define _OPENING_BOOK_H
#include "osl/move.h"
#include "osl/stl/vector.h"
#include "osl/state/simpleState.h"
#include "osl/record/compactBoard.h"
#include <fstream>
#include <functional>

namespace osl
{
  namespace record
  {
    namespace opening
    {
      class OMove
      {
      public:
	OMove(int i) { value = i; }
	OMove(Move m)
	{ 
	  const Square from = m.from();
	  const Square to = m.to();
	  const int bitFrom = (from.isPieceStand() ? 0 : 
			       (from.x() << 4 | from.y()));
	  const int bitTo = (to.isPieceStand() ? 0 : 
			     (to.x() << 12 | to.y() << 8));
	  value = (bitFrom | bitTo |
		   static_cast<unsigned int>(m.isPromotion()) << 19 |
		   static_cast<unsigned int>(m.capturePtype()) << 20 |
		   static_cast<unsigned int>(m.ptype()) << 24 |
		   static_cast<int>(m.player()) << 28);
	}
	Square getFrom()
	{
	  if ((value & 0xff) == 0)
	    return Square::STAND();
	  else
	    return Square((value >> 4) & 0xf, value & 0xf);
	}
	Square getTo()
	{
	  if (((value >> 8) & 0xff) == 0)
	    return Square::STAND();
	  else
	    return Square((value >> 12) & 0xf, (value >> 8) & 0xf);
	}
	bool isPromotion()
	{
	  return (value >> 19) & 1;
	}
	Ptype getCapturePtype()
	{
	  return static_cast<Ptype>((value >> 20) & 0xf);
	}
	Ptype getPtype()
	{
	  return static_cast<Ptype>((value >> 24) & 0xf);
	}
	Player getPlayer()
	{
	  return static_cast<Player>((value) >> 28);
	}

	operator Move() { return Move(getFrom(), getTo(), getPtype(),
					 getCapturePtype(), isPromotion(),
					 getPlayer()); }
	operator int() { return value; }
      private:
	int value;
      };

      class OBMove
      {
	Move move;
	int stateIndex;
      public:
	OBMove(Move m,int i) :move(m),stateIndex(i) {}
	Move getMove() const { return move; }
	int getStateIndex() const { return stateIndex; }
      };

      /**
       * StateとOBMoveを保持する.
       * Stateはvector<OBMove>と黒から見たwinCount, loseCountを保持する
       * OBMoveはMoveとそのMoveを採用した時のStateのindex
       * ファイル形式
       * state数 - 4byte
       * State - 16byte * state数
       *   + 黒のwinCount
       *   + 白のwinCount
       *   + OBMoveの数 
       *   + OBMoveの開始index
       * OBMove - 8byte * OBMove数
       *   + Move (4byte)
       *   + Stateのindex
       */
      class WinCountBook
      {
	int nStates;
	std::ifstream ifs;
      public:
	WinCountBook(const char *filename);
	~WinCountBook();
	int getWinCount(int stateIndex);
	int getLoseCount(int stateIndex);
	vector<OBMove> getMoves(int stateIndex);
      private:
	int readInt();
	void seek(int offset);
      };

      class WMove;
      std::ostream& operator<<(std::ostream& os, const WMove& w);
      std::istream& operator>>(std::istream& is, WMove& w);
      class WMove
      {
	Move move;
	int stateIndex;
	int weight;
      public:
	WMove() {}
	WMove(Move m, int i, int w)
	  : move(m), stateIndex(i), weight(w) {}
	Move getMove() const { return move; }
	int getStateIndex() const { return stateIndex; }
	int getWeight() const { return weight; }
        void setWeight(const int w) { weight = w; };
	friend std::ostream& operator<<(std::ostream& os, const WMove& w);
	friend std::istream& operator>>(std::istream& is, WMove& w);
      };
      inline bool operator==(const WMove& l, const WMove& r) 
      {
	return l.getMove() == r.getMove() && l.getStateIndex() == r.getStateIndex()
	  && l.getWeight() == r.getWeight();
      }

      /**
       * WMoveのWeightによるsort
       */
      struct WMoveSort : public std::binary_function<WMove, WMove, bool>
      {
	bool operator()(const WMove& l, const WMove& r) const 
	{
	  return l.getWeight() > r.getWeight();
	}
      };

      /**
       * WMoveのMoveによるsort
       */
      struct WMoveMoveSort : public std::binary_function<WMove, WMove, bool>
      {
        bool operator()(const WMove& l, const WMove& r) const 
        {
          return l.getMove().intValue() < r.getMove().intValue();
        }
      };

      /**
       * WMoveのWeightとMoveによるsort
       */
      struct WMoveWeightMoveSort : public std::binary_function<WMove, WMove, bool>
      {
        bool operator()(const WMove& l, const WMove& r) const 
        {
	  if (l.getWeight() != r.getWeight())
	    return l.getWeight() > r.getWeight();
          return l.getMove().intValue() < r.getMove().intValue();
        }
      };

      /**
       * StateとWMoveを保持する.
       * Stateはvector<WMove>を保持する
       * WMoveはMoveとそのMoveを採用した時のStateのindexと手番から見た
       * Moveの重み(0-1000)をもつ
       * ファイル形式
       * version番号 - 4byte
       * state数 - 4byte
       * move数 - 4byte
       * 開始state index - 4byte
       * State - 16byte * state数
       *   + WMoveの開始index
       *   + WMoveの数 
       *   + 先手の勝数
       *   + 後手の勝数
       * WMove - 12byte * WMove数
       *   + Move (4byte)
       *   + Stateのindex
       *   + Weight
       * CompactBoard形式の盤面 - 164byte * state数
       */
      class WeightedBook
      {
	int nStates;
	int nMoves;
	int startState;
	std::ifstream ifs;
      public:
        typedef vector<WMove> WMoveContainer;

	WeightedBook(const char *filename);
	~WeightedBook();
        /**
         * Return moves from the state of the stateIndex. If the zero_include is
         * true, all of the moves are returned. Otherwise, the moves that
         * have some weights (i.e. non-zero value) are returned.
         */
	WMoveContainer getMoves(int stateIndex, const bool zero_include = true);
	int getWhiteWinCount(int stateIndex);
	int getBlackWinCount(int stateIndex);
        osl::record::CompactBoard getCompactBoard(int stateIndex);
	SimpleState getBoard(int stateIndex);
	int getTotalState() const { return nStates; }
	int getStartState() const { return startState; }
	void validate();
        /** 
         * As traversing the 'tree', return all state indices of the state's
         * parents. 
         * @return state indexes; empty if there is none.
         */
        std::vector<int> getParents(const int stateIndex);
        /** 
         * As traversing the 'tree', find a state index of the state.  If
         * the visit_zero is true zero-weighted moves are visited (in this
         * case, player is ignored). Otherwise, the palyer's zero-weighted
         * moves are not visited.
         *
         * @param state to find
         * @param visit_zero
         * @param player
         * @return a state index of the state; if it is not found, return -1.
         */
        int getStateIndex(const SimpleState& state, 
                          const bool visit_zero = true, 
                          const Player player = BLACK);
        /** 
         * As traversing the 'tree', find a state index of the state reached
         * by applying the moves from the initial state.
         * Note that zero-weighted moves are visited.
         *
         * @param moves to apply
         * @return state index; if it is not found, return -1.
         */
        int getStateIndex(const vector<Move>& moves);
      private:
	void seek(int offset);
	static const int HEADER_SIZE = 16;
	static const int STATE_SIZE = 16;
	static const int MOVE_SIZE = 12;
	static const int BOARD_SIZE = 41 * 4;
      };
    }
  } // namespace record
} // namespace osl
#endif // _OPENING_BOOK_H
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
