/*
Copyright (c) 2003-2008, Dennis M. Sosnoski.
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
 * Neither the name of JiBX nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package org.jibx.binding.classes;

import org.apache.bcel.Constants;
import org.apache.bcel.generic.InstructionHandle;
import org.apache.bcel.generic.ObjectType;
import org.apache.bcel.generic.Type;

/**
 * Marshalling method builder. Tracks the creation of a marshalling method,
 * including special handling of exceptions that may be generated by object
 * accesses during the marshalling process.
 *
 * @author Dennis M. Sosnoski
 */
public class MarshalBuilder extends MarshalUnmarshalBuilder
{
    //
    // Constants for code generation.
    
    private static final String MARSHALCONTEXT_CLASS =
        "org.jibx.runtime.impl.MarshallingContext";
    protected static final String MARSHAL_EXCEPTION_TEXT =
        "Error while marshalling";
    private static final Type MARSHALCONTEXT_TYPE =
        new ObjectType(MARSHALCONTEXT_CLASS);

    /**
     * Constructor. This sets up for constructing a marshalling method with
     * public access and wrapped exception handling. If the method is being
     * generated directly to the class being marshalled it's built as a virtual
     * method; otherwise, it's done as a static method.
     *
     * @param name method name to be built
     * @param cf owning class file information
     * @param mf method generation class file information
     */
    public MarshalBuilder(String name, ClassFile cf, ClassFile mf) {
        super(name, Type.VOID, new Type[] {cf.getType(), MARSHALCONTEXT_TYPE},
            mf, Constants.ACC_PUBLIC | Constants.ACC_STATIC, 0, cf.getName(),
            1, MARSHALCONTEXT_CLASS);
    }

    /**
     * Add exception handler code. The implementation of this abstract base
     * class method provides handling specific to a marshalling method.
     * 
     * @return handle for first instruction in handler
     */
    public InstructionHandle genExceptionHandler() {
        
        // instruction sequence is create new exception object, duplicate two
        //  down (below caught exception), swap (so order is new, new, caught),
        //  load message, swap again, invoke constructor, and throw exception
        initStackState(new String[] {"java.lang.Exception"});
        InstructionHandle start =
            internalAppendCreateNew(FRAMEWORK_EXCEPTION_CLASS);
        appendDUP_X1();
        appendSWAP();
        appendLoadConstant(MARSHAL_EXCEPTION_TEXT);
        appendSWAP();
        appendCallInit(FRAMEWORK_EXCEPTION_CLASS,
            EXCEPTION_CONSTRUCTOR_SIGNATURE2);
        appendThrow();
        return start;
    }
}