/*
 * Copyright (c) 2007, Dennis M. Sosnoski All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 * following conditions are met:
 * 
 * Redistributions of source code must retain the above copyright notice, this list of conditions and the following
 * disclaimer. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 * following disclaimer in the documentation and/or other materials provided with the distribution. Neither the name of
 * JiBX nor the names of its contributors may be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

package org.jibx.schema.codegen.custom;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

import org.jibx.binding.util.StringArray;
import org.jibx.runtime.IUnmarshallingContext;
import org.jibx.runtime.QName;
import org.jibx.schema.elements.FacetElement;
import org.jibx.schema.elements.SchemaBase;
import org.jibx.schema.support.LazyList;
import org.jibx.schema.validation.ValidationContext;

/**
 * Base class for all standard schema customizations that can contain other customizations.
 * 
 * @author Dennis M. Sosnoski
 */
public abstract class NestingCustomBase extends CustomBase implements TypeReplacer
{
    /** Enumeration of allowed attribute names */
    public static final StringArray s_allowedAttributes =
        new StringArray(new String[] { "enforced-facets", "ignored-facets", "type-substitutions" });
    
    //
    // Bound fields.
    
    /** List of type substitution pairs. */
    private QName[] m_substitutions;
    
    /** Mask for facets enforced at this level. */
    private long m_enforcedFacetsMask;
    
    /** Mask for facets ignored at this level. */
    private long m_ignoredFacetsMask;
    
    /** Child customizations. */
    private final LazyList m_children;
    
    //
    // Constructed fields.
    
    /** Map of type substitutions. */
    private Map m_typeSubstitutionMap;
    
    /** Mask for facets active at this level (all facets not in scope of an ignore state). */
    private long m_activeFacetsMask;
    
    /**
     * Constructor.
     * 
     * @param parent
     */
    public NestingCustomBase(NestingCustomBase parent) {
        super(parent);
        m_children = new LazyList();
    }

    /**
     * Get type substitution pairs list.
     *
     * @return substitutions
     */
    public QName[] getSubstitutions() {
        return m_substitutions;
    }
    
    /**
     * Set type substitution pairs list.
     *
     * @param subs
     */
    public void setSubstitutions(QName[] subs) {
        m_substitutions = subs;
    }
    
    /**
     * Set the list of facet elements to be enforced.
     *
     * @param facets
     * @param ictx
     */
    public void setEnforcedFacets(String[] facets, IUnmarshallingContext ictx) {
        ValidationContext vctx = (ValidationContext)ictx.getUserContext();
        long mask = 0;
        if (facets != null) {
            for (int i = 0; i < facets.length; i++) {
                String facet = facets[i];
                int index = Arrays.binarySearch(FacetElement.FACET_ELEMENT_NAMES, facet);
                if (index >= 0) {
                    mask |= SchemaBase.ELEMENT_MASKS[FacetElement.FACET_ELEMENT_INDEXES[index]];
                } else {
                    vctx.addError("'" + facet + "' is not a facet name", this);
                }
            }
        }
        m_enforcedFacetsMask = mask;
    }
    
    /**
     * Set the list of facet elements to be ignored.
     *
     * @param facets
     * @param ictx
     */
    public void setIgnoredFacets(String[] facets, IUnmarshallingContext ictx) {
        ValidationContext vctx = (ValidationContext)ictx.getUserContext();
        long mask = 0;
        if (facets != null) {
            for (int i = 0; i < facets.length; i++) {
                String facet = facets[i];
                int index = Arrays.binarySearch(FacetElement.FACET_ELEMENT_NAMES, facet);
                if (index >= 0) {
                    mask |= SchemaBase.ELEMENT_MASKS[FacetElement.FACET_ELEMENT_INDEXES[index]];
                } else {
                    vctx.addError("'" + facet + "' is not a facet name", this);
                }
            }
        }
        m_ignoredFacetsMask = mask;
    }
    
    /**
     * Get the bitmask of facet element flags to be processed.
     *
     * @return bitmask
     */
    public long getActiveFacetsMask() {
        return m_activeFacetsMask;
    }
    
    /**
     * Get child customizations.
     *
     * @return children
     */
    public LazyList getChildren() {
        return m_children;
    }
    
    /**
     * Set a type replacement.
     *
     * @param original
     * @param replace
     */
    protected void setReplacement(QName original, QName replace) {
        m_typeSubstitutionMap.put(original, replace);
    }
    
    /**
     * Get replacement type.
     *
     * @param qname
     * @return replacement type (<code>null</code> if deletion; original type, if no replacement defined)
     */
    public QName getReplacement(QName qname) {
        if (m_typeSubstitutionMap.containsKey(qname)) {
            return (QName)m_typeSubstitutionMap.get(qname);
        } else {
            return qname;
        }
    }

    /**
     * Validate and finalize customization information. This creates a new type substitution map and active facets mask,
     * or inherits unchanged values from the parent customization.
     *
     * @param vctx validation context
     * @return <code>true</code> if valid, <code>false</code> if not
     */
    public boolean validate(ValidationContext vctx) {
        NestingCustomBase parent = getParent();
        if (m_substitutions == null || m_substitutions.length == 0) {
            m_typeSubstitutionMap = parent.m_typeSubstitutionMap;
        } else if ((m_substitutions.length % 2) == 0) {
            if (parent == null) {
                m_typeSubstitutionMap = new HashMap();
            } else {
                m_typeSubstitutionMap = new HashMap(parent.m_typeSubstitutionMap);
            }
            for (int i = 0; i < m_substitutions.length; i += 2) {
                m_typeSubstitutionMap.put(m_substitutions[i], m_substitutions[i+1]);
            }
        } else {
            vctx.addError("Type substitution list must be pairs, not an odd number of names", this);
        }
        // TODO: implement the facet handling
        m_activeFacetsMask = SchemaBase.ELEMENT_MASKS[SchemaBase.ENUMERATION_TYPE];
        return true;
    }
}