package org.jboss.cache.commands.write;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.InvocationContext;
import org.jboss.cache.NodeSPI;
import org.jboss.cache.commands.Visitor;
import org.jboss.cache.notifications.event.NodeModifiedEvent;
import org.jboss.cache.optimistic.DataVersion;
import org.jboss.cache.transaction.GlobalTransaction;

import java.util.HashMap;
import java.util.Map;

/**
 * Implements functionality defined by {@link org.jboss.cache.Cache#clearData(String)}}
 *
 * @author Mircea.Markus@jboss.com
 * @since 2.2
 */
public class ClearDataCommand extends AbstractVersionedDataCommand
{
   public static final int METHOD_ID = 7;
   public static final int VERSIONED_METHOD_ID = 42;

   private static final Log log = LogFactory.getLog(ClearDataCommand.class);
   private static boolean trace = log.isTraceEnabled();

   /* parameters*/
   private HashMap originalData;

   public ClearDataCommand(GlobalTransaction gtx, Fqn fqn)
   {
      this.globalTransaction = gtx;
      this.fqn = fqn;
   }

   public ClearDataCommand()
   {
   }

   /**
    * Clears the data map in the node referenced by the specified Fqn.
    *
    * @param ctx invocation context
    * @return null
    */
   public Object perform(InvocationContext ctx)
   {
      if (trace) log.trace("perform(" + globalTransaction + ", \"" + fqn + "\")");
      NodeSPI targetNode = dataContainer.peekVersioned(fqn, dataVersion);
      if (targetNode == null)
      {
         log.warn("node " + fqn + " not found");
         return null;
      }

      Map data = targetNode.getDataDirect();
      prepareDataForRollback(data);
      notifier.notifyNodeModified(fqn, true, NodeModifiedEvent.ModificationType.REMOVE_DATA, data, ctx);
      targetNode.clearDataDirect();
      notifier.notifyNodeModified(fqn, false, NodeModifiedEvent.ModificationType.REMOVE_DATA, data, ctx);
      return null;
   }

   private void prepareDataForRollback(Map data)
   {
      if (globalTransaction != null && !data.isEmpty())
      {
         originalData = new HashMap(data);
      }
   }

   public void rollback()
   {
      if (trace) log.trace("rollback(" + globalTransaction + ", \"" + fqn + "\", " + originalData + ")");
      NodeSPI nodeSpi = dataContainer.peek(fqn, false, true);
      if (nodeSpi == null)
      {
         if (trace) log.trace("Not rolling back node clearance for node: " + fqn + " as it does not exist in the cache. " +
               "This might be the result of an NoOp clear operation");
         return;
      }
      nodeSpi.putAllDirect(originalData);
   }

   public Object acceptVisitor(InvocationContext ctx, Visitor visitor) throws Throwable
   {
      return visitor.visitClearDataCommand(ctx, this);
   }

   public int getCommandId()
   {
      return isVersioned() ? VERSIONED_METHOD_ID : METHOD_ID;
   }

   @Override
   public Object[] getParameters()
   {
      if (isVersioned())
         return new Object[]{globalTransaction, fqn, true, dataVersion};
      else
         return new Object[]{globalTransaction, fqn, true};
   }

   @Override
   public void setParameters(int commandId, Object[] args)
   {
      globalTransaction = (GlobalTransaction) args[0];
      fqn = (Fqn) args[1];
      if (isVersionedId(commandId)) dataVersion = (DataVersion) args[3];
   }

   @Override
   public boolean equals(Object o)
   {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      if (!super.equals(o)) return false;
      ClearDataCommand that = (ClearDataCommand) o;
      return !(globalTransaction != null ? !globalTransaction.equals(that.globalTransaction) : that.globalTransaction != null);
   }

   @Override
   public int hashCode()
   {
      int result = super.hashCode();
      result = 31 * result + (globalTransaction != null ? globalTransaction.hashCode() : 0);
      return result;
   }

   @Override
   protected boolean isVersionedId(int id)
   {
      return id == VERSIONED_METHOD_ID;
   }

   @Override
   public String toString()
   {
      return "RemoveDataCommand{" +
            "fqn=" + fqn +
            ", dataVersion=" + dataVersion +
            ", globalTransaction=" + globalTransaction +
            ", originalData=" + originalData +
            '}';
   }
}
