/*
 * @(#)ErrorsImplJDK13UTest.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in
 *  all copies or substantial portions of the Software.
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.mbtf.v1.engine;

import org.easymock.EasyMock;
import org.easymock.MockControl;
import net.sourceforge.groboutils.junit.v1.iftc.*;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;

import net.sourceforge.groboutils.mbtf.v1.*;

/**
 * Tests the ErrorsImpl class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2003/05/29 13:05:55 $
 * @since     March 21, 2002
 */
public class ErrorsImplJDK13UTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = ErrorsImplJDK13UTest.class;
    
    public ErrorsImplJDK13UTest( String name )
    {
        super( name );
    }

    
    //-------------------------------------------------------------------------
    // setup

    private MockControl phControl;
    private IPathHistory mockPH;

    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
        this.phControl = EasyMock.controlFor( IPathHistory.class );
        this.mockPH = (IPathHistory)this.phControl.getMock();
    }


    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testConstructor1()
    {
        ErrorsImpl ei = new ErrorsImpl();
        assertEquals(
            "# of errors not zero",
            ei.getErrors().length,
            0 );
        assertEquals(
            "# of warnings not zero",
            ei.getWarnings().length,
            0 );
        assertTrue(
            "default path halt state wrong",
            !ei.isHaltPath() );
        assertTrue(
            "default test halt state wrong",
            !ei.isHaltTests() );
    }
    
    
    public void testHalt1()
    {
        ErrorsImpl ei = new ErrorsImpl();
        String msg = "msg";
        
        try
        {
            ei.halt( msg );
            fail("did not throw exception");
        }
        catch (TestHaltRuntimeException thre)
        {
            // check exception?
        }
        
        assertErrors( ei, msg, null, null );
        assertTrue(
            "path halt not set right",
            ei.isHaltPath() );
        assertTrue(
            "test halt not set right",
            ei.isHaltTests() );
    }
    
    
    public void testHalt2()
    {
        String msg = "msg";
        ErrorsImpl ei = new ErrorsImpl();
        this.mockPH.copy();
        this.phControl.setReturnValue( this.mockPH, 1 );
        this.phControl.activate();
        
        
        ei.setCurrentPathHistory( this.mockPH );
        try
        {
            ei.halt( msg );
            fail("did not throw exception");
        }
        catch (TestHaltRuntimeException thre)
        {
            // check exception?
        }
        assertTrue(
            "path halt not set right",
            ei.isHaltPath() );
        assertTrue(
            "test halt not set right",
            ei.isHaltTests() );
        assertErrors( ei, msg, this.mockPH, null );
        
        this.phControl.verify();
    }
    
    
    public void testAddFailure1()
    {
        String msg = "msg 2";
        ErrorsImpl ei = new ErrorsImpl();
        
        ei.addFailure( msg );
        
        assertTrue(
            "path halt not set right",
            ei.isHaltPath() );
        assertTrue(
            "test halt not set right",
            !ei.isHaltTests() );
        assertErrors( ei, msg, null, null );
    }
    
    
    public void testAddFailure2()
    {
        String msg = "msg < >";
        ErrorsImpl ei = new ErrorsImpl();
        this.mockPH.copy();
        this.phControl.setReturnValue( this.mockPH, 1 );
        this.phControl.activate();
        
        
        ei.setCurrentPathHistory( this.mockPH );
        ei.addFailure( msg );
        
        assertTrue(
            "path halt not set right",
            ei.isHaltPath() );
        assertTrue(
            "test halt not set right",
            !ei.isHaltTests() );
        assertErrors( ei, msg, this.mockPH, null );
        
        this.phControl.verify();
    }
    
    
    public void testAddFailure3()
    {
        String msg = "msg 2";
        Throwable t = new Throwable("ignore");
        ErrorsImpl ei = new ErrorsImpl();
        
        ei.addFailure( msg, t );
        
        assertTrue(
            "path halt not set right",
            ei.isHaltPath() );
        assertTrue(
            "test halt not set right",
            !ei.isHaltTests() );
        assertErrors( ei, msg, null, t );
    }
    
    
    public void testAddFailure4()
    {
        String msg = "msg < >";
        Throwable t = new Throwable("ignore");
        ErrorsImpl ei = new ErrorsImpl();
        this.mockPH.copy();
        this.phControl.setReturnValue( this.mockPH, 1 );
        this.phControl.activate();
        
        
        ei.setCurrentPathHistory( this.mockPH );
        ei.addFailure( msg, t );
        
        assertTrue(
            "path halt not set right",
            ei.isHaltPath() );
        assertTrue(
            "test halt not set right",
            !ei.isHaltTests() );
        assertErrors( ei, msg, this.mockPH, t );
        
        this.phControl.verify();
    }


    
    
    public void testAddError1()
    {
        String msg = "msg 2";
        ErrorsImpl ei = new ErrorsImpl();
        
        ei.addError( msg );
        
        assertTrue(
            "path halt not set right",
            !ei.isHaltPath() );
        assertTrue(
            "test halt not set right",
            !ei.isHaltTests() );
        assertErrors( ei, msg, null, null );
    }
    
    
    public void testAddError2()
    {
        String msg = "msg < >";
        ErrorsImpl ei = new ErrorsImpl();
        this.mockPH.copy();
        this.phControl.setReturnValue( this.mockPH, 1 );
        this.phControl.activate();
        
        
        ei.setCurrentPathHistory( this.mockPH );
        ei.addError( msg );
        
        assertTrue(
            "path halt not set right",
            !ei.isHaltPath() );
        assertTrue(
            "test halt not set right",
            !ei.isHaltTests() );
        assertErrors( ei, msg, this.mockPH, null );
        
        this.phControl.verify();
    }
    
    
    public void testAddError3()
    {
        String msg = "msg 2";
        Throwable t = new Throwable("ignore");
        ErrorsImpl ei = new ErrorsImpl();
        
        ei.addError( msg, t );
        
        assertTrue(
            "path halt not set right",
            !ei.isHaltPath() );
        assertTrue(
            "test halt not set right",
            !ei.isHaltTests() );
        assertErrors( ei, msg, null, t );
    }
    
    
    public void testAddError4()
    {
        String msg = "msg < >";
        Throwable t = new Throwable("ignore");
        ErrorsImpl ei = new ErrorsImpl();
        this.mockPH.copy();
        this.phControl.setReturnValue( this.mockPH, 1 );
        this.phControl.activate();
        
        
        ei.setCurrentPathHistory( this.mockPH );
        ei.addError( msg, t );
        
        assertTrue(
            "path halt not set right",
            !ei.isHaltPath() );
        assertTrue(
            "test halt not set right",
            !ei.isHaltTests() );
        assertErrors( ei, msg, this.mockPH, t );
        
        this.phControl.verify();
    }
    
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    protected void assertErrors( ErrorsImpl ei, String msg, IPathHistory hist,
            Throwable t )
    {
        IError e[] = ei.getErrors();
        assertErrorType( e, msg, hist, t );
    }
    
    
    protected void assertErrorType( IError[] e, String msg,
            IPathHistory hist, Throwable t )
    {
        assertNotNull(
            "error array is null",
            e );
        assertEquals(
            "did not add error correctly",
            e.length,
            1 );
        assertNotNull(
            "error contents are null",
            e[0] );
        assertEquals(
            "path history is not correct",
            e[0].getPathHistory(),
            hist );
        assertEquals(
            "throwable is not correct",
            e[0].getThrowable(),
            t );
        assertEquals(
            "message not right.",
            e[0].getMessage(),
            msg );
    }
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        InterfaceTestSuite suite = IErrorsUTestI.suite();
        
        // Test the implementation's interface conformity.
        suite.addFactory( new CxFactory( "A" ) {
            public Object createImplObject() {
                return new ErrorsImpl();
            }
        } );
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        super.tearDown();
    }
}
