/*
 * @(#)AnalysisMetaDataIOUTest.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.datastore;

import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import net.sourceforge.groboutils.autodoc.v1.AutoDoc;
import net.sourceforge.groboutils.codecoverage.v2.CCCreatorUtil;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisMetaData;


/**
 * Tests the AnalysisMetaDataIO class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:28 $
 * @since     December 28, 2002
 */
public class AnalysisMetaDataIOUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = AnalysisMetaDataIOUTest.class;
    private static final AutoDoc DOC = new AutoDoc( THIS_CLASS );
    
    public AnalysisMetaDataIOUTest( String name )
    {
        super( name );
    }


    //-------------------------------------------------------------------------
    // Tests
    
    public void testWriteAnalysisMetaData1() throws Exception
    {
        AnalysisMetaDataIO amdio = new AnalysisMetaDataIO();
        StringWriter sw = new StringWriter();
        IAnalysisMetaData amd = createIAnalysisMetaData(
            "c", "n", (byte)-1 );
        
        amdio.writeAnalysisMetaData( amd, sw );
        
        String res = sw.toString();
        DOC.getLog().info( "Wrote meta-data ["+res+"]" );
        assertEquals(
            "Incorrect result format.",
            "1:c,1:n,-1,",
            res );
    }
    
    public void testWriteAnalysisMetaData2() throws Exception
    {
        AnalysisMetaDataIO amdio = new AnalysisMetaDataIO();
        StringWriter sw = new StringWriter();
        IAnalysisMetaData amd = createIAnalysisMetaData(
            "123", "12345", (byte)100 );
        
        amdio.writeAnalysisMetaData( amd, sw );
        
        String res = sw.toString();
        DOC.getLog().info( "Wrote meta-data ["+res+"]" );
        assertEquals(
            "Incorrect result format.",
            "3:123,5:12345,100,",
            res );
    }
    
    
    public void testReadAnalysisMetaData1() throws Exception
    {
        AnalysisMetaDataIO amdio = new AnalysisMetaDataIO();
        StringReader sr = new StringReader( "1:c,1:n,-1," );
        IAnalysisMetaData amd = amdio.readAnalysisMetaData( sr );
        assertNotNull(
            "Returned null data.",
            amd );
        assertEquals(
            "Covered formatted text incorrect.",
            "c",
            amd.getCoveredFormattedText() );
        assertEquals(
            "Not covered formatted text incorrect.",
            "n",
            amd.getNotCoveredFormattedText() );
        assertEquals(
            "instruction weight incorrect.",
            (byte)-1,
            amd.getInstructionWeight() );
    }
    
    
    public void testReadAnalysisMetaData2() throws Exception
    {
        AnalysisMetaDataIO amdio = new AnalysisMetaDataIO();
        StringReader sr = new StringReader( "3:123,5:12345,100," );
        IAnalysisMetaData amd = amdio.readAnalysisMetaData( sr );
        assertNotNull(
            "Returned null data.",
            amd );
        assertEquals(
            "Covered formatted text incorrect.",
            "123",
            amd.getCoveredFormattedText() );
        assertEquals(
            "Not covered formatted text incorrect.",
            "12345",
            amd.getNotCoveredFormattedText() );
        assertEquals(
            "instruction weight incorrect.",
            (byte)100,
            amd.getInstructionWeight() );
    }
    
    
    public void testReadAnalysisMetaData3() throws Exception
    {
        AnalysisMetaDataIO amdio = new AnalysisMetaDataIO();
        StringReader sr = new StringReader( "" );
        try
        {
            amdio.readAnalysisMetaData( sr );
            fail( "Did not throw IOException." );
        }
        catch (IOException ioe)
        {
            // verify ioe
        }
    }
    
    
    public void testReadAnalysisMetaData4() throws Exception
    {
        AnalysisMetaDataIO amdio = new AnalysisMetaDataIO();
        StringReader sr = new StringReader( "1" );
        try
        {
            amdio.readAnalysisMetaData( sr );
            fail( "Did not throw IOException." );
        }
        catch (IOException ioe)
        {
            // verify ioe
        }
    }
    
    
    public void testReadAnalysisMetaData5() throws Exception
    {
        AnalysisMetaDataIO amdio = new AnalysisMetaDataIO();
        StringReader sr = new StringReader( "a:a" );
        try
        {
            amdio.readAnalysisMetaData( sr );
            fail( "Did not throw IOException." );
        }
        catch (IOException ioe)
        {
            // verify ioe
        }
    }
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    protected IAnalysisMetaData createIAnalysisMetaData( String c, String nc,
            byte w )
    {
        return CCCreatorUtil.createIAnalysisMetaData( c, nc, w );
    }
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

