/*
 * @(#)CallPairMeasure.java
 *
 * Copyright (C) 2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.codecoverage.v2.module;


import net.sourceforge.groboutils.codecoverage.v2.IAnalysisMetaData;
import net.sourceforge.groboutils.codecoverage.v2.IAnalysisModule;
import net.sourceforge.groboutils.codecoverage.v2.IMethodCode;

import org.apache.bcel.generic.Instruction;
import org.apache.bcel.generic.InstructionHandle;
import org.apache.bcel.generic.InvokeInstruction;


/**
 * Processes methods for call pair coverage analysis, where each function
 * call instruction are marked.
 * Currently, this does not support localization.
 * <P>
 * This is more accurately called "Object Code Branch Coverage", since
 * true branch coverage requires the originating source code to correctly
 * discover the branches.
 * <P>
 * This measure can be superior to line coverage due to the Java construct
 * of the <tt>?:</tt> operation.  This hides a branch inside a single
 * statement.  Also, some developers may put an <tt>if</tt> statement and
 * its one-line branch all on the same line, which will hide the branch
 * that was took.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2004/04/15 05:48:26 $
 * @since     February 17, 2003
 * @see       IAnalysisMetaData
 */
public class CallPairMeasure implements IAnalysisModule
{
    /**
     * Returns the human-readable name of the measure.
     */
    public String getMeasureName()
    {
        return "Call Pair";
    }
    
    /**
     * Returns the unit name for this particular coverage measure.
     */
    public String getMeasureUnit()
    {
        return "call";
    }
    
    
    /**
     * Returns the text format used in meta-data formatted text.  This should
     * be the mime encoding type, such as "text/plain" or "text/html".
     */
    public String getMimeEncoding()
    {
        return "text/plain";
    }
    
    
    /**
     * Perform the analysis on the method.
     */
    public void analyze( IMethodCode method )
    {
        BytecodeLineUtil blu = new BytecodeLineUtil( method );
        InstructionHandle handles[] = blu.getHandles();
        
        // find the positions of the instructions in the bytecode
        for (int i = 0; i < handles.length; ++i)
        {
            InstructionHandle h = handles[i];
            Instruction instr = h.getInstruction();
            if (instr instanceof InvokeInstruction)
            {
                // need to mark the target(s) and the next instruction, as it
                // will be the "else" or "default" instruction.
                int lineNo = blu.getLineNumber( h );
                method.markInstruction( i, createMetaData( i, lineNo ) );
            }
        }
    }
    
    
    private IAnalysisMetaData createMetaData( int bytecodePos, int lineNo )
    {
        return new DefaultAnalysisMetaData(
            "Invoke for line "+lineNo,
            "Didn't cover invocation for line "+lineNo,
            (byte)0 );
    }
}

