/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     disc_lagrange_2_2d.c                                           */
/*                                                                          */
/* description:  piecewise quadratic discontinuous Lagrange elements in 2d  */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

#define N_BAS_2D N_VERTICES_2D+N_EDGES_2D

static const REAL d_bary2_2d[6][N_LAMBDA] = {{1.0, 0.0, 0.0, 0.0},
					     {0.0, 1.0, 0.0, 0.0},
					     {0.0, 0.0, 1.0, 0.0},
					     {0.0, 0.5, 0.5, 0.0},
					     {0.5, 0.0, 0.5, 0.0},
					     {0.5, 0.5, 0.0, 0.0}};

/****************************************************************************/
/*  basisfunction at vertex 0                                               */
/****************************************************************************/

static REAL d_phi2v0_2d(const REAL lambda[N_LAMBDA])
{
  return(lambda[0]*(2.0*lambda[0] - 1.0));
}

static const REAL *d_grd_phi2v0_2d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[0] = 4.0*lambda[0] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2v0_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{4, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0}};

  return(D2);
}

/****************************************************************************/
/*  basisfunction at vertex 1                                               */
/****************************************************************************/

static REAL d_phi2v1_2d(const REAL lambda[N_LAMBDA])
{
  return(lambda[1]*(2.0*lambda[1] - 1.0));
}

static const REAL *d_grd_phi2v1_2d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[1] = 4.0*lambda[1] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2v1_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 4, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0}};

  return(D2);
}

/****************************************************************************/
/*  basisfunction at vertex 2                                               */
/****************************************************************************/

static REAL d_phi2v2_2d(const REAL lambda[N_LAMBDA])
{
  return(lambda[2]*(2.0*lambda[2] - 1.0));
}

static const REAL *d_grd_phi2v2_2d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[2] = 4.0*lambda[2] - 1.0;
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2v2_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 4, 0},
					      {0, 0, 0, 0}};

  return(D2);
}

/****************************************************************************/
/*  basisfunction at edge 0                                                 */
/****************************************************************************/

static REAL d_phi2e0_2d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[1]*lambda[2]);
}

static const REAL *d_grd_phi2e0_2d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[1] = 4.0*lambda[2];
  grd[2] = 4.0*lambda[1];
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2e0_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 0, 0},
					      {0, 0, 4, 0},
					      {0, 4, 0, 0},
					      {0, 0, 0, 0}};
  return(D2);
}

/****************************************************************************/
/*  basisfunction at edge 1                                                 */
/****************************************************************************/

static REAL d_phi2e1_2d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[0]*lambda[2]);
}

static const REAL *d_grd_phi2e1_2d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[0] = 4.0*lambda[2];
  grd[2] = 4.0*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2e1_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 0, 4, 0},
					      {0, 0, 0, 0},
					      {4, 0, 0, 0},
					      {0, 0, 0, 0}};
  return(D2);
}

/****************************************************************************/
/*  basisfunction at edge 2                                                 */
/****************************************************************************/

static REAL d_phi2e2_2d(const REAL lambda[N_LAMBDA])
{
  return(4.0*lambda[0]*lambda[1]);
}

static const REAL *d_grd_phi2e2_2d(const REAL lambda[N_LAMBDA])
{
  static REAL  grd[N_LAMBDA] = {0};

  grd[0] = 4.0*lambda[1];
  grd[1] = 4.0*lambda[0];
  return((const REAL *) grd);
}

static const REAL (*d_D2_phi2e2_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA] = {{0, 4, 0, 0},
					      {4, 0, 0, 0},
					      {0, 0, 0, 0},
					      {0, 0, 0, 0}};
  return(D2);
}

/****************************************************************************/
/*  functions for combining basisfunctions with coefficients                */
/****************************************************************************/

static const DOF *d_get_dof_indices2_2d(const EL *el, const DOF_ADMIN *admin,
					DOF *idof)
{
  static DOF  index_vec[N_BAS_2D];
  int         i, n0, node;
  DOF         *rvec = idof ? idof : index_vec;
  DOF         **dof = el->dof;

  n0 = admin->n0_dof[CENTER];
  node = admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_2D; i++)
    rvec[i] = dof[node][n0+i];

  return((const DOF *) rvec);
}

static const S_CHAR *d_get_bound2_2d(const EL_INFO *el_info, S_CHAR *bound)
{
  FUNCNAME("d_get_bound2_2d");
  static S_CHAR  bound_vec[N_BAS_2D];
  S_CHAR         *rvec = bound ? bound : bound_vec;
  int            i;

  TEST_FLAG(FILL_BOUND, el_info);

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = el_info->vertex_bound[i];
  for (i = 0; i < N_EDGES_2D; i++)
    rvec[N_VERTICES_2D+i] = el_info->edge_bound[i];

  return((const S_CHAR *) rvec);
}

static const int *d_get_int_vec2_2d(const EL *el, const DOF_INT_VEC *vec,
				    int *ivec)
{
  FUNCNAME("d_get_int_vec2_2d");
  int           i, n0, node;
  static int    local_vec[N_BAS_2D];
  int           *v = nil, *rvec = ivec ? ivec : local_vec;
  DOF           **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_2D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const int *) rvec);
}

static const REAL *d_get_real_vec2_2d(const EL *el, const DOF_REAL_VEC *vec,
				      REAL *Rvec)
{
  FUNCNAME("d_get_real_vec2_2d");
  int            i, n0, node;
  static REAL    local_vec[N_BAS_2D];
  REAL           *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_2D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const REAL *) rvec);
}

static const REAL_D *d_get_real_d_vec2_2d(const EL *el, 
					  const DOF_REAL_D_VEC *vec,
					  REAL_D *Rvec)
{
  FUNCNAME("d_get_real_d_vec2_2d");
  int            i, k, n0, node;
  static REAL_D  local_vec[N_BAS_2D];
  REAL_D         *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_2D; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[i][k] = v[dof[node][n0+i]][k];

  return((const REAL_D *) rvec);
}

static const U_CHAR *d_get_uchar_vec2_2d(const EL *el, const DOF_UCHAR_VEC *vec,
					 U_CHAR *uvec)
{
  FUNCNAME("d_get_uchar_vec2_2d");
  int            i, n0, node;
  static U_CHAR  local_vec[N_BAS_2D];
  U_CHAR         *v = nil, *rvec = uvec ? uvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_2D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const U_CHAR *) rvec);
}

static const S_CHAR *d_get_schar_vec2_2d(const EL *el, const DOF_SCHAR_VEC *vec,
					 S_CHAR *svec)
{
  FUNCNAME("d_get_schar_vec2_2d");
  int            i, n0, node;
  static S_CHAR  local_vec[N_BAS_2D];
  S_CHAR         *v = nil, *rvec = svec ? svec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);

  n0 = vec->fe_space->admin->n0_dof[CENTER];
  node = vec->fe_space->admin->mesh->node[CENTER];

  for (i = 0; i < N_BAS_2D; i++)
    rvec[i] = v[dof[node][n0+i]];

  return((const S_CHAR *) rvec);
}

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of scalar functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL(d_,2,2,6)

/*--------------------------------------------------------------------*/
/*--- function for local interpolaton of vector functions          ---*/
/*--------------------------------------------------------------------*/

GENERATE_INTERPOL_D(d_,2,2,6)


/****************************************************************************/
/*  functions for interpolation/ restriction during refinement/coarsening   */
/****************************************************************************/

static void d_real_refine_inter2_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("d_real_refine_inter2_2d");
  EL        *el, *child;
  REAL      *vec = nil;
  int       node, n0, i;
  DOF       dofc;
  const DOF *pdof;
  const DOF       *(*get_dof_indices)(const EL *, const DOF_ADMIN *, DOF *);
  const DOF_ADMIN *admin;

  if (n < 1) return;

  GET_DOF_VEC(vec, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  get_dof_indices = drv->fe_space->bas_fcts->get_dof_indices;
  GET_STRUCT(admin,drv->fe_space);

  for (i = 0; i < n; i++)
  {
    el = list[i].el_info.el;
    
    pdof = get_dof_indices(el, admin, nil);
    
    node = drv->fe_space->mesh->node[CENTER];
    n0 = admin->n0_dof[CENTER];

    /**** child[0] ****/
    child = el->child[0];

    /* vertex 0 */
    dofc = child->dof[node][n0];
    vec[dofc] = vec[pdof[2]];

    /* vertex 1 */
    dofc = child->dof[node][n0+1];
    vec[dofc] = vec[pdof[0]];

    /* vertex 2 */
    dofc = child->dof[node][n0+2];
    vec[dofc] = vec[pdof[5]];
    
    /* side 0 */
    dofc = child->dof[node][n0+3];
    vec[dofc] = 0.375*vec[pdof[0]] - 0.125*vec[pdof[1]]  + 0.75*vec[pdof[5]];
    
    /* side 1 */
    dofc = child->dof[node][n0+4];
    vec[dofc] = -0.125*(vec[pdof[0]] + vec[pdof[1]]) + 0.25*vec[pdof[5]]
      + 0.5*(vec[pdof[3]] + vec[pdof[4]]);
    
    /* side 2 */
    dofc = child->dof[node][n0+5];
    vec[dofc] = vec[pdof[4]];
    
    /**** child[1] ****/
    child = el->child[1];

    /* vertex 0 */
    dofc = child->dof[node][n0];
    vec[dofc] = vec[pdof[1]];

    /* vertex 1 */
    dofc = child->dof[node][n0+1];
    vec[dofc] = vec[pdof[2]];

    /* vertex 2 */
    dofc = child->dof[node][n0+2];
    vec[dofc] = vec[pdof[5]];
    
    /* side 0 */
    dofc = child->dof[node][n0+3];
    vec[dofc] = -0.125*(vec[pdof[0]] + vec[pdof[1]]) + 0.25*vec[pdof[5]]
      + 0.5*(vec[pdof[3]] + vec[pdof[4]]);
    
    /* side 1 */
    dofc = child->dof[node][n0+4];
    vec[dofc] = 0.375*vec[pdof[1]] - 0.125*vec[pdof[0]] + 0.75*vec[pdof[5]];
    
    /* side 2 */
    dofc = child->dof[node][n0+5];
    vec[dofc] = vec[pdof[3]];
  }    

  return;
}

static void d_real_coarse_inter2_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("d_real_coarse_inter2_2d");
  EL        *el, **child;
  REAL      *vec = nil, avg;
  int       i, node, n0;
  const DOF_ADMIN *admin;
  MESH      *mesh = nil;

  if (n < 1) return;

  GET_DOF_VEC(vec, drv);
  if (!drv->fe_space)
  {
    ERROR("no fe_space in dof_real_vec %s\n", NAME(drv));
    return;
  }
  else if (!drv->fe_space->bas_fcts)
  {
    ERROR("no basis functions in fe_space %s\n", NAME(drv->fe_space));
    return;
  }
  GET_STRUCT(admin,drv->fe_space);
  GET_STRUCT(mesh,drv->fe_space);

  node = mesh->node[CENTER];        
  n0 = admin->n0_dof[CENTER];

  for (i = 0; i < n; i++)
  {
    el = list[i].el_info.el;
    child = el->child;

    vec[el->dof[node][n0+0]] = vec[child[0]->dof[node][n0+1]];
    vec[el->dof[node][n0+1]] = vec[child[1]->dof[node][n0+0]];
    avg = 0.5*(vec[child[0]->dof[node][n0+0]]+vec[child[1]->dof[node][n0+1]]);
    vec[el->dof[node][n0+2]] = avg;
    vec[el->dof[node][n0+3]] = vec[child[1]->dof[node][n0+5]];
    vec[el->dof[node][n0+4]] = vec[child[0]->dof[node][n0+5]];
    avg = 0.5*(vec[child[0]->dof[node][n0+2]]+vec[child[1]->dof[node][n0+2]]);
    vec[el->dof[node][n0+5]] = avg;
  }

  return;
}


static BAS_FCT      *d_phi2_2d[6]     = {d_phi2v0_2d, d_phi2v1_2d, d_phi2v2_2d,
					 d_phi2e0_2d, d_phi2e1_2d, d_phi2e2_2d};
static GRD_BAS_FCT  *d_grd_phi2_2d[6] = {d_grd_phi2v0_2d, d_grd_phi2v1_2d,
					 d_grd_phi2v2_2d, d_grd_phi2e0_2d,
					 d_grd_phi2e1_2d, d_grd_phi2e2_2d};
static D2_BAS_FCT   *d_D2_phi2_2d[6]  = {d_D2_phi2v0_2d, d_D2_phi2v1_2d,
					 d_D2_phi2v2_2d, d_D2_phi2e0_2d,
					 d_D2_phi2e1_2d, d_D2_phi2e2_2d};

static BAS_FCTS  disc_lagrange2_2d = {"disc_lagrange2_2d", 2, N_BAS_2D, 2,
				      {0, N_BAS_2D, 0, 0}, 
				      nil,
				      d_phi2_2d, d_grd_phi2_2d, d_D2_phi2_2d, 
				      d_get_dof_indices2_2d, 
				      d_get_bound2_2d,
				      d_interpol2_2d,
				      d_interpol_d2_2d,
				      d_get_int_vec2_2d,
				      d_get_real_vec2_2d,
				      d_get_real_d_vec2_2d,
				      d_get_uchar_vec2_2d,
				      d_get_schar_vec2_2d,
				      d_real_refine_inter2_2d,
				      d_real_coarse_inter2_2d,
				      nil,
				      nil,
				      nil,
				      nil,
				      d_bary2_2d, };

#undef N_BAS_2D
