;;;;; -*-coding: raw-text;mode: emacs-lisp;-*-
;;;;; Copyright (C) 1991-2002  Lysator Academic Computer Association.
;;;;;
;;;;; This file is part of the LysKOM Emacs LISP client.
;;;;; 
;;;;; LysKOM is free software; you can redistribute it and/or modify it
;;;;; under the terms of the GNU General Public License as published by 
;;;;; the Free Software Foundation; either version 2, or (at your option) 
;;;;; any later version.
;;;;; 
;;;;; LysKOM is distributed in the hope that it will be useful, but WITHOUT
;;;;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;;;;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;;;;; for more details.
;;;;; 
;;;;; You should have received a copy of the GNU General Public License
;;;;; along with LysKOM; see the file COPYING.  If not, write to
;;;;; Lysator, c/o ISY, Linkoping University, S-581 83 Linkoping, SWEDEN,
;;;;; or the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
;;;;; MA 02139, USA.
;;;;;
;;;;; Please mail bug reports to bug-lyskom@lysator.liu.se. 
;;;;;
;;;; ================================================================
;;;; ================================================================
;;;;
;;;; File: vars.el
;;;;
;;;; This file contains almost all the variables used in lyskom.
;;;;

;;;; Note: put "**" at the front of docstrings that are ready
;;;; for use in DocBook autogeneration.
;;;;
;;;; Rules for documentation that can be translated to DocBook
;;;; is in interndoc/vardoc


(defvar lyskom-dummy-variable-to-fool-the-byte-compiler)

(defvar lyskom-mule-compiled
  (eval-when-compile (and (fboundp 'multibyte-string-p)
                          (multibyte-string-p "")))
  "Non-nil if the client was compiled with multibyte characters enabled")

(provide 'lyskom)

(require 'lyskom-defvar "defvar")

(defconst lyskom-global-variable-types
  '((boolean (read . lyskom-flag-read-boolean)
             (write . lyskom-flag-write-boolean))
    (integer (read . lyskom-flag-read-integer)
             (write . lyskom-flag-write-integer))
    (symbol-list (read . lyskom-flag-read-symbol-list)
                 (write . lyskom-flag-write-symbol-list))
    (t (read . lyskom-flag-read-from-string)
       (write . prin1-to-string))))


(defun lyskom-protect-variable (sym) 
  "Protect SYM from being killed when deleting local variables."
  (put sym 'permanent-local t)
  (lyskom-local-variable sym)
  (add-to-list 'lyskom-protected-variables sym))

(defun lyskom-local-variable (sym)
  "Define SYM as a buffer-local variable"
  (add-to-list 'lyskom-local-variables sym))

(defun lyskom-setup-local-variables ()
  "Set up all globally defined local variables in the current buffer."
  (mapc 'make-local-variable lyskom-local-variables)
  (mapc 'lyskom-make-local-hook lyskom-local-hooks))

(defvar lyskom-is-loaded nil
  "Non-nil when lyskom has been loaded.")

(defvar lyskom-have-one-login nil
  "Non-nil after the first login")


(def-kom-var kom-dont-read-saved-variables '(kom-dont-read-saved-variables
                                             lyskom-login-hook)
  "**Determines which LysKOM variables to not store in the server.

For the most part you do not have to modify this variable. Any variables
set before LysKOM is loaded will not be read from the server.
This variable should `nil', `t' or a list of symbols. 

The value `nil' means read all variables from the server. The value
`t' means read no variables from the server. When the value is a list
of symbols, those variables will not be read from the server.

Any other values are reserved for future use.")

(defmacro lyskom-maybe-setq (var value)
  "This is a wrapper around setq that does nothing
if the variable is in kom-dont-read-saved-variables."
  `(cond ((eq kom-dont-read-saved-variables t) nil)
         ((memq ,var kom-dont-read-saved-variables) nil)
         (t (set ,var ,value))))

(defmacro lyskom-maybe-setq-default (var value)
  "This is a wrapper around setq-default that does nothing
if the variable is in kom-dont-read-saved-variables."
  `(cond ((eq kom-dont-read-saved-variables t) nil)
         ((memq ,var kom-dont-read-saved-variables) nil)
         (t (set-default ',var ,value))))

;;;; ================================================================
;;;;                     Variables and constants.


;;; User flags

(def-kom-var kom-auto-confirm-new-conferences nil
  "**Determines behavior of `kom-list-new-conferences' and `kom-list-new-persons'.

When this variable is set to `t', `kom-list-new-conferences' and
`kom-list-new-persons' will automatically mark all listed conferences
and persons as known. When set to `nil', these commands will ask for
confirmation before marking the listed conferences and persons as known.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-dont-complain-about-missing-presentation nil
  "**Controls whether the client complains when you don't have
a presentation. When set to non-nil, the client will never
complain.

This variable should not be documented. Anybody asocial enough
to want this feature ought to have to work to find it."
  server)

(def-kom-var kom-show-sync-messages nil
  "**Controls display of database saving message.

When this variable is set to `t', the client will show a message in
the echo area when the LysKOM server signals that it is saving the
database. When set to `nil', no message is shown (the mode line is
changed regardless of the setting of this variable).

Values other than `t' and `nil' are reserved for future use."
  server)


(def-kom-var kom-text-links
  '((t ("\\<bug[ \t\n]+\\([0-9]+\\)\\>"
	"http://bugzilla.lysator.liu.se/show_bug.cgi?id=\\1"
	0 t)))
  "**Defines patterns for automatic links in the text.

This variable defines patterns to look for in text and convert to
links to URLs. The value of this variable must be an alist, where the
key (car) is a conference number and the value (cdr) is a list of link
specifications for that conference. The special value `t' for the key
indicates a list of patterns to apply in all conferences. Key values
other than positive integers and `t' are reserved for future use.

Each link specification is in turn a list \(`PATTERN' `REPLACEMENT'
`HIGHLIGHT' `IGNORE-CASE'), where `PATTERN' is a regular expression to
look for in the text, `REPLACEMENT' is used to generate a URL from the
text matching `PATTERN', `HIGHLIGHT' is the pattern group to highlight
as a URL and `IGNORE-CASE' specifies whether matching is
case-sensitive or not.

Each text is examined for each `PATTERN'. If a match is found, a URL
is created from `REPLACEMENT'. All characters except backslash are
copied verbatim. The backslash character starts one of the following
sequences:

    Sequence    Meaning
    -----------------------------------------------------------------
    `\\&'          Substitute the matched text

    `\\N'          Substitute match for text matching the Nth (...)
                group in `PATTERN'

    `\\\\'          Insert one backslash.
    -----------------------------------------------------------------

The `HIGHLIGHT' parameter specifies which (...) group in `PATTERN'
to highlight as an URL. Use 0 to highlight the entire match.

Finally, if `IGNORE-CASE' is `t', then ignore character case while
looking for `PATTERN'. Values other than `t' and `nil' are reserved
for future use.

When a text has more than one recipient, links will be generated for
all recipients. The order in which the recipients are examined and
the priority among links that match the same text is undefined."
  server)


(def-kom-var lyskom-default-conference-strategy
  '((kom-start-anew (what-is-your-name
                     (default . (lyskom-default-conference-empty))))

    (kom-delete-conf (what-conf-to-delete
                      (default . (lyskom-default-conference-empty))))

    (kom-sub-self (leave-what-conf
                   (filter . (lyskom-default-conference-not-self))))

    (kom-change-priority (what-to-change-pres-you
                          (default . (lyskom-default-conference-at-point
                                      lyskom-default-conference-self))))

    (kom-change-presentation (what-to-change-pres-you
                              (default . (lyskom-default-conference-at-point
                                          lyskom-default-conference-self))))

    (kom-change-conf-motd (who-to-put-motd-for
                           (default . (lyskom-default-conference-at-point
                                       lyskom-default-conference-self))))

    (kom-set-presentation (what-to-set-pres-you
                           (default . (lyskom-default-conference-at-point
                                       lyskom-default-conference-self))))

    (kom-set-motd-text (what-to-set-motd-you
                        (default . (lyskom-default-conference-at-point
                                    lyskom-default-conference-self))))

    (kom-remove-presentation (who-to-remove-pres-for
                              (default . (lyskom-default-conference-at-point
                                          lyskom-default-conference-self))))

    (kom-unset-conf-motd (who-to-remove-motd-for
                          (default . (lyskom-default-conference-at-point
                                      lyskom-default-conference-self))))

    (kom-go-to-conf (go-to-conf-p
                     (default . (lyskom-default-conference-at-point
                                 lyskom-default-conference-self))
                     (filter . (lyskom-default-conference-not-current))))

    (kom-list-created-conferences (list-confs-created-by
                                   (default . (lyskom-default-conference-at-point
                                               lyskom-default-conference-self))))

    (kom-change-name (name-to-be-changed
                      (default . (lyskom-default-conference-at-point
                                  lyskom-default-conference-self))))

    (kom-change-parenthesis (name-to-be-changed
                             (default . (lyskom-default-conference-at-point
                                         lyskom-default-conference-self))))

    (kom-change-password (whos-passwd
                          (default . (lyskom-default-conference-at-point
                                      lyskom-default-conference-self))))

    (kom-redirect-comments (redirect-for-whom
                            (default . (lyskom-default-conference-at-point
                                        lyskom-default-conference-self)))
                           (redirect-to-which-conf))

    (kom-filter-author (filter-author
                        (default . (lyskom-default-conference-at-point
				    lyskom-default-conference-author-of-text-at-point
                                    lyskom-default-conference-last-author))
                        (filter . (lyskom-default-conference-not-self)))
                       (filter-in-conf
                        (filter . (lyskom-default-conference-not-self))))


    (kom-review-by-to (review-by-whom
                       (default . (lyskom-default-conference-at-point
                                   lyskom-default-conference-empty)))
                      (review-to-conf))

    (kom-review-roots-by-to (review-by-whom
			     (default . (lyskom-default-conference-at-point
					 lyskom-default-conference-empty)))
			    (review-to-conf))

    (kom-review-first (review-by-whom
                       (default . (lyskom-default-conference-at-point
                                   lyskom-default-conference-empty)))
                      (review-to-conf))

    (kom-review-first-roots (review-by-whom
			     (default . (lyskom-default-conference-at-point
					 lyskom-default-conference-empty)))
			    (review-to-conf))

    (kom-review-all (review-by-whom
                     (default . (lyskom-default-conference-at-point
                                 lyskom-default-conference-empty)))
                    (review-to-conf))

    (kom-review-all-roots (review-by-whom
			   (default . (lyskom-default-conference-at-point
				       lyskom-default-conference-empty)))
			  (review-to-conf))

    (kom-unread-by-to (unread-by-whom
                       (default . (lyskom-default-conference-at-point
                                   lyskom-default-conference-empty)))
                      (unread-to-conf))

    (kom-unread-roots-by-to (unread-by-whom
			     (default . (lyskom-default-conference-at-point
					 lyskom-default-conference-empty)))
			    (unread-to-conf))

    (kom-unread-first (unread-by-whom
                      (default . (lyskom-default-conference-at-point
                                  lyskom-default-conference-empty)))
                      (unread-to-conf))

    (kom-unread-first-roots (unread-by-whom
			     (default . (lyskom-default-conference-at-point
					 lyskom-default-conference-empty)))
			    (unread-to-conf))

    (kom-unread-all (unread-by-whom 
                     (default . (lyskom-default-conference-at-point
                                 lyskom-default-conference-empty)))
                    (unread-to-conf))

    (kom-unread-all-roots (unread-by-whom 
			   (default . (lyskom-default-conference-at-point
				       lyskom-default-conference-empty)))
			  (unread-to-conf))

    (kom-set-permitted-submitters (conf-to-set-permitted-submitters-q)
                                  (new-permitted-submitters-q
                                   (default . (lyskom-default-conference-empty))))

    (lyskom (what-is-your-name 
             (default . (lyskom-default-conference-empty))))


    (kom-add-recipient (who-to-add-q 
                        (default . ((lyskom-default-conference-saved last-added-rcpt)))
                        (save last-added-rcpt)))

    (kom-add-copy (who-to-add-copy-q 
                   (default . ((lyskom-default-conference-saved last-added-cc)))
                   (save last-added-cc)))

    (kom-add-bcc (who-to-add-bcc-q 
                  (default . ((lyskom-default-conference-saved last-added-bcc)))
                  (save last-added-bcc)))

    (kom-sub-recipient (who-to-sub-q
                        (default . (lyskom-default-conference-at-point
                                    (lyskom-default-conference-saved last-sub-recipient)
                                    lyskom-default-conference-current
                                    lyskom-default-conference-restriction))
                        (save last-sub-recipient)))

    (kom-move-text (who-to-move-from-q
                    (default . (lyskom-default-conference-at-point
                                (lyskom-default-conference-saved last-sub-recipient)
                                lyskom-default-conference-current
                                lyskom-default-conference-restriction))
                    (save last-sub-recipient))
                   (who-to-move-to-q
                    (default . ((lyskom-default-conference-saved last-added-rcpt)
                                lyskom-default-conference-current))
                    (save last-added-rcpt)))

    (kom-move-text-tree (who-to-move-from-q
                         (default . (lyskom-default-conference-at-point
                                     (lyskom-default-conference-saved last-sub-recipient)
                                     lyskom-default-conference-current
                                     lyskom-default-conference-restriction))
                         (save last-sub-recipient))
                        (who-to-move-to-or-sub-q
                         (default . ((lyskom-default-conference-saved last-added-rcpt)
                                     lyskom-default-conference-current))
                         (save last-added-rcpt))
                        (who-to-move-to-q
                         (default . ((lyskom-default-conference-saved last-added-rcpt)
                                     lyskom-default-conference-current))
                         (save last-added-rcpt))
                        (who-to-add-q
                         (default . ((lyskom-default-conference-saved last-added-rcpt)
                                     lyskom-default-conference-current))
                         (save last-added-rcpt)))

    (kom-send-message (who-to-send-message-to
                       (default . (lyskom-default-conference-for-send-message))))

    (kom-moronify (moronify-whom
		   (default . (lyskom-default-conference-at-point
			       lyskom-default-conference-author-of-text-at-point
			       lyskom-default-conference-last-author))))

    (kom-befriend (befriend-whom
		   (default . (lyskom-default-conference-at-point
			       lyskom-default-conference-author-of-text-at-point
			       lyskom-default-conference-last-author))))

    (kom-send-alarm (who-to-send-message-to
                     (default . (lyskom-default-conference-for-send-message))))

    (kom-is-person-member-of-conference (pers-to-check-mship-for)
                                        (conf-to-check-mship-of))



    ;; Defaults and commands that use the defaults

    (kom-remote-autoreply (remote-control-who))
    (kom-remote-set-message (remote-control-who))
    (kom-remote-list-messages (remote-control-who))
    (kom-remote-erase-messages (remote-control-who))
    (kom-remote-quit (remote-control-who))
    (kom-review-presentation (presentation-for-whom))
    (kom-unread-presentation (unread-presentation-for-whom))
    (kom-add-member (who-to-add) (where-to-add))
    (kom-add-self (where-to-add-self))
    (kom-change-priority (change-priority-for-q))
    (kom-sub-member (who-to-exclude) (where-from-exclude))
    (kom-write-text (who-send-text-to))
    (kom-send-letter (who-letter-to))
    (kom-force-logoout (who-to-throw-out))
    (kom-change-supervisor (what-to-change-supervisor-for) (new-supervisor))
    (kom-who-is-on-in-conference (who-is-on-in-what-conference))
    (kom-who-is-present-in-conference (who-is-present-in-what-conference))
    (kom-add-cross-reference (which-conf-to-xref) (which-pers-to-xref))
    (kom-edit-add-cross-reference (which-conf-to-xref) (which-pers-to-xref))
    (kom-edit-insert-link (which-conf-to-link) (which-pers-to-link))
    (kom-status-conf (conf-for-status))
    (kom-status-session (session-for-status))
    (kom-force-logout (who-to-throw-out))
    (kom-status-person (pers-for-status))
    (kom-set-garb-nice (conf-to-set-garb-nice-q))
    (kom-set-super-conf (conf-to-set-super-conf-q) (new-super-conf-q))
    (kom-set-personal-label (label-what-pers) (label-what-conf))
    (kom-will-person-read-text (pers-to-check-will-read-for))
    (kom-create-aux-item (which-conf-to-add-aux-to))
    (kom-recommend-conference (recommend-which-conf))
    (kom-limit-import (limit-import-to-conf))
    (kom-edit-add-recipient (added-recipient))
    (kom-edit-add-bcc (added-blank-carbon-copy))
    (kom-edit-add-copy (added-carbon-copy))
    (kom-edit-move-text (who-to-move-to-q))
    (kom-add-faq (conf-to-add-faq))
    (kom-del-faq (conf-to-del-faq))
    (kom-review-faq (view-which-faq))
    (kom-unread-faq (unread-which-faq))
    (kom-change-conf-faq (what-to-change-faq-you))
    (kom-list-faqs (conf-to-list-faqs))
    (kom-filter-subject (filter-in-conf))
    (kom-filter-recipient (filter-recipient))
    (kom-filter-text (filter-in-conf))
    (kom-change-conf-type (what-conf-to-change))
    (kom-change-privileges (what-pers-privs-to-change))
    (kom-change-message-flag (set-message-flag-for-conf))

    (t (t (default . (lyskom-default-conference-at-point
                      lyskom-default-conference-current))))
    )

  "Strategy for the initial value for conference defaults.

Each element in this list identifies an Emacs och LysKOM command, and
defines the initial values for queries it makes for conference names.

Each element has the following format:

    \(COMMAND . \(\(PROMPT-1 . \(\(default . \(INITIAL-VALUES\\)\\)
                             \(filter  . \(EXCLUDED-CONFS\)\)
                             \(save    . \(SAVE-GROUPS\)\)\)\)
                \(PROMPT-2 . \(\(default . \(INITIAL-VALUES\)\)
                             \(filter  . \(EXCLUDED-CONFS\)\)
                             \(save    . \(SAVE-GROUPS\)\)\)\)
                ...\)\)

COMMAND is the name of a LysKOM or Emacs command. PROMPT-1 and
PROMPT-2 \(there can be any number of prompts liste in a single
command\) are symbols representing the prompt. These are defined
in lyskom-messages in swedish-strings.el and english-strings.el. 

INITIAL-VALUES are functions to call to generate a list of possible
initial values for the prompt. These functions must return lists of
conference numbers. EXCLUDED-CONFS are functions that are called to
eliminate possible initial values. These must take a single argument,
an uconf-stat, and should return the uconf-stat itself if it is a
valid candidate for the initial value, or nil if it is not. Finally,
SAVE-GROUPS are symbols under which the value the user inputs will be
saved.

As a special case, a list in INITIAL-VALUES is interpreted as a
function with arguments. The first element in the list is applied to
the remaining elements.

If `t' is used for COMMAND, it indicates the default to use when no
COMMAND listed matches the current command. If `t' is used for PROMPT,
it indicates defaults, filters and save groups to use when no other
PROMPT matches.


Some useful functions for INITIAL-VALUES are:

lyskom-default-conference-at-point returns the conference shown where
point is. If point is on a conference name, that name will be used as
the initial value.

lyskom-default-conference-current returns the current conference.

lyskom-default-conference-self returns the logged-in person.

lyskom-default-conference-last-author returns the author of the most
recently read text.

lyskom-default-conference-author-of-text-at-point returns the author
of the text at point.

lyskom-default-conference-saved returns saved input. This must be
specified as a list in INITAL-VALUES, where the argument (second
element of the list) is the symbol under which the input was saved,
from \(SAVE-GROUPS\) in one or more commands.


Some useful functions for EXCLUDE-CONFS are:

lyskom-default-conference-not-self excludes the logged-in person.

lyskom-default-conference-not-current excludes the current conference.


The format of this variable may change in the future.
")


(defconst kom-old-farts-text-prompt-strategy
  '((kom-comment-previous (nil lyskom-get-previous-text))
    (kom-view-previous-commented-text (nil lyskom-get-previous-text))
    (kom-private-answer-previous (nil lyskom-get-previous-text))
    (t  (t   lyskom-get-last-read-text)
        (nil     lyskom-get-last-read-text)
        (0       lyskom-get-text-at-point)
        (lyskom-eq-dash   (lambda () (lyskom-get-text-above-point 1)))
        (listp   (lambda () (lyskom-tnpa-prompt (lyskom-get-last-read-text))))
        (plusp   current-prefix-arg)
        (minusp  (lambda () (lyskom-get-text-above-point (- current-prefix-arg))))))
"Put in your `kom-pick-text-no-strategy-alist' to get the 0.46 behaviour:
 * No prefix argument refers to the most recently read text.
 * The prefix argument zero refers to the text under point.
 * A positive prefix argument is interpreted as a text-no.
 * A negative prefix argument will try to find the text-no
   of the text `arg' messages above point from the current
   kom buffer.")

(defconst lyskom-old-farts-text-prompt-strategy
  kom-old-farts-text-prompt-strategy
  "Obsolete: Alias for kom-old-farts-text-prompt-strategy.")

(def-kom-var kom-pick-text-no-strategy-alist
  '(

    ;; Abstract specifications

    (read-or-written
     (t   (lambda () (lyskom-tnpa-prompt (lyskom-get-text-at-point))))
     (nil lyskom-get-text-at-point
          (lambda () (lyskom-tnpa-valid
                      (lyskom-tnpa-prompt (lyskom-get-last-written-or-read-by-me))))
          :constraint lyskom-text-written-by-me-p))
    (previous-text (nil lyskom-get-previous-text))
    (commented-text (nil lyskom-maybe-get-commented-text))
    (footnoted-text (nil lyskom-maybe-get-footnoted-text))

    ;; Specifications for actual commands

    (kom-comment-previous             :refer previous-text)
    (kom-view-previous-commented-text :refer previous-text)
    (kom-unread-previous-commented-text :refer previous-text)
    (kom-private-answer-previous      :refer previous-text)
    (kom-sub-comment                  :refer commented-text)
    (kom-sub-footnote                 :refer footnoted-text)
    (kom-write-footnote               :refer read-or-written)
    (kom-add-no-comments              :refer read-or-written)
    (kom-add-private-answer           :refer read-or-written)
    (kom-add-request-confirm          :refer read-or-written)
    (kom-move-comment		      :refer read-or-written)

    ;; Default specification

    (t (nil     lyskom-get-text-at-point)
       (0       (lambda () (lyskom-tnpa-prompt (lyskom-get-text-at-point))))
       (lyskom-eq-dash (lambda () (lyskom-get-text-above-point 1)))
       (car     (lambda () (lyskom-get-text-at-point-ancestor (/ (logb (car current-prefix-arg)) 2))))
       (plusp   (lambda () (lyskom-get-text-below-point current-prefix-arg)))
       (minusp  (lambda () (lyskom-get-text-above-point (- current-prefix-arg))))
       (t       (lambda () (lyskom-tnpa-prompt (lyskom-get-text-at-point))))
       :constraint nil
       :filter  nil
       )
    )
  "**Defines how prefix arguments are used by commands to find a text to operate on.

The format of this list is as follows:

<
\((FN-KEY . ((PFX-KEY-1 . (FN1 FN2 ... PROP1 VAL1 PROP2 VAL2 ...)) 
            (PFX-KEY-2 . (FN1 FN2 ... PROP1 VAL1 PROP2 VAL2 ...))
            GPROP1 GVAL1
            GPROP2 GVAL2
            ...))
 ...)
>

`FN-KEY' is the name of a command, `t' or a symbol that does not
correspond to any function. Keys corresponding to commands indicate
configuration for that command. The `t' key indicates fallback
condiguration that applies to all commands, unless overridden by a
command-specific configuration. Other keys can be referenced from
command-specific configurations.

`PFX-KEY' is a key that indicates a prefix. If it is an atom not bound
to a function, it indicates configuration for prefix arguments eq to
the atom \(e.g. 0, - or `nil'). The special key `t' indicates fallback
configuration, used when all prefix-specific configurationas failed to
generate valid values.

`FN' is a function or variable used to generate lists of text numbers.
If a variable name, the value of the variable is used to generate a
list of text numbers. If a function, the function is called with no
arguments, and is expected to return a list of text numbers.

`PROP' is a property name \(see below) and VAL is the corresponding
value. Properties defined at the inner level apply only to results
generated for that prefix. Properties defined at the outer level apply
irrespective of prefix arguments. Properties defined in the default
element \(with `FN-KEY' set to `t'\) always apply, unless overridden by
non-nil properties in command-specific elements.

The following properties are available:

    Key             Value
    ------------------------------------------------------------------
    `:filter'       A function that is called on all text numbers. The
                    filter function is expected to return a new or 
                    modified text number. This can be used to indicate 
                    that the command should prompt or bypass the 
                    constraint.
    
    `:constraint'  A function that is called on all text numbers. It 
                   should return non-nil for text numbers that are 
                   valid for the command.
    
    `:refer'       A symbol indicating a configuration to use. Useful 
                   when several commands share the same (complex)
                   configuration. May not be used in the default 
                   element.
    ------------------------------------------------------------------


The following filter functions may be useful:

    Name                    What it does
    ------------------------------------------------------------------
    `lyskom-tnpa-prompt'    Prompt the user for the text number. The 
                            number used as the argument will be the 
                            default. 
    
    `lyskom-tnpa-valid'     Declare that the text number is valid, 
                            regardless of any constraint functions.
    ------------------------------------------------------------------

When `lyskom-read-text-no-prefix-arg' is called, this list is searched
both for a command-specific element and for the default element. Any
`:refer' links are resolved in the command-specific element. The
filter and constraint properties are extracted. These elements are
then searched for prefix-specific elements that match the current
prefix argument. The first text number that is generated that passes
the filter and constraint functions is used as the text number
argument for the command."
)

(def-kom-var kom-url-transformation-rules 
  '(
    ("^http://[^/]*aftonbladet\\.se/.*/story/.*html?$" . "\\&.")
    )
  "**An alist specifying transformations to be applied to URLs.

Elements in this list are of the form \(`PATTERN' . `REPLACEMENT').
Before an URL is opened, it is transformed by matching the URL against
each `PATTERN' in turn, and when a match is found, generating a new
URL according to `REPLACEMENT'.

All characters in `REPLACEMENT' except \\ are copied verbatim. The
backslash character starts one of the following sequences:

    Sequence    Meaning
    ---------------------------------------------------------------
    `\\&'        Substitute the matched text

    `\\N'        Substitute match for text matching the Nth (...)
                group in `PATTERN'

    `\\\\'        Insert one backslash.
    ---------------------------------------------------------------
"
  server
  )


(def-kom-var kom-check-configuration-on-startup t
  "**When non-nil, check Emacs configuration on client startup.

Valid values for this variable are `t' and `nil'. All other values are
reserved for future extensions. When non-nil, several checks will be
performed:

* Check that multibyte character support is enabled, if available.

* Check that the character coding Emacs uses matches the coding system
  used by the server.

Warnings will be shown in the LysKOM buffer on logon.
")


(def-kom-var kom-keyboard-menu-immediate-selection nil
  "**Controls how keyboard menus work.

When displaying text-based menus (typically by pressing the `=' key
when the cursor is on an active area), there are two modes of
selection. Setting this variable to `t' means pressing the letter
corresponding to an item selects that item. When this variable is
`nil', selection has to be confirmed by pressing enter.

Values other tha `t' and `nil' are reserved for future use."
  server)


(def-kom-var kom-edit-hide-add-button nil
  "**Controls the \"add\" button in the edit buffer.

When set to `t', hide the add button shown after the headers when
editing a text. When set to `nil' (the default), show the button.

Values other than `t' and `nil' are reserved for future use."
  server
)

(def-kom-var kom-max-overlays 120
  "**Maximum number of overlays to use to highlight texts.

When a text is displayed with its own background color (see
`kom-highlight-text-body', `kom-highlight-first-line',
`kom-highlight-dashed-lines', `kom-async-highlight-text-body' and
`kom-async-highlight-dashed-lines') one or more overlays are created.
Since a large number of overlays has a serious impact on performance,
the total number of overlays in the buffer can be limited using this
variable.

When set to a positive integer, limit the number of overlays to that
number. When set to `nil', do not limit the number of overlays. Values
other than a positive integer and `nil' are reserved for future use
and may cause unpredictible results."
  server
)

(def-kom-var kom-highlight-text-body t
  "**Controls highlighting of text bodies.

If `t', overlay `kom-text-body-face' on printed text bodies in LysKOM.
When this is enabled, an overlay or extent will be created that may
override certain aspects of the underlying text's formatting. Values
other than `t' or `nil' are reserved for future use.

See `kom-max-overlays' for information on how to limit the number of
overlays."
  server
)

(def-kom-var kom-highlight-first-line t
  "**Controls highlighting of the first line of text headers.

If `t', overlay `kom-first-line-face' on the first header line of
texts. When this is enabled, an overlay or extent will be created that
may override certain aspects of the underlying text's formatting.
Values other than `t' or `nil' are reserved for future use.

See `kom-max-overlays' for information on how to limit the number of
overlays."
  server
)

(def-kom-var kom-highlight-dashed-lines t
  "**Controls highlighting of dashed lines around texts.

If `t', overlay `kom-dashed-lines-face' on the lines surrounding
texts. When this is enabled, an overlay or extent will be created that
may override certain aspects of the underlying text's formatting.
Values other than `t' or `nil' are reserved for future use.

See `kom-max-overlays' for information on how to limit the number of
overlays."
  server
)

(def-kom-var kom-async-highlight-text-body t
  "**Controls highlighting of asynchronous messages.

If `t', overlay `kom-async-text-body-face' on asynchronous messages in
LysKOM. When this is enabled, an overlay or extent will be created
that may override certain aspects of the underlying text's formatting.
Values other than `t' or `nil' are reserved for future use.

See `kom-max-overlays' for information on how to limit the number of
overlays."
  server
)

(def-kom-var kom-async-highlight-dashed-lines t
  "**Controls highlighting of dashed lines around asynchronous messages.

If `t', overlay `kom-dashed-lines-face' on the lines surrounding
messages. When this is enabled, an overlay or extent will be created
that may override certain aspects of the underlying text's formatting.
Values other than `t' or `nil' are reserved for future use.

See `kom-max-overlays' for information on how to limit the number of
overlays."
  server
)

(def-kom-var kom-extended-status-information nil
  "**Controls display of extra information when displaying LysKOM objects.

If `t', extended status information is listed for all objects in
LysKOM. If set to an association list, each element is a pair of tag
and value. The tag specifies a type of extended information. If the
corresponding value is `t', the information is shown. If the
corresponding value is `nil', the information is not shown.

Valid tags are:

    Tag                   Meaning
    ------------------------------------------------------------------
    `conf'                Show all extended information for conferences

    `pers'                Show all extended information for persons

    `server'              Show all extended information for the server

    `read-faq'            Show read FAQs

    `raw-server-stats'    Show raw server statistics

    `raw-boottime-info'   Show raw boot-time server information

    `t'                   Any information not explicitly listed
    ------------------------------------------------------------------

The information controlled by this variable is the kind of data that
most users are not interested in, and, if displayed, would make more
important information hard to find.

Currently this includes read FAQs (for letterboxes) and full
statistics and boot-time information (for the server).

Values other than those specified are reserved for future extensions."
server )

(def-kom-var kom-auto-list-faqs t
  "**Controls automatic display of FAQs.

When this variable is set to `t', list unread FAQs when entering a
conference or logging on to the server. When entering a conference,
list FAQs for that conference that have not been read. When logging
on, list FAQs for the server that have not been read.

Values other than `t' or `nil' are reserved for future extension."
  server
)

(def-kom-var kom-auto-review-faqs t
  "**Controls automatic review of FAQs.

When set to `t', automatically review unread FAQs when entering a
conference or logging on to the server. When entering a conference,
FAQs that are unread for that conference will be reviewed. When
logging on, FAQs that are unread for the server will be reviewed.

Values other than `t' or `nil' are reserved for future use."

  server
)

(def-kom-var kom-allow-incompleteness nil
  "**Allow or disallow operation on incomplete membership information.

If `nil', commands like `kom-list-news' will wait for all membership
information to be read. If this flag is set to `t', commands will not
wait for all membership information. Not waiting for complete
information allows certain commands to execute faster, particularly
during the login phase, but may result in incorrect or incomplete
answers.

Values other than `t' or `nil' are reserved for future use."
  server
)

(def-kom-var kom-bury-buffers t
  "**Controls the behaviour of `kom-next-kom' and its cousins.

If this variable is `t' the current buffer is sent to the back of the
buffer list when one of the commands `kom-next-kom',
`kom-previous-kom' or `kom-next-unread-kom' is invoked.

Values other than `t' or `nil' are reserved for future use."
  server)

(def-kom-var kom-write-texts-in-window nil
  "**Determines where to edit texts.

The value must be one of `nil', `other', `new-frame', `other-frame', a
string or a buffer.

When set to `nil', edit texts in the same window as the LysKOM buffer.
When set to `other', edit in another window, creating it if necessary.
When set to `other-frame', edit in another frame, if there is one
\(otherwise edit in the same window as the LysKOM buffer). When set to
`new-frame', create a new frame for editing. This frame will be
removed when editing is finished. A string means edit in the buffer
with that name. A buffer means edit in that buffer."
  server)

(def-kom-var kom-view-commented-in-window 'other
  "**Where to view commented texts. 

See the documentation for `kom-write-texts-in-window' for details."
  server)

(def-kom-var kom-edit-filters-in-window nil
  "**Where to edit filters with `kom-filter-edit'.

See the documentation for `kom-write-texts-in-window' for details."
  server)

(def-kom-var kom-list-membership-in-window 'other
  "**Where to list membership with `kom-list-membership'.

See the documentation for `kom-write-texts-in-window' for details."
  server)

(def-kom-var kom-personal-messages-in-window 'other
  "**Where to display personal messages. 

See the documentation for `kom-write-texts-in-window' for details."
  server)

(def-kom-var kom-customize-format 'long
  "**Determines the format of the customize buffer.

This variable currently only controls initial display of documentation
in the customize buffer (the buffer shown by `kom-customize'. When set
to the symbol `long', display documentation. When set to the symbol
`short', don't display documentation.

Values other than `long' or `short' are reserved for future use."
  server)

(def-kom-var kom-user-prompt-format "%[%c% %m%] - "
  "**Format of LysKOM prompt when waiting for input.

The value of this variable must be a string, which is displayed as the
LysKOM prompt, while waiting for a command. The string may contain the
following special sequences:

    Sequence      Meaning
    -----------------------------------------------------------------
    `%c'          Inserts the current default command.

    `%['          Inserts `[' if the ansaphone is on.

    `%]'          Inserts `]' is the ansaphone is on.

    `%m'          Inserts information about recorded messages.

    `%s'          Inserts the name of the LysKOM system

    `%S'          Inserts the server name.

    `%p'          Inserts the name of the user currently logged on.

    `%w'          Inserts the name of the current conference.

    `%a'          Inserts `anonymous' in the current language.

    `%A'          Inserts `Anonymous' in the current language.

    `%#'          Inserts the current session number.

    `% '          Inserts a space if it seems necessary (percent SPC).

    `%%'          Inserts a percent sign.
    -----------------------------------------------------------------

Here are a few examples:

    Format string                 What it does
    -----------------------------------------------------------------
    `\"%[%c% %m%] - \"'           The default prompt

    `\"%[%s: %c% %m%] - \"'       Could display \"LysKOM: Time - \"
    -----------------------------------------------------------------

Note that multiline prompts are not supported."
  server)


(def-kom-var kom-user-prompt-format-executing "%[%c% %m%]."
  "**Format of LysKOM prompt when executing a default command.

The value of this variable must be a string. See
`kom-user-prompt-format' for details."
  server)


(def-kom-var kom-enabled-prompt-format "%[%c% %m%] # "
  "**Format of LysKOM prompt when privileges are enabled..

The value of this variable must be a string. See
`kom-user-prompt-format' for details."
  server)


(def-kom-var kom-enabled-prompt-format-executing "%[%c% %m%]."
  "**Format of LysKOM prompt when executing a default command when
privileges are enabled. 

The value of this variable must be a string. See
`kom-user-prompt-format' for details."
  server)

(def-kom-var kom-anonymous-prompt-format "%[%c% %m%] (%a) - "
  "**Format of the LysKOM prompt when running anonymously.

The value of this variable must be a string. See
`kom-user-prompt-format' for details. See `kom-become-anonymous' for
information on anonymous mode.
"
  server)

(def-kom-var kom-anonymous-prompt-format-executing "%[%c% %m%] (%a)."
  "**Format of the LysKOM prompt when executing a command anonymously.

The value of this variable must be a string. See
`kom-user-prompt-format' for details. See `kom-become-anonymous' for
information on anonymous mode."
  server)

(def-kom-var kom-show-week-number t
  "**Controls display of week numbers in `kom-display-time'.

If set to `t' show the ISO week number when displaying the time. If
set to `nil', don't display the time. All other values are reserved
for future use.

For this feature to work, the calendar package must be installed."
  server)

(def-kom-var kom-show-mx-date-as-local t
  "**Controls display of creation date of imported texts.

If set to `t' then display timestamps of imported texts in the local
timezone instead of the timezone specified in the timestamp."
  server)


(def-kom-var kom-cite-string ">"
  "**String to insert before each line of a commented text.

When citing a text in edit-mode, this string will be inserted before
each line. Although citing is not common and usually not useful in
LysKOM, there is occasionally reason to cite a commented text."
  server)

(def-kom-var kom-created-texts-are-saved nil
  "**If non-nil, save all created texts to a file. 

The value of this variable is the file name on which to save new
texts. Each time a text is created, it is appended to this file.
Values other than a string may produce unpredictible results, and are
reserved for future extensions."
  server
  inherited)

(def-kom-var kom-created-texts-are-read nil
  "**non-nil means automatically mark texts that you create as read.

When set to `t', all text you write are automatically marked as read,
except when running in anonymous mode. When set to `nil', your texts
will be shown as all others.

Values other than `t' and `nil' are reserved for future extensions."
  common created-texts-are-read boolean)

(def-kom-var kom-mark-read-texts-as-read-in-new-recipient t
  "**Controls whether to mark read texts as read in new recipients the acquire.

When this is set to `t', silently mark texts as read when they are
added to a new recipient, if they have been read in any conference.
When `nil', do not mark as read. This feature only works when you are
logged on. Texts added to new conferences while you are not logged on
will not be marked as read in the new recipient.

Values other than `t' or `nil' are reserved for future extensions."
  server)

(def-kom-var kom-customize-in-window nil
  "**Where to customize LysKOM with `kom-customize'.

See the documentation for `kom-write-texts-in-window' for details."
  server)

(def-kom-var kom-prioritize-in-window nil
  "**Where to prioritize conferences with `kom-prioritize'.

See the documentation for `kom-write-texts-in-window' for details."
  server)

(def-kom-var kom-default-mark nil
  "**The default mark used by `kom-mark-text'.

When set to an integer, `kom-mark-text' will not ask for a mark, but
simply mark with that number. When set to `nil', `kom-mark-text' will
ask for a mark.

Values other than integers and `nil' are reserved for future use."
  common default-mark integer
  server)

(def-kom-var kom-symbolic-marks-alist '(("Standard" . 100))
  "**Association list that maps symbolic marks to mark numbers.

Internally, texts can only be marked with numbers. The elisp client
implements symbolic marks through this mapping from mark names to
numbers. Each element of this list is a pair \(`NAME' . `MARK'), where
`NAME' is the name of the mark, and `MARK' is the corresponding
numeric mark."
  server)

(def-kom-var kom-reading-puts-comments-in-pointers-last t
  "**Controls the location where comment pointers are shown.

When set to `t', comment references are shwon at the end of texts.
When set to `nil', comment references are shwon in the headers.

`kom-reading-puts-comments-in-pointers-last' set to `nil'

<
    398331 1996-09-24  13:22  /2 lines/ George Berkeley
    Recipient: Philosophy <1226>
    Comment in text 398374 by John Locke
    Subject: 
    ------------------------------------------------------------
    An abstract idea is a contradiction in terms.
    (398331) -----------------------------------
>

`kom-reading-puts-comments-in-pointers-last' set to `t'

<
    398331 1996-09-24  13:22  /2 lines/ George Berkeley
    Recipient: Philosophy <1226>
    Subject: 
    ------------------------------------------------------------
    An abstract idea is a contradiction in terms.
    (398331) -----------------------------------
    Comment in text 398374 by John Locke
>
"
  common reading-puts-comments-in-pointers-last boolean
  inherited)

(def-kom-var kom-review-uses-cache t
  "**Controls whether review commands use the cache or not.

When set to `t', review commands (such as `kom-review-by-to') will get
texts from the client cache, if possible. This improves performance,
but texts may have changed since they were cached. When set to `nil',
review commands will always get texts from the server.

Values other than `t' and `nil' are reserved for future use."
  server
  inherited)

(def-kom-var kom-review-marks-texts-as-read nil
  "**Controls whether viewing texts with review commands marks then as read.

When set to `t', viewing texts with review commands (e.g.
`kom-review-by-to') will mark unread texts as read. When set to `nil',
texts are not marked as read. 

It is possible to change this setting for a single review command with
the key sequence for `kom-toggle-mark-as-read-prefix'. The value can
be changed for the entire session by using
`kom-make-review-mark-as-read' or `kom-make-review-not-mark-as-read'.

Values other than `t' and `nil' are reserved for future use."
  server
  inherited)

(def-kom-var kom-postpone-default 17
  "**The default number of texts to postpone with `kom-postpone'.

The value of this variable must be a positive integer or zero. All
other values are reserved for future use."
  server)


(def-kom-var kom-dashed-lines t
  "**Controls display of dashed lines before and after texts.

when set to `t' display dashed lines. When set to `nil', don't display
dashed lines.

`kom-dashed-lines' set to `t'

<
    892343 1996-09-24  19:21  /2 lines/ Tycho Brahe
    Recipien: Presentation (of new) Members
    Subject: Tycho Brahe
    ------------------------------------------------------------
    Astronomer and discoverer of stars resident on the island of Ven.
    (892343) -----------------------------------
>

`kom-dashed-lines' set to `nil'

<
    892343 1996-09-24  19:21  /2 lines/ Tycho Brahe
    Recipien: Presentation (of new) Members
    Subject: Tycho Brahe

    Astronomer and discoverer of stars resident on the island of Ven.
    (892343)
>

Values other than `t' and `nil' are reserved for future use. 

See `kom-highlight-dashed-lines' and
`kom-async-highlight-dashed-lines' for additional settings that affect
these lines."
  common dashed-lines boolean
  inherited)

(def-kom-var kom-long-lines nil
  "**If non-nil, some lines and borders will be made longer.

When set to `t', most dashed lines will be longer than the default.
When set to `nil',  use the standard length.

See `kom-text-footer-dash-length', `kom-text-header-dash-length' and
`kom-text-footer-format' for other related settings."
  server
  inherited)

(def-kom-var kom-text-footer-dash-length 52
  "**Control the length of the text footer.

When dashed lines (see `kom-dashed-lines') are in effect, and this is
set to a positive integer, make the text footer at least that many
characters long. Values other than a positive integer are reserved for
future use.

Note that the footer may end up longer than this if one or more elements
together are longer than this length.

This length is currently ignored when `kom-text-footer-format' is used."
  server
  inherited)

(def-kom-var kom-text-header-dash-length 60
  "**Control the length of the text footer.

When dashed lines (see `kom-dashed-lines') are in effect, and this is
set to a positive integer, make the dashed line before the text that
many characters long. 

Values other than a positive integer are reserved for future use."
  server
  inherited)



(def-kom-var kom-text-footer-format nil
  "**Defines the format of the text footer.

When set to a string, use that string as the text footer, overriding
all other settings that affect the text footer. The string may contain
the following special sequences:

    Directive     Meaning
    -------------------------------------------------------------
    `%n'          Insert the text number

    `%p'          Insert the number of the author

    `%P'          Insert the name of the author

    `%-'          Insert a bunch of dashes

    `%f'          Insert special formatting information
    -------------------------------------------------------------

Format directives can be prefixed with a number specifying the minimum
field width (e.g. `%20-'. The field width can be prefixed with an
equals sign (e.g. `%=20p' which means that the field is exactly as
wide as specified (contents may be truncated). A negative field width
means left justify the contents.

The field width of %- is special. It specifies the maximum number of
dashes printed. The actual number will be the maximum minus the length
of the author's name, if it is included anywhere in the format string.

When set, this variable overrides `kom-dashed-lines' and
`kom-show-author-at-end'.

The default format is equivalent to the following strings, depending on 
the settings of kom-dashed-lines and kom-show-author-at-end.

    `kom-dashed-lines'  `kom-show-author-at-end'   Format
    --------------------------------------------------------------------
    `t'                  `t'                       `\"(%n) /%P/%42-%f\"'

    `t'                  `nil'                     `\"(%n) %42-%f\"'

    `nil'                `t'                       `\"(%n) /%P/ %f\"'

    `nil'                `nil'                     `\"(%n) %f\"'
    --------------------------------------------------------------------
"
  server
  inherited)



(def-kom-var kom-show-creating-software nil
  "**Controls display of the software used to create each text.

When creating a text, it is possible to attach information about which
client was used to create the text. When this variable is set to `t',
this information, when available,  is displayed in all text headers.
When this variable is set to `nil', the information is not shown.

To display creating software for a text when this is off, review the
text using `kom-review-noconversion'.

Values other than `t' and `nil' are reserved for future use."
  server
  inherited)

(def-kom-var kom-show-author-at-end t
  "**Controls display of author information in the text footer.

When this is set to `t', display the author of each text immediately
after the text. When set to `nil', the author is not displayed.

`kom-show-author-at-end' set to `t' (with dashed lines on):

<
  892342 1996-09-24  19:21  /2 lines/ Claude Shannon
  Mottagare: Presentation (of new) Members
  rende: Claude Shannon
  ------------------------------------------------------------
  Information theoretician
  (892342) /Claude Shannon/------------------------------
>

`kom-show-author-at-end' set to `nil':

<
  892342 1996-09-24  19:21  /2 lines/ Claude Shannon
  Recipient: Presentation (of new) Members
  Subject: Claude Shannon
  ------------------------------------------------------------
  Information theoretician
  (892342) -----------------------------------
>

If `kom-text-footer-format' is set, the value of this variable is
ignored. Values other than `t' and `nil' are reserved for future use."
  server
  inherited)

(def-kom-var kom-truncate-threshold nil
  "**Controls truncation of text when reviewing.

When set to a positive integer, truncate long texts when reviewing. If
the text has more lines than the value of this variable, it will be
truncated to `kom-truncate-show-lines' lines. When a message is
truncated, a note to that effect will be shown in the message footer
unless `kom-text-footer-format' has been set to a string that does not
include the `%f' directive.

When set to `nil', no truncation will occur. Values other than
positive integers and `nil' are reserved for future use."
  server)

(def-kom-var kom-truncate-show-lines 10
  "**Number of lines to display when truncating texts.

The value must be a positive integer. When a text is truncated \(see
`kom-truncate-threshold', the minimum of `kom-truncate-threshold' and
`kom-truncate-show-lines' determines the number of lines to show.

Values other than positive integers are reserved for future use."
  server)


(def-kom-var kom-print-number-of-unread-on-entrance t
  "**Controls display of number of unread when entering a conference.

When set to `t', the number of unread texts in a conference is
displayed when entering that conference. When set to `nil', the number
of unread texts is not shown. Shortly after logging on, using
`kom-set-unread' or using `kom-recover', the number shown may be
incorrect.

Values other than `t' and `nil' are reserved for future use."
  common print-number-of-unread-on-entrance boolean)

(def-kom-var kom-show-unread-in-frame-title t
  "**Controls display of the unread indicator in the frame title.

When set to `t', an indicator will be shown in the title of each frame
containing a selected LysKOM session with unread texts. When set to
`nil' no indicator will be shown.

Values other than `t' and `nil' are reserved for future extensions."
  server)

;; In the common block of the user-area is a simple boolean variable
;; `presence-messages'.  The control of presence messages in the Emacs
;; Lisp client is more advanced, using two non-boolean variables instead
;; (`kom-presence-messages-in-echo' and `kom-presence-messages-in-buffer').
;;
;; Keep track of the common block variable too, although it isn't used here.
;; `lyskom-save-options' sets it before saving it in obvious cases,
;; but generally it's not possible to know what value it should have.
;; (It could also be used for default values for the presence options for
;; users converting from another client, but that isn't done.)

(def-kom-var kom-presence-messages t
  "Equivalent to the variable `presence-messages' in the user-area.
This is a boolean where other clients may store whether the user wants
messages about people logging in and out of LysKOM.  Here this is instead
controlled by `kom-presence-messages-in-echo-area' and
`kom-presence-messages-in-buffer', so only use this variable to
influence what this client stores in the common block of the user-area."
  common presence-messages boolean)

(def-kom-var kom-presence-messages-in-echo-area t
  "**Controls display of presence messages in the echo area.

If non-nil, LysKOM prints continuous info about what other people
are doing. Info is printed on the echo area and never in the buffer.
If minibuffer is used, no message is printed.

The value `t' means show messages for everyone.

A list is treated as an `indirect association list', and is the
most flexible way to configure this variable. When a user logs
in, the person's number is matched against the list, and if the
value of the match is non-nil, a message is shown. The matching
rules are as follows:

  Element           Meaning
  ------------------------------------------------------------------
  `ATOM'            Matches with the default value if ATOM equals 
                    the person number (this means that you can set 
                    this variable to a list of person numbers).

  `(t . VAL)'       Matches with value VAL if no other match is 
                    found (i.e. sets the default value).

  `(FN . VAL)'      Matches with value VAL if FN is a function that
                    returns non-nil when called with the person 
                    number as its sole argument.

  `(SYM . VAL)'     If SYM is bound to a list, match the person 
                    number against that list using these rules. 
                    VAL is used as the default value for the 
                    matching process.

  `(SYM . VAL)'     If SYM is not bound to a list, matches with 
                    the value VAL if the value of SYM equals the
                    person number.

  `(LIST . VAL)'    Match the person number against LIST using
                    these rules. VAL is used as the default value.

  `(ATOM . VAL)'    Match with value VAL if ATOM equals the 
                    person number.
  ------------------------------------------------------------------

For example, the value `((5 . nil) (kom-friends . t))' will show
a message for every person listed in `kom-friends', except for
person number 5.

The following values work but are obsolete as they can be
expressed by a list:

The value `friends' means show messages for the users in
`kom-friends'. Equivalent to the value `((kom-friends . t))

The value `morons' means show messages for the users in
`kom-morons'. Equivalent to `((kom-morons . t)).

The value `friends-and-morons' means show messages for the users
in `kom-friends' and `kom-morons'. Equivalent to the
value `((kom-friends . t) (kom-morons . t))'.

If you want the messages in the buffer, set the variable 
`kom-presence-messages-in-buffer'.

Values other than those listed are reserved for future use."
  server
  copy-transition lyskom-copy-indirect-assq)

(def-kom-var kom-presence-messages-in-buffer nil
  "**Controls display of presence messages in the LysKOM buffer.

If non-nil, LysKOM prints information about what other people are
doing in the LysKOM buffer. All printing is done just before the
prompt.

See `kom-presence-messages-in-echo-area for a list of permitted
values."
  server
  copy-transition lyskom-copy-indirect-assq)

(def-kom-var kom-unread-mode-line-type nil
  "**Controls how information about unread sessions is shown in the mode line.

If this variable is `nil', there will be a single prompt indicating
whether you have unread texts or letters in any active session.

If this variable is `t', the mode line will indicate each session with
unread texts. Unread letters are indicated by upper-casing the session
name or (if the session name is in upper case already) surrounding it
with asterisks.

All other values are reserved for future use."
  server
  )


(def-kom-var kom-show-where-and-what t
  "**Controls display of hostname and activity information in `kom-who-is-on'. 

When this is set to `t', `kom-who-is-on' and related commands will
display the machine the session is connected from and what the user is
doing. When set to `nil', this information will not be shwon.

Listing from `kom-who-is-on' with `kom-show-where-and-what' set to `t':

<
      User                                   Is in conference             
      At                                     Activity                        
  --------------------------------------------------------------------------
   6810 George Berkeley                      Philosophy                     
        berkeley@emp1.tcd.ie                 (Writing a comment.)           
   7571 John Locke                           Philosophy                     
        eridy@cc.ox.ac.uk                    (Waiting.)                     
  --------------------------------------------------------------------------
>

Listing from `kom-who-is-on' with `kom-show-where-and-what' set to `nil':

<
      User                                   Is in conference             
  --------------------------------------------------------------------------
   6810 George Berkeley                      Philosophy                     
   7571 John Locke                           Philosophy
  --------------------------------------------------------------------------
>

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-show-since-and-when nil
  "**Controls display of connection time in `kom-who-is-on'.

When set to `t', `kom-who-is-on' and related commands will display the
time when each session connected to the server and when each session
last indicated activity.

Listing from `kom-who-is-on' with `kom-show-since-and-when' set to `t':

<
        User                                 Is in conference             
        Connected                            Active last                  
  --------------------------------------------------------------------------
   6810 George Berkeley                      Philosophy                     
        Thursday 2003-01-09 09:24:56         Active                         
   7571 John Locke                           Philosophy                     
        Saturday 2003-01-11 14:02:47         12 minutes                     
  --------------------------------------------------------------------------
>

Listing from `kom-who-is-on' with `kom-show-since-and-when' set to `nil':

<
        User                                 Is in conference             
  --------------------------------------------------------------------------
   6810 George Berkeley                      Philosophy                     
   7571 John Locke                           Philosophy
  --------------------------------------------------------------------------
>

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-idle-hide 30
  "**Controls display of idle sessions in `kom-who-is-on'.

When set to a positive integer, the number of minutes of idletime
before a user is excluded from the list of users shown by
`kom-who-is-on' and related commands. This can be overridden by a
prefix argument to these commands. When set to `nil', do not hide idle
sessions.

Variables other than positive integers and `nil' are reserved for
future use."
  server)

(def-kom-var kom-show-footnotes-immediately t
  "**Controls display of footnotes.

When set to `t', footnotes will be displayed immediately following the
text. When set to `nil', footnotes will be displayed like regular
comments.

Values other than t and `nil' are reserved for future use."
  server)

(def-kom-var kom-follow-comments-outside-membership nil
  "**Controls display of comments in conferences you are not a member of.

If this variable is set to `nil', texts with no recipient you are a
member of will be ignored when selecting comments to display when
reading. When set to `t', the client will follow comment links
regardless of whether you are a member of any of the recipients of the
comment.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-follow-attachments t
  "**Controls how attachments to imported e-mail messages are handled.

When this is set to `t', attachments are followed like regular
comments. When set to `nil', attachments are not followed. To see them
you have to review them manually.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-read-depth-first t
  "**Determines the order in which to read texts.

When set to `t' \(the default), texts are read in comment order. When
set to `nil', texts are read in chronological order.

When reading in comment order, the next text to read is the first
comment of the most recently read text. If there are no comments, the
next text is a comment to another text. Reading in comment order lets
you view threads of discussion.

Values other than `t' or `nil' are reserved for future use."
  common read-depth-first boolean)

(def-kom-var kom-read-related-first 'oldest-ancestor
  "**Read unread related texts first?

Once a candidate text to read has been found (using the rules defined
by the `kom-read-depth-first' variable), the choice can be further
refined as specified by this variable.

When set to `nil', no refinement is made.  When set to
`oldest-ancestor' \(the default), unread parents are shown first.
When set to `oldest-related', children of the parents are also
considered.

Other values are reserved for future use."
  server)

(def-kom-var kom-continuous-scrolling t
  "**Controls frequency of scrolling.

When set to `t', the LysKOM buffer is scrolled whenever text is
inserted. The last viewed position \(usually the most recent prompt)
will always be visible.

When set to `nil', scrolling takes place less frequently. This is
a suitable setting when running Emacs over a slow link, such as an old
modem.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-deferred-printing t
  "**Controls background display of non-cached information.

When set to `t', delay display of \(some) information that has to be
retreived from the server. A placeholder is shown where the
information will be printed, until it is available. This setting
results in a significant improvement in interactive performance.

When set to `nil', do not delay display of any information. This
lowers interactive performance, but may be suitable when running Emacs
over a slow connection.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var lyskom-overlay-pool nil  
  "Pool of overlays"
  local)

(def-kom-var lyskom-defer-indicator "[...]"
  "String to display while LysKOM is waiting for the real string.")

(def-kom-var kom-review-priority nil
  "**Priority to use when reviewing texts.

When set to a positive integer, use that as the priority to use when
reviewing texts. By setting `kom-review-priority' higher than 255 (the
maximum conference priority), new texts to conferences will not break
in while reviewing.

If set to `nil', use the current priority while reviewing. This allows
texts to conferences with priorities higher than that of the current
conference to break in while reviewing (depending on the setting of
`kom-higher-priority-breaks').

Values other than positive integers and `nil' are reserved for future
use."
  server)

(def-kom-var kom-higher-priority-breaks nil
  "**Controls how conference priorities are handled.

When set to `express', texts to conferences with a higher priority
than the current conferences will be shown as soon as the arrive. When
set to `t', texts to prioritized conferences will be shown after the
current comment tree. When set to `nil', prioritized conferences will
be visited when all texts in the current conference have been read.

Values other than `express', `t' and `nil' are reserved for future
use."
  server)

(def-kom-var kom-server-priority-breaks nil
  "**Controls how server priorities are handled.

A non-nil value allows servers with a higher priority than the
current server to break in when new texts arrive. This can be used to
give a work-related server a higher priority than a server used for
frivolous purposes (or, indeed, the other way around).

The following settings are available:

  Value                 Meaning
  --------------------------------------------------------------------
  `express'             Break in immediately when there are unread
                        texts.

  `express-letters'     Break in immediately when there are unread
                        letters.

  `t'                   Break in after the current comment chain when
                        there are unread texts.

  `letters'             Break in after the current comment-chain when
                        there are unread letters.

  `after-conf'          Break in after the current conference when
                        there are unread texts.

  `after-conf-letters'  Break in after the current conference when
                        there are unread letters.

  `when-done'           Prompt the user to go to the next session with
                        unreads after everything has been read. This
                        overrides `kom-do-when-done' as long as there
                        are sessions with unread texts. 
  --------------------------------------------------------------------

Values other than those listed above are reserved for future use and
may result in unpredictible behavior.

See `kom-server-priority' for information on setting the server
priority."
  server)


(def-kom-var kom-session-nickname nil
  "**Nickname for the current LysKOM session.

This variable contains the name of the current LysKOM session (a string)
or `nil'. If set, it will be used as the nickname for this LysKOM session.

See also `kom-server-aliases' and `kom-builtin-server-aliases'."
  server
  non-portable)

(def-kom-var kom-view-text-hook nil
  "**Hook that is called before a text is shown. 

When the hooks are called, `lyskom-view-text-text' is bound to the text
mass of the text and `lyskom-view-text-text-stat' to the text-stat of the
text to be shown."
  local-hook)

(def-kom-var kom-send-message-hook '(lyskom-send-message-trim-newlines)
  "**Hook that is called before a personal, group or common message is sent. 

When called, `lyskom-message-string' is bound to the message that will
be sent and `lyskom-message-recipient' to the conf-stat of the
recipient or nil if the recipient does not exist or if the message is
a common message.

If `lyskom-message-string' is set to nil by a hook, the message will not
be sent."
  local-hook)

(def-kom-var kom-send-message-setup-hook nil
  "**Hook that is called when the minibuffer is entered to read a message.

This hook can be used to set up the minibuffer in a way suitable for
writing messages. For example, the hook might enable `auto-fill-mode'
or set up automatic resizing of the minibuffer."
  local-hook)

(def-kom-var kom-send-message-exit-hook nil
  "**Hook that is called when the minibuffer is exited after reading a message.

Typically this hook will be used to undo the effects of
`kom-send-message-setup-hook'."
  local-hook)

(def-kom-var kom-send-text-hook nil
  "**Hook that is called before sending a text.

This hook is called before the headers are parsed, so it is possible
for the headers to be modified after this hook is called. It is also
possible that the text will not be sent at all."
  local-hook)

(def-kom-var kom-after-load-hook nil
  "**Hook to run once after LysKOM is loaded.

This is similar in effect to using `eval-after-load', but is
independent of the file name of the client.")

(def-kom-var lyskom-after-load-hook nil
  "Obsolete synonym for kom-after-load-hook.")

(def-kom-var kom-change-conf-hook nil
  "**Hook to run when changing conferences.

The functions in this list are run with two arguments. The first is the 
current conference number and the second is the number of the
conference being changed to.

This hook is run before `lyskom-current-conf' is changed, and before
any standard messages have been printed."
  local-hook)

(def-kom-var kom-after-change-conf-hook nil
  "**Hook to run when changing conferences.

The functions in this list are run with two arguments. The first is the 
current conference number and the second is the conference number
being changed to.

This hook is run after `lyskom-current-conf' is changed, and after any
standard messages have been printed."
  local-hook)

(def-kom-var lyskom-login-hook nil
  "**Hook called while logging in.

This hook is called after the session is logged in but before any
command is accepted from the keyboard. It is called immediately before
`kom-login-hook'."
  local-hook)

(def-kom-var kom-login-hook nil
  "**Hook called while logging in.

This hook is called after the session is logged in but before any
command is accepted from the keyboard. Unlike `lyskom-login-hook' it
can be stored in the server."
  server)

(def-kom-var kom-relogin-inhibit-commands '(kom-next-kom 
					    kom-previous-kom 
					    kom-where-is 
					    kom-next-unread-kom)
  "**Commands for which relogin is disabled.

This should be set to a list of commands for which relogin should not
be offered or used."
  server)

(def-kom-var kom-relogin-behaviour 'ask
  "**Controls how to behave when commands are issued in dead sessions.

When this is set to `t', the client will try to login automatically.
When set to `ask', the client will ask whatever the user wants to
reattach. When set to `nil', the client will not attempt to reattach."
  server)

(def-kom-var kom-remember-password nil
  "**Controls whatever to store the password in the session buffer.

When this is set to `t', the client will store the password for the 
current session as a buffert local variable. This can be used in
conjunction with `kom-relogin-behaviour' to reattach dead sessions
automatically."
  server)

(def-kom-var kom-confirm-add-recipients t
  "**Controls confirmation when adding recipients.

When this is set to `t', confirm the recipient type when adding
recipients. Commands like `kom-add-recipient' will pose a question to
determine the type of recipient to add. When set to `nil', assume that
the user always wants full recipients when using `kom-add-recipient'
and similar commands.

Values other than `t' and `nil' are reserved for future use."
  server)


(def-kom-var kom-do-when-done '(kom-review-all-marked-texts kom-display-time)
  "**What to do when all texts are read.

This is a list of commands and lists of commands that are prompted for
and executed when there are no more new texts. The last command in the
list will be prompted for over and over until new texts arrive.

The values in the list can be a LysKOM command (a symbol), an Emacs
command or a keyboard macro (a string).

Values other than those listed are reserved for future use."
  server)

(def-kom-var kom-page-before-command nil
  "**Controls clearing of the screen prior to certain commands.

When this variable is set to `t', all commands will execute at the top
of the LysKOM window. When a command is issued, the buffer will be
scroll so the last prompt is on the first line of the buffer.

When set to a list of LysKOM commands \(symbols), those commands will
execute at the top of the window. All others will execute without
scrolling the buffer.

When set to `nil', never scroll the buffer in the way described.

Values other than those listed are reserved for future use."
  server)

(def-kom-var kom-permissive-completion t
  "**Controls completion of logged-in sessions.

When the client reads names of logged-in users in the minibuffer \(e.g.
`kom-status-session'), it can permit completion of all users or just
those that are logged on.

If this variable is set to `t', completion will include all users,
including those that are not logged in. If this variable is set to
`nil', completion will be restricted to users who are logged in.

Setting this variable to `t' may improve performance significantly,
particularly on servers with many sessions.

Values other than `t' or `nil' are reserved for future use."
  server)

(def-kom-var kom-unsubscribe-makes-passive t
  "**Controls behavior of `kom-sub-self'.

If this variable is set to `t', leaving a conference with
`kom-sub-self' will make the membership passive. Leaving a second time
\(while the membership is still passive) will remove the membership
entirely. 

When set to `nil', `kom-sub-self' removes the membership immediately.

Values other than `t' and `nil' are reserved for future use."
  server)


(def-kom-var kom-membership-default-priority 'ask
  "**Default priority when joining a new conference.

If set to a valid priority (integer from 0 to 255) then new
conferences are read with this priority. When set to the symbol `ask',
the client will ask for a priority when joining new conferences.

Values other than those listed are reserved for future use."
  server)

(def-kom-var kom-membership-default-message-flag 'ask
  "**Default message flag when joining a new conference.

If set to `nil', messages will not be received for new conferences,
if set to `t', messages will be received. When set to the symbol `ask',
the client will ask if you want to receive messages when you join new 
conferences.

Values other than those listed are reserved for future use."
  server)

(def-kom-var kom-membership-default-placement 'last
  "**Default placement of new memberships.

This variable controls the placement of memberships within the
membership list when you join a conference. Note that the membership
priority has precedence over this position.

The value can be `first', `last' or a number. When set to `first', the
membership will be entered before all others. When set to `last', it
will be entered after all others. When set to an integer, the
membership is entered at that position in the list.

All other values are reserved for future use."
  server)

(def-kom-var lyskom-current-prompt nil
  "The current prompt or nil.
This is either nil, indicating that there is currently no prompt, or
a symbol indicating which command is prompted in the LysKOM buffer."
  local)

(def-kom-var lyskom-current-prompt-text nil
  "The current prompt text or nil.
This is either nil, indicating that there is currently no prompt, or
a string indicating the prompt shown in the LysKOM buffer."
  local)

(def-kom-var lyskom-current-prompt-args nil
  "The current prompt arguments.
These are arguments used to format the current prompt."
  local)

(def-kom-var lyskom-current-prompt-timestamp nil
  "The creationtime of the current prompt.
This is used when updating the prompt and on `lyskom-start-of-command'."
  local)

(def-kom-var lyskom-need-prompt-update nil
  "Non-nil if all prompts need to be updated."
  local)

(def-kom-var kom-show-personal-messages-in-buffer t
  "**Buffer to show personal, group and alarm messages in.

This variable controls which buffer personal, group and alarm messages
are shown in. When set to `nil', all messages are silently discarded.
When set to `t', messages are shown in the main LysKOM buffer. When
set to a string, messages will be inserted in a buffer by that name
\(one will be created if necessary). When set to a buffer, messages
are inserted in that buffer.

All values other than those listed are reserved for future use."
  server)

(def-kom-var kom-pop-personal-messages nil
  "**Non-nil means pop up a buffer with personal messages as they arrive.

When this variable is set to `nil', personal, group and alarm messages
are simply inserted in the appropriate buffer \(see
`kom-show-personal-messages-in-buffer'). The buffer is not displayed
automatically. When this variable is set to `t', the buffer in which a
new message is shown will be displayed automatically, possibly
splitting windows and uniconifying frames. When this variable is set
to `yes', the behavior is identical to when it is set to `t', but
frames will not be uniconified.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-lost-session-notification 'all-buffers
  "**How (and if) notification of lost sessions will be handled.

Lost (abnormally terminated) sessions are always notified in the
session buffer. If this parameter is set to `beep', the client will
also beep.  If it is set to `all-buffers', the client will beep and
insert a notification message in any active LysKOM buffer.

Values other than `all-buffers', `beep' and `nil' are reserved for
future use."
  server)

(def-kom-var kom-ding-pause-amount 0.1
  "**Number of seconds to wait between successive beeps.

When multiple beeps are used as audible notification, this determines
the amount of time (in seconds) to wait between successive beeps. The
value must be a positive number (it may be a floating point number).")

(def-kom-var kom-ding-on-priority-break 1
  "**Non-nil means ding if a higher priority text or conference breaks in.

This variable must be set to a `t', `nil', a positive integer, a
string or a function. When set to `t' use the default beep. When set
to `nil', do not beep. When set to a number, beep that many times
\(see `kom-ding-pause-amount' for additional settings). When set to a
function, call that function. When set to a string, run the program
indicated by `kom-audio-player' with the string as its sole argument."
  server)

(def-kom-var kom-ding-on-new-letter nil
  "**Non-nil means ding if a message arrives in the letter box. 

See `kom-ding-on-priority-break' for valid values."
  server)


(def-kom-var kom-ding-on-wait-done 1
  "**Non-nil means ding when `kom-busy-wait' terminates.

See `kom-ding-on-priority-break' for valid values."
  server)

(def-kom-var kom-ding-on-common-messages 0
  "**Non-nil means ding when an alarm message arrives. 

See `kom-ding-no-priority-break' for valid values. In addition to
those listed there, the value of this variable may be a list of
elements like \(`KEY' . `VALUE'). If the sender (a conference number) is
found as the key of any element, the value of that element will be
used to generate the ding (valid values are those listed for
`kom-ding-on-priority-break'). The special key `t' is used when no other
key matches the sender."
  server)

(def-kom-var kom-ding-on-group-messages 1
  "**non-nil means ding when a group messages arrives.

See `kom-ding-on-common-messages' for valid values."
  server)

(def-kom-var kom-ding-on-personal-messages 2
  "**non-nil means ding when a personal message arrives.

See `kom-ding-on-common-messages' for valid values."
server)


(def-kom-var kom-ding-on-no-subject 2
  "**How to ding if the user has not entered a subject line.

When attempting to submit a text without a subject line, this variable
determines how to beep. See `kom-ding-on-priority-break' for valid
values."
  server)


(def-kom-var kom-audio-player "audioplay"
  "**Program used to play audio files.

If a of the variables that control audio signals (dings, beeps) is set
to a string, the command (shell command, not Emacs command) indicated
by this variable will be called with that string as an argument.
Therefore, this variable should be set to the name of an external
command that takes a single argument: a sound file to play.

Non-string values are reserved for future use."
  server)


(def-kom-var kom-ignore-message-senders nil
  "**List of senders whose personal, group and alarm messages are ignored.

The value of this variable must be a list of person numbers. Personal,
group and alarm messages sent by users in this list will be silently
ignored.

Values other than those listed are reserved for future use."
  server
  copy-transition lyskom-copy-indirect-assq)

(def-kom-var kom-ignore-message-recipients nil
  "**List of recipients you do not want group messages to.

The value of this variable must be a list of conference numbers.
Messages sent to a recipient listed will be silently ignored.

Values other than those listed are reserved for future use."
  server
  copy-transition lyskom-copy-indirect-assq)

(def-kom-var kom-show-personal-message-date t
  "**Controls display of date on personal messages.

When set to `t', display the date and time when personal, group and
alarm messages arrived. When set to `nil', only display the time.

`kom-show-personal-message-date' set to `t'>:

<
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  Alarm message from Claude Chappe (1805-01-23 17:34):
  
  Je me donne la mort our viter l'ennui de la vie qui m'accable; je
  n'ai point de reproches  me faire.
  ----------------------------------------------------------------
>

`kom-show-personal-message-date' set to `nil':

<
  ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  Alarm message from Claude Chappe (17:34):
  
  Je me donne la mort our viter l'ennui de la vie qui m'accable; je
  n'ai point de reproches  me faire.
  ----------------------------------------------------------------
>

See Bulletin of Les Amis de Paris Central Tlgraphe, no. 20, July
1993 for information on the quote above.
"
  server)

(def-kom-var kom-default-message-recipient 'group
  "**Determines default recipient of personal messages.

When set to `everybody', the default recipient of all messages will be
all users (i.e. the default is to send alarm messages). This may not
work properly. 

If set to `group', the default recipient depends on the message most
recently received. If that message was a group message, then the
default recipient will be the recipient of that message. If that
message was a personal message or an alarm message, the default
recipient will be the sender of the most recent message.

If set to `sender', the default recipient is always the sender of the
most recently received message, regardless of its type.

Values other than those listed are reserved for future use."
  server)


(def-kom-var kom-filter-outgoing-messages t
  "**Determines whether automatic outgoing messages are shown.

When set to `t', outgoing remote control messages and automatic
replies are not displayed. When set to `nil', these messages are shown
the same way normal messages are shown when sent.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-highlight-conferences 
  '((kom-friends . kom-friends-face)
    (kom-morons  . kom-morons-face)
    (lyskom-highlight-has-no-presentation . kom-active-strikethrough-face)
    (lyskom-pers-no . kom-me-face)
    (lyskom-highlight-i-am-supervisor . kom-active-highlight-face))
  "**How to highlight conference an person names.

The value of this variable is an alist whose keys are matched against
conference numbers and whose values are the names of faces to use for
matching conferences.

The following values are legal for keys:

    Value       Meaning
    -----------------------------------------------------------------
    Symbol      If the value is a list, matches if the conference
                number is in the list. If the value is an integer,
                matches if the conference number matches the integer.

    List        Matches if the conference number is in the list.

    Function    Matches if the function returns non-nil (see below).
    -----------------------------------------------------------------

When using a function as the key, the function will be called with a
single argument, the object to be printed. The type of the argument
will vary depending on how it is printed. Functions should be written
so they work regardless of what is passed to them.

All other values are reserved for future use."
  server
  copy-transition lyskom-copy-indirect-assq)

(def-kom-var kom-person-list-1 nil
  "**General list of persons. 

This variable has no predefined meanind and is intended for use
in variables such as `kom-presence-messages-in-buffer' and
`kom-highlight-conferences'."
  server
  non-portable)

(def-kom-var kom-person-list-2 nil
  "**General list of persons. 

This variable has no predefined meanind and is intended for use
in variables such as `kom-presence-messages-in-buffer' and
`kom-highlight-conferences'."
  server
  non-portable)

(def-kom-var kom-person-list-3 nil
  "**General list of persons. 

This variable has no predefined meanind and is intended for use
in variables such as `kom-presence-messages-in-buffer' and
`kom-highlight-conferences'."
  server
  non-portable)

(def-kom-var kom-person-list-4 nil
  "**General list of persons. 

This variable has no predefined meanind and is intended for use
in variables such as `kom-presence-messages-in-buffer' and
`kom-highlight-conferences'."
  server
  non-portable)

(def-kom-var kom-friends nil
  "**List of friends and other nice people.

The value of this variable is a list of person numbers. People listed
here will be displayed using the face in `kom-friends-face'. They can
also receive special treatment in other cases (see
`kom-presence-messages').

See `kom-morons' for a related variable.

Values other than a list of integers are reserved for future use."
  server
  non-portable)

(def-kom-var kom-morons nil
  "**List of people morons and other nasty people.

The value of this variable is a list of person numbers. People listed
here will be displayed using the face in `kom-morons-face'.

See `kom-friends' for a related variable.

Values other than a list of integers are reserved for future use."
  server
  non-portable)

(def-kom-var kom-dont-check-commented-authors nil
  "**A list of recipients that don't need to see comments to their texts.

When writing a comment, the client can check that the author of the
commented text is a member of at least one of the recipients. If that
is not the case, the user will be offered to add the author as a
recipient.

This variable lists authors who shold not be checked in this manner.
Typically it will contain a list of import agents.

Values other than a list of integers are reserved for future use."
  server
  inherited
  copy-transition lyskom-copy-indirect-assq)

(def-kom-var kom-smileys t
  "**Controls display of graphical smileys.

When set to `t', display graphical smileys instead of `:-)' and
similar character sequences. When set to `nil', do not display
graphical smileys.

For this to work at all you must have the `smiley' or `smiley-ems'
installed (usually installed with Gnus) and support for graphics (part
of XEmacs and Gnu Emacs 21.x and later).

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-text-properties t
  "**Controls use of text properties in LysKOM (fonts and stuff).

When set to `t', the client will freely use text properties, like
fonts. When set to `nil', the client will not use text properties.
This will disable all font usage and clickable stuff.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-fontify-text t
  "**Controls whether plaintext messages are fontified.

When set to `t', the client will make words and phrases delimited by
asterisks bold, and words and phrases delimited by underscore italic.
Other similar features may be added in the future.

Values other than `t' or `nil' are reserved for future use."
  server)


(def-kom-var kom-use-button-hints t
  "**Controls use of context sensitive actions on clickable areas.

When set to `t', clickable areas of the same type may behave
differently depending on what command created them. For example, in
one case clicking a conference name may show the presentation of that
conference and in another it will go to the conference.

Values other than `t' and `nil' are reserved for future use.")

(def-kom-var kom-autowrap t
  "**Controls automatic text wrapping.

When set to `t', automatically break long lines in texts. A set of
rules attempts to limit line breaking to regular, unformatted text.
This works most of the time, but occasionally fails (in which case
`kom-review-noconversion' comes in handy).

When set to an integer, perform automatic line breaking in text no
longer than that many characters. 

When set to `nil', do not break lines automatically at all.

Values other than `t', `nil' and integers are reserved for future
use."
  server)

(def-kom-var kom-autowrap-timeout 5
  "**Limits how much time automatic text wrapping may take.

When set to an integer, limits the number of seconds automatic line
breaking may take. Without a limitation, breaking lines of a very long
text can take a very, very long time. Note that this setting is
approximate; the actual time spent breaking lines may be several
seconds longer than this value. Furthermore, using this setting slows
down all text display slightly.

When set to `nil', do not limit the amout of time automatic line
breaking may take.

Values other than `nil' and integers are reserved for future use."
  server)

(def-kom-var kom-keep-alive-interval 180
  "**Polling interval for `kom-keep-alive'.

The command `kom-keep-alive' polls the server periodically to keep the
connection active. This variable specifies how many seconds to wait
between these periodic requests to the server.

Values other than integers are reserved for future use."
  server)

(defvar lyskom-transforming-external-text nil
  "Dynamically bound to non-nil when transforming text in which text,
conference and person buttons are not expected.")

(def-kom-var lyskom-url-protocol-regexp
  "\\(file\\|ftp\\|gopher\\|http\\|https\\|news\\|wais\\|mailto\\|telnet\\):"
  "Regexp to match the protocol part of a URL.")

(def-kom-var lyskom-text-buttons
  '(
    ;; Text numbers
    ("\\(\\<[0-9][0-9][0-9][0-9]\\([0-9]\\)?\\([0-9]\\)?\\([0-9]\\)?\\([0-9]\\)?\\([0-9]\\)?\\>\\)" 
                                        ; Match
     text                               ; Button type
     0                                  ; Portion that's a button
     1                                  ; Portion that's the arg
     nil                                ; Face or nil (=default)
     )

    ;; Email

    ("\\(\\b\\|^\\)[^()<>@,;:\"\\\\\000- ]+@[^\000- <>;,.'\"!:?) \t\012\014]+\\(\\.[^\000- <>;,.'\"!:?)]+\\)+"
     email 0 0 kom-url-face)

    ;; URLs

    ("\\b\\(www\\|ftp\\|home\\)\\.[^\t \012\014\"<>|\\]*[^][\t \012\014\"<>|.,!(){}?'`:;]" 
     pseudo-url 0 nil kom-url-face)

    ("\\(google:\\|file://\\|ftp://\\|gopher://\\|rtsp://\\|http://\\|https://\\|news:\\|wais://\\|mailto:\\|telnet:\\)[^\t \012\014\"<>|\\]*[^][\t \012\014\"<>|.,!(){}?'`:;]" 
     url 0 nil kom-url-face)

    ("<URL:\\([^<>]+\\)>"
     pseudo-url 1 1 kom-url-face lyskom-is-url)
    ("<\\([^<>]+\\)>"
     pseudo-url 1 1 kom-url-face lyskom-is-url)

    ;; JySKom enhancements

    ("<(?m[|]te[ \t\n\r]*\\([0-9]+\\)\\([^0-9>]?\\|[^0-9>][^>]*\\))?>"
     conf 0 1 nil)
    ("<(?text[ \t\n\r]*\\([0-9]+\\)\\([^0-9>]?\\|[^0-9>][^>]*\\))?>"
     text 0 1 nil)
    ("<(?person[ \t\n\r]*\\([0-9]+\\)\\([^0-9>]?\\|[^0-9>][^>]*\\))?>"
     pers 0 1 nil)

    ;; Info node reference

    ("\\*Note[ \n\t]+\\([^:\n]*\\(\n[^:\n]*\\)?\\):\\s-*\\(\\(([^\)]+)\\)?[^.,\t\n]*\\(\n[^.,\t\n]*\\)?\\)[.,\t]" 
     info-node 1 3 kom-url-face)

    )
  "List of buttons to install in the text mass of LysKOM objects. Each 
element is a list consisting of REGEXP TYPE BUTTON-MATCH BUTTON-ARG-MATCH
FACE &optional PRED.

REGEXP is the regexp to look for in the text.
TYPE is the button type. Valid button types are defined in lyskom-button-actions.
BUTTON-MATCH is the number of the parenthesized expression that is the actual button.
BUTTON-ARG-MATCH is the number of the expression to be used as the button argument.
FACE is the text face to apply to the button, or nil to use the default face.
If PRED is given, it is a function that will be passed the matched string; if 
it returns non-nil, the match is considered valid.")

(def-kom-var kom-url-viewer-preferences '("emacs"
                                          "windows"
                                          "w3")

  "**Specifies application preferences for opening URLs.

This is a list of URL handlers to try when opening a URL. Each handler
is associated with a set of protocols. An URL will be opened by the
first handler in the list that is associated with the URLs protocol.

Value values for elements in the list are:

    Handler          What it does                     Handles protocols
    ------------------------------------------------------------------------
    \"default\"      Use `browse-url' to open URLs    All

    \"windows\"      Microsoft Windows default        All

    \"netscape\"     Opens URLs in Netscape, Mozilla, All
                     Firefox or Opera.

    \"mosaic\"       Opens URLs in NCSA Mosaic        All common

    \"lynx\"         Opens URLs in Lynx               All common

    \"galeon\"       Opens URLs in Galeon             All common

    \"w3\"           Opens URLs in Emacs W3           http,gopher,ftp

    \"emacs\"        Opens URLs in Emacs              ftp,telnet,file,mailto

    \"dired\"        Opens URLs in Emacs              ftp,file

    \"telnet-mode\"  Opens URLs in Emacs              telnet

    \"mail-mode\"    Opens URLs in Emacs              mailto
    ------------------------------------------------------------------------

The variable `kom-url-managers' contains a list of all handlers. 

See `kom-mosaic-command', `kom-netscape-command',
`kom-galeon-command', `kom-lynx-terminal-command',
`kom-lynx-xterm-command', and `kom-windows-browser-command' for
additional settings that affect opening URLs."
  server)


(def-kom-var kom-url-managers '(("default"
                                 ".*"
                                 "Browse-URL"
                                 lyskom-view-url-browse-url)
                                ("w3" 
                                 "\\(google\\|http\\|gopher\\|ftp\\)"
                                 "Emacs W3" 
                                 lyskom-view-url-w3)
                                ("windows"
                                 ".*"
                                 "web browser"
                                 lyskom-view-url-windows)
                                ("netscape"
                                 ".*"
                                 "Netscape Navigator/Mozilla/Firefox/Opera"
                                 lyskom-view-url-netscape)
                                ("\\(emacs\\|dired\\)"
                                 "\\(ftp\\|file\\)"
                                 "dired"
                                 lyskom-view-url-dired)
                                ("\\(emacs\\|telnet-mode\\)"
                                 "telnet"
                                 "emacs telnet"
                                 lyskom-view-url-telnet)
                                ("\\(emacs\\|mail-mode\\)"
                                 "mailto"
                                 "mail-mode"
                                 lyskom-view-url-mailmode)
                                ("mosaic"
                                 "\\(http\\|gopher\\|ftp\\|mailto\\|news\\|wais\\|file\\|telnet\\)"
                                 "NCSA Mosaic"
                                 lyskom-view-url-mosaic)
                                ("lynx"
                                 "\\(http\\|gopher\\|ftp\\|mailto\\|news\\|wais\\|file\\|telnet\\)"
                                 "Lynx"
                                 lyskom-view-url-lynx)
                                ("galeon"
                                 "\\(http\\|gopher\\|ftp\\|mailto\\|news\\|wais\\|file\\|telnet\\)"
                                 "Galeon"
                                 lyskom-view-url-galeon))

  "List of URL managers. Each element is a list consisting of
\(MANAGER-REGEXP PROTOCOLS NAME VIEW-FUNCTION). When LysKOM attempts to
view a URL, kom-url-viewer-preferences is scanned, and the URL
-manager whose MANAGER-REGEXP first matches an element in
kom-url-viewer-preferences and whose PROTOCOLS matches the protocol of
the selected URL is used to view the URL by calling its VIEW-FUNCTION
with the URL and the manager entry as arguments.")

(def-kom-var kom-windows-browser-command ""
  "**Program to open a URL in Windows. 

If it is the empty string, a couple of commands that are likely to
work on Windows will be tried."
  server)

(def-kom-var kom-mosaic-command "/usr/local/bin/mosaic"
  "**Command to run Mosaic.

Note that Mosaic uses its own special conventions to open URLs
remotely that are probably not suitable for other browsers."
  server)

(def-kom-var kom-netscape-command "netscape"
  "**Command to start Firefox, Mozilla, Opera or Netscape.

If set to a string, it should be a command that starts Firefox,
Mozilla, Netscape or Opera with no arguments. If a list, the first
element must be a command that starts the web browser. The remaining 
elements are used as arguments to the browser.

For instance, a value of \"netscape\" is valid, but \"netscape -d
host:0\" is not. Instead, the latter should be \(\"netscape\" \"-d\"
\"host:0\"\)"
  server)

(def-kom-var kom-netscape-variant nil
  "**Netscape-specific options.

When set to `nil', open URLs in whatever window Firefox, Netscape,
Opera or Mozilla chooses. Usually an old window is recycled. When set
to `new-window', open URLs in a new window. When set to `new-tab',
open URLs in a new tab. These options may not work with all versions
of the different browsers, or on all operating systems.

Values other than those listed above are reserved for future use."
  server)


(def-kom-var kom-galeon-command "galeon"
  "**Command used to run to start Galeon.

If a string, it should be a command that starts Galeon with no
arguments. If a list, the first element must be a command that starts
Galeon. The remaining elements are used as arguments to Galeon.

For instance, a value of \"galeon\" is valid, but \"galeon
--display host:0\" is not. Instead, the latter should be
\(\"galeon\" \"--display\" \"host:0\"\)"
  server)

(def-kom-var kom-lynx-terminal 'xterm
  "**Where to start Lynx when opening URLs.

Valid values are `xterm' \(start Lynx in an xterm) and `terminal'
\(start Lynx in Emacs terminal mode)."
  server)

(def-kom-var kom-lynx-xterm-command
  '("xterm" "-geometry" "90x50+100+100" "-e" "lynx")
  "**Command to run to start Lynx in an xterm.

Must be a list of strings, where the first element is the name of the
xterm program, and the remaining elements are arguments to the
xterm. The last elements should be \"-e\" \"lynx\", or something similar,
to start Lynx."
  server)

(def-kom-var kom-lynx-terminal-command "lynx"
  "**Command to run Lynx in Emacs terminal mode.

This can be either a string, to start Lynx with no arguments, or a
list of strings, where the first element is the command, and the rest
are arguments to Lynx."
  server)

(def-kom-var kom-confirm-multiple-recipients 'after
  "**Non-nil means ask the user for confirmation about recipients.

When the user writes a comment to a text with more than one recipient,
ask for confirmation that all recipients are relevant, or one question
for each recipient.

If this variable is set to `before', ask once for each recipient
before opening the edit buffer.

If this variable is set to `after', ask once for all recipients after
editing the text. This is the preferred value since it is difficult to
judge the relevance of a recipient until the text has been written.

Values other than those listed are reserved for future use."
  common confirm-multiple-recipients boolean)

(def-kom-var kom-check-for-new-comments t
  "**Controls check for new comments when writing comments.

When posting a comment, the client can check if the text being
commented has unread comments. This frequently happens when several
people attempt to answer the same question at the same time.

If this variable is set to `t' check for new comments before posting a
comment. If there are new comments, ask for confirmation before
posting.

If this variable is `nil', dont' check.

If this variable is set to a function name, call the function to see
if check should be performed. The function is called with the
commented text's text-stat as its sole argument. If it returns
non-nil, the check is performed.

A list of conference numbers means perform the check for all
conferences other than those listed.

All other values are reserved for future use."
  server
  copy-transition lyskom-copy-indirect-assq)


(def-kom-var kom-check-commented-author-membership t
  "**Controls check that authors of commented texts will see a new comment.

When writing a comment, the client can check that the author of the
commented text is a member of at least one of the recipients. If that
is not the case, the user will be offered to add the author as a
recipient.

If this variable is set to `t', perform the check. If not, don't
perform the check.

See `kom-dont-check-commented-authors' for a way to exclude certain
authors from this check.

Values other than `t' and `nil' are reserved for future use."
  server
  copy-transition lyskom-copy-indirect-assq)


(def-kom-var kom-inhibit-typeahead nil
  "**Controls how command typed while the client is busy are handled.

When this variable is set to `t', input that arrives while a command
is running will be discarded before the next prompt is shown. When set
to `nil', that input will be handled normally.

If you find yourself accidentally executing commands, it may make
sense to set this variable to `t'.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-max-buffer-size nil
  "**Controls the maximum size of the LysKOM buffer.

When this variable is set to an integer, limit the size of the LysKOM
buffer to that many characters. When set to `nil', don't limit the
buffer size.

Before anything is deleted, `lyskom-trim-buffer-hook' (note that this
hook variable will probably be renamed in the future). Also see
`kom-trim-buffer-minimum' for more information.

Values other than `nil' and positive integers are reserved for future
use."
  server)

(def-kom-var kom-trim-buffer-minimum 4096
  "**The amount of text to trim from the buffer when limiting its size.

To avoid cutting tiny bits from the top of the buffer all the time,
delete this many bytes (rounded to a whole line) from the buffer at a
time.

See `kom-max-buffer-size' for information on how to limit the buffer
size.

Values other than positive integers are reserved for future use."
  server)


(def-kom-var kom-print-relative-dates t
  "**Controls display of relative dates.

When set to `t', print today's date as \"today\" and yesterday's as
\"yesterday\" in most places. When set to `nil', print all dates using
the default numeric format.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-print-seconds-in-time-strings nil
  "**Controls display of seconds in timestamps.

When set to `t', display many timestamps with seconds. When set to
`nil', only show hours and minutes.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-show-namedays nil
  "**Controls display of namedays.

When this variable is set to a non-nil value,`kom-display-time' can
display names of the day.

If set to `t', show the namedays for the currently selected languag
\(if there are any). If set to a symbol, that symbol should indicate a
list of names (use `kom-list-nameday-lists' to see a list of all
possible lists). If set to a list, each element should be a symbol
indicating a list of names; display namedays from all indicated lists.

All values other than those listed are reserved for future use."
  server)

(def-kom-var kom-ssh-relay-host nil
  "**Controls relay of LysKOM sessions through ssh.

It is possible to automatically tunnel LysKOM sessions through ssh.
For this to work you need to have command-line ssh installed, and
connecting to the remote host indicated by this variable must succeed
without asking for a password or passphrase.

This variable, when set to a string, enables tunneling ssh to the host
indicated by the value of this variable.

Note that storing this variable in the LysKOM server makes no sense.

See `kom-ssh-command' for additional configuration options.

Values other than strings are reserved for future use.")

(def-kom-var kom-ssh-command "ssh"
  "**Command to start ssh.

This variable should be set to a command that runs ssh. Note that it
is not possible to specify command-line arguments using this variable.

See `kom-ssh-relay-host' for more information.

Non-string values are reserved for future use.")

(def-kom-var lyskom-ssh-proxy nil
  "When non-nil, the ssh proxy used for this buffer."
  local
  protected)

(def-kom-var kom-ssh-general-errors
  "\\<\\(Enter passphrase.*$\\|^.*password.*$\\)\\|refused\\|disconnect\\|denied\\|error\\|key not found\\|cannot listen\\|[Aa]ddress already in use\\>"
  "**Regexp for messages from ssh that indicate that ssh has failed.")

(def-kom-var kom-ssh-forwarding-errors
  "\\<cannot listen\\|[Aa]ddress already in use\\>"
  "**Regexp for messages from ssh that indicate that forwarding failed.")


(def-kom-var kom-www-proxy nil
  "**Controls use of an HTTP proxy for the LysKOM session.

When this variable is set to a string, it should be the host name of
an HTTP proxy that supports the CONNECT method. All LysKOM sessions
will be transparently tunneled through this proxy.

When set to a list, each element must be a pair \(`SERVER' . `PROXY'),
where `SERVER' is a LysKOM server and `PROXY' is the proxy to use for
that server (or `nil' to not use a proxy). The special value `t' for
`SERVER' indicates the proxy to use for unlisted servers.

The proxy string has the form \"`HOST':`PORT'\", where `HOST' is the
proxy host and `PORT' is the port on which the proxy is running. The
port part is optional. If it is not specified, port 80 is assumed.

Values other than those described are reserved for future use.")

(def-kom-var kom-www-proxy-headers
  "User-Agent: Mozilla/4.7C-CCK-MCD  [en] (X11; I; SunOS 5.6 sun4u)"
  "**Extra HTTP headers to use when connecting through a Proxy.

The value of this variable should either be a single string, which is
sent verbatim to the proxy, or a list of strings which will be sent to
the proxy separated by CRLF, or a list of elements like \(`NAME' `H1'
`H2' ... `Hn') where `NAME' is the name of a proxy and the remaining
elements are headers to send when connecting through that proxy.

Do not use this variable for proxy authentication.

Values other than those listed are reserved for future use.")


(def-kom-var kom-server-aliases nil
  "**An alist mapping server names to shorter identification strings.

Each value in this string should be of the form \(`SERVER' . `NICKNAME'),
where `NICKNAME' is the short name for the server `SERVER'. You can
set this in init files before loading LysKOM.

See `kom-builtin-server-aliases' for more information.

Values other than those described are reserved for future use.")

(def-kom-var kom-builtin-server-aliases
  '(("kom.lysator.liu.se" . "LysKOM")
    ("com.lysator.liu.se" . "LysCOM (LysKOM in English)")
    ("kom.ludd.luth.se" . "LuddKOM")
    ("kom.hem.liu.se" . "RydKOM")
    ("kom.update.uu.se" . "UppKOM")
    ("kom.mds.mdh.se" . "MdS-KOM")
    ("kom.stacken.kth.se" . "TokKOM")
    ("com.helsinki.fi" . "HesaKOM")
    ("kom.cd.chalmers.se" . "CD-KOM")
    ("community.roxen.com" . "Roxen Community KOM")
    ("kom.ds.hj.se" . "DSKOM")
    ("kom.sno.pp.se" . "SnoppKOM")
    ("myskom.kfib.org" . "MysKOM"))
  "**An alist mapping server names to shorter identification strings.

Each value in this string should be of the form (`SERVER' . `NICKNAME'),
where `NICKNAME' is the short name for the server `SERVER'. Avoid
setting this variable since that will override the list compiled into
the client. Use `kom-server-aliases' instead.

Values other than those described are reserved for future use.")

(def-kom-var kom-ansaphone-on nil
  "*'Controls automatic replies to personal messages.

When set to `t', send automatically send replies to personal messages.
See `kom-ansaphone-replies' and `kom-ansaphone-default-reply' for ways
to configure the replies to send.

When set to `nil', don't send automatic replies.

This can be set temporarily with `kom-toggle-auto-reply'. The reply
message can be changed temporarily with `kom-change-auto-reply'.

Values other than `t' and `nil' are reserved for future use."
  local)

(def-kom-var kom-silent-ansaphone nil
  "**Controls beeps when automatic replies are enabled.

When set to `t', don't beep or otherwise signal receipt of personal
messages while automatic replies are enabled (see `kom-ansaphone-on').
When set to `nil', signal receipt of personal messages as usual.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-ansaphone-record-messages t
  "**Controls recording of personal messages while automatic replies are on.

When set to `t', record personal messages that are received while
automatic replies are on. Recorded messages can be listed with
`kom-list-messages' and erased with `kom-erase-messages'.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-ansaphone-show-messages t
  "**Controls display or personal messages while automatic replies are on.

When set to `t', personal messages received while automatic replies
are enabled will be shown as usual. When set to `nil' the will not be
shown. Note that if this variable is set to `nil' and
`kom-ansaphone-record-messages' is also set to `nil', messages are
simply discarded.

Values other than `t' and `nil' are reserved for future use."
  server)


(def-kom-var lyskom-ansaphone-messages nil
  "Messages collected by the automatic reply facility.
The most recent message is the first message in the list."
  local)

(def-kom-var lyskom-ansaphone-when-set (current-time-string)
  "Time when the auto-reply facility was enabled."
  local)

(def-kom-var kom-remote-control t
  "**Enables and disable remote control.

When this is set to `t', it will be possible to control the client
from other sessions using `kom-remote-autoreply',
`kom-remote-list-messages', `kom-remote-set-message',
`kom-remote-erase-messages' and `kom-remote-quit' (and possibly others
in the future). When set to `nil', remote control is disabled.

See `kom-remote-controllers' and `kom-self-control' for ways to
control access through remote control.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-remote-controllers nil 
  "**List of people who may use remote control.

This should be `nil' or a list of person numbers. The persons listed
may control the session using remote control commands (see
`kom-remote-control' for more information).

See `kom-self-control' for another variable that affects remote
control.

Values other than `nil' and a list of persons are reserved for future
use."
  server
  non-portable)

(def-kom-var kom-self-control t
  "**Enable or disable remote control for the logged-in user.

When set to `t', the user who is currently logged in may control the
session remotely from sessions that where the same user is logged in.
When set to `nil', only permit users listed in
`kom-remote-controllers'.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-ansaphone-replies
  '((group nil nil nil nil)
    (common nil nil nil nil))
  "**List of automatic replies to various messages.

This variable determines non-default automatic replies to personal
messages. Automatic replies are only send when `kom-ansaphone-on' is
set to `t'.

The value of this variable is a list of rules. Each rule is a list
with five elements: `MESSAGE-TYPE', `SENDER', `RECIPIENT', `TEXT', and
`REPLY'. Incoming messages are compared against these rules in order.
A message matches a rule if it matches all non-nil elements of the
rule.

If `MESSAGE-TYPE' is non-nil, match against the message type. Valid
values are `personal', `group' and `common' (for alarm messages).

If `SENDER' is non-nil, match against the message sender. The value
is either an integer (a person number) or a list of person numbers. A
message matches if its sender is any of the listed persons.

If `RECIPIENT' is non-nil, match against the message recipient. The
value is either an integer (a conference number) or a list of
conference numbers. A message matches if its recipient is any of the
listed conferences.

If `TEXT' is non-nil, match against the message text. The value is
a regular expression. A message matches if it contains a match for the
regular expression.

If all non-nil components of a rule match, rule processing is
terminated. If `REPLY' is a string, send that string as the reply. If
`REPLY' is `nil', don't send a reply.

If all rules are processed without finding a match, the value of
`kom-ansaphone-default-reply' is sent.

The default value for this variable disables automatic replies to all
group and alarm messages."
  server)

(def-kom-var kom-agree-text nil
  "**Determines the text used in `kom-agree'.

When non-nil, this is the default text for `kom-agree'. The value
may be a string, which is used verbatim as the default text; a
function, which is called and should return a text to use; or a list
whose elements must be strings, functions or lists, and from which an
element will be chosen at random and used in the same manner as
`kom-agree-text' itself.

Values other than those described are reserved for future use."
  server)

(def-kom-var kom-default-language nil
  "**The default language for LysKOM.

When set to non-nil, this variable should be set to a list of
symbols or a symbol indicating the prefered language(s) for LysKOM.
Each symbol name must be the ISO 630 code for a desired language (e.g.
`sv' for Swedish and `en' for English). Available languages depend on
how LysKOM was built.

All valid choices are listed in `lyskom-languages'."
  common language symbol-list
  transition (lambda (x) (cond ((listp x) x) (t (list x))))
  inherited
  protected)

(def-kom-var lyskom-language kom-default-language
  local
  inherited
  minibuffer
  protected
  "The language currently in use for messages.")

(def-kom-var lyskom-global-language kom-default-language
  "The language for language-specific things that affect multiple sessions.")

(def-kom-var lyskom-edit-mode-map nil
  "Mode map for LysKOM edit."
  local)

(def-kom-var lyskom-edit-prefix nil
  "Mode map for LysKOM edit mode.")

(def-kom-var lyskom-customize-map nil
  "Keymap for the customize buffer"
  local)

(def-kom-var lyskom-command-alternatives nil
  "Possible command completions."
  local
  minibuffer)


;;; =================================================================
;;;
;;; Language-dependent variables
;;;

(def-kom-var lyskom-month-names nil
  "A list of month names.
Each element is a cons cell consisting of the name of the month
\(a symbol) and the number of the month (1-12). Each month may
appear more than once, but the first occurence should be the
preferred name of the month."
  local
  inherited
  language-force)

(def-kom-var lyskom-help-data nil
  "Help strings."
  local
  language-force)

(def-kom-var lyskom-onoff-table nil
  "A completion table for on and off selections."
  local
  language-force)

(def-kom-var lyskom-move-tree-actions nil
  "A completion table for actions in kom-move-text-tree"
  local
  language-force)

(def-kom-var lyskom-language-codes nil
  "A list of ISO 639 language codes"
  local
  language-force)

(def-kom-var lyskom-filter-predicate-list nil
      "A list of legal filter comparison predicates."
      local
      language-force)

(def-kom-var lyskom-filter-what nil
      "A list of legal filter conditions and their textual representation."
      local
      language-force)

(def-kom-var lyskom-filter-actions nil
      "A list of legal filter actions an their textual representation."
      local
      language-force)

(def-kom-var lyskom-filter-edit-map nil
  "Keymap for LysKOM filter edit."
  local)

(def-kom-var lyskom-prioritize-mode-map nil
  "Keymap used in lyskom-prioritize-mode."
  local)

(def-kom-var kom-ansaphone-default-reply nil
  "**Default message to send when automatic replies are on.

The value of this variable must be a string or `nil'. When set to
`nil', no automatic replies are generated. See `kom-ansaphone-replies'
for a more flexible way of specifying automatic replies.

Values other than strings ans `nil' are reserved for future use."
  server)

(def-kom-var kom-ispell-dictionary nil
  "**Dictionary to use for spell checking.

When spell checking is used when writing texts, this variable can be
used to specify an alternate dictionary. When set to `nil', use the
default dictionary. When set, it should be set to a string indicating
the alternate dictionary (see `ispell-dictionary').

Values other than `nil' and strings are reserved for future use."
  server
  inherited)

(def-kom-var lyskom-unread-mode-line nil
  "This variable will become part of mode-line-format"
  language-force)

(def-kom-var lyskom-unread-title-format nil
  "This variable will become part of frame-title-format"
  language-force)

(def-kom-var lyskom-button-actions
  '((text
     text-popup-title
     lyskom-button-view-text
     ((lyskom-button-view-text-action . lyskom-button-view-text)
      (lyskom-button-unread-text-action . lyskom-button-unread-text)
      (lyskom-button-copy-text-no-action . lyskom-button-copy-text-no)
      (lyskom-button-review-noconversion-action . lyskom-button-review-noconversion)
      (lyskom-button-review-converted-action . lyskom-button-review-converted)
      (lyskom-button-review-rot13-action . lyskom-button-review-rot13)
      (lyskom-button-find-root-review-action . lyskom-button-find-root-review)
      (lyskom-button-find-root-action . lyskom-button-find-root)
      (lyskom-button-review-comments-action . lyskom-button-review-comments)
      (lyskom-button-review-tree-action . lyskom-button-review-tree)
      (lyskom-button-comment-text-action . lyskom-button-comment-text)
      (lyskom-button-private-comment-text-action . lyskom-button-private-comment-text)
      (lyskom-button-write-footnote-action . lyskom-button-write-footnote)
      (lyskom-button-fast-reply-action . lyskom-button-fast-reply)
      (lyskom-button-mark-text-action . lyskom-button-mark-text)
      (lyskom-button-unmark-text-action . lyskom-button-unmark-text)
      (lyskom-button-save-text-action . lyskom-button-save-text)
      (lyskom-button-save-text-body-action . lyskom-button-save-text-body)
      )
     nil
     ;; ((nil lyskom-print-text footer lyskom-button-comment-text))
     )
    (conf 
     conf-popup-title
     lyskom-button-view-conf-presentation
     ((lyskom-button-view-conf-presentation-action . lyskom-button-view-conf-presentation)
      (lyskom-button-view-conf-status-action . lyskom-button-view-conf-status)
      (lyskom-button-goto-conf-action . lyskom-button-goto-conf)
      (lyskom-button-send-message-action . lyskom-button-send-message)
      (lyskom-button-add-self-action . lyskom-button-add-self)
      (lyskom-button-sub-self-action . lyskom-button-sub-self))
     ((kom-list-news . lyskom-button-goto-conf)
      (kom-membership . lyskom-button-goto-conf)))
    (pers 
     pers-popup-title
     lyskom-button-view-pers-presentation
     ((lyskom-button-view-pers-presentation-action . lyskom-button-view-pers-presentation)
      (lyskom-button-view-pers-status-action . lyskom-button-view-pers-status)
      (lyskom-button-goto-conf-action . lyskom-button-goto-conf)
      (lyskom-button-view-session-status-action . lyskom-button-view-session-status)
      (lyskom-button-mail-action . lyskom-button-mail)
      (lyskom-button-send-message-action . lyskom-button-send-message)
      (lyskom-button-moronify-action . lyskom-button-moronify)
      (lyskom-button-befriend-action . lyskom-button-befriend))
     ; Hints
     ((kom-list-news . lyskom-button-goto-conf)
      (kom-membership . lyskom-button-goto-conf))
     ; Filter
     (lambda (item arg) 
       (cond ((not (numberp arg)) t)
             ((eq 'lyskom-button-goto-conf item)
              (or (eq arg lyskom-pers-no) (lyskom-is-supervisor arg lyskom-pers-no)))
             (t t)))
     )
    (url 
     url-popup-title
     lyskom-button-open-url
     ((lyskom-button-open-url-action . lyskom-button-open-url)
      (lyskom-button-copy-url-action . lyskom-button-copy-url))
     nil)
    (info-node 
     generic-popup-title
     lyskom-button-goto-info-node
     ((lyskom-button-goto-info-node-action . lyskom-button-goto-info-node))
     nil)
    (email
     generic-popup-title
     lyskom-button-open-email
     ((lyskom-button-open-email-action . lyskom-button-open-email)
      (lyskom-button-copy-email-action . lyskom-button-copy-email))
     nil)
    (aux
     aux-popup-title
     lyskom-button-info-aux
     ((lyskom-button-info-aux-action . lyskom-button-info-aux)
      (lyskom-button-delete-aux-action . lyskom-button-delete-aux))
     nil)
    (aux-edit-menu
     nil
     nil
     ((lyskom-edit-toggle-secret-aux-action . lyskom-edit-toggle-secret-aux)
      (lyskom-edit-toggle-anonymous-aux-action . lyskom-edit-toggle-anonymous-aux)
      (lyskom-edit-toggle-inherit-aux-action . lyskom-edit-toggle-inherit-aux)
      (lyskom-edit-delete-aux-action . lyskom-edit-delete-aux))
     nil)
    (prioritize-flag-menu
     nil
     lyskom-prioritize-flag-toggle
     ((lyskom-prioritize-flag-toggle-action . lyskom-prioritize-flag-toggle)
      (lyskom-prioritize-flag-set-action . lyskom-prioritize-flag-set)
      (lyskom-prioritize-flag-clear-action . lyskom-prioritize-flag-clear))
     nil)
    (func
     nil
     lyskom-button-apply
     nil
     nil)
    (timestamp
     timestamp-popup-title
     (lambda (buffer argument text) nil)
     ((lyskom-button-copy-timestamp-action . lyskom-button-copy-timestamp))
     nil)
    (recpt-type
     recpt-type-popup-title
     (lambda (buffer argument text) nil)
     ((lyskom-button-recpt-type-recipient
       . (lambda (buffer recpt-and-buffer text)
	   (lyskom-edit-do-add-recipient/copy
	    'RECPT (car recpt-and-buffer)
	    (car (cdr recpt-and-buffer)))))
      (lyskom-button-recpt-type-copy
       . (lambda (buffer recpt-and-buffer text)
	   (lyskom-edit-do-add-recipient/copy
	    'CC-RECPT (car recpt-and-buffer)
	    (car (cdr recpt-and-buffer)))))
      (lyskom-button-recpt-type-bcc
       . (lambda (buffer recpt-and-buffer text)
	   (lyskom-edit-do-add-recipient/copy
	    'BCC-RECPT (car recpt-and-buffer)
	    (car (cdr recpt-and-buffer)))))
      (lyskom-button-recpt-type-sub
       . (lambda (buffer recpt-and-buffer text)
	   (lyskom-edit-sub-recipient/copy
	    (car recpt-and-buffer)
	    (car (cdr recpt-and-buffer)))))))
    (add-recipient-or-xref
     add-recipient-or-xref
     nil
     ((lyskom-button-recpt-add-recipient
       . (lambda (buffer buffer text)
           (set-buffer buffer)
	   (kom-edit-add-recipient)))
      (lyskom-button-recpt-add-copy
       . (lambda (buffer buffer text)
           (set-buffer buffer)
	   (kom-edit-add-copy)))
      (lyskom-button-recpt-add-bcc
       . (lambda (buffer buffer text)
           (set-buffer buffer)
	   (kom-edit-add-bcc)))
      (lyskom-button-aux-type-xref
       . (lambda (buffer recpt-and-buffer text)
           (save-excursion
             (set-buffer recpt-and-buffer)
             (kom-edit-add-cross-reference))))
      (lyskom-button-aux-type-no-comments
       . (lambda (buffer recpt-and-buffer text)
           (save-excursion
             (set-buffer recpt-and-buffer)
             (kom-edit-add-no-comments))))
      (lyskom-button-aux-type-personal-comments
       . (lambda (buffer recpt-and-buffer text)
           (save-excursion
             (set-buffer recpt-and-buffer)
             (kom-edit-add-personal-comments))))))
    (view-pdf
     nil
     (lambda (buffer argument text) (lyskom-doc-view-text argument))
     nil
     nil)
    )
  "This variable defines valid button types in LysKOM. Each element is a
list consisting of (TYPE LABEL DEFAULT ACTIONS HINTS).
TYPE is the button type the entry defines
LABEL is a textual representation for the button type, used in menu titles. If
      it is a symbol, that symbol will be looked up using lyskom-get-string.
DEFAULT is the default action to take on a click. It must be a function.
ACTIONS are other possible actions. The format of this entry is described 
        below.
HINTS is a list of hints to override the default action. This is described 
      below.

The ACTIONS entry is used to construct a pop-up menu. It is a list consisting 
of lists with the format (STRING . FUNCTION). STRING is the menu label and
FUNCTION is the function to call when the menu item is selected.

The HINTS entry is used to generate hints that the default action should be 
overridden. It is a list containing elements (COMMAND . HINT) where COMMAND is
as interactive LysKOM command and HINT is a function to call. When a button
is generated while the command COMMAND is being executed, HINT is used as a 
hint for a new default action. The user has the option to ignore or used the 
hint.

Also see the function \"lyskom-add-button-action\"."
  local
  inherited)

(def-kom-var kom-show-imported-envelope-sender t
  "**Controls display of envelope sender of imported e-mails.

When set to `t', dispaly the envelope sender of imported e-mails (if
the importer has recorded such information).

`kom-show-imported-envelope-sender' set to `t':

<
  2912231 today 10:07 +0100 /41 lines/  Super-User <root@spareribs.meat.com>
  Sent by: root@meat.spareribs.com
  Imported: today 00:07 by Mailman
  External recipient: BBQ <531@kom.meat.com>
>

`kom-show-imported-envelope-sender' set to `nil':

<
  2912231 today 10:07 +0100 /41 lines/  Super-User <root@spareribs.meat.com>
  Imported: today 00:07 by Mailman
  External recipient: BBQ <531@kom.meat.com>
>

Other variables that control display of imported e-mail include
`kom-show-imported-importer' and
`kom-show-imported-external-recipients'.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-show-imported-importer t
  "**Controls display of the importer name in imported mail.

When set to `t', display information about what imported the e-mail
\(if the importer has recorded such information). When set to `nil',
don't.

`kom-show-imported-importer' set to `t':

<
  2912231 today 10:07 +0100 /41 lines/  Super-User <root@spareribs.meat.com>
  Imported: today 00:07 by Mailman
  External recipient: BBQ <531@kom.meat.com>
>

`kom-show-imported-importer' set to `nil':

<
  2912231 today 10:07 +0100 /41 lines/  Super-User <root@spareribs.meat.com>
  External recipient: BBQ <531@kom.meat.com>
>

Other variables that control display of imported e-mail include
`kom-show-imported-importer' and
`kom-show-imported-external-recipients'.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-show-imported-message-id nil
  "**Controls display of the message id in imported mail.

When set to `t', display the message id of the imported e-mail. When
set to `nil', don't.

`kom-show-imported-message-id' set to `t':

<
  2912231 today 10:07 +0100 /41 lines/  Super-User <root@spareribs.meat.com>
  Message-ID: <D1696C471C6CD511A0BE00D0B7A932DE01BBB4FA@spareribs.meat.com>
  External recipient: BBQ <531@kom.meat.com>
>

`kom-show-imported-message-id' set to `nil':

<
  2912231 today 10:07 +0100 /41 lines/  Super-User <root@spareribs.meat.com>
  External recipient: BBQ <531@kom.meat.com>
>

Other variables that control display of imported e-mail include
`kom-show-imported-envelope-sender', `kom-show-imported-importer' and
`kom-show-imported-external-recipients'.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-show-imported-external-recipients t
  "**Controls display of external recipients of imported e-mail.

When set to `t', display all external recipients of each imported
e-mail (the to and cc headers). When set to `nil', don't display these
headers.

`kom-show-imported-external-recipients' set to `t':

<
  2912231 today 10:07 +0100 /41 lines/  Super-User <root@spareribs.meat.com>
  Imported: today 00:07 by Mailman
  External recipient: BBQ <531@kom.meat.com>
  External recipient: \"Mac the Hack\" <mac.hack@steak.meat.com>
>

`kom-show-imported-external-recipients' set to `nil':

<
  2912231 today 10:07 +0100 /41 lines/  Super-User <root@spareribs.meat.com>
  Imported: today 00:07 by Mailman
>

Other variables that control display of imported e-mail include
`kom-show-imported-envelope-sender', `kom-show-imported-importer' and
`kom-show-imported-message-id'.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-complete-numbers-before-names t
  "**Controls completion of numeric person and conference names.

When set to `t', reading conference and user names accepts the special
forms \"m 4711\" or \"p 42\" as numeric references to conference 4711 and
person 42 instead of trying to look for an object with a matching
name. If `nil', any name matching the input will be preferred to a
numeric reference.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-preferred-charsets '(iso-8859-1 utf-8)
  "**Character sets to try encoding texts with.

Thie setting may contain a list of valid MIME character sets. When
encoding a text to send to the server, the client will use the default
server character set whenever possible. If it is insufficient, it will
try the character sets listed in this setting, in order. If they are
also insufficient, it will attempt to find some other character set
that works.

WARNING: this setting only applies to Gnu Emacs. In XEmacs, customize
the variable mm-coding-system-priorities instead; you may have to load
Gnus in order to do so."
  server)


(def-kom-var kom-mercial nil
  "**What you are doing when you're done reading.

The value of this variable should be a string. This string is sent to
the server as the user's current activity when everything is read.

Users are encouraged to use their best sense of humor."
  server)


(defconst lyskom-commands-not-in-menu
  '(
    kom-unread-previous-commented-text
    kom-create-aux-item
    kom-become-nonanonymous
    kom-become-anonymous
    kom-send-alarm
    kom-get-abuse
    kom-get-appreciation
    kom-list-clients
    kom-view-previous-commented-text
    kom-membership
    kom-delete-text
    kom-quick-mode
    kom-slow-mode
    kom-busy-wait
    kom-comment-previous
    kom-private-answer-previous
    kom-change-language
    kom-delete-user-area
    )
  "Commands that are not supposed to appear in menus.")

(defconst lyskom-commands
  '(
    kom-help
    kom-slow-mode
    kom-quick-mode
    kom-send-message
    kom-moronify
    kom-befriend
    kom-create-conf
    kom-delete-conf
    kom-delete-text
    kom-display-time
    kom-go-to-conf
    kom-go-to-next-conf
    kom-jump
    kom-list-created-conferences
    kom-list-conferences 
    kom-list-persons
    kom-list-news
    kom-list-re
    kom-list-sessions
    kom-membership
    kom-list-marks
    kom-postpone
    kom-set-session-priority
    kom-prioritize
    kom-status-person
    kom-status-conf
    kom-add-self
    kom-change-priority
    kom-list-summary
    kom-sub-self
    kom-quit
    kom-recover 
    kom-start-anew
    kom-view
    kom-find-root-review
    kom-review-comments
    kom-review-tree
    kom-review-cross-references
    kom-review-clear
    kom-review-last-normally-read
    kom-review-noconversion
    kom-review-converted
    kom-review-rot13
    kom-review-next
    kom-find-root
    kom-review-by-to
    kom-review-roots-by-to
    kom-review-more
    kom-review-first
    kom-review-first-roots
    kom-review-all
    kom-review-all-roots
    kom-view-commented-text
    kom-view-previous-commented-text
    kom-review-stack
    kom-review-presentation
    kom-review-backward
    kom-view-next-text
    kom-who-is-on
    kom-who-is-on-in-conference
    kom-who-is-on-and-friend
    kom-who-am-i
    ;;   kom-display-who-buffer
    kom-list-clients
    kom-busy-wait
    kom-write-comment
    kom-comment-previous
    kom-write-footnote
    kom-private-answer
    kom-private-answer-previous
    kom-set-unread
    kom-write-text
    kom-send-letter
    kom-change-name
    kom-change-parenthesis
    kom-change-password
    kom-change-supervisor
    kom-change-presentation
    kom-get-appreciation
    kom-get-abuse
    kom-mark-text
    kom-unmark-text
    kom-review-marked-texts
    kom-review-all-marked-texts
    kom-add-recipient
    kom-add-copy
    kom-add-bcc
    kom-sub-recipient
    kom-move-text
    kom-move-text-tree
    kom-add-comment
    kom-sub-comment
    kom-move-comment
    kom-add-cross-reference
    kom-add-member
    kom-sub-member
    kom-change-conf-motd
    kom-set-garb-nice
    kom-set-super-conf
    kom-set-permitted-submitters
    kom-unset-conf-motd
    kom-save-text
    kom-save-text-body
    kom-save-options
    kom-shutdown-server
    kom-sync-database
    kom-enable-adm-caps
    kom-disable-adm-caps
    kom-set-motd
    kom-remove-motd
    kom-force-logout
    kom-filter-author
    kom-filter-subject
    kom-filter-text
    kom-filter-recipient
    kom-super-jump
    kom-filter-edit
    kom-list-filters
    kom-show-user-area
    kom-delete-user-area
    kom-change-conf-type

    kom-change-auto-reply
    kom-toggle-auto-reply
    kom-list-messages
    kom-erase-messages

    kom-remote-autoreply
    kom-remote-set-message
    kom-remote-list-messages
    kom-remote-erase-messages
    kom-remote-quit

    kom-status-session
    kom-customize
    kom-change-language
    kom-calculate

    kom-where-is

    kom-next-kom
    kom-previous-kom
    kom-next-unread-kom

    kom-send-alarm
    kom-agree
    kom-fast-reply
    kom-add-faq
    kom-del-faq
    kom-review-faq

    kom-add-footnote
    kom-sub-footnote

    kom-add-private-answer
    kom-add-no-comments
    kom-add-request-confirm

    kom-review-mail-headers

    kom-compare-texts
    kom-diff-texts

    kom-view-url-in-text

    kom-become-anonymous
    kom-become-nonanonymous

    kom-keep-alive
    kom-stop-keep-alive

    kom-who-is-present-in-conference
    kom-is-person-member-of-conference
    kom-change-conf-faq

    kom-make-review-mark-as-read
    kom-make-review-not-mark-as-read
    kom-set-presentation
    kom-set-motd-text
    kom-remove-presentation
    kom-create-aux-item
    kom-status-server
    kom-add-server-faq
    kom-del-server-faq
    kom-change-server-faq
    kom-review-server-faq
    kom-recommend-conference
    kom-redirect-comments
    kom-copy-options
    kom-mark-unread
    kom-unread-by-to
    kom-unread-roots-by-to
    kom-unread-last-normally-read
    kom-unread-root-review
    kom-unread-root
    kom-unread-tree
    kom-unread-comments
    kom-unread-previous-commented-text
    kom-unread-commented-text
    kom-unread-more
    kom-unread-all
    kom-unread-all-roots
    kom-unread-first
    kom-unread-first-roots
    kom-unread-all-marked-texts
    kom-unread-marked-texts
    kom-unread-faq
    kom-unread-server-faq
    kom-unread-presentation
    kom-join-all-conferences
    kom-leave-all-conferences
    kom-will-person-read-text
    kom-limit-import
    kom-change-message-flag
    kom-list-faqs
    kom-list-server-faqs
    kom-list-new-conferences
    kom-list-new-persons
    kom-change-privileges

    kom-review-more-comments
    ))

;;; ================================================================
;;;                  Internal variables and constants


(def-kom-var lyskom-settings-version lyskom-clientversion
  "Version of saved settings."
  server)

(defvar lyskom-max-int 8388607
  "The largest int Emacs, and thus this LysKOM client, can handle.")

(def-kom-var lyskom-server-uses-utc nil
  "When non-nil, assume that the server uses UTC timestamps."
  inherited
  local)

(defconst lyskom-server-features
  '((10 lyskom-bcc-flag
        lyskom-extended-types-flag)
    (9  lyskom-accept-async-flag
        lyskom-dynamic-session-info-flag
        lyskom-idle-time-flag)
    (8  lyskom-long-conf-types-flag 
        lyskom-set-last-read-flag 
        lyskom-uconf-stats-flag
        lyskom-set-last-read-flag)
    (7  lyskom-z-lookup-flag))
  "List describing which features a certain server has. Each
element is a list containing the protocol version and what
it supports. The format of each element is:

\(VERSION . SUPPORTS\)

Version is simply a protocol version. Protocol equal to or above the
version support the supports list.

SUPPORTS is a list of pairs and symbols. Cons pairs are treated as
arguments to setq, symbols are interpreted as variable names set
 to 't'.")

;;;(defconst lyskom-server-features
;;;  '(((>= 2 0 0) (lyskom-bcc-flag
;;;                 lyskom-aux-items-flag))
;;;    ((>= 1 9 0) (lyskom-accept-async-flag
;;;                 lyskom-dynamic-session-info-flag
;;;                 lyskom-idle-time-flag))
;;;    ((>= 1 8 0) (lyskom-long-conf-types-flag 
;;;                 lyskom-set-last-read-flag
;;;                 lyskom-uconf-stats-flag))
;;;    ((>= 1 7 0) (lyskom-z-lookup-flag))
;;;    ((= 2 0 0) ((protocol-version 10)))
;;;    ((= 1 9 0) ((protocol-version 9)))
;;;    ((= 1 8 0) ((protocol-version 8)))
;;;    ((= 1 7 0) ((protocol-version 7)))
;;;    ((= 1 7 1) ((protocol-version 7)))
;;;    ((< 1 7 0) ((protocol-version 6))))
;;;  "List describing which features a certain server version has.
;;;Each element is a list containing the server version and what it
;;;supports:
;;;")


(def-kom-var lyskom-server-version '(0 0 0)
  "The version of the server.  A list of three integers: major
version, minor version and revision."
  local)

(def-kom-var lyskom-server-coding-system 'iso-8859-1
  "The default coding system used by the server for all strings."
  inherited)

(def-kom-var lyskom-current-user-area 0
  "Text-no of the user area last saved or read for lyskom-pers-no."
  local)

(def-kom-var lyskom-max-packet-size lyskom-max-int
  "The largest possible packet size that can be transmitted to a
TCP/IP connection. This should be unlimited, but in practise there
are systems that limits this. This variable is automatically adjusted
if any problems are detected.")

(def-kom-var lyskom-pending-commands nil
  "Commands pending to be executed.
When a command finishes, it checks this variable to see if another command
should be run.

It should be a list where each element should be either a symbol or an
expression. If it is a symbol, it is invoked with `call-interactively', and
an expression is evaluated with `eval'."
  local)

(def-kom-var lyskom-do-when-done nil
  "Internal of kom-do-when-done."
  local)

(def-kom-var lyskom-do-when-starting nil
  "Internal of kom-do-when-starting. Obsolete.")

(def-kom-var lyskom-sessions-with-unread nil
  "List of LysKOM sessions with unread texts.
This is not buffer-local.")

(def-kom-var lyskom-sessions-with-unread-letters nil
  "List of LysKOM sessions with unread letters.
This is not buffer-local.")

(def-kom-var lyskom-session-has-unread-letters nil
  "Non-nil if this session has unread letters."
  local
  protected
  inherited)

(def-kom-var lyskom-session-has-unreads nil
  "Non-nil if this session has unread texts."
  local
  protected
  inherited)

(def-kom-var lyskom-buffer nil
  "The LysKOM buffer we are connected to."
  inherited
  minibuffer)

(def-kom-var lyskom-buffer-type nil
  "Type of the current buffer."
  local
  protected)

(def-kom-var lyskom-errno nil
  "Errno of last lyskom error."
  local)

(def-kom-var lyskom-err-stat nil
  "Err-stat of last lyskom error."
  local)

(def-kom-var lyskom-parser-recovering nil
  "Non-nil if the parser is recovering from an error."
  local)

(def-kom-var lyskom-parse-pos nil
  "Position of parsing.")

(def-kom-var lyskom-unparsed-buffer nil
  "Buffer containing unparsed information from the server."
  local)

(def-kom-var lyskom-unparsed-marker nil
  "Where we now are inserting."
  local)

(def-kom-var lyskom-to-be-printed-before-prompt nil
  "Contains the strings to be printed before the next prompt."
  local)

(def-kom-var lyskom-other-clients-user-areas nil
  "Contains the parts of the user areas of unknown clients.
The area is a pair: name . info (both strings)."
  local)

(def-kom-var lyskom-saved-unknown-variables nil
  "Variables read from the user area that this client version
knows nothing about and that will be saved back unaltered."
  local)

(def-kom-var lyskom-pending-calls nil
  "Assoc list of calls to LysKOM server that have not yet completed.
Each element on the list has the format

	(REF-NO . KOM-QUEUE)
REF-NO	   unique number assigned by lyskom-send-packet.
KOM-QUEUE is a kom-queue. (See lyskom-call-data.)"
  local)

(def-kom-var lyskom-output-queues nil
  "Pending output to the server.
This is a vector of ten elements, each of which is a kom-queue. Calls from
queues with a higher index (priority) are always sent first.

At most lyskom-max-pending-calls calls are sent at once."
  local)

(def-kom-var lyskom-max-pending-calls 20
  "Max number of calls that are transmitted to the server at once.
Extra calls are queued in lyskom-output-queue and sent when the replies
returns.

This variable is not saved in the LysKOM server.")

(def-kom-var lyskom-number-of-pending-calls 0
  "Number of pending calls that are transmitted to the server."
  local)

;; This variable is used to prevent "starvation" of the blocking-do call.
;; When there is heavy prefetch going on in the background and a
;; blocking-do call is made there is a good chance that the
;; accept-process-output call will not return within a reasonable
;; time, because there will always be data to read from the server,
;; which means that Emacs will call lyskom-filter instead of returning
;; from accept-process-output.
(defvar lyskom-ok-to-send-new-calls t
  "This variable controls whether calls are passed to the server.
If it is nil, all outgoing calls are inhibited.")

(def-kom-var lyskom-ref-no 0
  "Next ref-no to use. These ref-nos are used to keep track of the
different packets.")

(def-kom-var lyskom-pers-no 0
  "The pers-no of the current user."
  inherited)

(def-kom-var lyskom-session-no 0
  "Session number in the server for this connection."
  local)

(def-kom-var kom-default-session-priority 0
  "**The default session priority.

Tha value of this variable must be an integer. Only texts in
conferences with a priority equal to or higher than this will be shown
by default.

To set the session priority in a running session, use
`kom-set-session-priority'. Setting this value directly will have no
effect on the actual session priority.

Non-integer values are reserved for future use."
  local
  server)

(def-kom-var kom-server-priority -1
  "**The default server priority.

The value of this variable must be an integer.

When `kom-server-priority-breaks' is set to a non-nil value, this
priority is used to decide when to go to a prioritized session. For a
session to be prioritized its priority must be higher than the current
session's `kom-server-priority' and higher than the priority of
whatever conference is currently being read.

Non-integer values are reserved for future use."
  server)


(def-kom-var lyskom-session-priority 0
  "This sessions priority.
Only texts in conferences with a priority equal to or higher than this
will be shown."
  local)

(def-kom-var lyskom-proc nil
  "The process (network connection) that is associated with this buffer."
  inherited
  minibuffer)

(def-kom-var lyskom-server-info nil
  "Info about the server."
  local)

(def-kom-var lyskom-server-version-info nil
  "Version information about the client."
  local)

(def-kom-var lyskom-server-name ""
  "The name of the server."
  inherited)

(def-kom-var lyskom-server-port nil
  "The port we are connected to."
  local)

(def-kom-var lyskom-buffer-list nil
  "List of all LysKOM buffers.")

(def-kom-var lyskom-static-session-info-cache nil
  "Cache of session."
  local)

(def-kom-var lyskom-conf-cache nil
  "Cache of conference statuses."
  local)

(def-kom-var lyskom-static-server-info nil
  "Cache of static-server-info"
  local)

(def-kom-var lyskom-stats-description nil
  "Cache of get-stats-description"
  local)

(def-kom-var lyskom-uconf-cache nil
  "Cache of small conference statuses."
  local)

(def-kom-var lyskom-pers-cache nil
  "Cache of person statuses."
  local)

(def-kom-var lyskom-text-cache nil
  "Cache of text statuses."
  local)

(def-kom-var lyskom-text-mass-cache nil
  "Cache of texts."
  local)

(def-kom-var lyskom-marked-text-cache nil
  "Cache of marks of all texts the current user has marked. "
  local)

(def-kom-var lyskom-is-parsing t
  "True when parsing a result.
This is used to prevent parallel parsing since the parser is not reentrant."
  local)

(def-kom-var lyskom-string-bytes-missing 0
  "Number of bytes missing in the unparsed buffer when parsing a string.
Set when parsing a string and there were not enough bytes in the buffer
with the unparsed bytes. This variable is used to prevent reparsing before 
the string is complete.
This variable is buffer-local in the unparsed-buffer."
  local
  inherited)

(def-kom-var lyskom-last-viewed 0       ;
  "Position of the first char of the last line that the user has had
time to view. This is normally the pos of the first char of the prompt."
  local)

(def-kom-var lyskom-mode-map nil
  "Keymap used in LysKOM mode."
  local)


(def-kom-var lyskom-reading-list nil
  "List of articles to read in the current conference.
Each element is a read-info. Only one of the elements is of the type CONF.
This one is located last in the list (except for the elements of the type 
REVIEW, REVIEW-TREE or REVIEW-MARK).
When reading an article with comments a list of the comments is built
recursively if the flag kom-read-depth-first is non-nil.
This is to keep track of the reading order.
Articles can exist in several of the read-info elements. All unread articles
in the conference are always present in the CONF type entry in this list even
if also in other entries. (COMM-IN, FOOTN-IN)

Some powerful reviewing commands requires to construct a list of articles that
should be read. These use the type REVIEW. When reviewing trees and when
every viewed article is supposed to be followed by all its comments then the
type REVIEW-TREE is used.

The first element is a dummy."
  local)

(def-kom-var lyskom-to-do-list nil
  "List of conferences with unread texts.
Each element is a read-info. All have the type 'CONF and there is one for
every conference with unread articles that have already been prefetched.
The list is sorted in falling priority. 
When going to a conference the first element (the one with the highest
priority) is copied from this list to lyskom-reading-list.

The first element is a dummy."
  local)

(def-kom-var lyskom-quit-flag nil
  "A flag indicating if the filter was interrupted by C-g.
 It is set to the same value as quit-flag on filter exit.")

(def-kom-var lyskom-ignoring-async-list nil
  "A list of async messages we are currently ignoring.
Each element is a list. The car of the list is the message and the
remaining elements are whatever is suitable for that type of message.
See the checks in lyskom-parse-async for details."
  local)

(def-kom-var lyskom-inhibit-minibuffer-messages nil
  "A flag indicating whether asynchronous minibuffer messages are allowed.
If this variable is non-nil, no asynchronous messages will appear.")

(def-kom-var lyskom-is-saving nil
  "A flag indicating whether the server is saving at the moment.")

;;; These variables control prefetch of conf-stats, text-stats and texts:

(def-kom-var lyskom-prefetch-conf-tresh 50
  "If fewer than lyskom-prefetch-conf-tresh texts are known, ask for more
conf-stats from server.

This is currently not used."
  local)

(def-kom-var lyskom-prefetch-confs 10
  "Number of confs to ask about at once when checking for unread texts.

This is currently not used."
  local)

(def-kom-var lyskom-fetch-map-nos 100
  "Number of text-nos lyskom will fetch when fetching maps."
  local)

(def-kom-var lyskom-fetch-membership-length 100
  "Number of entries in the membership-list that is fetched at a time.")

(def-kom-var lyskom-prefetch-limit 10
  "Number of prefetch requests the client will try to keep going
at a time.")


;;;

(def-kom-var lyskom-unread-confs nil
  "List containing all unread confs."
  local)

(def-kom-var lyskom-dont-change-prompt nil
  "Non-nil during the entry of a text."
  local)

(def-kom-var lyskom-command-to-do 'unknown
  "Atom describing what command to do. See the function lyskom-what-to-do."
  local)

(def-kom-var lyskom-is-waiting nil
  "If non-nil, this is the condition for the waiting to be stopped.
If t, however, it means that the user is waiting for a text with a prompt.
It is a form that will be evaluated (using eval) every time the asynchronous
message \"new text\" is received.

This is used by the command kom-busy-wait."
  local)

(def-kom-var lyskom-current-conf 0
  "Current conference. 0 means user is not reading any conf."
  local)

(def-kom-var lyskom-current-text nil
  "Text-no of current text. nil means no text is current."
  local)

(def-kom-var lyskom-last-written nil
  "Text-no of last text written. nil means no text written."
  local)

(def-kom-var lyskom-last-seen-written nil
  "Text-no of last text read or written by the current user.
When a new text is written, this is set to the text number of that text.
When a text is read that was written by the current user, this is
set to that text."
  local)

(def-kom-var lyskom-previous-text nil
  "Text-no of previous text. Nil means no text."
  local)

(def-kom-var lyskom-normally-read-texts nil
  "Stack of texts that are read normally. Used for kom-review-last-normally-read."
  local)

(def-kom-var lyskom-current-subject ""
  "Current subject."
  local)

(def-kom-var lyskom-current-user-area nil
  "Text-no of user area that current settings come from."
  local)

(def-kom-var kom-saved-file-name (concat default-directory "kom-text")
  "**The default filename when saving a LysKOM text.

This file name is used by `kom-save-text' as the default filename
until a new filename has been chosen."
  server)

(def-kom-var lyskom-saved-file-name nil
  "After saving once, the default file name when saving a LysKOM text."
  local)

(def-kom-var lyskom-mode-hook nil
  "*'Hook to run when `lyskom-mode' is entered.")

(def-kom-var kom-quit-hook nil
  "**Hook to run when the LysKOM session is ended.

This hook is only run when the session ends by using `kom-quit' or
`kom-remote-quit'. The hook is not run is the session is forecfully
terminated."
  server-hook)

(def-kom-var kom-quit-when-idle t
  "Controls automatic quitting when LysKOM is full.

When set to `t', automatically end the session when LysKOM is full and
the session has been idle for a predetermined amount of time.

This is currently not implemented.")

(def-kom-var kom-permanent-filter-list nil
  "List of patterns to filter permanently."
  server
  non-portable)

(def-kom-var kom-session-filter-list nil
  "List of patterns to filter during this session."
  local)

(def-kom-var lyskom-text-no-prompts-defaults
  '(kom-delete-text
    kom-view
    kom-write-footnote
    kom-mark-text
    kom-unmark-text
    kom-add-recipient
    kom-add-copy
    kom-add-bcc
    kom-sub-recipient
    kom-move-text
    kom-move-text-tree
    kom-add-comment
    kom-sub-comment
    kom-add-cross-reference
    kom-save-text-body
    kom-add-footnote
    kom-sub-footnote
    kom-add-faq
    kom-add-no-comments
    kom-add-private-answer
    kom-add-request-confirm
    kom-add-server-faq
    kom-mark-unread
    kom-set-presentation
    kom-set-motd-text
    kom-will-person-read-text
    kom-compare-texts
    kom-diff-texts
    )
  "Commands that prompt for a text number rather than assume a default."
  inherited)

(def-kom-var kom-text-no-prompts nil
  "**Determines which commands ask for text numbers.

The value of this variable must be a list containing pairs of 
(`COMMAND' . `VALUE'). If `VALUE' is `t', `COMMAND' will prompt for text
numbers. If `VALUE' is nil, `COMMAND' will use a suitable default value."
  server
  inherited)


(def-kom-var lyskom-filter-list nil
  "List of patterns that are filtered."
  local)

(def-kom-var kom-create-text-hook nil
  "**Hook to run before creating a new text.
This hook is run just before the server call to create the text is made.

The hook is currently called with the following arguments:

  Argument          Contents
  ----------------------------------------------------------------
  `MESSAGE'         The message text

  `MISC-LIST'       The misc-info list

  `AUX-LIST'        The aux-item list

  `BUFFER'          The edit buffer

  `IS-ANONYMOUS'    Non-nil if the user is currently anonymous.
  ----------------------------------------------------------------

Additional arguments may be added in the future, so include 
`&rest reserved' as the last element of the parameter list.

The hook can change the message by modifying the variable
`full-message', the misc-info list by modifying `misc-list' and the
aux-item list by modifying `aux-list'. This is not encouraged and may
break in the future."
  local-hook)

(def-kom-var kom-new-text-hook nil
  "**Hook to run when a new text is created.

This hook is run after the prompt is removed if it shall be changed
but before the text \"Text 4711 created\" is printed in the message
area and before the new prompt is printed."
  local-hook)

(def-kom-var kom-deleted-text-hook nil
  "**Hook to run when a text is deleted.

This hook is run after the prompt is removed if it shall be changed but 
before the new prompt is printed."
  local-hook)

(def-kom-var kom-new-recipient-hook nil
  "**Hook to run when a text receives a new recipient.

This hook is run after the prompt is removed if it shall be changed
but before the new prompt is printed. It may not run if the text has
been marked as read in any conference other than the person's
letterbox."
  local-hook)

(def-kom-var kom-personal-message-hook nil
  "**Hook to run when a personal message is received.

When the hook is run, `sender' is bound to the conf-stat of the sender
of the message (or possibly `nil'), `recipient' is 0 if the message is
an alarm message and otherwise the conf-stat of the recipient, and
`message' is a string that holds the message."
  local-hook)

(def-kom-var lyskom-executing-command t
  "Non-nil means the client is executing a command.
Most commands can't be interrupted by another command."
  local)

(def-kom-var lyskom-current-command nil
  "The command currently being executed."
  local)

(def-kom-var lyskom-command-point nil
  "The point where the most recent command was started."
  local)

(def-kom-var lyskom-current-function nil
  "Sometimes set to the current high-level function being executed."
  local)

(def-kom-var lyskom-current-function-phase nil
  "Sometimes set to the phase of the current high-level function being
executed."
  local)

(def-kom-var lyskom-membership-is-read nil
  "t when the membership has been read."
  local)

(def-kom-var lyskom-is-writing nil
  "t when the user is writing a text."
  local)

(def-kom-var lyskom-debug-communications-to-buffer nil
  "Non-nil means all communications with the server is stored in a buffer.
The name is stored in lyskom-debug-communications-to-buffer-buffer.")

(def-kom-var lyskom-debug-communications-limit 60000
  "When set to an integer limits the communications log to that many
bytes. When set to a non-integer, the communications log is unlimited.
If a protocol error is detected, this limit will be reset to nil.

See lyskom-debug-communications-to-buffer")

(def-kom-var lyskom-backtrace-list nil
  "List containing debugging information.")
  
(def-kom-var lyskom-debug-what-i-am-doing t
  "Non-nil means asynchronous message 5 will be logged to the debug
  buffer. ")

(def-kom-var lyskom-is-anonymous nil
  "Non-nil means be a bit secretive about things. Not totally
  secretive of course, since the server doesn't allow that yet."
  local)

(def-kom-var lyskom-debug-communications-to-buffer-buffer "*kom*-debugs"
  "Name of the buffer to insert the communications with the server into if
lyskom-debug-communications-to-buffer is non-nil.")

(def-kom-var lyskom-doing-default-command nil
  "Non-nil if LysKOM is executing the default command."
  local)

(def-kom-var lyskom-first-time-around nil
  "Non-nil if LysKOM is being entered for the first time."
  local)

(def-kom-var lyskom-experimental-features nil
  "If non-nil, LysKOM is likely to blow up in your face."
  local)

(def-kom-var lyskom-format-experimental nil
  "If non-nil, LysKOM is likely to make a fool out of you."
  local)

(def-kom-var lyskom-count-var 0
  "This variable is used for counting things in the client, such as
unread texts in list-unread."
  local)

(def-kom-var lyskom-default-conf-string nil
  "The default string to use for an unknown conference.
Set this locally when inserting a conference name using
lyskom-format-insert if you want to replace the usual description of
an unknown conference.")

(def-kom-var lyskom-default-pers-string nil
  "The default string to use for an unknown person.
Set this locally when inserting a conference name using
lyskom-format-insert if you want to replace the usual description of
an unknown person.")


(def-kom-var lyskom-is-administrator nil
  "This variable is t if the user is in administrator mode and nil otherwise."
  local
  minibuffer)

(def-kom-var lyskom-last-personal-message-sender nil
  "Name of sender of last personal message received."
  local)

(def-kom-var lyskom-last-group-message-recipient nil
  "Name of target for last group message received."
  local)

(def-kom-var lyskom-last-message-recipient nil
  "Number of last async message recipient sent to."
  local)

(def-kom-var lyskom-is-new-user nil
  "An internal variable used in kom-start-anew.")

(def-kom-var lyskom-apo-timeout-s 1
  "Seconds timeout for accept-process-output.")

(def-kom-var lyskom-apo-timeout-ms nil
  "Microseconds timeout for accept-process-output.")

(def-kom-var lyskom-collate-table nil
  "Table mapping characters to an equivalence class."
  inherited)

(def-kom-var lyskom-char-classes nil
  "An assoc list from character to a list of equivalent strings.
See lyskom-compute-char-classes."
  inherited)

(def-kom-var lyskom-dont-read-user-area nil
  "If non-nil the user area will not be read on login."
  local)

(def-kom-var lyskom-allow-missing-subject nil
  "If non-nil allow texts without subjects.")

(def-kom-var kom-w3-simplify-body t
  "**Control stripping of color information from HTML.

When this variable is set to `t', strip color information from body
tags in HTML passed to w3. This is generally a good idea. When `nil',
don't strip color information from body tags. This is generally a bad
idea.

Values other than `t' and `nil' are reserved for future use."
  server)

(def-kom-var kom-format-show-images t 
  "**Controls in-line display of images.

When set to `t', attempt to display images in-line. When set to `nil', 
images are handled as normal texts.

Values other than `t' and `nil' are reserved for future use."
  server)


(def-kom-var kom-format-html-authors '((t . t))
  "**Determines from which authors we accept HTML.

If non-nil, LysKOM will attempt to format texts marked as HTML.

The value is treated as an `indirect association list'. When
reading a text he author of a text is matched against the list,
and if the value of the match is non-nil, the HTML formatting may
be attempted. The matching rules are as follows:

  Element           Meaning
  ------------------------------------------------------------------
  `ATOM'            Matches with the default value if ATOM equals 
                    the author number (this means that you can set 
                    this variable to a list of person numbers).

  `(t . VAL)'       Matches with value VAL if no other match is 
                    found (i.e. sets the default value).

  `(FN . VAL)'      Matches with value VAL if FN is a function that
                    returns non-nil when called with the author
                    number as its sole argument.

  `(SYM . VAL)'     If SYM is bound to a list, match the author
                    number against that list using these rules. 
                    VAL is used as the default value for the 
                    matching process.

  `(SYM . VAL)'     If SYM is not bound to a list, matches with 
                    the value VAL if the value of SYM equals the
                    author number.

  `(LIST . VAL)'    Match the person number against LIST using
                    these rules. VAL is used as the default value.

  `(ATOM . VAL)'    Match with value VAL if ATOM equals the 
                    author number.
  ------------------------------------------------------------------

For example, the value `((5 . nil) (kom-friends . t))' will
prevent texts by person 5 from being shown as formatted HTML,
while texts by persons listed in the variable `kom-friends' can
be shown as formatted HTML.

Values other than those listed are reserved for future use." 
  server
  copy-transition lyskom-copy-indirect-assq)

(def-kom-var lyskom-format-special 
  '(("html"                . (lyskom-format-html-w3m lyskom-format-html-w3 lyskom-format-html-plaintext))
    ("enriched"            . lyskom-format-enriched)
    ("^image/"             . lyskom-format-image)
    ("^text/html"          . (lyskom-format-html-w3m lyskom-format-html-w3 lyskom-format-html-plaintext))
    ("^text/enriched"      . lyskom-format-enriched)
    ("^text/"              . lyskom-format-plaintext)
    ("^x-kom/text"         . lyskom-format-plaintext)  ;Archaic alias for text/x-kom-basic.
    ("^x-kom/basic"        . lyskom-format-plaintext)  ;Archaic alias for text/x-kom-basic.
    ("^application/pdf"    . lyskom-format-pdf)
    ("^x-kom/user-area"    . lyskom-format-x-kom/user-area)
    ("^x-kom/\\."      . lyskom-format-)
    ("^audio/"		   . lyskom-format-audio)
    ("^video/"		   . lyskom-format-video)
    ("^multipart/"	   . lyskom-format-multipart)
    ("^message/rfc822"	   . lyskom-format-plaintext)
    ("^message/delivery-status" . lyskom-format-plaintext)
    ("^message/disposition-notification" . lyskom-format-plaintext)
    ("^message/news"       . lyskom-format-plaintext)
    ("^message/"	   . lyskom-format-message)
    ("^model/"		   . lyskom-format-model)
    ("^application/"	   . lyskom-format-application)
    (""	                   . lyskom-format-unknown)
    )
  
  "AList of (FORMAT . FUNCTION) specifying functions that format texts
of that type. FORMAT is a symbol and FUNCTION is a function taking one
argument and returning a formatted string.")

(def-kom-var kom-respect-ancient-content-types nil
  "**When set to non-nil, LysKOM respects an ancient method for specifying
the content type of texts (in the first line of the text body). Should be
set to nil for most users."
  server)

(def-kom-var lyskom-send-text-transform-hook nil
  "Functions to call to transform text before sending it to the server.

When each function is run, lyskom-edit-text is bound to the text mass,
lyskom-edit-subject is bound to the subject, lyskom-edit-aux-list is
bound to the list of aux-items, and lyskom-edit-misc-list is bound to
the list of misc items. Functions may change these variables.

The first function to return a non-nil value will be the last function
to run.")

(def-kom-var lyskom-slow-mode nil
  "Non-nil when in slow mode."
  local)

(def-kom-var lyskom-saved-read-only nil
  "Saved value of buffer-read-only when in slow mode."
  local)

(def-kom-var lyskom-read-conf-saved-inputs nil
  "Saved inputs from lyskom-read-conf."
  local)


(defvar lyskom-line-start-chars-string
  "\"$&'()*+-./0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ[\\]_`abcdefghijklmnopqrstuvwxyz"
  "Characters that may start a line in a paragraph to be broken.")

(def-kom-var lyskom-line-start-chars nil
  "Computer-friendly version of lyskom-line-start-string.")


(def-kom-var lyskom-last-text-format-flags nil
  "List of flags specifying how the last text was reformatted. This variable
should be dynamically bound whenever it needs to be used.")

(def-kom-var lyskom-read-faqs nil
  "List of FAQs that have been read."
  inherited)

(def-kom-var lyskom-rejected-recommendations nil
  "List of invitations that have been rejected."
  inherited)

(def-kom-var lyskom-last-known-conf-no nil
  "Last known conference number."
  server
  non-portable)

(def-kom-var lyskom-last-known-pers-no nil
  "Last known conference number."
  server
  non-portable)

(defvar lyskom-xface-cache (make-vector 29 0))


;;; ======================================================================
;;; Event hooks
;;;

(def-kom-var lyskom-add-membership-hook nil
  "Functions to call when a membership is added."
  local-hook)

(def-kom-var lyskom-replace-membership-hook nil
  "Functions to call when a membership is replaced."
  local-hook)

(def-kom-var lyskom-remove-membership-hook nil
  "Functions to call when a membership is removed."
  local-hook)

(def-kom-var lyskom-new-membership-list-hook nil
  "Functions to call when the entire membership list is replaced."
  local-hook)


;;; ======================================================================
;;;; lyskom-tell-phrases-validation-keyword-list
;;; This is a list of keywords for kom-tell-phrases.
;;; These are the only keywords that are allowed in kom-tell-phrases.

;;; To coders of the elisp-client:
;;; If you add/delete a reference to any of these keywords make sure
;;; you update these changes.

;;; To everyone:
;;; The kom-tell-phrases list is checked against this list when the
;;; client is loaded, i.e. by lyskom-tell-phrases-validate that causes
;;; an error if any keyword is not present or any non-keyword is
;;; present.

(defvar kom-tell-phrases nil
  "**A list of phrases describing what the client is doing. 

Each element in the list is a pair \(`KEY' . `PHRASE') where KEY is one
of the keywords in `lyskom-tell-phrases-validation-keyword-list' and
`PHRASE' is the phrase to sent to the server then the client is doing
what `KEY' describes.

If the value of this variable is `nil', suitable defaults for the
currently selected language will be selected.")

(defconst lyskom-tell-phrases-validation-keyword-list
  '(
    (kom-tell-silence)
    (kom-tell-send)
    (kom-tell-login)
    (kom-tell-read)
    (kom-tell-1st-pres)
    (kom-tell-write-comment)
    (kom-tell-write-footnote)
    (kom-tell-write-letter)
    (kom-tell-write-reply)
    (kom-tell-write-text)
    (kom-tell-conf-pres)
    (kom-tell-recover)
    (kom-tell-wait)
    (kom-tell-regret)
    (kom-tell-review)
    (kom-tell-change-name)
    (kom-tell-change-supervisor)
    (kom-tell-next-lyskom)
    )
  "Users must not change this constant, but are encouraged to change
the value of  kom-tell-phrases for fun.")

;;; ================================================================
;;; Commands lists that are removed from extended command depending on
;;; administrator status.

(defconst lyskom-admin-removed-commands
  '(kom-enable-adm-caps))
(defconst lyskom-noadmin-removed-commands
  '(kom-disable-adm-caps
    kom-remove-motd
    kom-set-motd
    kom-shutdown-server
    kom-sync-database
    kom-add-server-faq 
    kom-del-server-faq
    kom-change-server-faq
    kom-recommend-conference
    kom-change-privileges))


;;; ================================================================
;;; Symbolic error codes

(defvar lyskom-error-codes
  '((no-error . 0)
    (not-implemented . 2)
    (obsolete-call . 3)
    (invalid-password . 4)
    (string-too-long . 5)
    (login-first . 6)
    (login-disallowed . 7)
    (conference-zero . 8)
    (undefined-conference . 9)
    (undefined-person . 10)
    (access-denied . 11)
    (permission-denied . 12)
    (not-member . 13)
    (no-such-text . 14)
    (text-zero . 15)
    (no-such-local-text . 16)
    (local-text-zero . 17)
    (bad-name . 18)
    (index-out-of-range . 19)
    (conference-exists . 20)
    (person-exists . 21)
    (secret-public . 22)
    (letterbox . 23)
    (ldb-error . 24)
    (illegal-misc . 25)
    (illegal-info-type . 26)
    (already-recipient . 27)
    (already-comment . 28)
    (already-footnote . 29)
    (not-recipient . 30)
    (not-comment . 31)
    (not-footnote . 32)
    (recipient-limit . 33)
    (comment-limit . 34)
    (footnote-limit . 35)
    (mark-limit . 36)
    (not-author . 37)
    (no-connect . 38)
    (out-of-memory . 39)
    (server-is-crazy . 40)
    (client-is-crazy . 41)
    (undefined-session . 42)
    (regexp-error . 43)
    (not-marked . 44)
    (temporary-failure . 45)
    (long-array . 46)
    (anonymous-rejected . 47)
    (illegal-aux-item . 48)
    (aux-item-permission . 49)
    (unknown-async . 50)
    (internal-error . 51)
    (feature-disabled . 52)
    (message-not-sent . 53)
    (invalid-membership-type . 54)
    (invalid-range . 55)
    (invalid-range-list . 56)
    (undefined-measurement . 57)
    (priority-denied . 59)
    (weight-zero . 60)
    (bad-bool . 61)
    )
  )

;;; ================================================================
;;;          Externally defined variables (environment)


(def-kom-var kom-default-server nil
  "**Default LysKOM server.

Use this server to log in to LysKOM. Can be overridden by the
`KOMSERVER' environment variable.

The value of this variable must be a string or `nil'."
  local)

(def-kom-var lyskom-default-server nil
  "**This variable is reserved for internal use. See 
`kom-default-server' instead." 
  local)

(def-kom-var kom-default-user-name nil
  "**Default LysKOM user name.

The value of this variable must be a string, indicating the
default user name, an integer indicating the conference number or 
`nil' to indicate no default. If set to an integer, log in as the
user with that conference number. If set to a string, log in as the 
user with that name.

Overrides the environment variable `KOMNAME'."
  local)

(def-kom-var kom-default-password nil
  "**Default LysKOM password.

The value of this variable must be a string or `nil'. Overrides the 
environment variable `KOMPASSWORD' \(which should never be used)."
  local)

(def-kom-var lyskom-default-password nil
  "**This variable is reserved for internal use. See 
`kom-default-password' instead."
  local)

(def-kom-var mode-line-conf-name nil
  "Conf name that is present on the mode-line."
  local)

(def-kom-var mode-line-server-name nil
  "Server name that is present on the mode-line."
  local)


;;; ============================================================
;;; History lists
;;;

(defvar lyskom-command-history nil)
(defvar lyskom-expression-history nil)
(defvar lyskom-message-history nil)
(defvar lyskom-language-history nil)
(defvar lyskom-fast-reply-history nil)
(defvar lyskom-help-history nil)


;;; ============================================================

(defconst lyskom-comment-types-list '(COMM-IN FOOTN-IN))
(defconst lyskom-recpt-types-list '(RECPT CC-RECPT BCC-RECPT)
  "Types of recipients in LysKOM.
This variable must be sorted in order of importance, with full recipients
first, followed by successively more and more minor recipients. The reason
for this is in kom-sub-recipient, which sorts the recipient list using
the order of this list.")
(defconst lyskom-review-types-list '(REVIEW REVIEW-TREE
                                            REVIEW-MARK REVIEW-FAQ
                                            REVIEW-FAQ-TREE))

;;; ============================================================


(eval-and-compile (provide 'lyskom-vars))

;;; vars.el ends here
