/*
 *  Definition of Lorene class Star_rot
 *				 
 */

/*
 *   Copyright (c) 2010 Eric Gourgoulhon 
 *
 *   This file is part of LORENE.
 *
 *   LORENE is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   LORENE is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with LORENE; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


#ifndef __STAR_ROT_H_ 
#define __STAR_ROT_H_ 

/*
 * $Id: star_rot.h,v 1.6 2015/05/19 09:30:55 j_novak Exp $
 * $Log: star_rot.h,v $
 * Revision 1.6  2015/05/19 09:30:55  j_novak
 * New methods for computing the area of the star and its mean radius.
 *
 * Revision 1.5  2014/10/13 08:52:36  j_novak
 * Lorene classes and functions now belong to the namespace Lorene.
 *
 * Revision 1.4  2010/02/08 10:56:30  j_novak
 * Added a few things missing for the reading from resulting file.
 *
 * Revision 1.3  2010/02/02 13:35:00  e_gourgoulhon
 * Remove the "under construction" mark.
 *
 * Revision 1.2  2010/01/25 18:14:05  e_gourgoulhon
 * Added member unsurc2
 * Added method is_relativistic()
 * Suppressed method f_eccentric
 *
 * Revision 1.1  2010/01/24 16:07:45  e_gourgoulhon
 * New class Star_rot.
 *
 *
 * $Header: /cvsroot/Lorene/C++/Include/star_rot.h,v 1.6 2015/05/19 09:30:55 j_novak Exp $
 *
 */

// Headers Lorene
#include "star.h"

namespace Lorene {
class Eos ;

			//--------------------------//
			//    class Star_rot        //
			//--------------------------//

/**
 * Class for isolated rotating stars. \ingroup (star)
 * 
 * The metric is
 * \f[
 *   ds^2 = - N^2 dt^2 + A^2 (dr^2 + r^2 d\theta^2)
 *		       + B^2 r^2 \sin^2\theta (d\varphi - N^\varphi dt)^2
 * \f]
 *
 * A star of class \c Star_rot  can be either relativistic or Newtonian, 
 * depending on the boolean indicator \c relativistic . For a Newtonian
 * star, the metric coefficients \e N, \e A, and \e B  are set to 1,  and \f$N^\varphi\f$ is
 * set to zero; the only relevant gravitational quantity in this case is
 * \c logn  which represents the Newtonian gravitational potential generated by the star. 
 * 
 * \version $Id: star_rot.h,v 1.6 2015/05/19 09:30:55 j_novak Exp $
 */
class Star_rot : public Star {

    // Data : 
    // -----
    protected:
	/** Indicator of relativity: \c true  for a relativistic star,
	 *	\c false  for a Newtonian one. 
	 */
	bool relativistic ;

	/** \f$1/c^2\f$ : \c unsurc2=1 for a relativistic star,
	 *  0 for a Newtonian one. 
	 */
	double unsurc2 ; 	     

	double omega ;	    ///< Rotation angular velocity (\c [f_unit] ) 

	/// Square of the metric factor \e A 
	Scalar a_car ; 

	/// Metric factor \e B 
	Scalar bbb ; 

	/// Square of the metric factor \e B 
	Scalar b_car ; 

	/// Metric coefficient \f$N^\varphi\f$
	Scalar nphi ; 

	/** Component \f$\tilde N^\varphi = N^\varphi r\sin\theta\f$ of the
	 *  shift vector
	 */
	Scalar tnphi ; 

	/// Norm of \c u_euler 
	Scalar uuu ;		
	
	/** Part of the Metric potential \f$\nu = \ln N\f$ = \c logn 
	 *  generated by the matter terms
	 */
	Scalar nuf ;	

	/** Part of the Metric potential \f$\nu = \ln N\f$ = \c logn 
	 *  generated by the quadratic terms
	 */
	Scalar nuq ;	

	/// Metric potential \f$\zeta = \ln(AN)\f$ 
	Scalar dzeta ;	

	/// Metric potential \f$\tilde G = (NB-1) r\sin\theta\f$
	Scalar tggg ; 

	/** Vector \f$W^i\f$ used in the decomposition of \c shift ,
	 *  following Shibata's prescription 
	 *  [\a Prog. \a Theor. \a Phys. \b 101 , 1199 (1999)] :
	 * \f[
	 *  N^i = {7\over 8} W^i - {1\over 8} 
	 *			\left(\nabla^i\chi+\nabla^iW^kx_k\right)
	 * \f]
	 * NB: \c w_shift  contains the components of \f$W^i\f$
	 *      with respect to the Cartesian triad associated with the 
	 *	mapping \c mp . 
	 */
	Vector w_shift ; 
	
	/** Scalar \f$\chi\f$ used in the decomposition of \c shift ,
	 *  following Shibata's prescription 
	 *  [\a Prog. \a Theor. \a Phys. \b 101 , 1199 (1999)] :
	 * \f[
	 *  N^i = {7\over 8} W^i - {1\over 8} 
	 *			\left(\nabla^i\chi+\nabla^iW^kx_k\right)
	 * \f]
	 */
	Scalar khi_shift ; 

	/** Tensor \f${\tilde K_{ij}}\f$ related to the extrinsic curvature
	 *  tensor by \f${\tilde K_{ij}} = B^{-2} K_{ij}\f$.
	 *  \c tkij  contains the Cartesian components of
	 *  \f${\tilde K_{ij}}\f$. 
	 */
	Sym_tensor tkij ; 

	/** Scalar \f$A^2 K_{ij} K^{ij}\f$.
	 *  For axisymmetric stars, this quantity is related to the 
	 *  derivatives of \f$N^\varphi\f$ by
	 * \f[
	 *	A^2 K_{ij} K^{ij} = {B^2 \over 2 N^2} \, r^2\sin^2\theta \,  
	 *    \left[ \left( {\partial N^\varphi \over \partial r} \right) ^2
	 *	    + {1\over r^2} \left( {\partial N^\varphi \over 
	 *		    \partial \theta} \right) ^2 \right] \ . 
	 * \f]
	 * In particular it is related to the quantities \f$k_1\f$ and \f$k_2\f$
	 * introduced by Eqs.~(3.7) and (3.8) of 
	 * Bonazzola et al. \a Astron. \a Astrophys. \b 278 , 421 (1993)
	 * by 
	 * \f[
	 *	A^2 K_{ij} K^{ij} = 2 A^2 (k_1^2 + k_2^2) \ . 
	 * \f]
	 */
	 Scalar ak_car ; 

	/** Effective source at the previous step for the resolution of 
	 *  the Poisson equation for \c nuf  by means of
	 *  \c Map_et::poisson .
	 */
	Scalar ssjm1_nuf ; 

	/** Effective source at the previous step for the resolution of 
	 *  the Poisson equation for \c nuq  by means of
	 *  \c Map_et::poisson .
	 */
	Scalar ssjm1_nuq ; 

	/** Effective source at the previous step for the resolution of 
	 *  the Poisson equation for \c dzeta .
	 */
	Scalar ssjm1_dzeta ; 

	/** Effective source at the previous step for the resolution of 
	 *  the Poisson equation for \c tggg .
	 */
	Scalar ssjm1_tggg ; 

	/** Effective source at the previous step for the resolution of 
	 *  the Poisson equation for the scalar \f$\chi\f$ by means of
	 *  \c Map_et::poisson . 
	 *  \f$\chi\f$ is an intermediate quantity for the resolution of the
	 *  elliptic equation for the shift vector \f$N^i\f$
	 */
	 Scalar ssjm1_khi ; 
	 
	/** Effective source at the previous step for the resolution of 
	 *  the vector Poisson equation for \f$W^i\f$.
	 *  \f$W^i\f$ is an intermediate quantity for the resolution of the
	 *  elliptic equation for the shift vector \f$N^i\f$
	 *  (Components with respect to the Cartesian triad associated with 
	 *   the mapping \c mp )
	 */
	 Vector ssjm1_wshift ; 
	 

    // Derived data : 
    // ------------
    protected:
	
	mutable double* p_angu_mom ;	///< Angular momentum 
	mutable double* p_tsw ;		///< Ratio T/W
	mutable double* p_grv2 ;	///< Error on the virial identity GRV2
	mutable double* p_grv3 ;	///< Error on the virial identity GRV3
	mutable double* p_r_circ ;	///< Circumferential radius
	mutable double* p_aplat ;	///< Flatening r_pole/r_eq
	mutable double* p_area ;        ///< Integrated surface area 
	mutable double* p_z_eqf ;	///< Forward redshift factor at equator
	mutable double* p_z_eqb ;	///< Backward redshift factor at equator
	mutable double* p_z_pole ;	///< Redshift factor at North pole
	mutable double* p_mom_quad ;	///< Quadrupole moment
	mutable double* p_r_isco ;	///< Circumferential radius of the ISCO
	mutable double* p_f_isco ;	///< Orbital frequency of the ISCO
	/// Specific energy of a particle on the ISCO 
	mutable double* p_espec_isco ;	
	/// Specific angular momentum of a particle on the ISCO
	mutable double* p_lspec_isco ;	
        mutable double* p_f_eq ;        ///< Orbital frequency at the equator
	
	 

    // Constructors - Destructor
    // -------------------------
    public:
	/** Standard constructor. 
	 * 
	 * @param mp_i Mapping on which the star is contructed
	 * @param nzet_i Number of domains occupied by the star
	 * @param relat \c true  for a relativistic
	 *			star,  \c false  for a Newtonian one
	 * @param eos_i Equation of state of the stellar matter
	 */
	Star_rot(Map& mp_i, int nzet_i, bool relat, const Eos& eos_i) ;			
	
	
	Star_rot(const Star_rot& ) ;		///< Copy constructor

	/** Constructor from a file (see \c sauve(FILE*) ). 
	 * 
	 * @param mp_i Mapping on which the star is constructed
	 * @param eos_i Equation of state of the stellar matter
	 * @param fich	input file (must have been created by the function
	 *	\c Star_rot::sauve )
	 */
	Star_rot(Map& mp_i, const Eos& eos_i, FILE* fich) ;    		

	virtual ~Star_rot() ;			///< Destructor


    // Memory management
    // -----------------
    protected:
	/// Deletes all the derived quantities
	virtual void del_deriv() const ; 
	
	/// Sets to \c 0x0  all the pointers on derived quantities
	virtual void set_der_0x0() const ; 

	/** Sets to \c ETATNONDEF  (undefined state) the hydrodynamical 
	 *  quantities relative to the Eulerian observer.
	 */
	virtual void del_hydro_euler() ; 
	

    // Mutators / assignment
    // ---------------------
    public:
	/// Assignment to another \c Star_rot 
	void operator=(const Star_rot& ) ;	
	
    // Accessors
    // ---------
    public:
	/** Returns \c true  for a relativistic star, \c false  for 
	 *  a Newtonian one
	 */
	bool is_relativistic() const {return relativistic; } ; 	

	/** Returns the central value of the rotation angular velocity 
	 *  (\c [f_unit] )
	 */ 
	virtual double get_omega_c() const ;	    

	/// Returns the metric factor \e B 
	const Scalar& get_bbb() const {return bbb;} ; 

	/// Returns the square of the metric factor \e A 
	const Scalar& get_a_car() const {return a_car;} ; 

	/// Returns the square of the metric factor \e B 
	const Scalar& get_b_car() const {return b_car;} ; 

	/// Returns the metric coefficient \f$N^\varphi\f$
	const Scalar& get_nphi() const {return nphi;} ; 

	/** Returns the component \f$\tilde N^\varphi = N^\varphi r\sin\theta\f$ 
	 *  of the shift vector
	 */
	const Scalar& get_tnphi() const {return tnphi;} ; 

	/// Returns the norm of \c u_euler 
	const Scalar& get_uuu() const {return uuu;} ;		
	
	/** Returns the part of the Metric potential \f$\nu = \ln N\f$ = \c logn 
	 *  generated by the matter terms
	 */
	const Scalar& get_nuf() const {return nuf;} ;	

	/** Returns the Part of the Metric potential \f$\nu = \ln N\f$ = \c logn 
	 *  generated by the quadratic terms
	 */
	const Scalar& get_nuq() const {return nuq;} ;	

	/// Returns the Metric potential \f$\zeta = \ln(AN)\f$ 
	const Scalar& get_dzeta() const {return dzeta;} ;	

	/// Returns the Metric potential \f$\tilde G = (NB-1) r\sin\theta\f$
	const Scalar& get_tggg() const {return tggg;} ; 

	/** Returns the vector \f$W^i\f$ used in the decomposition of 
	 *  \c shift ,
	 *  following Shibata's prescription 
	 *  [\a Prog. \a Theor. \a Phys. \b 101 , 1199 (1999)] :
	 * \f[
	 *  N^i = {7\over 8} W^i - {1\over 8} 
	 *			\left(\nabla^i\chi+\nabla^iW^kx_k\right)
	 * \f]
	 * NB: \c w_shift  contains the components of \f$W^i\f$
	 *      with respect to the Cartesian triad associated with the 
	 *	mapping \c mp . 
	 */
	const Vector& get_w_shift() const {return w_shift;} ; 
	
	/** Returns the scalar \f$\chi\f$ used in the decomposition of 
	 *  \c shift  
	 *  following Shibata's prescription 
	 *  [\a Prog. \a Theor. \a Phys. \b 101 , 1199 (1999)] :
	 * \f[
	 *  N^i = {7\over 8} W^i - {1\over 8} 
	 *			\left(\nabla^i\chi+\nabla^iW^kx_k\right)
	 * \f]
	 * NB: \c w_shift  contains the components of \f$W^i\f$
	 *      with respect to the Cartesian triad associated with the 
	 *	mapping \c mp . 
	 */
	const Scalar& get_khi_shift() const {return khi_shift;} ; 

	/** Returns the tensor \f${\tilde K_{ij}}\f$ related to the extrinsic 
	 *  curvature tensor by \f${\tilde K_{ij}} = B^{-2} K_{ij}\f$.
	 *  \c tkij  contains the Cartesian components of
	 *  \f${\tilde K_{ij}}\f$. 
	 */
	const Sym_tensor& get_tkij() const {return tkij;} ; 

	/** Returns the scalar \f$A^2 K_{ij} K^{ij}\f$.
	 *  For axisymmetric stars, this quantity is related to the 
	 *  derivatives of \f$N^\varphi\f$ by
	 * \f[
	 *	A^2 K_{ij} K^{ij} = {B^2 \over 2 N^2} \, r^2\sin^2\theta \,  
	 *    \left[ \left( {\partial N^\varphi \over \partial r} \right) ^2
	 *	    + {1\over r^2} \left( {\partial N^\varphi \over 
	 *		    \partial \theta} \right) ^2 \right] \ . 
	 * \f]
	 * In particular it is related to the quantities \f$k_1\f$ and \f$k_2\f$
	 * introduced by Eqs. (3.7) and (3.8) of 
	 * Bonazzola et al. \a Astron. \a Astrophys. \b 278 , 421 (1993)
	 * by 
	 * \f[
	 *	A^2 K_{ij} K^{ij} = 2 A^2 (k_1^2 + k_2^2) \ . 
	 * \f]
	 */
	 const Scalar& get_ak_car() const {return ak_car;} ; 

    // Outputs
    // -------
    public:
	virtual void sauve(FILE* ) const ;	    ///< Save in a file
    
	/// Display in polytropic units
	virtual void display_poly(ostream& ) const ; 

    protected:
	/// Operator >> (virtual function called by the operator <<). 
	virtual ostream& operator>>(ostream& ) const ;    

	/// Printing of some informations, excluding all global quantities
	virtual void partial_display(ostream& ) const ;    

    // Global quantities
    // -----------------
    public:
	
	/** Description of the stellar surface: returns a 2-D \c Itbl 
	 *	containing the 
	 *	values of the domain index \e l  on the surface at the 
	 *	collocation points in \f$(\theta', \phi')\f$.
	 *	The stellar surface is defined as the location where
	 *	the enthalpy (member \c ent ) vanishes.
	 */
	virtual const Itbl& l_surf() const ; 
	
	virtual double mass_b() const ;	    ///< Baryon mass
	virtual double mass_g() const ;	    ///< Gravitational mass
	virtual double angu_mom() const ;	///< Angular momentum 
	virtual double tsw() const ;		///< Ratio T/W

	/** Error on the virial identity GRV2.
	 *  This indicator is only valid for relativistic computations.
	 */
	virtual double grv2() const ;	

	/** Error on the virial identity GRV3.
	 *  The error is computed as the integral defined
	 *  by Eq. (43) of [Gourgoulhon and Bonazzola, 
	 *  \a Class. \a Quantum \a Grav. \b 11, 443 (1994)] divided by
	 *  the integral of the matter terms.
	 * 
	 *  @param ost output stream to give details of the computation;
	 *		if set to 0x0 [default value], no details will be
	 *		given.
	 *   
	 */
	virtual double grv3(ostream* ost = 0x0) const ;	

	virtual double r_circ() const ;	///< Circumferential radius
	virtual double aplat() const ;	///< Flatening r_pole/r_eq
	virtual double area() const ;   ///< Integrated surface area in \f${\rm km}^2\f$
	/// Mean star radius from the area \f$ r_{\rm mean} = \sqrt{\mathcal{A}} / 4\pi\f$
	virtual double mean_radius() const ; 
	virtual double z_eqf() const ;	///< Forward redshift factor at equator
	virtual double z_eqb() const ;	///< Backward redshift factor at equator
	virtual double z_pole() const ;	///< Redshift factor at North pole
    
	/** Quadrupole moment.
	 *  The quadrupole moment \e Q is defined according to Eq. (7) of
	 *  [Salgado, Bonazzola, Gourgoulhon and Haensel, \a Astron. \a Astrophys.
	 *   \b 291 , 155 (1994)]. At the Newtonian limit it is related to
	 *  the component \f${\bar I}_{zz}\f$ of the MTW (1973) reduced quadrupole 
	 *  moment \f${\bar I}_{ij}\f$ by: \f$Q = -3/2 {\bar I}_{zz}\f$. 
	 *  Note that \e Q is the negative of the quadrupole moment defined 
	 *  by Laarakkers and Poisson, \a Astrophys. \a J. \b 512 , 282 (1999).
	 */
	virtual double mom_quad() const ;	

	/** Circumferential radius of the innermost stable circular orbit (ISCO).	
	 *
	 *  @param ost output stream to give details of the computation;
	 *		if set to 0x0 [default value], no details will be
	 *		given.
	 */
	virtual double r_isco(ostream* ost = 0x0) const ;	
 	
 	/// Orbital frequency at the innermost stable circular orbit (ISCO).	
 	virtual double f_isco() const ;	

	/// Energy of a particle on the ISCO 
 	virtual double espec_isco() const ;	
	
	/// Angular momentum of a particle on the ISCO
 	virtual double lspec_isco() const ;	

        /// Orbital frequency at the equator.
	virtual double f_eq() const ;
	

    // Computational routines
    // ----------------------
    public: 
	/** Computes the hydrodynamical quantities relative to the Eulerian
	 *  observer from those in the fluid frame.
	 *
	 *  The calculation is performed starting from the quantities
	 *  \c ent , \c ener , \c press , and \c a_car ,  
	 *  which are supposed to be up to date.  
	 *  From these,  the following fields are updated:
	 *  \c gam_euler , \c u_euler , \c ener_euler , \c s_euler . 
	 * 
	 */
	virtual void hydro_euler() ; 
	
	/** Computes metric coefficients from known potentials. 
	 * 
	 *  The calculation is performed starting from the quantities
	 *  \c logn ,  \c dzeta , \c tggg  and \c shift , 
	 *  which are supposed to be up to date.  
	 *  From these,  the following fields are updated:
	 *  \c nnn , \c a_car ,  \c bbb  and \c b_car, as well as 
	 *  the 3-metric \c gamma. 
	 * 
	 */
	void update_metric() ; 
		
	/** Computes \c shift  from \c w_shift  and \c khi_shift 
	 *  according to Shibata's prescription 
	 *  [\a Prog. \a Theor. \a Phys. \b 101 , 1199 (1999)] :
	 * \f[
	 *  N^i = {7\over 8} W^i - {1\over 8} 
	 *			\left(\nabla^i\chi+\nabla^iW^kx_k\right)
	 * \f]
	 */
	void fait_shift() ; 
	
	/** Computes \c tnphi  and \c nphi  from the Cartesian 
	 *   components of the shift, stored in \c shift .
	 */
	void fait_nphi() ; 
		
	/** Computes \c tkij  and \c ak_car  from 
	 *  \c shift , \c nnn  and \c b_car .
	 */
	void extrinsic_curvature() ;
	
	/** Computes the coefficient \f$\lambda\f$ which ensures that the
	 *	GRV2 virial identity is satisfied.
	 *  \f$\lambda\f$ is the coefficient by which one must multiply
	 *  the quadratic source term \f$\sigma_q\f$ of the 2-D Poisson equation
	 *	\f[
	 *		\Delta_2 u = \sigma_m + \sigma_q
	 *	\f]
	 *  in order that the total source does not contain any monopolar term,
	 *  i.e. in order that
	 *  \f[
	 *		\int_0^{2\pi} \int_0^{+\infty} \sigma(r, \theta)
	 *				\, r \, dr \, d\theta = 0	    \ ,
	 *  \f]
	 *  where \f$\sigma = \sigma_m + \sigma_q\f$.
	 *	\f$\lambda\f$ is computed according to the formula
	 *  \f[
	 *		\lambda = - { \int_0^{2\pi} \int_0^{+\infty} \sigma_m(r, \theta)
	 *				\, r \, dr \, d\theta	    \over
	 * 			\int_0^{2\pi} \int_0^{+\infty} \sigma_q(r, \theta)
	 *				\, r \, dr \, d\theta } \ .
	 *  \f]
	 *  Then, by construction, the new source
	 *	\f$\sigma' = \sigma_m + \lambda \sigma_q\f$ has a vanishing monopolar
	 *  term.
	 *
	 *	@param sou_m [input] matter source term \f$\sigma_m\f$
	 *	@param sou_q [input] quadratic source term \f$\sigma_q\f$
	 *  @return	value of \f$\lambda\f$
	 */
	static double lambda_grv2(const Scalar& sou_m, const Scalar& sou_q) ;
		
	/** Computes an equilibrium configuration.
	 *  
	 *  @param ent_c  [input] Central enthalpy 
	 *  @param omega0  [input] Requested angular velocity 
	 *			     (if \c fact_omega=1. )
	 *  @param fact_omega [input] 1.01 = search for the Keplerian frequency,
	 *			      1. = otherwise.
	 *  @param nzadapt  [input] Number of (inner) domains where the mapping 
	 *			    adaptation to an iso-enthalpy surface
	 *			    should be performed
	 *  @param ent_limit [input] 1-D \c Tbl  of dimension \c nzet  which
	 *				defines the enthalpy at the outer boundary
	 *				of each domain
	 *  @param icontrol [input] Set of integer parameters (stored as a
	 *			    1-D \c Itbl  of size 8) to control the 
	 *			    iteration: 
	 *	\li \c icontrol(0) = mer_max  : maximum number of steps 
	 *	\li \c icontrol(1) = mer_rot  : step at which the rotation is 
	 *				      switched on 
	 *	\li \c icontrol(2) = mer_change_omega  : step at which the rotation
	 *			  velocity is changed to reach the final one  
	 *	\li \c icontrol(3) = mer_fix_omega  :  step at which the final
	 *			    rotation velocity must have been reached  
	 *	\li \c icontrol(4) = mer_mass  : the absolute value of 
	 *			    \c mer_mass  is the step from which the 
	 *			    baryon mass is forced to converge, 
	 *			    by varying the central enthalpy 
	 *			    (\c mer_mass>0 ) or the angular 
	 *			    velocity (\c mer_mass<0 ) 
	 *	\li \c icontrol(5) = mermax_poisson  : maximum number of steps in 
	 *				\c Map_et::poisson  
	 *	\li \c icontrol(6) = mer_triax  : step at which the 3-D 
	 *				perturbation is switched on 
	 *	\li \c icontrol(7) = delta_mer_kep  : number of steps
	 *			    after \c mer_fix_omega  when \c omega 
	 *			    starts to be increased by \c fact_omega 
	 *			    to search for the Keplerian velocity
	 * 	 
	 *  @param control [input] Set of parameters (stored as a 
	 *			    1-D \c Tbl  of size 7) to control the 
	 *			    iteration: 
	 *	\li \c control(0) = precis  : threshold on the enthalpy relative 
	 *				change for ending the computation 
	 *	\li \c control(1) = omega_ini  : initial angular velocity, 
	 *			    switched on only if \c mer_rot<0 , 
	 *			    otherwise 0 is used  
	 *	\li \c control(2) = relax  : relaxation factor in the main 
	 *				   iteration  
	 *	\li \c control(3) = relax_poisson  : relaxation factor in 
	 *				   \c Map_et::poisson  
	 *	\li \c control(4) = thres_adapt  :  threshold on dH/dr for 
	 *			    freezing the adaptation of the mapping 
	 *	\li \c control(5) = ampli_triax  :  relative amplitude of 
	 *			    the 3-D perturbation 
	 *	\li \c control(6) = precis_adapt  : precision for 
	 *			    \c Map_et::adapt 
	 *
	 *  @param mbar_wanted [input] Requested baryon mass (effective only 
	 *				if \c mer_mass > \c mer_max )
	 *  @param aexp_mass [input] Exponent for the increase factor of the 
	 *			      central enthalpy to converge to the 
	 *			      requested baryon mass
	 *  @param diff [output]   1-D \c Tbl  of size 7 for the storage of 
	 *			    some error indicators : 
	 *	    \li \c diff(0)  : Relative change in the enthalpy field
	 *			      between two successive steps 
	 *	    \li \c diff(1)  : Relative error in the resolution of the
	 *			    Poisson equation for \c nuf    
	 *	    \li \c diff(2)  : Relative error in the resolution of the
	 *			    Poisson equation for \c nuq    
	 *	    \li \c diff(3)  : Relative error in the resolution of the
	 *			    Poisson equation for \c dzeta    
	 *	    \li \c diff(4)  : Relative error in the resolution of the
	 *			    Poisson equation for \c tggg    
	 *	    \li \c diff(5)  : Relative error in the resolution of the
	 *			    equation for \c shift  (x comp.)   
	 *	    \li \c diff(6)  : Relative error in the resolution of the
	 *			    equation for \c shift  (y comp.)   
	 */
	virtual void equilibrium(double ent_c, double omega0, double fact_omega, 
			 int nzadapt, const Tbl& ent_limit,
			 const Itbl& icontrol, const Tbl& control,
			 double mbar_wanted, double aexp_mass, 
			 Tbl& diff, Param* = 0x0) ;
	

};

}
#endif
