package tests::ReportParserAsciiDocBookFormatterTest;

use strict;

use base qw/ Lire::Test::TestCase /;

use Lire::ReportParser::AsciiDocBookFormatter qw/ dbk2txt /;

sub set_up {
    my $self = $_[0];
    $self->SUPER::set_up();

    return;
}

sub tear_down {
    my $self = $_[0];
    $self->SUPER::tear_down();

    return;
}


sub _element_is_normalized {
    my ( $self, $element ) = @_;

    $self->annotate( "_element_is_normalized: $element\n" );
    my $dbk_text = "<para>Wawa <$element>bobo</$element> zozo</para>";
    my $e_text = <<_EOD;
    Wawa bobo zozo

_EOD
    $self->assert_str_equals( $e_text, dbk2txt( $dbk_text ) );

    return;
}

sub _test_admonition {
    my ( $self, $admonition ) = @_;

    $self->annotate( "testing admonition '$admonition'\n" );
    my $dbk_text = "<$admonition><para>test</para></$admonition>";
    my $uc_admonition = ucfirst( $admonition );
    my $e_text = <<_EOD;
    $uc_admonition:
      test

_EOD
    $self->assert_str_equals( $e_text, dbk2txt( $dbk_text ) );

    return;
}

sub _test_autopcdata {
    my ( $self, $data ) = @_;

    $self->annotate( "testing autopcdata '$data'\n" );
    $self->_element_is_normalized( $data );

    return;
}

sub test_admonitions {
    my $self = $_[0];

    my @admonitions = ( 'note', 'tip', 'important', 'caution', 'warning' );
    foreach my $admonition ( @admonitions ) {
        $self->_test_admonition( $admonition );
    }

    return;
}

sub test_all_autopcdata {
    my $self = $_[0];

    foreach my $data ( qw/emphasis phrase trademark wordasword action 
                          application literal constant markup option
                          optional parameter prompt property
                          replaceable returnvalue sgmltag structfield structname
                          symbol systemitem token type userinput varname/ )
    {
        $self->_test_autopcdata( $data );
    }

    return;
}

sub test_orderedlist {
    my $self = $_[0];

    my $dbk_text = '<orderedlist><listitem><para>test1</para></listitem><listitem><para>test2</para></listitem></orderedlist>';
    my $e_text = <<_EOD;
      1.  test1

      2.  test2

_EOD
    $self->assert_str_equals( $e_text, dbk2txt( $dbk_text ) );

    return;
}

sub test_itemizedlist {
    my $self = $_[0];

    my $dbk_text = '<itemizedlist><listitem><para>test1</para></listitem><listitem><para>test2</para></listitem></itemizedlist>';
    my $e_text = <<_EOD;
      - test1

      - test2

_EOD
    $self->assert_str_equals( $e_text, dbk2txt( $dbk_text ) );

    return;
}

# varlistentry and listitem are tested here too
sub test_variablelist {
    my $self = $_[0];

    my $dbk_text = <<_EOD;
    <variablelist>
      <title>The wawa generator</title>
      <varlistentry>
        <term>Report store</term>
        <listitem>
          <para>A folder containing report data</para>
        </listitem>
      </varlistentry>
    </variablelist>
_EOD
    my $e_text = <<_EOD;
      The wawa generator

      Report store
\t  A folder containing report data

_EOD
    $self->assert_str_equals( $e_text, dbk2txt( $dbk_text ) );

    $dbk_text = <<_EOD;
    <variablelist>
      <varlistentry>
        <listitem>
          <para>A folder containing report data</para>
        </listitem>
      </varlistentry>
    </variablelist>
_EOD
    $e_text = <<_EOD;
\t  A folder containing report data

_EOD
    $self->assert_str_equals( $e_text, dbk2txt( $dbk_text ) );

    return;
}

sub test_para {
    my $self = $_[0];

    my $dbk_text = '<para>This is a very very very very very very very very very very very very very very very long paragraph</para>';
    my $e_text = <<_EOD;
    This is a very very very very very very very very very very
    very very very very very long paragraph

_EOD
    $self->assert_str_equals( $e_text, dbk2txt( $dbk_text ) );

    $self->assert_str_equals( "    Test accentu.\n\n",
                              dbk2txt( "<para>Test accentu.</para>" ) );
    return;
}

sub test_ulink {
    my $self = $_[0];

    my $dbk_text = '<para>Go to <ulink url="http://www.logreport.org/">The LogReport Foundation</ulink> website.</para>';
    my $e_text = <<_EOD;
    Go to The LogReport Foundation (http://www.logreport.org/)
    website.

_EOD
    $self->assert_str_equals( $e_text, dbk2txt( $dbk_text ) );

    return;
}

sub test_quote {
    my $self = $_[0];

    my $dbk_text = '<para><quote>Linux</quote></para>';
    my $e_text = <<_EOD;
    "Linux"

_EOD
    $self->assert_str_equals( $e_text, dbk2txt( $dbk_text ) );

    return;
}

1;
