/*
 * Copyright (c) 2007-2010, Erik Lindroos <gliptic@gmail.com>
 * Copyright (c) 2010, "basro"
 * Copyright (c) 2012, Martin Erik Werner <martinerikwerner@gmail.com>
 * This software is released under the The BSD-2-Clause License:
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * * Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 * * Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef LIERO_WORM_HPP
#define LIERO_WORM_HPP

#include "math.hpp"
#include "rand.hpp"
#include <SDL/SDL.h>
#include <string>
#include <cstring>
#include <memory>
#include <gvl/resman/shared_ptr.hpp>
#include "version.hpp"
#include <gvl/serialization/archive.hpp> // For gvl::enable_when
#include <gvl/crypt/gash.hpp>


#include <iostream> // TEMP

struct Worm;
struct Game;

struct Ninjarope
{
	Ninjarope()
	: out(false)
	, attached(false)
	, anchor(0)
	{
	}

	bool out;            //Is the ninjarope out?
	bool attached;
	Worm* anchor;			// If non-zero, the worm the ninjarope is attached to
	fixed x, y, velX, velY; //Ninjarope props
	// Not needed as far as I can tell: fixed forceX, forceY;
	int length, curLen;

	void process(Worm& owner);
};

/*
struct Controls
{
	bool up, down, left, right;
	bool fire, change, jump;
};*/

struct WormWeapon
{
	WormWeapon()
	: id(0)
	, ammo(0)
	, delayLeft(0)
	, loadingLeft(0)
	, available(true)
	{
	}

	int id;
	int ammo;
	int delayLeft;
	int loadingLeft;
	bool available;
};

struct WormSettingsExtensions
{
	enum Control
	{
		Up, Down, Left, Right,
		Fire, Change, Jump,
		Dig,
		MaxControl = Dig,
		MaxControlEx
	};
	//static const int MaxControl = Dig;

	uint32_t controlsEx[MaxControlEx];
};

struct WormSettings : gvl::shared, WormSettingsExtensions
{
	WormSettings()
	: health(100)
	, controller(0)
	, randomName(true)
	, color(0)
	{
		rgb[0] = 26;
		rgb[1] = 26;
		rgb[2] = 62;

		std::memset(weapons, 0, sizeof(weapons));
	}

	gvl::gash::value_type& updateHash();

	void saveProfile(std::string const& newProfileName);
	void loadProfile(std::string const& newProfileName);

	int health;
	uint32_t controller; // CPU / Human
	uint32_t controls[MaxControlEx];
	int weapons[5]; // TODO: Adjustable
	std::string name;
	int rgb[3];
	bool randomName;

	int color;

	std::string profileName;

	gvl::gash::value_type hash;
};

template<typename Archive>
void archive(Archive ar, WormSettings& ws)
{
	ar
	.ui32(ws.color)
	.ui32(ws.health)
	.ui16(ws.controller);
	for(int i = 0; i < WormSettings::MaxControl; ++i)
		ar.ui16(ws.controls[i]);
	for(int i = 0; i < 5; ++i)
		ar.ui16(ws.weapons[i]);
	for(int i = 0; i < 3; ++i)
		ar.ui16(ws.rgb[i]);
	ar.b(ws.randomName);
	ar.str(ws.name);
	if(ar.context.replayVersion <= 1)
	{
		ws.WormSettingsExtensions::operator=(WormSettingsExtensions());
		return;
	}

	int wsVersion = myGameVersion;
	ar.ui8(wsVersion);

	for(int c = 0; c < WormSettings::MaxControl; ++c)
	{
		int dummy = 0;
		gvl::enable_when(ar, wsVersion >= 2)
			.ui8(dummy, 255)
			.ui8(dummy, 255);
	}

	for(int c = 0; c < WormSettings::MaxControlEx; ++c)
	{
		gvl::enable_when(ar, wsVersion >= 3)
			.ui32(ws.controlsEx[c], ws.controls[c]);
	}
}

/*
typedef struct _settings
{
 long m_iHealth[2];
 char m_iController[2];
	char m_iWeapTable[40];
 long m_iMaxBonuses;
 long m_iBlood;
 long m_iTimeToLose;
 long m_iFlagsToWin;
 char m_iGameMode;
 bool m_bShadow;
 bool m_bLoadChange;
 bool m_bNamesOnBonuses;
 bool m_bRegenerateLevel;
 BYTE m_iControls[2][7];
 BYTE m_iWeapons[2][5];
	long m_iLives;
 long m_iLoadingTime;
	bool m_bRandomLevel;
 char m_bWormName[2][21];
 char m_bLevelFile[13];
 BYTE m_iWormRGB[2][3];
 bool m_bRandomName[2];
 bool m_bMap;
 bool m_bScreenSync;
} SETTINGS, *PSETTINGS;
*/

struct Viewport;

struct WormAI
{
	WormAI(Worm& worm)
	: worm(worm)
	{
	}

	virtual void process() = 0;

	Worm& worm;
};

struct DumbLieroAI : WormAI
{
	DumbLieroAI(Worm& worm)
	: WormAI(worm)
	{
	}

	void process();

	Rand rand;
};

struct Worm : gvl::shared
{
	enum
	{
		RFDown,
		RFLeft,
		RFUp,
		RFRight
	};

	enum Control
	{
		Left = WormSettings::Left,
		Right = WormSettings::Right,
		Up = WormSettings::Up,
		Down = WormSettings::Down,
		Fire = WormSettings::Fire,
		Change = WormSettings::Change,
		Jump = WormSettings::Jump,
		MaxControl
		//Dig = WormSettings::Dig
	};
	//static const unsigned int MaxControl = Dig;


	struct ControlState
	{
		ControlState()
		: istate(0)
		{
		}

		bool operator==(ControlState const& b) const
		{
			return istate == b.istate;
		}

		uint32_t pack() const
		{
			return istate; // & ((1 << MaxControl)-1);
		}

		void unpack(uint32_t state)
		{
			istate = state & 0x7f;
		}

		bool operator!=(ControlState const& b) const
		{
			return !operator==(b);
		}

		bool operator[](std::size_t n) const
		{
			return ((istate >> n) & 1) != 0;
		}

		void set(std::size_t n, bool v)
		{
			if(v)
				istate |= 1 << n;
			else
				istate &= ~(uint32_t(1u) << n);
		}

		uint32_t istate;
	};

	Worm(/*gvl::shared_ptr<WormSettings> settings, int index, int wormSoundID, */Game& game)
	: x(0), y(0), velX(0), velY(0)
	, hotspotX(0), hotspotY(0)
	, aimingAngle(0), aimingSpeed(0)
	, ableToJump(false), ableToDig(false)   //The previous state of some keys
	, keyChangePressed(false)
	, movable(false)
	, animate(false)                 //Should the worm be animated?
	, visible(false)                 //Is the worm visible?
	, ready(false)                   //Is the worm ready to play?
	, flag(false)                    //Has the worm a flag?
	, makeSightGreen(false)          //Changes the sight color
	, health(0)                  //Health left
	, lives(0)                   //lives left
	, kills(0)                   //Kills made
	, timer(0)                   //Timer for GOT
	, killedTimer(0)             //Time until worm respawns
	, currentFrame(0)
	, flags(0)                   //How many flags does this worm have?
	, currentWeapon(0)
	, fireConeActive(false)
	, lastKilledBy(0)
	, fireCone(0)
	, leaveShellTimer(0)
//	, viewport(0)
	, index(index)
	, direction(0)
	, game(game)
	{
		//std::memset(controlStates, 0, sizeof(controlStates));

		makeSightGreen = false;

		ready = true;
		movable = true;

		//health = settings->health;
		visible = false;
		killedTimer = 150;

		//currentWeapon = 1; // This is later changed to 0, why is it here?
	}

	bool pressed(Control control) const
	{
		return controlStates[control];
	}

	bool pressedOnce(Control control)
	{
		bool state = controlStates[control];
		controlStates.set(control, false);
		return state;
	}

	void release(Control control)
	{
		controlStates.set(control, false);
	}

	void press(Control control)
	{
		controlStates.set(control, true);
	}

	void setControlState(Control control, bool state)
	{
		controlStates.set(control, state);
	}

	void toggleControlState(Control control)
	{
		controlStates.set(control, !controlStates[control]);
	}

	void beginRespawn();
	void doRespawning();
	void process();
	void processWeapons();
	void processPhysics();
	void processMovement();
	void processTasks();
	void processAiming();
	void processWeaponChange();
	void processSteerables();
	void fire();
	void processSight();
	void calculateReactionForce(int newX, int newY, int dir);
	void initWeapons();
	int angleFrame() const;

	fixed x, y;                    //Worm position
	fixed velX, velY;              //Worm velocity

	int logicRespawnX, logicRespawnY;

	int hotspotX, hotspotY;      //Hotspots for laser, laser sight, etc.
	fixed aimingAngle, aimingSpeed;

	//Controls controls;
	bool ableToJump, ableToDig;   //The previous state of some keys
	bool keyChangePressed;
	bool movable;

	bool animate;                 //Should the worm be animated?
	bool visible;                 //Is the worm visible?
	bool ready;                   //Is the worm ready to play?
	bool flag;                    //Does the worm have a flag?
	bool makeSightGreen;          //Changes the sight color
	int health;                  //Health left
	int lives;                   //lives left
	int kills;                   //Kills made

	int timer;                   //Timer for GOT
	int killedTimer;             //Time until worm respawns
	int currentFrame;

	int flags;                   //How many flags does this worm have?

	Ninjarope ninjarope;

	int currentWeapon;           //The selected weapon
	bool fireConeActive;          //Is the firecone showing
	Worm* lastKilledBy;          // What worm that last killed this worm
	int fireCone;                //How much is left of the firecone
	int leaveShellTimer;         //Time until next shell drop

	gvl::shared_ptr<WormSettings> settings; // !CLONING
	//Viewport* viewport; // !CLONING
	int index; // 0 or 1

	std::auto_ptr<WormAI> ai;

	int reacts[4];
	WormWeapon weapons[5];
	int direction;
	ControlState controlStates;
	ControlState prevControlStates;
	Game& game; // !CLONING

	// Data for LocalController
	ControlState cleanControlStates; // This contains the real state of real and extended controls

};

bool checkForWormHit(int x, int y, int dist, Worm* ownWorm);
bool checkForSpecWormHit(int x, int y, int dist, Worm& w);

#endif // LIERO_WORM_HPP
