/*
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*! \mainpage LCMAPS - Local Credential MAPping Service
 *
 *  \section intro Introduction
 *
 *  This document describes the LCMAPS API and the LCMAPS plugins. Please check the links above.
 *
 *  \section interfaces the LCMAPS Interfaces
 *
 *  -# The interface to the LCMAPS credential mapping framework is described in \ref LcmapsInterface
 *  -# The LCMAPS plugins should use the LCMAPS API described in \ref APIforLcmapsPlugins
 *  -# The interface that the plugins should provide to the LCMAPS framework is described in
 *     \ref LcmapsPluginInterface
 *
 *  \section plugins The LCMAPS plugins
 *  A description of the LCMAPS plugins can be found
 *  here ...
 *
 *  ... the basic plugins:
 *  -# \ref lcmaps_posix_enf.mod
 *  -# \ref lcmaps_ldap_enf.mod
 *  -# \ref lcmaps_localaccount.mod
 *  -# \ref lcmaps_poolaccount.mod
 *
 *  ... the voms-aware plugins:
 *  -# \ref lcmaps_voms.mod
 *  -# \ref lcmaps_voms_poolaccount.mod
 *  -# \ref lcmaps_voms_localgroup.mod
 *  -# \ref lcmaps_voms_poolgroup.mod
 *
 *  ... miscellaneous:
 *  -# \ref lcmaps_afs.mod
 *  -# \ref lcmaps_jobrep.mod
 *  -# \ref lcmaps_dummy_good.mod
 *  -# \ref lcmaps_dummy_bad.mod
 *
 */

/*!
    \file   lcmaps.c
    \brief  the LCMAPS module - the local credential mapping service.
    \author Martijn Steenbakkers for the EU DataGrid.

    The interface to the LCMAPS module is composed of:
    -# lcmaps_init(): start the PluginManager --> load plugins, start evaluation manager
    -# lcmaps_run():  run the PluginManager --> run evaluation manager --> run plugins
    -# lcmaps_term(): stop the PluginManager
*/

#define _XOPEN_SOURCE	500

#include "lcmaps_config.h"

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <pwd.h>

#ifdef LCMAPS_GSI_MODE
#   include <grp.h>
#   include <stdlib.h>
#   include <openssl/x509.h>
#   include <openssl/err.h>
#   include <gssapi.h>
#endif

/* LCMAPS includes */
/* #include "lcmaps_return_account_from_pem.h" */
/* #include "lcmaps_account.h" */
#include "lcmaps.h"
#include "lcmaps_types.h"
#include "lcmaps_cred_data.h"
#include "pluginmanager/_lcmaps_pluginmanager.h"
#include "pluginmanager/_lcmaps_log.h"
#include "grid_credential_handling/_lcmaps_credential.h"

/******************************************************************************
                       Define module specific variables
******************************************************************************/
static lcmaps_cred_id_t    lcmaps_cred; /*!< \internal */
static int                 lcmaps_initialized = 0; /*!< \internal */



/******************************************************************************
Function:   lcmaps_init_and_logfile
Description:
    Select logging type
    Setup logging by providing a file handle or file name, error handling (not yet).
    Start PluginManager:
    read from LCMAPS config file, the plugins to be loaded

Parameters:
    logfile: name of logfile
    fp: file handle for logging (from gatekeeper or other previously opened file handle)
        If the file handle is zero, assume that only syslogging is requested
    logtype: type of logging (usrlog and/or syslog)

Returns:
    0: initialization succeeded
    1: initialization failed
******************************************************************************/
int lcmaps_init_and_logfile(
        char * logfile,
        FILE* fp,
        unsigned short logtype
)
{
    if (lcmaps_initialized)
    {
        lcmaps_log_debug(LOG_DEBUG, "LCMAPS already initialized\n");
        return 0;
    }

    /* start logging */
    if (lcmaps_log_open(logfile,fp,logtype))
        goto fail_lcmaps_init_and_logfile;
    lcmaps_log_time(LOG_DEBUG,"Initialization LCMAPS version %s\n",VERSION);

    /* Start PluginManager */
    if (lcmaps_startPluginManager()) {
        lcmaps_log(LOG_ERR,"lcmaps_init() error: could not start plugin manager\n");
        goto fail_lcmaps_init_and_logfile;
    }

    /* success */
    lcmaps_initialized++;
    return 0;

 fail_lcmaps_init_and_logfile:
    return 1;

}
/******************************************************************************
Function:   lcmaps_init_and_log
Description:
    Select logging type
    Start PluginManager:
    read from LCMAPS config file, the plugins to be loaded

Parameters:
    fp: file handle for logging (from gatekeeper or other previously opened file handle)
        If the file handle is zero, assume that only syslogging is requested
    logtype: type of logging (usrlog and/or syslog)

Returns:
    0: initialization succeeded
    1: initialization failed
******************************************************************************/
int lcmaps_init_and_log(
        FILE* fp,
        unsigned short logtype
)
{
    return lcmaps_init_and_logfile(NULL, fp, logtype);
}

/******************************************************************************
Function:   lcmaps_init
Description:
    Start PluginManager:
    read from LCMAPS config file, the plugins to be loaded

Parameters:
    fp: file handle for logging (from gatekeeper or other previously opened file handle)
        If the file handle is zero, assume that only syslogging is requested
Returns:
    0: initialization succeeded
    1: initialization failed
******************************************************************************/
int lcmaps_init(
        FILE* fp
)
{
    /* set logging file descriptor, for the moment the gatekeeper logfile */
    /* if fp == NULL --> syslogging, otherwise only DO_USRLOG */
    if (fp) {
        return lcmaps_init_and_log(fp,DO_USRLOG);
    } else {
        return lcmaps_init_and_log(NULL,DO_SYSLOG);
    }
}

/******************************************************************************
Function:   lcmaps_term
Description:
    Terminate LCMAPS module:

Parameters:
Returns:
    0: termination succeeded
    1: termination failed
******************************************************************************/
int lcmaps_term(void)
{


/* although this a good idea to do here, the values are already cleaned, which kinda beat the point. */
#if 0
    char lcmapsmappingresultoneliner[1024];

    int i = 0;

    char *  dn       = NULL;
    int     cntDNs   = 0;
    uid_t   uid      = -1;
    int     uid_cnt  = 0;
    gid_t * pgids    = NULL;
    int     pgid_cnt = 0;
    gid_t * sgids    = NULL;
    int     sgid_cnt = 0;
    char ** fqans    = NULL;
    int     nfqans   = 0;

    /* I need to publish the DN in most cases for the interaction,
     * so why not publish all the stuff I have on this mapping in the magic oneliner that was
     * requested somewhere way back when...
     */
    bzero (lcmapsmappingresultoneliner, 1024);

    if ((dn = getCredentialData(DN, &cntDNs)) && cntDNs)
    {
        snprintf (lcmapsmappingresultoneliner, 1024, "%s", dn);
    }
    if ((uid = getCredentialData(UID, &uid_cnt)) && uid_cnt)
    {
        snprintf (lcmapsmappingresultoneliner, 1024, "%s:%d", lcmapsmappingresultoneliner, (int) uid);
    }
    if ((pgids = getCredentialData(PRI_GID, &pgid_cnt)) && pgid_cnt)
    {
        for (i = 0; i < pgid_cnt; i++)
        {
            snprintf (lcmapsmappingresultoneliner, 1024, "%s:%d", lcmapsmappingresultoneliner, (int) pgids[i]);
        }
    }
    if ((sgids = getCredentialData(SEC_GID, &sgid_cnt)) && sgid_cnt)
    {
        for (i = 0; i < sgid_cnt; i++)
        {
            snprintf (lcmapsmappingresultoneliner, 1024, "%s:%d", lcmapsmappingresultoneliner, (int) sgids[i]);
        }
    }
    if ((fqans = getCredentialData(LCMAPS_VO_CRED_STRING, &nfqans)) && nfqans)
    {
        for (i = 0; i < nfqans; i++)
        {
            snprintf (lcmapsmappingresultoneliner, 1024, "%s:%s", lcmapsmappingresultoneliner, fqans[i]);
        }
    }
    lcmaps_log_time(1, "lcmaps.mod-lcmaps_term(): %s\n", lcmapsmappingresultoneliner);
    fprintf (stderr, "lcmaps.mod-lcmaps_term(): %s\n", lcmapsmappingresultoneliner);
    /* end of new publishment */
#endif

    lcmaps_log_time(LOG_DEBUG,"Termination LCMAPS\n");
    lcmaps_log_time(LOG_DEBUG, "%s(): terminating\n", __func__);
    if (lcmaps_stopPluginManager() != 0)
        return 1;
    if (lcmaps_log_close() != 0)
        return 1;
    if (lcmaps_initialized > 0)
        lcmaps_initialized--;

#if 0
/* NOTE: these calls should be called before program end, but will also cleanup
 * structures which might have been from others... */
#ifdef LCMAPS_GSI_MODE
    /* Most reasonable place to cleanup openssl stuff, see also Jan Just's
     * grid-proxy-verify.c */
    /* This cleans up the error queue, pid==0 means current thread */
    lcmaps_log(LOG_DEBUG,"lcmaps_term(): Calling ERR_remove_state(0)\n");
    ERR_remove_state(0);
    /* Cleans up mem from ERR_load_crypto_strings(), called by e.g. VOMS and
     * verify-proxy plugin */
    lcmaps_log(LOG_DEBUG,"lcmaps_term(): Calling ERR_free_strings()\n");
    ERR_free_strings();
    /* cleans up OpenSSLs internal object table */
    lcmaps_log(LOG_DEBUG,"lcmaps_term(): Calling OBJ_cleanup()\n");
    OBJ_cleanup();
    /* OpenSSL keeps an internal table of digest algorithms and ciphers. It uses
     * this table to lookup ciphers via functions such as
     * EVP_get_cipher_byname(). EVP_cleanup() removes all ciphers and digests
     * from the table. LCMAPS probably uses them only via VOMS */
    lcmaps_log(LOG_DEBUG,"lcmaps_term(): Calling EVP_cleanup()\n");
    EVP_cleanup();
    /* See also http://www.openssl.org/support/faq.html#PROG13,
     *	"I think I've detected a memory leak, is this a bug?" */
    lcmaps_log(LOG_DEBUG,"lcmaps_term(): Calling CRYPTO_cleanup_all_ex_data()\n");
    CRYPTO_cleanup_all_ex_data();
#endif
#endif

    return 0;
}

/******************************************************************************
Function:   lcmaps_run_without_credentials_and_return_username
Description:
    Based on the only the user DN do the following:
    Do the user mapping based on the provided list of policies (first successful
    policy found in the lcmaps policy file (lcmaps.db) will result in the user
    mapping) and return user name.
    This interface can be used to provide the legacy
        GLOBUS_GSS_ASSIST_GRIDMAP()
    interface.

Parameters:
    user_dn_tmp : user DN
    request     : RSL string
    usernamep   : pointer to user name (to be freed by calling application)
    npols       : number of policies to be considered for evaluation
    policynames : the names of the policies to be considered for evaluation

Returns:
    0: mapping succeeded
    1: mapping failed
******************************************************************************/
int lcmaps_run_without_credentials_and_return_username(
        char * user_dn_tmp,
        lcmaps_request_t request,
        char ** usernamep,
        int npols,
        char ** policynames
)
{
    char *            user_dn     = NULL;
    uid_t *           uid;
    int               cntUid;
    struct passwd *   user_info   = NULL;
    int               rc          = 0;


    if (lcmaps_initialized == 0)
    {
        lcmaps_log(LOG_ERR,"LCMAPS has to be initialized first !\n");
        goto fail_lcmaps_run_without_credentials_and_return_username;
    }
    lcmaps_log_time(LOG_DEBUG,"LCMAPS credential mapping request\n");

    lcmaps_log_debug(3, "Using \"%s\" interface of LCMAPS\n",__func__);
    if (usernamep == NULL)
        goto fail_lcmaps_run_without_credentials_and_return_username;
    *usernamep = NULL;

    /*
     * Create lcmaps credential:
     *     fill it with the dn and extract it again
     */
    if ((rc = lcmaps_credential_init(&lcmaps_cred)) != LCMAPS_CRED_SUCCESS)
    {
        if (rc == LCMAPS_CRED_INVOCATION_ERROR)
        {
            lcmaps_log(LOG_ERR, "%s() error: lcmaps_cred does not exist (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_without_credentials_and_return_username;
        }
        else
        {
            lcmaps_log(LOG_ERR, "%s() error: cannot initialize lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_without_credentials_and_return_username;
        }
    }
    if ((rc = lcmaps_credential_store_dn(user_dn_tmp, &lcmaps_cred)) != LCMAPS_CRED_SUCCESS)
    {
        if (rc == LCMAPS_CRED_NO_DN)
        {
            lcmaps_log(LOG_ERR, "%s() error: storing EMPTY dn in lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_without_credentials_and_return_username;
        }
        else
        {
            lcmaps_log(LOG_ERR, "%s() error: storing dn in lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_without_credentials_and_return_username;
        }
    }
    user_dn = lcmaps_credential_get_dn(lcmaps_cred);
    if (user_dn == NULL)
    {
        lcmaps_log(LOG_ERR, "%s() error: user DN empty\n", __func__);
        goto fail_lcmaps_run_without_credentials_and_return_username;
    }

    /* Run PluginManager */
    if (lcmaps_runPluginManager(request, lcmaps_cred, NULL, npols, policynames, LCMAPS_NORMAL_MODE)) {
        lcmaps_log_debug(1, "%s: Error: could not run plugin manager\n", __func__);
        goto fail_lcmaps_run_without_credentials_and_return_username;
    }

    /* Now try to get the userid drom the credential data */
    uid    = getCredentialData(UID,     &cntUid);
    if (uid)
    {
        if ( (user_info = getpwuid(uid[0])) == NULL )
        {
            lcmaps_log(LOG_ERR,"LCMAPS could not find the username related to uid: %d\n",uid[0]);
            return 1;
        }
        (*usernamep) = strdup(user_info->pw_name);
	if (*usernamep == NULL)	{
	    lcmaps_log(LOG_ERR, "%s: Out of memory\n", __func__);
	    goto fail_lcmaps_run_without_credentials_and_return_username;
	}
    }
    else
    {
        lcmaps_log(LOG_ERR,"LCMAPS could not find any uid\n");
        return 1;
    }

    /* succes */
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2,"%s(): succeeded\n",__func__);
    return 0;

 fail_lcmaps_run_without_credentials_and_return_username:
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2, "%s(): failed\n", __func__);
    return 1;
}


/******************************************************************************
Function:   lcmaps_run_with_fqans_mapcounter_and_return_account
Description:
    LCMAPS will run with a list of FQANs and the DN as an input. In addition a
    list of policies may be provided.
    The allocated uid, gids and the poolindex will be returned to the calling
    application.
    This interface is intended to be used by a wrapper function for the
        DYNAMIC ACCOUNTS SERVICE (DAS).

Parameters:
    user_dn     : the DN of the user
    fqan_list   : the list of (VOMS) FQANs that have been asserted to the user
    nfqan       : the number of FQANs in fqan_list
    mapcounter: : The counter which will be added to the poolindex, effectively enabling
                  multiple account mappings
    request     : RSL string
    npols       : number of policies to be considered for evaluation
    policynames : the names of the policies to be considered for evaluation
    puid        : pointer to the uid found (output parameter)
    ppgid_list  : pointer to the list of primary gids found (output parameter)
    pnpgid      : pointer to the number of primary gids found (output parameter)
    psgid_list  : pointer to the list of secondary gids found (output parameter)
    pnsgid      : pointer to the number of secondary gids found (output parameter)
    poolindexp  : pointer to poolindex string (output parameter)

Returns:
    0: mapping succeeded
    1: mapping failed
******************************************************************************/
int lcmaps_run_with_fqans_mapcounter_and_return_account(
        char *            user_dn,
        char **           fqan_list,
        int               nfqan,
        int               mapcounter,
        lcmaps_request_t  request,
        int               npols,
        char **           policynames,
        uid_t *           puid,
        gid_t **          ppgid_list,
        int *             pnpgid,
        gid_t **          psgid_list,
        int *             pnsgid,
        char **           poolindexp
)
{
    int      rc             = 0;
    uid_t *  uid            = NULL;
    int      cntUid         = -1;
    gid_t *  priGid         = NULL;
    int      cntPriGid      = -1;
    gid_t *  secGid         = NULL;
    int      cntSecGid      = -1;
    char **  poolindex_list = NULL;
    char *   poolindex      = NULL;
    int      cntpoolindex   = 0;

    if (lcmaps_initialized == 0)
    {
        lcmaps_log(LOG_ERR,"LCMAPS has to be initialized first !\n");
        goto fail_lcmaps_run_with_fqans_and_return_account;
    }
    lcmaps_log_time(LOG_DEBUG, "LCMAPS credential mapping request\n");

    lcmaps_log_debug(3, "Using \"%s\" interface of LCMAPS\n", __func__);
    /*
     * Create lcmaps credential:
     *     fill it with the dn and extract it again
     */
    if ((rc = lcmaps_credential_init(&lcmaps_cred)) != LCMAPS_CRED_SUCCESS)
    {
        if (rc == LCMAPS_CRED_INVOCATION_ERROR)
        {
            lcmaps_log(LOG_ERR, "%s() error: lcmaps_cred does not exist (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_fqans_and_return_account;
        }
        else
        {
            lcmaps_log(LOG_ERR, "%s() error: cannot initialize lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_fqans_and_return_account;
        }
    }
    if ((rc = lcmaps_credential_store_dn(user_dn, &lcmaps_cred)) != LCMAPS_CRED_SUCCESS)
    {
        if (rc == LCMAPS_CRED_NO_DN)
        {
            lcmaps_log(LOG_ERR, "%s() error: storing EMPTY dn in lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_fqans_and_return_account;
        }
        else
        {
            lcmaps_log(LOG_ERR, "%s() error: storing dn in lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_fqans_and_return_account;
        }
    }
    if ((rc = lcmaps_credential_store_fqan_list(nfqan, fqan_list, &lcmaps_cred)) != LCMAPS_CRED_SUCCESS)
    {
        if (rc == LCMAPS_CRED_INVOCATION_ERROR)
        {
            lcmaps_log(LOG_ERR, "%s() error: lcmaps_cred does not exist (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_fqans_and_return_account;
        }
        else if (rc == LCMAPS_CRED_NO_FQAN) /* Only issue a warning, no failure */
        {
            lcmaps_log_debug(1, "%s() warning: fqan list is empty (rc = 0x%x)\n", __func__, rc);
        }
        else /* LCMAPS_CRED_ERROR */
        {
            lcmaps_log(LOG_ERR, "%s() error: storing fqan list! (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_fqans_and_return_account;
        }
    }
    if (lcmaps_credential_store_mapcounter(mapcounter, &lcmaps_cred) != LCMAPS_CRED_SUCCESS)
    {
        lcmaps_log(LOG_ERR, "%s() error: storing mapcounter in lcmaps_cred\n",__func__);
        goto fail_lcmaps_run_with_fqans_and_return_account;
    }

    /* Run PluginManager */
    if (lcmaps_runPluginManager(request, lcmaps_cred, NULL, npols, policynames, LCMAPS_NORMAL_MODE)) {
        lcmaps_log_debug(1, "%s() error: could not run plugin manager\n", __func__);
        goto fail_lcmaps_run_with_fqans_and_return_account;
    }

    /* At this stage extract uid, gids and poolindex */
    /* First the uid */
    uid              = (uid_t *) getCredentialData(UID,     &cntUid);
    if (uid == NULL)
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS could not find any uid\n", __func__);
        goto fail_lcmaps_run_with_fqans_and_return_account;
    }
    else if (cntUid != 1)
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS found %d uids, only 1 allowed\n", __func__, cntUid);
        goto fail_lcmaps_run_with_fqans_and_return_account;
    }
    else
    {
        *puid = uid[0];
    }
    /* Then the primary gids */
    priGid           = (gid_t *) getCredentialData(PRI_GID, &cntPriGid);
    if (priGid == NULL)
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS could not find any gid, at least one required!\n", __func__);
        goto fail_lcmaps_run_with_fqans_and_return_account;
    }
    else
    {
        *pnpgid = cntPriGid;
        *ppgid_list = priGid;
    }
    /* Then the secondary gids */
    secGid           = (gid_t *) getCredentialData(SEC_GID, &cntSecGid);
    if (secGid == NULL)
    {
        lcmaps_log_debug(1, "%s: LCMAPS found no secondary groups\n", __func__);
    }
    else
    {
        *pnsgid = cntSecGid;
        *psgid_list = secGid;
    }
    /* And finally the poolindex */
    poolindex_list = (char **) getCredentialData(POOL_INDEX, &cntpoolindex);
    if (poolindex_list && (cntpoolindex > 0))
    {
        poolindex = (char *)(*(poolindex_list));
        lcmaps_log_debug(5, "%s: found %d poolindices starting at address = %p\n", __func__, cntpoolindex, (void*)poolindex_list);
        lcmaps_log_debug(5, "%s(): found this poolindex %s\n", __func__, poolindex);
        *poolindexp = poolindex;
    }
    else
    { /* Version 1.3.3 and lower: Error if no poolindex found                */
      /* Now just continue. It's not up to us to decide if this is an error. */
      /* It may just be a statically assigned account                        */
        lcmaps_log_debug(5, "%s: LCMAPS could not find a poolindex (a statically assigned account?)\n", __func__);
    }

    /* succes */
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2, "%s: succeeded\n", __func__);
    return 0;

 fail_lcmaps_run_with_fqans_and_return_account:
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2, "%s: failed\n", __func__);
    return 1;
}

/******************************************************************************
Function:   lcmaps_run_with_fqans_and_return_account
Description:
    LCMAPS will run with a list of FQANs and the DN as an input. In addition a
    list of policies may be provided.
    The allocated uid, gids and the poolindex will be returned to the calling
    application.
    This interface is intended to be used by a wrapper function for the
        DYNAMIC ACCOUNTS SERVICE (DAS).

Parameters:
    user_dn     : the DN of the user
    fqan_list   : the list of (VOMS) FQANs that have been asserted to the user
    nfqan       : the number of FQANs in fqan_list
    request     : RSL string
    npols       : number of policies to be considered for evaluation
    policynames : the names of the policies to be considered for evaluation
    puid        : pointer to the uid found (output parameter)
    ppgid_list  : pointer to the list of primary gids found (output parameter)
    pnpgid      : pointer to the number of primary gids found (output parameter)
    psgid_list  : pointer to the list of secondary gids found (output parameter)
    pnsgid      : pointer to the number of secondary gids found (output parameter)
    poolindexp  : pointer to poolindex string (output parameter)

Returns:
    0: mapping succeeded
    1: mapping failed
******************************************************************************/
int lcmaps_run_with_fqans_and_return_account(
        char *            user_dn,
        char **           fqan_list,
        int               nfqan,
        lcmaps_request_t  request,
        int               npols,
        char **           policynames,
        uid_t *           puid,
        gid_t **          ppgid_list,
        int *             pnpgid,
        gid_t **          psgid_list,
        int *             pnsgid,
        char **           poolindexp
)
{
    return lcmaps_run_with_fqans_mapcounter_and_return_account(
        user_dn,
        fqan_list,
        nfqan,
        -1,
        request,
        npols,
        policynames,
        puid,
        ppgid_list,
        pnpgid,
        psgid_list,
        pnsgid,
        poolindexp
    );
}

/*
 * HERE FOLLOW THE GSI DEPENDENT INTERFACES
 */

#ifdef LCMAPS_GSI_MODE


/******************************************************************************
Function:   lcmaps_run
Description:
    Do the user mapping based on the user's gss (gsi) credential and the job
    request.
    This is the legacy lcmaps interface and is used by
        the GATEKEEPER.

Parameters:
    request: RSL string
    user_cred : user globus credential handle
Returns:
    0: mapping succeeded
    1: mapping failed
******************************************************************************/
#if ALLOW_EMPTY_CREDENTIALS
int lcmaps_run(
        char * user_dn_tmp,
        gss_cred_id_t user_cred,
        lcmaps_request_t request
)
#else
int lcmaps_run(
        gss_cred_id_t user_cred,
        lcmaps_request_t request
)
#endif
{
    char *  user_dn = NULL;
    int     rc = 0;

    if (lcmaps_initialized == 0) {
        lcmaps_log(LOG_ERR,"LCMAPS has to be initialized first !\n");
        goto fail_lcmaps_run;
    }
    lcmaps_log_debug(LOG_DEBUG, "LCMAPS credential mapping request\n");

    /*
     * Create lcmaps credential:
     *     fill it with the dn and extract it again
     */
    if ((rc = lcmaps_credential_init(&lcmaps_cred)) != LCMAPS_CRED_SUCCESS) {
        if (rc == LCMAPS_CRED_INVOCATION_ERROR) {
            lcmaps_log(LOG_ERR, "%s() error: lcmaps_cred does not exist (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run;
        } else {
            lcmaps_log(LOG_ERR, "%s() error: cannot initialize lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run;
        }
    }

    /* Store the gss_cred_id_t and the derivatives */
    rc = lcmaps_credential_store_gss_cred_id_t_and_sub_elements(user_cred, &lcmaps_cred);
    if (rc != LCMAPS_CRED_SUCCESS) {
        if (rc == LCMAPS_CRED_INVOCATION_ERROR) {
            lcmaps_log(LOG_ERR, "%s() error: lcmaps_cred does not exist\n", __func__);
            goto fail_lcmaps_run;
        } else if (rc == LCMAPS_CRED_NO_GSS_CRED) {
            /* Use provided DN */
            lcmaps_log(LOG_ERR, "%s() WARNING: empty credential found !\n", __func__);
        } else if (rc == LCMAPS_CRED_NO_FQAN) {
            /* No FQANs found or valid */
            lcmaps_log(LOG_DEBUG, "%s() Debug: No VOMS FQANs were found, continuing without them.\n", __func__);
        } else {
            lcmaps_log(LOG_ERR, "%s() error: storing gss_credential or its derivative credentials\n", __func__);
            goto fail_lcmaps_run;
        }
    }

    /* Double check if I got a proper DN */
    user_dn = lcmaps_credential_get_dn(lcmaps_cred);
    if (user_dn == NULL) {
        lcmaps_log(LOG_ERR, "%s() error: user DN empty\n", __func__);
        goto fail_lcmaps_run;
    }

    /* Run PluginManager */
    if (lcmaps_runPluginManager(request, lcmaps_cred, NULL, 0, NULL, LCMAPS_NORMAL_MODE)) {
        lcmaps_log_debug(1, "%s() error: could not run plugin manager\n", __func__);
        goto fail_lcmaps_run;
    }

    /* succes */
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2, "%s(): succeeded\n", __func__);
    return 0;

 fail_lcmaps_run:
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2, "%s(): failed\n", __func__);
    return 1;
}

/******************************************************************************
Function:   lcmaps_run_and_return_username
Description:
    do the user mapping based on the provided list of policies (first successful
    policy found in the lcmaps policy file (lcmaps.db) will result in the user
    mapping) and return user name.
    This interface is used by the
        GRIDFTP SERVER.

Parameters:
    request     : RSL string
    user_cred   : user globus credential handle
    usernamep   : pointer to user name (to be freed by calling application)
    npols       : number of policies to be considered for evaluation
    policynames : the names of the policies to be considered for evaluation

Returns:
    0: mapping succeeded
    1: mapping failed
******************************************************************************/
#if ALLOW_EMPTY_CREDENTIALS
int lcmaps_run_and_return_username(
        char * user_dn_tmp,
        gss_cred_id_t user_cred,
        lcmaps_request_t request,
        char ** usernamep,
        int npols,
        char ** policynames
)
#else
int lcmaps_run_and_return_username(
        gss_cred_id_t user_cred,
        lcmaps_request_t request,
        char ** usernamep,
        int npols,
        char ** policynames
)
#endif
{
    char *  user_dn = NULL;
    int     rc      = 0;

    uid_t *          uid;
    int              cntUid;
    struct passwd *  user_info   = NULL;
    char * req_username;

    if (lcmaps_initialized == 0)
    {
        lcmaps_log(LOG_ERR,"LCMAPS has to be initialized first !\n");
        goto fail_lcmaps_run_and_return_username;
    }
    lcmaps_log_time(LOG_DEBUG, "LCMAPS credential mapping request\n");

    lcmaps_log_debug(3, "Using \"%s\" interface of LCMAPS\n", __func__);
    if (usernamep == NULL)
        goto fail_lcmaps_run_and_return_username;

    /* Save input username */
    req_username=*usernamep;
    *usernamep = NULL;

    /*
     * Create lcmaps credential:
     *     fill it with the dn and extract it again
     */
    if ((rc = lcmaps_credential_init(&lcmaps_cred)) != LCMAPS_CRED_SUCCESS) {
        if (rc == LCMAPS_CRED_INVOCATION_ERROR) {
            lcmaps_log(LOG_ERR, "%s() error: lcmaps_cred does not exist (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_return_username;
        } else {
            lcmaps_log(LOG_ERR, "%s() error: cannot initialize lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_return_username;
        }
    }

    /* Store the gss_cred_id_t and the derivatives */
    rc = lcmaps_credential_store_gss_cred_id_t_and_sub_elements(user_cred, &lcmaps_cred);
    if (rc != LCMAPS_CRED_SUCCESS) {
        if (rc == LCMAPS_CRED_INVOCATION_ERROR) {
            lcmaps_log(LOG_ERR, "%s() error: lcmaps_cred does not exist\n", __func__);
            goto fail_lcmaps_run_and_return_username;
        } else if (rc == LCMAPS_CRED_NO_GSS_CRED) {
            /* Use provided DN */
            lcmaps_log(LOG_ERR, "%s() WARNING: empty credential found !\n", __func__);
        } else if (rc == LCMAPS_CRED_NO_FQAN) {
            /* No FQANs found or valid */
            lcmaps_log(LOG_DEBUG, "%s() Debug: No VOMS FQANs were found, continuing without them.\n", __func__);
        } else {
            lcmaps_log(LOG_ERR, "%s() error: storing gss_credential or its derivative credentials\n", __func__);
            goto fail_lcmaps_run_and_return_username;
        }
    }

#if 0
#if ALLOW_EMPTY_CREDENTIALS
    if ((rc = lcmaps_credential_store_dn(user_dn_tmp, &lcmaps_cred)) != LCMAPS_CRED_SUCCESS)
    {
        if (rc == LCMAPS_CRED_NO_DN)
        {
            lcmaps_log(LOG_ERR, "%s() error: storing EMPTY dn in lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_return_username;
        }
        else
        {
            lcmaps_log(LOG_ERR, "%s() error: storing dn in lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_return_username;
        }
    }
#endif
#endif
    user_dn = lcmaps_credential_get_dn(lcmaps_cred);
    if (user_dn == NULL)
    {
        lcmaps_log(LOG_ERR, "%s() error: user DN empty\n", __func__);
        goto fail_lcmaps_run_and_return_username;
    }

    /* Run PluginManager */
    if (lcmaps_runPluginManager(request, lcmaps_cred, req_username, npols, policynames, LCMAPS_NORMAL_MODE)) {
        lcmaps_log_debug(1,"%s() error: could not run plugin manager\n", __func__);
        goto fail_lcmaps_run_and_return_username;
    }

    /*
     * Apparently lcmaps succeeded, so let's get the username from the credential repository
     * and return happily
     */

    /* Now try to get the userid drom the credential data */
    uid    = getCredentialData(UID,     &cntUid);
    if (uid)
    {
        if ( (user_info = getpwuid(uid[0])) == NULL )
        {
            lcmaps_log_debug(1, "%s(): LCMAPS could not find the username related to uid: %d\n", __func__, uid[0]);
            return 1;
        }
        (*usernamep) = strdup(user_info->pw_name);
	if (*usernamep == NULL)	{
	    lcmaps_log(LOG_ERR, "%s: Out of memory\n", __func__);
	    goto fail_lcmaps_run_and_return_username;
	}
    }
    else
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS could not find any uid\n", __func__);
        return 1;
    }

    /* succes */
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2,"%s(): succeeded\n", __func__);
    return 0;

 fail_lcmaps_run_and_return_username:
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2,"%s(): failed\n", __func__);
    return 1;
}


/******************************************************************************
Function:   lcmaps_run_and_return_poolindex
Description:
    do the user mapping based on the provided list of policies (first successful
    policy found in the lcmaps policy file (lcmaps.db) will result in the user
    mapping) and return the poolindex
    This interface was intended to be used by a wrapper function for the
        DYNAMIC ACCOUNTS SERVICE (DAS).

Parameters:
    request     : RSL string
    user_cred   : user globus credential handle
    poolindexp  : pointer to poolindex (to be freed by calling application).
                  Note: poolindexp should be non-NULL at the start !
    npols       : number of policies to be considered for evaluation
    policynames : the names of the policies to be considered for evaluation

Returns:
    0: mapping succeeded
    1: mapping failed
******************************************************************************/
#if ALLOW_EMPTY_CREDENTIALS
int lcmaps_run_and_return_poolindex(
        char * user_dn_tmp,
        gss_cred_id_t user_cred,
        lcmaps_request_t request,
        char ** poolindexp,
        int npols,
        char ** policynames
)
#else
int lcmaps_run_and_return_poolindex(
        gss_cred_id_t user_cred,
        lcmaps_request_t request,
        char ** poolindexp,
        int npols,
        char ** policynames
)
#endif
{
    char *           user_dn      = NULL;
    char *           poolindex    = NULL;
    char **          tmpptr       = NULL;
    int              cntpoolindex = -1;
    int              rc           = 0;

    if (lcmaps_initialized == 0)
    {
        lcmaps_log(LOG_ERR,"LCMAPS has to be initialized first !\n");
        goto fail_lcmaps_run_and_return_poolindex;
    }
    lcmaps_log_time(LOG_DEBUG, "LCMAPS credential mapping request\n");

    lcmaps_log_debug(3, "Using \"%s\" interface of LCMAPS\n", __func__);
    if (poolindexp == NULL)
        goto fail_lcmaps_run_and_return_poolindex;

    *poolindexp = NULL;

    /*
     * Create lcmaps credential:
     *     fill it with the dn and extract it again
     */
    if ((rc = lcmaps_credential_init(&lcmaps_cred)) != LCMAPS_CRED_SUCCESS) {
        if (rc == LCMAPS_CRED_INVOCATION_ERROR) {
            lcmaps_log(LOG_ERR, "%s() error: lcmaps_cred does not exist (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_return_poolindex;
        } else {
            lcmaps_log(LOG_ERR, "%s() error: cannot initialize lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_return_poolindex;
        }
    }

    /* Store the gss_cred_id_t and the derivatives */
    rc = lcmaps_credential_store_gss_cred_id_t_and_sub_elements(user_cred, &lcmaps_cred);
    if (rc != LCMAPS_CRED_SUCCESS) {
        if (rc == LCMAPS_CRED_INVOCATION_ERROR) {
            lcmaps_log(LOG_ERR, "%s() error: lcmaps_cred does not exist\n", __func__);
            goto fail_lcmaps_run_and_return_poolindex;
        } else if (rc == LCMAPS_CRED_NO_GSS_CRED) {
            /* Use provided DN */
            lcmaps_log(LOG_ERR, "%s() WARNING: empty credential found !\n", __func__);
        } else if (rc == LCMAPS_CRED_NO_FQAN) {
            /* No FQANs found or valid */
            lcmaps_log(LOG_DEBUG, "%s() Debug: No VOMS FQANs were found, continuing without them.\n", __func__);
        } else {
            lcmaps_log(LOG_ERR, "%s() error: storing gss_credential or its derivative credentials\n", __func__);
            goto fail_lcmaps_run_and_return_poolindex;
        }
    }

#if 0
#if ALLOW_EMPTY_CREDENTIALS
    if ((rc = lcmaps_credential_store_dn(user_dn_tmp, &lcmaps_cred)) != LCMAPS_CRED_SUCCESS)
    {
        if (rc == LCMAPS_CRED_NO_DN)
        {
            lcmaps_log(LOG_ERR, "%s() error: storing EMPTY dn in lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_return_poolindex;
        }
        else
        {
            lcmaps_log(LOG_ERR, "%s() error: storing dn in lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_return_poolindex;
        }
    }
#endif
#endif

    user_dn = lcmaps_credential_get_dn(lcmaps_cred);
    if (user_dn == NULL)
    {
        lcmaps_log(LOG_ERR, "%s() error: user DN empty\n", __func__);
        goto fail_lcmaps_run_and_return_poolindex;
    }

    /* Run PluginManager */
    if (lcmaps_runPluginManager(request, lcmaps_cred, NULL, npols, policynames, LCMAPS_NORMAL_MODE)) {
        lcmaps_log_debug(1,"%s() error: could not run plugin manager\n", __func__);
        goto fail_lcmaps_run_and_return_poolindex;
    }

    /*
     * Apparently lcmaps succeeded, so let's get the poolindex from the credential repository
     * and return happily
     */
    tmpptr = (char **) getCredentialData(POOL_INDEX, &cntpoolindex);
    if (tmpptr && (cntpoolindex > 0))
    {
        poolindex = (char *)(*(tmpptr));
        lcmaps_log_debug(5, "%s(): found %d poolindeces at address = %p\n", __func__, cntpoolindex, (void*)tmpptr);
        lcmaps_log_debug(5, "lcmaps_run_and_return_poolindex(): found this poolindex %s\n", poolindex);
        *poolindexp = strdup(poolindex);
	if ( *poolindexp ==NULL)    {
	    lcmaps_log(LOG_ERR, "%s: Out of memory\n", __func__);
	    goto fail_lcmaps_run_and_return_poolindex;
	}
    }
    else
    {
        lcmaps_log(LOG_ERR, "%s(): LCMAPS could not find the poolindex\n", __func__);
        goto fail_lcmaps_run_and_return_poolindex;
    }

    /* succes */
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2,"%s(): succeeded\n", __func__);
    return 0;

 fail_lcmaps_run_and_return_poolindex:
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2, "%s(): failed\n", __func__);
    return 1;
}


/******************************************************************************
Function:   lcmaps_run_with_pem_and_return_account
Description:
    LCMAPS runs receiving a proxy credential in a PEM formatted string.
    A list of policies may be provided.
    The allocated uid, gids and the poolindex will be returned to the calling
    application.
    This interface is intended to be used by the modified suexec wrapper.

Parameters:
    user_dn     : the user DN
    pem_string: : the PEM-encoded string containing the user proxy
    mapcounter: : The counter which will be added to the poolindex, effectively enabling
                  multiple account mappings
    request     : RSL string
    npols       : number of policies to be considered for evaluation
    policynames : the names of the policies to be considered for evaluation
    puid        : pointer to the uid found (output parameter)
    ppgid_list  : pointer to the list of primary gids found (output parameter)
    pnpgid      : pointer to the number of primary gids found (output parameter)
    psgid_list  : pointer to the list of secondary gids found (output parameter)
    pnsgid      : pointer to the number of secondary gids found (output parameter)
    poolindexp  : pointer to poolindex string (output parameter)

Returns:
    0: mapping succeeded
    1: mapping failed
******************************************************************************/
int lcmaps_run_with_pem_and_return_account(
        char *            user_dn,
        char *            pem_string,
        int               mapcounter,
        lcmaps_request_t  request,
        int               npols,
        char **           policynames,
        uid_t *           puid,
        gid_t **          ppgid_list,
        int *             pnpgid,
        gid_t **          psgid_list,
        int *             pnsgid,
        char **           poolindexp
)
{
    int      rc             = 0;
    uid_t *  uid            = NULL;
    int      cntUid         = -1;
    gid_t *  priGid         = NULL;
    int      cntPriGid      = -1;
    gid_t *  secGid         = NULL;
    int      cntSecGid      = -1;
    char **  poolindex_list = NULL;
    char *   poolindex      = NULL;
    int      cntpoolindex   = 0;

    if (lcmaps_initialized == 0)
    {
        lcmaps_log(LOG_ERR,"LCMAPS has to be initialized first !\n");
        goto fail_lcmaps_run_with_pem_and_return_account;
    }
    lcmaps_log_time(LOG_DEBUG, "LCMAPS credential mapping request\n");

    lcmaps_log_debug(3, "Using \"%s\" interface of LCMAPS\n", __func__);
    /*
     * Create lcmaps credential:
     *     fill it with the dn and extract it again
     */
    if ((rc = lcmaps_credential_init(&lcmaps_cred)) != LCMAPS_CRED_SUCCESS) {
        if (rc == LCMAPS_CRED_INVOCATION_ERROR) {
            lcmaps_log(LOG_ERR, "%s() error: lcmaps_cred does not exist (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_pem_and_return_account;
        } else {
            lcmaps_log(LOG_ERR, "%s() error: cannot initialize lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_pem_and_return_account;
        }
    }

    /* Store the credentials to be used by the plug-ins */
    if ((rc = lcmaps_credential_store_pem_string_and_sub_elements(pem_string, &lcmaps_cred)) != LCMAPS_CRED_SUCCESS) {
        if (rc == LCMAPS_CRED_NO_PEM_STRING)
        {
            lcmaps_log(LOG_ERR, "%s() error: PEM string is empty (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_pem_and_return_account;
        }
        else if ( (rc & LCMAPS_CRED_NO_X509_CHAIN) == LCMAPS_CRED_NO_X509_CHAIN)
        {
            lcmaps_log(LOG_ERR, "%s() error: Cannot retrieve X509 chain from PEM string (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_pem_and_return_account;
        }
        else if ( (rc & LCMAPS_CRED_NO_X509_CRED) == LCMAPS_CRED_NO_X509_CRED)
        {
            lcmaps_log(LOG_ERR, "%s() error: Cannot retrieve proxy certificate from PEM string (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_pem_and_return_account;
        }
        else if ( (rc & LCMAPS_CRED_NO_DN) == LCMAPS_CRED_NO_FQAN)
        {
            lcmaps_log(LOG_ERR, "%s() error: Cannot retrieve DN from proxy certificate (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_pem_and_return_account;
        }
        /* If no fqans are found, this should not be fatal, but only if no other errors are found !*/
        else if (rc == LCMAPS_CRED_NO_FQAN) /* Only issue a warning, no failure */
        {
            lcmaps_log_debug(1, "%s() warning: fqan list is empty (rc = 0x%x)\n", __func__, rc);
        }
        else
        {
            lcmaps_log(LOG_ERR, "%s() error: Error storing PEM string (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_pem_and_return_account;
        }
    }
    if (lcmaps_credential_store_mapcounter(mapcounter, &lcmaps_cred) != LCMAPS_CRED_SUCCESS)
    {
        lcmaps_log(LOG_ERR, "%s() error: storing mapcounter in lcmaps_cred\n", __func__);
        goto fail_lcmaps_run_with_pem_and_return_account;
    }

    /* Run PluginManager */
    if (lcmaps_runPluginManager(request, lcmaps_cred, NULL, npols, policynames, LCMAPS_NORMAL_MODE)) {
        lcmaps_log_debug(1, "%s() error: could not run plugin manager\n", __func__);
        goto fail_lcmaps_run_with_pem_and_return_account;
    }

    /* At this stage extract uid, gids and poolindex */
    /* First the uid */
    uid              = (uid_t *) getCredentialData(UID,     &cntUid);
    if (uid == NULL)
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS could not find any uid\n", __func__);
        goto fail_lcmaps_run_with_pem_and_return_account;
    }
    else if (cntUid != 1)
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS found %d uids, only 1 allowed\n", __func__, cntUid);
        goto fail_lcmaps_run_with_pem_and_return_account;
    }
    else
    {
        *puid = uid[0];
    }
    /* Then the primary gids */
    priGid           = (gid_t *) getCredentialData(PRI_GID, &cntPriGid);
    if (priGid == NULL)
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS could not find any gid, at least one required!\n", __func__);
        goto fail_lcmaps_run_with_pem_and_return_account;
    }
    else
    {
        *pnpgid = cntPriGid;
        *ppgid_list = priGid;
    }
    /* Then the secondary gids */
    secGid           = (gid_t *) getCredentialData(SEC_GID, &cntSecGid);
    if (secGid == NULL)
    {
        lcmaps_log_debug(1, "%s(): LCMAPS found no secondary groups\n", __func__);
    }
    else
    {
        *pnsgid = cntSecGid;
        *psgid_list = secGid;
    }
    /* And finally the poolindex */
    poolindex_list = (char **) getCredentialData(POOL_INDEX, &cntpoolindex);
    if (poolindex_list && (cntpoolindex > 0))
    {
        poolindex = (char *)(*(poolindex_list));
        lcmaps_log_debug(5, "%s(): found %d poolindeces at address = %p\n", __func__, cntpoolindex, poolindex);
        *poolindexp = strdup(poolindex);
	if ( *poolindexp ==NULL)    {
	    lcmaps_log(LOG_ERR, "%s: Out of memory\n", __func__);
	    goto fail_lcmaps_run_with_pem_and_return_account;
	}
    }
    else
    {
        lcmaps_log_debug(5, "%s(): LCMAPS could not find the poolindex\n", __func__);
        /* goto fail_lcmaps_run_and_return_poolindex; */
    }

    /* succes */
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2, "%s(): succeeded\n", __func__);
    return 0;

 fail_lcmaps_run_with_pem_and_return_account:
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2, "%s(): failed\n", __func__);
    return 1;
}



/******************************************************************************
Function:   lcmaps_run_and_verify_account_from_pem
Description:
    LCMAPS verifies the account mapping based on the proxy credential
    in a PEM formatted string.
    A list of policies may be provided.
    This interface is intended to be used by the modified suexec wrapper.

Parameters (only input):
    user_dn     : the user DN
    pem_string: : the PEM-encoded string containing the user proxy
    uid         : the uid of the account that should be verified
    pgid_list   : the list of primary gids of the account that should be verified
    npgid       : the number of primary gids of the account that should be verified
    sgid_list   : the list of secondary gids of the account that should be verified
    nsgid       : the number of secondary gids of the account that should be verified
    poolindex   : poolindex string of the account that should be verified
    request     : RSL string
    npols       : number of policies to be considered for evaluation
    policynames : the names of the policies to be considered for evaluation

Returns:
    0: verification succeeded
    1: verification failed
******************************************************************************/
int lcmaps_run_and_verify_account_from_pem(
        char *            user_dn,
        char *            pem_string,
        uid_t             uid,
        gid_t *           pgid_list,
        int               npgid,
        gid_t *           sgid_list,
        int               nsgid,
        char *            poolindex,
        lcmaps_request_t  request,
        int               npols,
        char **           policynames
)
{
    int                 rc                  = 0;
    uid_t *             puid                = NULL;
    gid_t *             priGid              = NULL;
    gid_t *             secGid              = NULL;
    uid_t               uid_found;
    int                 nuid_found;
    gid_t *             pgid_list_found     = NULL;
    int                 npgid_found;
/*    gid_t *             sgid_list_found     = NULL;*/
    int                 nsgid_found;
    char *              poolindex_found     = NULL;
    char **             poolindex_list      = NULL;
    int                 cntpoolindex        = 0;
    struct passwd *     user_info           = NULL;
    struct group *      group_info          = NULL;
    char *              lcmaps_verify_type  = NULL;

    if (lcmaps_initialized == 0)
    {
        lcmaps_log(LOG_ERR,"LCMAPS has to be initialized first !\n");
        goto fail_lcmaps_run_and_verify_account_from_pem;
    }
    lcmaps_log_time(LOG_DEBUG, "LCMAPS credential mapping request\n");

    lcmaps_log_debug(3, "Using \"%s\" interface of LCMAPS\n", __func__);
    /*
     * Create lcmaps credential:
     *     fill it with the dn and extract it again
     */
    if ((rc = lcmaps_credential_init(&lcmaps_cred)) != LCMAPS_CRED_SUCCESS) {
        if (rc == LCMAPS_CRED_INVOCATION_ERROR) {
            lcmaps_log(LOG_ERR, "%s() error: lcmaps_cred does not exist (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_verify_account_from_pem;
        } else {
            lcmaps_log(LOG_ERR, "%s() error: cannot initialize lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_verify_account_from_pem;
        }
    }

    /* Store the credentials to be used by the plug-ins */
    if ((rc = lcmaps_credential_store_pem_string_and_sub_elements(pem_string, &lcmaps_cred)) != LCMAPS_CRED_SUCCESS) {
        if (rc == LCMAPS_CRED_NO_PEM_STRING)
        {
            lcmaps_log(LOG_ERR, "%s() error: PEM string is empty (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_verify_account_from_pem;
        }
        else if ( (rc & LCMAPS_CRED_NO_X509_CHAIN) == LCMAPS_CRED_NO_X509_CHAIN)
        {
            lcmaps_log(LOG_ERR, "%s() error: Cannot retrieve X509 chain from PEM string (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_verify_account_from_pem;
        }
        else if ( (rc & LCMAPS_CRED_NO_X509_CRED) == LCMAPS_CRED_NO_X509_CRED)
        {
            lcmaps_log(LOG_ERR, "%s() error: Cannot retrieve proxy certificate from PEM string (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_verify_account_from_pem;
        }
        else if ( (rc & LCMAPS_CRED_NO_DN) == LCMAPS_CRED_NO_FQAN)
        {
            lcmaps_log(LOG_ERR, "%s() error: Cannot retrieve DN from proxy certificate (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_verify_account_from_pem;
        }
        /* If no fqans are found, this should not be fatal, but only if no other errors are found !*/
        else if (rc == LCMAPS_CRED_NO_FQAN) /* Only issue a warning, no failure */
        {
            lcmaps_log_debug(1, "%s() warning: fqan list is empty (rc = 0x%x)\n", __func__, rc);
        }
        else
        {
            lcmaps_log(LOG_ERR, "%s() error: Error storing PEM string (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_verify_account_from_pem;
        }
    }

    if ((rc = lcmaps_credential_store_requested_account(
        &uid,
        &pgid_list,
        &npgid,
        &sgid_list,
        &nsgid,
        &poolindex,
        &lcmaps_cred)) != LCMAPS_CRED_SUCCESS)
    {
        if (rc == LCMAPS_CRED_ERROR)
        {
            lcmaps_log(LOG_ERR, "%s() error: Error filling lcmaps_account_info_t (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_verify_account_from_pem;
        }
        else
        {
            lcmaps_log(LOG_ERR, "%s() error: Unknown error (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_and_verify_account_from_pem;
        }
    }

    /* Run PluginManager in verification mode ! */
    if (lcmaps_runPluginManager(request, lcmaps_cred, NULL, npols, policynames, LCMAPS_VERIFICATION_MODE)) {
        lcmaps_log_debug(1, "%s() error: could not run plugin manager\n", __func__);
        goto fail_lcmaps_run_and_verify_account_from_pem;
    }

    /* At this stage extract uid, gids and poolindex and compare them to input */
    /* First the uid */
    puid = (uid_t *) getCredentialData(UID, &nuid_found);
    if (puid == NULL)
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS could not find any uid\n", __func__);
        goto fail_lcmaps_run_and_verify_account_from_pem;
    }
    else if (nuid_found != 1)
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS found %d uids, only 1 allowed\n", __func__, nuid_found);
        goto fail_lcmaps_run_and_verify_account_from_pem;
    }
    else
    {
        uid_found = puid[0];
    }
    /* Then the primary gids */
    priGid = (gid_t *) getCredentialData(PRI_GID, &npgid_found);
    if ((npgid_found < 1) || (priGid == NULL))
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS could not find any (primary) gid, at least one required!\n", __func__);
        goto fail_lcmaps_run_and_verify_account_from_pem;
    }
    else
    {
        pgid_list_found = priGid;
    }

    /* Then the secondary gids */
    secGid           = (gid_t *) getCredentialData(SEC_GID, &nsgid_found);
    if (secGid == NULL)
    {
        lcmaps_log_debug(1, "%s(): LCMAPS found no secondary groups\n", __func__);
    }
/*    else
    {
        sgid_list_found = secGid;
    }*/

    /* And finally the poolindex */
    poolindex_list = (char **) getCredentialData(POOL_INDEX, &cntpoolindex);
    if (poolindex_list && (cntpoolindex > 0))
    {
        poolindex_found = (char *)(*(poolindex_list));
        lcmaps_log_debug(5, "%s(): found %d poolindeces starting at address = %p\n", __func__, cntpoolindex, (void*)poolindex_list);
        lcmaps_log_debug(5, "%s(): found this poolindex %s\n", __func__, poolindex_found);
    }
    else
    {
        lcmaps_log_debug(5, "%s(): LCMAPS could not find the poolindex\n", __func__);
    }

    /*
     * Compare account-to-be-verified with account-found
     * Default: only verify uid
     * if LCMAPS_VERIFY_TYPE is set to
     *     "uid": only verify the uid
     *     "uid_pgid": only verify the uid and primary gid
     *     "all": verify uid, primary gid and all secondary gids
     */
    user_info = getpwuid(uid);
    lcmaps_verify_type = getenv("LCMAPS_VERIFY_TYPE");
    if ( (lcmaps_verify_type == NULL) || (strcmp("uid", lcmaps_verify_type) == 0) )
    {
        if (uid_found != uid) /* uid verification failure */
        {
            lcmaps_log(LOG_ERR, "%s(): LCMAPS could not verify the requested account (uid=%d, name=%s)\n",
                       __func__, uid, (user_info?user_info->pw_name:NULL));
            goto fail_lcmaps_run_and_verify_account_from_pem;
        }
    }
    else if (strcmp("uid_pgid", lcmaps_verify_type) == 0)
    {
        if (uid_found != uid) /* uid verification failure */
        {
            lcmaps_log(LOG_ERR, "%s(): LCMAPS could not verify the requested account (uid=%d, name=%s)\n",
                       __func__, uid, (user_info?user_info->pw_name:NULL));
            goto fail_lcmaps_run_and_verify_account_from_pem;
        }
        if ( (pgid_list == NULL) || (npgid < 1) )
        {
            lcmaps_log(LOG_ERR, "%s(): LCMAPS was requested to verify the primary gids, but did not receive any on input (failure)", __func__);
            goto fail_lcmaps_run_and_verify_account_from_pem;
        }
        group_info = getgrgid(pgid_list[0]);
        if (pgid_list[0] != pgid_list_found[0]) /* pgid verification failure */
        {
            lcmaps_log(LOG_ERR, "%s(): LCMAPS could not verify the requested primary gid (gid=%d, gname=%s)\n",
                       __func__, pgid_list[0], (group_info?group_info->gr_name:NULL));
            goto fail_lcmaps_run_and_verify_account_from_pem;
        }
    }
    else
    {
        lcmaps_log(LOG_ERR, "%s: Unknown verification type: %s() (failure)\n", __func__, lcmaps_verify_type);
        goto fail_lcmaps_run_and_verify_account_from_pem;
    }


    /* succes */
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2, "%s(): succeeded\n", __func__);
    return 0;

 fail_lcmaps_run_and_verify_account_from_pem:
    lcmaps_release_cred(&lcmaps_cred);
    lcmaps_log_debug(2, "%s(): failed\n", __func__);
    return 1;
}


/******************************************************************************
Function:   lcmaps_run_with_stack_of_x509_and_return_account
Description:
    LCMAPS runs receiving a certificate chain, containing at least an End-Entity Certificate.
    A list of policies may be provided.
    The allocated uid, gids and the poolindex will be returned to the calling
    application.
    This interface is intended to be used by the modified suexec wrapper.

Parameters:
    cert_chain  : The certificate chain to use for the mapping
    mapcounter: : The counter which will be added to the poolindex, effectively enabling
                  multiple account mappings
    request     : RSL string
    npols       : number of policies to be considered for evaluation
    policynames : the names of the policies to be considered for evaluation
    puid        : pointer to the uid found (output parameter)
    ppgid_list  : pointer to the list of primary gids found (output parameter)
    pnpgid      : pointer to the number of primary gids found (output parameter)
    psgid_list  : pointer to the list of secondary gids found (output parameter)
    pnsgid      : pointer to the number of secondary gids found (output parameter)
    poolindexp  : pointer to poolindex string (output parameter)

Returns:
    0: mapping succeeded
    1: mapping failed
******************************************************************************/
int lcmaps_run_with_stack_of_x509_and_return_account(
        STACK_OF(X509) *  cert_chain,
        int               mapcounter,
        lcmaps_request_t  request,
        int               npols,
        char **           policynames,
        uid_t *           puid,
        gid_t **          ppgid_list,
        int *             pnpgid,
        gid_t **          psgid_list,
        int *             pnsgid,
        char **           poolindexp
)
{
    int      rc             = 0;
    int      i              = 0;
    uid_t *  uid            = NULL;
    int      cntUid         = -1;
    gid_t *  priGid         = NULL;
    int      cntPriGid      = -1;
    gid_t *  secGid         = NULL;
    int      cntSecGid      = -1;
    char **  poolindex_list = NULL;
    char *   poolindex      = NULL;
    int      cntpoolindex   = 0;

    if (lcmaps_initialized == 0)
    {
        lcmaps_log(LOG_ERR,"LCMAPS has to be initialized first !\n");
        goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
    }
    lcmaps_log_time(LOG_DEBUG, "LCMAPS credential mapping request\n");

    lcmaps_log_debug(3, "Using \"%s\" interface of LCMAPS\n", __func__);
    /*
     * Create lcmaps credential:
     *     fill it with the dn and extract it again
     */
    if ((rc = lcmaps_credential_init(&lcmaps_cred)) != LCMAPS_CRED_SUCCESS) {
        if (rc == LCMAPS_CRED_INVOCATION_ERROR) {
            lcmaps_log(LOG_ERR, "%s() error: lcmaps_cred does not exist (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
        } else {
            lcmaps_log(LOG_ERR, "%s() error: cannot initialize lcmaps_cred (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
        }
    }

    /* Store the passed X509 credentials */
    if ((rc = lcmaps_credential_store_x509_and_sub_elements(NULL, cert_chain, &lcmaps_cred)) != LCMAPS_CRED_SUCCESS)
    {
        if ( (rc & LCMAPS_CRED_NO_X509_CHAIN) == LCMAPS_CRED_NO_X509_CHAIN)
        {
            lcmaps_log(LOG_ERR, "%s() error: Cannot retrieve X509 chain from PEM string (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
        }
        else if ( (rc & LCMAPS_CRED_NO_X509_CRED) == LCMAPS_CRED_NO_X509_CRED)
        {
            lcmaps_log(LOG_ERR, "%s() error: Cannot retrieve EEC or proxy certificate from certificate chain (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
        }
        else if ( (rc & LCMAPS_CRED_NO_DN) == LCMAPS_CRED_NO_FQAN)
        {
            lcmaps_log(LOG_ERR, "%s() error: Cannot retrieve DN from certificate chain (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
        }
        /* If no fqans are found, this should not be fatal, but only if no other errors are found !*/
        else if (rc == LCMAPS_CRED_NO_FQAN) /* Only issue a warning, no failure */
        {
            lcmaps_log_debug(1, "%s() warning: fqan list is empty (rc = 0x%x)\n", __func__, rc);
        }
        else
        {
            lcmaps_log(LOG_ERR, "%s() error: Error storing X.509 chain string (rc = 0x%x)\n", __func__, rc);
            goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
        }
    }
    if (lcmaps_credential_store_mapcounter(mapcounter, &lcmaps_cred) != LCMAPS_CRED_SUCCESS)
    {
        lcmaps_log(LOG_ERR, "%s() error: storing mapcounter in lcmaps_cred\n", __func__);
        goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
    }

    /* Run PluginManager */
    if (lcmaps_runPluginManager(request, lcmaps_cred, NULL, npols, policynames, LCMAPS_NORMAL_MODE)) {
        lcmaps_log_debug(1, "%s() error: could not run plugin manager\n", __func__);
        goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
    }

    /* At this stage extract uid, gids and poolindex */
    /* First the uid */
    uid              = (uid_t *) getCredentialData(UID,     &cntUid);
    if (uid == NULL)
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS could not find any uid\n", __func__);
        goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
    }
    else if (cntUid != 1)
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS found %d uids, only 1 allowed\n", __func__, cntUid);
        goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
    }
    else
    {
        *puid = uid[0];
    }
    /* Then the primary gids */
    priGid           = (gid_t *) getCredentialData(PRI_GID, &cntPriGid);
    if (priGid == NULL || cntPriGid<=0)
    {
        lcmaps_log_debug(1, "%s() error: LCMAPS could not find any gid, at least one required!\n", __func__);
        goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
    }
    else
    {
        *pnpgid = cntPriGid;
	/* Note: cntPriGid>0 */
        *ppgid_list = malloc(sizeof(gid_t) * (size_t)cntPriGid);
	if (*ppgid_list == NULL)    {
	    lcmaps_log(LOG_ERR, "%s: Out of memory\n",__func__);
	    goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
	}

        for (i = 0; i < cntPriGid; i++)
        {
            (*ppgid_list)[i] = priGid[i];
        }
    }
    /* Then the secondary gids */
    secGid           = (gid_t *) getCredentialData(SEC_GID, &cntSecGid);
    if (secGid == NULL || cntSecGid<=0)
    {
        lcmaps_log_debug(1, "%s(): LCMAPS found no secondary groups\n", __func__);
    }
    else
    {
        *pnsgid = cntSecGid;
	/* Note: cntPriGid>0 */
        *psgid_list = malloc(sizeof(gid_t) * (size_t)cntSecGid);
	if (*psgid_list == NULL)    {
	    lcmaps_log(LOG_ERR, "%s: Out of memory\n",__func__);
	    goto fail_lcmaps_run_with_stack_of_x509_and_return_account;
	}

        for (i = 0; i < cntSecGid; i++)
        {
            (*psgid_list)[i] = secGid[i];
        }
    }

    /* And finally the poolindex */
    poolindex_list = (char **) getCredentialData(POOL_INDEX, &cntpoolindex);
    if (poolindex_list && (cntpoolindex > 0))
    {
        poolindex = (char *)(*(poolindex_list));
        /* lcmaps_log_debug(3, "%s(): found %d poolindeces starting at address = %p\n", __func__, cntpoolindex, poolindex_list); */
        lcmaps_log_debug(5, "%s(): found this poolindex %s\n", __func__, poolindex);
        *poolindexp = poolindex;
    }
    else
    {
        lcmaps_log_debug(5, "%s(): LCMAPS could not find the poolindex\n", __func__);
        /* goto fail_lcmaps_run_with_stack_of_x509_and_return_account; */
    }

    /* succes */
    lcmaps_release_cred_leave_STACK_OF_X509(&lcmaps_cred);
    lcmaps_log_debug(2, "%s(): succeeded\n", __func__);
    return 0;

 fail_lcmaps_run_with_stack_of_x509_and_return_account:
    lcmaps_release_cred_leave_STACK_OF_X509(&lcmaps_cred);
    lcmaps_log_debug(2, "%s(): failed\n", __func__);
    return 1;
}

#endif /* LCMAPS_GSI_MODE */
/*
 * END OF THE GSI DEPENDENT INTERFACES
 */



/* Return the Version information of LCMAPS. This information is set in the configure.ac file and parsed here */
int lcmaps_get_major_version (void)
{
    int major = 0, minor = 0, patch = 0;
    if (sscanf(VERSION, "%d.%d.%d", &major, &minor, &patch) != 3)
    {
        lcmaps_log(LOG_ERR, "%s() error: LCMAPS could parse compile-time version information.\n", __func__);
        return 0;
    }
    else
        return major;
}
int lcmaps_get_minor_version (void)
{
    int major = 0, minor = 0, patch = 0;
    if (sscanf(VERSION, "%d.%d.%d", &major, &minor, &patch) != 3)
    {
        lcmaps_log(LOG_ERR, "%s() error: LCMAPS could parse compile-time version information.\n", __func__);
        return 0;
    }
    else
        return minor;
}
int lcmaps_get_patch_version (void)
{
    int major = 0, minor = 0, patch = 0;
    if (sscanf(VERSION, "%d.%d.%d", &major, &minor, &patch) != 3)
    {
        lcmaps_log(LOG_ERR, "%s() error: LCMAPS could parse compile-time version information.\n", __func__);
        return 0;
    }
    else
        return patch;
}


/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/lcmaps.c,v $
    $Date: 2014-07-07 21:28:25 +0200 (Mon, 07 Jul 2014) $
    $Revision: 17839 $
    $Author: msalle $
******************************************************************************/
