# Migration guide

This guide intends to describe the major changes between the JerryScript 1.0 and 2.0 versions.
In addtion it is designed to provide a guide on how to modify the 1.0 version code to a
2.0 compliant code.

During the development it was important to minimize the changes in the API functions and types.
Each API method removal or chang is described below providing a ***before*** and ***after***
code example.
For more information on the current API methods please check the [API reference](02.API-REFERENCE.md) document.

# Short list of removed/renamed headers, types, functions, and macros

***Removed legacy headers***

- `jerry-internal.h`

***Renamed headers***

- `jerry-api.h` to `jerryscript.h`
- `jerry-port.h` to `jerryscript-port.h`

***Removed API types***

- `jerry_char_ptr_t` usage replaced with `jerry_char_t *`
- `jerry_object_free_callback_t` replaced by `jerry_object_native_free_callback_t`

***Removed API methods***

- `jerry_get_memory_limits`
- `jerry_get_object_native_handle` replaced by `jerry_get_object_native_pointer`
- `jerry_set_object_native_handle` replaced by `jerry_set_object_native_pointer`
- `jerry_value_set_abort_flag` replaced by `jerry_create_abort_from_value`
- `jerry_value_has_abort_flag` replaced by `jerry_value_is_abort`
- `jerry_value_set_error_flag` replaced by `jerry_create_error_from_value`
- `jerry_value_has_error_flag` replaced by `jerry_value_is_error`
- `jerry_value_clear_error_flag` replaced by `jerry_get_value_from_error`
- `jerry_get_value_without_error_flag` replaced by `jerry_get_value_from_error`
- `jerry_parse_and_save_snapshot` replaced by `jerry_generate_snapshot`
- `jerry_parse_and_save_function_snapshot` replaced by `jerry_generate_function_snapshot`


***Removed unused configuration macros***

- `CONFIG_MEM_DATA_LIMIT_MINUS_HEAP_SIZE`
- `CONFIG_MEM_STACK_LIMIT`
- `CONFIG_VM_STACK_FRAME_INLINED_VALUES_NUMBER`
- `CONFIG_ECMA_GLOBAL_ENVIRONMENT_DECLARATIVE`
- All `CONFIG_..` macros have been renamed to use the `JERRY_` prefix format.


# Modified API functions

## Error manipulating functions

The most important changes in the API are releated to error handling and manipulation.

### jerry_value_set_abort_flag

This function was replaced with  [`jerry_create_abort_from_value`](02.API-REFERENCE.md#jerry_create_abort_from_value).
Take note of the second argument of the new `jerry_create_abort_from_value` function which controls if the
first argument should be usable after the call or not.

**Before**

```c
{
  jerry_value_t value;
  // create or acquire value
  // ...

  jerry_value_set_abort_flag (&value);

  jerry_release_value (value);
}
```

**After**

```c
{
  jerry_value_t value;
  // create or acquire value
  // ...

  jerry_value_t abort = jerry_create_abort_from_value (value, true);
  // using the 'value' variable after release is invalid

  jerry_release_value (abort);
}
```

- OR

```c
{
  jerry_value_t value;
  ... // create or acquire value

  jerry_value_t abort = jerry_create_abort_from_value (value, false);
  // both 'abort' and 'value' can be used and must be released when they are no longer needed

  jerry_release_value (abort);
  jerry_release_value (value);
}
```

### jerry_value_has_abort_flag

This function was renamed to  [`jerry_value_is_abort`](02.API-REFERENCE.md#jerry_value_is_abort).

**Before**

```c
{
  jerry_value_t value;
  // create or acquire value
  // ...

  if (jerry_value_has_abort_flag (value))
  {
    // ...
  }

  jerry_release_value (value);
}
```

**After**

```c
{
  jerry_value_t value;
  // create or acquire value
  // ...

  if (jerry_value_is_abort (value))
  {
    // ...
  }

  jerry_release_value (value);
}
```

### jerry_value_set_error_flag

This function was replaced with  [`jerry_create_error_from_value`](02.API-REFERENCE.md#jerry_create_error_from_value).
Take note of the second argument of the new `jerry_create_error_from_value` function which controls if the
first argument should be usable after the call or not.

**Before**

```c
{
  jerry_value_t value;
  // create or acquire value
  // ...

  jerry_value_set_error_flag (&value);

  jerry_release_value (value);
}
```

**After**

```c
{
  jerry_value_t value;
  // create or acquire value
  // ...

  jerry_value_t error = jerry_create_error_from_value (value, true);
  // using the 'value' variable after release is invalid

  jerry_release_value (error);
}
```

- OR

```c
{
  jerry_value_t value;
  // create or acquire value
  // ...

  jerry_value_t error = jerry_create_error_from_value (value, false);
  // both 'error' and 'value' can be used and must be released when they are no longer needed

  jerry_release_value (error);
  jerry_release_value (value);
}
```

### jerry_value_has_error_flag

This function was renamed to  [`jerry_value_is_error`](02.API-REFERENCE.md#jerry_value_is_error).

**Before**

```c
{
  jerry_value_t value;
  // create or acquire value
  // ...

  if (jerry_value_has_error_flag (value))
  {
    // ...
  }

  jerry_release_value (value);
}
```

**After**

```c
{
  jerry_value_t value;
  // create or acquire value
  // ...

  if (jerry_value_is_error (value))
  {
    // ...
  }

  jerry_release_value (value);
}
```

### jerry_value_clear_error_flag AND jerry_get_value_without_error_flag

These functions were merged into [`jerry_get_value_from_error`](02.API-REFERENCE.md#jerry_get_value_from_error).
Please note the second argument of the new function which controls if the first argument passed should be released
or not.

**Before**

```c
{
  jerry_value_t value;
  // create or acquire value
  // ...

  jerry_value_set_error_flag (&value);
  jerry_value_clear_error_flag (&value);
  // or
  jerry_value_t real_value = jerry_get_value_without_error_flag (value);

  jerry_release_value (value);
  jerry_release_value (real_value);
}
```

**After**

```c
{
  jerry_value_t value;
  // create or acquire value
  // ...

  jerry_value_t error = jerry_create_error_from_value (value, true);

  jerry_value_t real_value = jerry_get_value_from_error (error, true);

  jerry_release_value (real_value);
}
```

## Other functions changed

### jerry_register_magic_strings

In case of the `jerry_register_magic_strings` function the change is that
the first argument's base type `jerry_char_ptr_t` was changed to `jerry_char_t*`.
For more details see: [`jerry_register_magic_strings`](02.API-REFERENCE.md#jerry_register_magic_strings).

In the following code parts please take note of the type used for the `magic_string_items` array.

**Before**

```c
{
  // must be static, because 'jerry_register_magic_strings' does not copy
  // the items must be sorted by size at first, then lexicographically
  static const jerry_char_ptr_t magic_string_items[] = {
                                                          (const jerry_char_ptr_t) "magicstring1",
                                                          (const jerry_char_ptr_t) "magicstring2",
                                                          (const jerry_char_ptr_t) "magicstring3"
                                                        };
  uint32_t num_magic_string_items = (uint32_t) (sizeof (magic_string_items) / sizeof (jerry_char_ptr_t));

  // must be static, because 'jerry_register_magic_strings' does not copy
  static const jerry_length_t magic_string_lengths[] = { 12, 12, 12 };
  jerry_register_magic_strings (magic_string_items, num_magic_string_items, magic_string_lengths);
}
```

**After**

```c
{
  // must be static, because 'jerry_register_magic_strings' does not copy
  // the items must be sorted by size at first, then lexicographically
  static const jerry_char_t *magic_string_items[] = {
                                                      (const jerry_char_t *) "magicstring1",
                                                      (const jerry_char_t *) "magicstring2",
                                                      (const jerry_char_t *) "magicstring3"
                                                    };
  uint32_t num_magic_string_items = (uint32_t) (sizeof (magic_string_items) / sizeof (jerry_char_t *));

  // must be static, because 'jerry_register_magic_strings' does not copy
  static const jerry_length_t magic_string_lengths[] = { 12, 12, 12 };
  jerry_register_magic_strings (magic_string_items, num_magic_string_items, magic_string_lengths);
}
```

## Snapshot generating API

### jerry_parse_and_save_snapshot

This function was replaced with [`jerry_generate_snapshot`](02.API-REFERENCE.md#jerry_generate_snapshot).
The function returns an error object if there was any problem during snapshot generation and
if there was no problem the return value is a number value containing the snapshot size in bytes.

**Before**

```c
{
  static uint32_t global_mode_snapshot_buffer[256];
  const jerry_char_t *code_to_snapshot_p = (const jerry_char_t *) "(function () { return 'string from snapshot'; }) ();";

  size_t global_mode_snapshot_size =
    jerry_parse_and_save_snapshot (code_to_snapshot_p,
                                   strlen ((const char *) code_to_snapshot_p),
                                   true,
                                   false,
                                   global_mode_snapshot_buffer,
                                   sizeof (global_mode_snapshot_buffer) / sizeof (uint32_t));
  // use "global_mode_snapshot_buffer"
}
```

**After**

```c
{
  static uint32_t global_mode_snapshot_buffer[256];
  const jerry_char_t *code_to_snapshot_p = (const jerry_char_t *) "(function () { return 'string from snapshot'; }) ();";

  jerry_value_t generate_result;
  generate_result = jerry_generate_snapshot (NULL,
                                             0,
                                             code_to_snapshot_p,
                                             strlen ((const char *) code_to_snapshot_p),
                                             global_mode_snapshot_buffer,
                                             sizeof (global_mode_snapshot_buffer) / sizeof (uint32_t));
  if (jerry_value_is_error (generate_result))
  {
    // There was a problem during snapshot generation, for example there is a SyntaxError.
    // Use the "generate_result" to check the error.
  }
  else
  {
    size_t snapshot_size = (size_t) jerry_get_number_value (generate_result);
    // use "global_mode_snapshot_buffer"
  }
  jerry_release_value (generate_result);
}
```

### jerry_parse_and_save_function_snapshot

This function was replaced with [`jerry_generate_function_snapshot`](02.API-REFERENCE.md#jerry_parse_and_save_function_snapshot).
The function returns an error object if there was any problem during snapshot generation and
if there was no problem the return value is a number value containing the snapshot size in bytes.

**Before**

```c
{
  static uint32_t func_snapshot_buffer[1024];

  const jerry_char_t *args_p = (const jerry_char_t *) "a, b";
  const jerry_char_t *src_p = (const jerry_char_t *) "return a + b;";

  size_t func_snapshot_size =
  jerry_parse_and_save_function_snapshot (src_p,
                                          strlen ((const char *) src_p),
                                          args_p,
                                          strlen ((const char *) args_p),
                                          false,
                                          func_snapshot_buffer,
                                          sizeof (func_snapshot_buffer) / sizeof (uint32_t));
  // check "function_snapshot_size" and use "func_snapshot_buffer"
}
```

**After**

```c
{
  static uint32_t func_snapshot_buffer[1024];

  const jerry_char_t *args_p = (const jerry_char_t *) "a, b";
  const jerry_char_t *src_p = (const jerry_char_t *) "return a + b;";

  jerry_value_t generate_result;
  generate_result = jerry_generate_function_snapshot (NULL,
                                                      0,
                                                      src_p,
                                                      strlen ((const char *) src_p),
                                                      args_p,
                                                      strlen ((const char *) args_p),
                                                      0,
                                                      func_snapshot_buffer,
                                                      sizeof (func_snapshot_buffer) / sizeof (uint32_t));
  if (jerry_value_is_error (generate_result))
  {
    // There was a problem during snapshot generation, for example there is a SyntaxError.
    // Use the "generate_result" to check the error.
  }
  else
  {
    size_t snapshot_size = (size_t) jerry_get_number_value (generate_result);
    // use "func_snapshot_buffer"
  }

  jerry_release_value (generate_result)
}
```

## Garbage collection

### jerry_gc

The [`jerry_gc`](02.API-REFERENCE.md#jerry_gc) function was modified to handle an argument which represents the pressure for the garbage collector.
For more information checkout the [`jerry_gc_mode_t`](02.API-REFERENCE.md#jerry_gc_mode_t) reference.

**Before**

```c
{
  jerry_gc ();
}
```

**After**

```c
{
  jerry_gc (JERRY_GC_PRESSURE_LOW);
}
```

## jerry_eval

The third argument of [`jerry_eval`](02.API-REFERENCE.md#jerry_eval) has been changed
from `bool` to [`jerry_parse_opts_t`](02.API-REFERENCE.md#jerry_parse_opts_t).

**Before**

```c
const jerry_char_t *str_to_eval = (const jerry_char_t *) "1 + 1";
jerry_value_t ret_val = jerry_eval (str_to_eval,
                                    strlen ((const char *) str_to_eval),
                                    false);
```

**After**

```c
const jerry_char_t *str_to_eval = (const jerry_char_t *) "1 + 1";
jerry_value_t ret_val = jerry_eval (str_to_eval,
                                    strlen ((const char *) str_to_eval),
                                    JERRY_PARSE_NO_OPTS);
```

## Port API

### jerry_port_get_time_zone

The port API of handling timezones has been changed. The previous interface did not
allow timezones to be handled correctly, even if the host system was up to the task.
Check [the related issue](https://github.com/jerryscript-project/jerryscript/issues/1661)
for more details.

The new port API function name is [jerry_port_get_local_time_zone_adjustment](05.PORT-API.md#date-1].

Below is the default implementations for both versions:

**Before**

```c
bool jerry_port_get_time_zone (jerry_time_zone_t *tz_p)
{
  struct timeval tv;
  struct timezone tz;

  /* gettimeofday may not fill tz, so zero-initializing */
  tz.tz_minuteswest = 0;
  tz.tz_dsttime = 0;

  if (gettimeofday (&tv, &tz) != 0)
  {
    return false;
  }

  tz_p->offset = tz.tz_minuteswest;
  tz_p->daylight_saving_time = tz.tz_dsttime > 0 ? 1 : 0;

  return true;
} /* jerry_port_get_time_zone */
```

**After**

```c
double jerry_port_get_local_time_zone_adjustment (double unix_ms,
                                                  bool is_utc)
{
  struct tm tm;
  time_t now = (time_t) (unix_ms / 1000);
  localtime_r (&now, &tm);
  if (!is_utc)
  {
    now -= tm.tm_gmtoff;
    localtime_r (&now, &tm);
  }
  return ((double) tm.tm_gmtoff) * 1000;
} /* jerry_port_get_local_time_zone_adjustment */
```

## Native pointers

The assignment of native pointers (previously called handles) have been changed
since v1.0. In the previous version only one native pointer could be assigned to
a `jerry_value_t`. Now it is allowed to register multiple native infos, which
can be accessed with the corresponding
[`jerry_object_native_info_t`](02.API-REFERENCE.md#jerry_object_native_info_t).
The old functions were removed and replaced by new ones.

- `jerry_object_free_callback_t` callback type is replaced by `jerry_object_native_info_t`
- `jerry_get_object_native_handle` is replaced by [`jerry_get_object_native_pointer`](02.API-REFERENCE.md#jerry_get_object_native_pointer)
- `jerry_set_object_native_handle` is replaced by [`jerry_set_object_native_pointer`](02.API-REFERENCE.md#jerry_set_object_native_pointer)

**Before**

```c
struct
{
  int data;
} my_info;

static void
handler_construct_freecb (uintptr_t native_p)
{
  // Invoked when the JS object is released and the
  // native data should be freed.

  struct my_info *info = (struct my_info *) native_p;
  free (info);
}

void
demo (void)
{
  jerry_value_t this_val;
  // create or acquire this_val
  // ...

  struct my_info *info = (struct my_info *) malloc (sizeof (struct my_info));
  info->data = 11;

  // setting the native handle
  jerry_set_object_native_handle (this_val,
                                  (uintptr_t) info,
                                  handler_construct_freecb);
  // ...
  // reading back the native handle
  uintptr_t ptr = (uintptr_t) NULL;
  bool is_ok = jerry_get_object_native_handle (this_val, &ptr);
  if (is_ok)
  {
    struct my_info *obj_info = (struct my_info *) ptr;
    // use "obj_info"
  }
}
```


**After**

```c
struct
{
  int data;
} my_info;

static void
handler_construct_freecb (void *native_p)
{
  // Invoked when the JS object is released and the
  // native data should be freed.

  struct my_info *info = (struct my_info *) native_p;
  free (info);
}

static const jerry_object_native_info_t my_info_type_info =
{
  .free_cb = handler_construct_freecb
};

void
demo (void)
{
  jerry_value_t this_val;
  // create or acquire this_val
  // ...

  struct my_info *info = (struct my_info *) malloc (sizeof (struct my_info));
  info->data = 11;

  // setting the native handle
  jerry_set_object_native_pointer (this_val,
                                   info,
                                   &my_info_type_info);
  // ...
  // reading back the native handle pointed by the "my_info_type_info" variable
  void *ptr = NULL;
  bool has_p = jerry_get_object_native_pointer (this_val, &ptr, &my_info_type_info);
  if (has_p)
  {
    struct my_info *obj_info = (struct my_info *) ptr;
    // use "obj_info"
  }
}
```

# New API functions

In this section the new API functions are listed.

## Built-in objects

***ArrayBuffer***

- [`jerry_create_arraybuffer`](02.API-REFERENCE.md#jerry_create_arraybuffer)
- [`jerry_create_arraybuffer_external`](02.API-REFERENCE.md#jerry_create_arraybuffer_external)
- [`jerry_get_arraybuffer_pointer`](02.API-REFERENCE.md#jerry_get_arraybuffer_pointer)

***DataView***

- [`jerry_create_dataview`](02.API-REFERENCE.md#jerry_create_dataview)
- [`jerry_value_is_dataview`](02.API-REFERENCE.md#jerry_value_is_dataview)
- [`jerry_get_dataview_buffer`](02.API-REFERENCE.md#jerry_get_dataview_buffer)

***JSON***

- [`jerry_json_parse`](02.API-REFERENCE.md#jerry_json_parse)
- [`jerry_json_stringify`](02.API-REFERENCE.md#jerry_json_stringify)

***Number***

- [`jerry_create_number_infinity`](02.API-REFERENCE.md#jerry_create_number_infinity)
- [`jerry_create_number_nan`](02.API-REFERENCE.md#jerry_create_number_nan)

***Promise***

- [`jerry_run_all_enqueued_jobs`](02.API-REFERENCE.md#jerry_run_all_enqueued_jobs)
- [`jerry_create_promise`](02.API-REFERENCE.md#jerry_create_promise)
- [`jerry_resolve_or_reject_promise`](02.API-REFERENCE.md#jerry_resolve_or_reject_promise)
- [`jerry_value_is_promise`](02.API-REFERENCE.md#jerry_value_is_promise)

***RegExp***

- [`jerry_create_regexp`](02.API-REFERENCE.md#jerry_create_regexp)
- [`jerry_create_regexp_sz`](02.API-REFERENCE.md#jerry_create_regexp_sz)

***String***

- [`jerry_substring_to_utf8_char_buffer`](02.API-REFERENCE.md#jerry_substring_to_utf8_char_buffer)
- [`jerry_get_utf8_string_size`](02.API-REFERENCE.md#jerry_get_utf8_string_size)
- [`jerry_get_utf8_string_length`](02.API-REFERENCE.md#jerry_get_utf8_string_length)
- [`jerry_create_string_from_utf8`](02.API-REFERENCE.md#jerry_create_string_from_utf8)
- [`jerry_create_string_sz_from_utf8`](02.API-REFERENCE.md#jerry_create_string_sz_from_utf8)

***Symbol***

- [`jerry_create_symbol`](02.API-REFERENCE.md#jerry_create_symbol)
- [`jerry_get_symbol_descriptive_string`](02.API-REFERENCE.md#jerry_get_symbol_descriptive_string)
- [`jerry_value_is_symbol`](02.API-REFERENCE.md#jerry_value_is_symbol)

***TypedArray***

- [`jerry_create_typedarray`](02.API-REFERENCE.md#jerry_create_typedarray)
- [`jerry_create_typedarray_for_arraybuffer`](02.API-REFERENCE.md#jerry_create_typedarray_for_arraybuffer)
- [`jerry_create_typedarray_for_arraybuffer_sz`](02.API-REFERENCE.md#jerry_create_typedarray_for_arraybuffer_sz)
- [`jerry_get_typedarray_type`](02.API-REFERENCE.md#jerry_get_typedarray_type)
- [`jerry_get_typedarray_length`](02.API-REFERENCE.md#jerry_get_typedarray_length)
- [`jerry_get_typedarray_buffer`](02.API-REFERENCE.md#jerry_get_typedarray_buffer)
- [`jerry_value_is_typedarray`](02.API-REFERENCE.md#jerry_value_is_typedarray)


## Instances and memory management

***JerryScript instances***

- [`jerry_create_context`](02.API-REFERENCE.md#jerry_create_context)
- [`jerry_get_context_data`](02.API-REFERENCE.md#jerry_get_context_data)

***Memory management***

- [`jerry_heap_alloc`](02.API-REFERENCE.md#jerry_heap_alloc)
- [`jerry_heap_free`](02.API-REFERENCE.md#jerry_heap_free)


## Operations with JavaScript values

***Binary operations***

- [`jerry_binary_operation`](02.API-REFERENCE.md#jerry_binary_operation)

***Error manipulating***

- [`jerry_get_error_type`](02.API-REFERENCE.md#jerry_get_error_type)
- [`jerry_get_backtrace`](02.API-REFERENCE.md#jerry_get_backtrace)

***Native pointers***

- [`jerry_delete_object_native_pointer`](02.API-REFERENCE.md#jerry_delete_object_native_pointer)
- [`jerry_objects_foreach_by_native_info`](02.API-REFERENCE.md#jerry_objects_foreach_by_native_info)

***Property***

- [`jerry_delete_property_by_index`](02.API-REFERENCE.md#jerry_delete_property_by_index)
- [`jerry_objects_foreach`](02.API-REFERENCE.md#jerry_objects_foreach)


## Debugger

- [`jerry_debugger_is_connected`](07.DEBUGGER.md#jerry_debugger_is_connected)
- [`jerry_debugger_stop`](07.DEBUGGER.md#jerry_debugger_stop)
- [`jerry_debugger_continue`](07.DEBUGGER.md#jerry_debugger_continue)
- [`jerry_debugger_stop_at_breakpoint`](07.DEBUGGER.md#jerry_debugger_stop_at_breakpoint)
- [`jerry_debugger_wait_for_client_source`](07.DEBUGGER.md#jerry_debugger_wait_for_client_source)
- [`jerry_debugger_send_output`](07.DEBUGGER.md#jerry_debugger_send_output)
- [`jerry_debugger_send_log`](07.DEBUGGER.md#jerry_debugger_send_log)


## Other

- [`jerry_is_feature_enabled`](02.API-REFERENCE.md#jerry_is_feature_enabled)
- [`jerry_parse_and_save_literals`](02.API-REFERENCE.md#jerry_parse_and_save_literals)
- [`jerry_set_vm_exec_stop_callback`](02.API-REFERENCE.md#jerry_set_vm_exec_stop_callback)


## Port API functions

- [`jerry_port_normalize_path`](05.PORT-API.md#jerry_port_normalize_path)
- [`jerry_port_read_source`](05.PORT-API.md#jerry_port_read_source)
- [`jerry_port_release_source`](05.PORT-API.md#jerry_port_release_source)
- [`jerry_port_print_char`](05.PORT-API.md#jerry_port_print_char)
- [`jerry_port_get_current_context`](05.PORT-API.md#jerry_port_get_current_context)
- [`jerry_port_fatal`](05.PORT-API.md#jerry_port_fatal)
- [`jerry_port_sleep`](05.PORT-API.md#jerry_port_sleep)
