/* 
 * File: gmlsearch.c
 *
 * Medialib search and playlist editor
 * Copyright (c) 2005-2008 Johannes Heimansberg
 *
 * requires GTK+ 2.6 or better and XMMS2
 *
 * Released under the GNU General Public License v2
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkdialog.h>
#include "gmedialib.h"
#include "gtrackinfo/trackinfo.h"
#include "gtrackinfo/gtrackinfo.h"
#include "sdecode/sdecode.h"

void gmedialib_refresh(GMedialib*, xmmsc_connection_t*);

extern xmmsc_connection_t *connection; /* available from gxmms2.c */

struct search_params
{
	GMedialib          *gml;
	xmmsc_connection_t *conn;
	gchar              *category;
	gchar              *search_str;
	gboolean            start;
	xmmsc_coll_t       *coll_query;
};

struct rehash_notify_data
{
	GMedialib *gml;
	guint      remaining_ids;
};

void gml_search_do_search(GMedialib *gml, guint search_for, gchar* search_str)
{
		gtk_entry_set_text(GTK_ENTRY(gml->search.entry_searchstring), 
		                   search_str);
		gtk_combo_box_set_active(GTK_COMBO_BOX(gml->search.search_combo), 
		                         search_for);
		gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(gml->search.checkbutton_exact), TRUE);
		gtk_button_clicked(GTK_BUTTON(gml->search.button_search));
}

static void cb_search_on_tree_view_row_activated(GtkTreeView       *view, 
                                                 GtkTreePath       *path,
                                                 GtkTreeViewColumn *col,
                                                 gpointer           userdata)
{
	GtkTreeModel   *model;
	GtkTreeIter     iter;
	xmmsc_result_t *res;
	guint           id;
	
	model = gtk_tree_view_get_model(view);

	if (gtk_tree_model_get_iter(model, &iter, path)) {
		gtk_tree_model_get(model, &iter, 0, &id, -1);

		res = xmmsc_playlist_add_id(connection, NULL, id);
		xmmsc_result_unref(res);
	}
}

static void add_selected_foreach_func(GtkTreeModel *model,
                                      GtkTreePath  *path,
                                      GtkTreeIter  *selected_row,
                                      gpointer      userdata)
{
	xmmsc_result_t *res;
	guint           id;

	gtk_tree_model_get(model, selected_row, 0, &id, -1);

	res = xmmsc_playlist_add_id(connection, NULL, id);
	xmmsc_result_unref(res);
}

static void cb_search_add_button_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib        *gml = (GMedialib *)pt;
	GtkTreeSelection *sel;

	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->search.list_results));
	gtk_tree_selection_selected_foreach(sel, add_selected_foreach_func, 
	                                    (gpointer)gml);
}

static void insert_selected_foreach_func(GtkTreeModel *model,
                                         GtkTreePath  *path,
                                         GtkTreeIter  *selected_row,
                                         gpointer      userdata)
{
	xmmsc_result_t *res;
	guint           id;
	GMedialib      *gml = (GMedialib *)userdata;
	gint            pos = gml_pl_get_selected_pos(gml);

	gtk_tree_model_get(model, selected_row, 0, &id, -1);

	res = xmmsc_playlist_insert_id(connection, NULL, pos + 1, id);
	xmmsc_result_unref(res);
}

static void cb_search_popup_insert_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib        *gml = (GMedialib *)pt;
	gint              pos = gml_pl_get_selected_pos(gml);
	GtkTreeSelection *sel;

	if (pos != -1) {
		sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->search.list_results));
		gtk_tree_selection_selected_foreach(sel, insert_selected_foreach_func, 
		                                    (gpointer)gml);
	}


}

static int n_rehash(xmmsv_t *val, void *userdata)
{   
	struct rehash_notify_data* data = (struct rehash_notify_data *)userdata;

	data->remaining_ids--;
	
	if (data->remaining_ids == 0)
		gmedialib_refresh(data->gml, connection);
	return FALSE;
}   

static void cb_search_rehash_button_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib        *gml = (GMedialib *)pt;
	GtkTreeView      *tree_view = GTK_TREE_VIEW(gml->search.list_results);
	GtkTreeSelection *selection = gtk_tree_view_get_selection(tree_view);
	GtkTreeModel     *model = gtk_tree_view_get_model(tree_view);
	GList            *sel, *cur;
	static struct rehash_notify_data data;

	sel = gtk_tree_selection_get_selected_rows(selection, &model);
	data.gml = gml;
	data.remaining_ids = g_list_length(sel);
	
	for (cur = sel; cur; cur = g_list_next(cur)) {
		GtkTreeIter     iter;
		xmmsc_result_t *res;
		guint           id;

		gtk_tree_model_get_iter(model, &iter, (GtkTreePath*)cur->data);
		gtk_tree_model_get(model, &iter, 0, &id, -1);

		res = xmmsc_medialib_rehash (connection, id);
		xmmsc_result_notifier_set(res, n_rehash, &data);
		xmmsc_result_unref(res);
	}
	g_list_free(sel);
}

static void cb_search_remove_button_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib        *gml = (GMedialib *)pt;
	GtkTreeView      *tree_view = GTK_TREE_VIEW(gml->search.list_results);
	GtkTreeSelection *selection = gtk_tree_view_get_selection(tree_view);
	GtkTreeModel     *model = gtk_tree_view_get_model(tree_view);
	GList            *sel, *cur;

	sel = g_list_last(gtk_tree_selection_get_selected_rows(selection, &model));
	for (cur = sel; cur; cur = g_list_previous(cur)) {
		GtkTreeIter     iter;
		xmmsc_result_t *res;
		guint           id;

		gtk_tree_model_get_iter(model, &iter, (GtkTreePath*) cur->data);
		gtk_tree_model_get(model, &iter, 0, &id, -1);

		res = xmmsc_medialib_remove_entry(connection, id);
		xmmsc_result_unref(res);

		gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
	}
	g_list_free(sel);
}

gchar* map_category(gint category)
{
	gchar *str = NULL;

	switch (category) {
		case 0:
			break;
		case 1:
			str = "artist";
			break;
		case 2:
			str = "title";
			break;
		case 3:
			str = "album";
			break;
		case 4:
			str = "url";
			break;
		default:
			g_return_val_if_reached(NULL);
	}

	return g_strdup(str);
}

static void cb_search_search_button_pressed(GtkWidget *widget, gpointer pt)
{
	GMedialib *gml = (GMedialib *)pt;
	gchar     *str, *category;
	gint       active_category;
	gboolean   exact;

	str = (gchar *)gtk_entry_get_text(GTK_ENTRY(gml->search.entry_searchstring));
	active_category = gtk_combo_box_get_active(
		GTK_COMBO_BOX(gml->search.search_combo));
	category = map_category(active_category);	
	exact = gtk_toggle_button_get_active(
	        	GTK_TOGGLE_BUTTON(gml->search.checkbutton_exact));

	if (strlen(str) > 2 || exact) {
		gtk_widget_set_sensitive(gml->search.button_search,      FALSE);
		gtk_widget_set_sensitive(gml->search.entry_searchstring, FALSE);
		gtk_widget_set_sensitive(gml->search.search_combo,       FALSE);
		gtk_widget_set_sensitive(gml->search.checkbutton_exact,  FALSE);
		if (gml->search.ls != NULL)
			gtk_list_store_clear(gml->search.ls);

		g_free(gml->search.last_search_str);
		gml->search.last_search_str = g_strdup(str);
		gml->search.last_category = active_category;
		gml->search.last_exact = exact;

		gmedialib_search(gml, connection, category, str, exact);
		g_free(category);
	}
}

static void cb_search_popup_search_artist_pressed(GtkWidget *menuitem,
                                                  gpointer   userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel = 
		gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->search.list_results));
	GtkTreeModel     *model = GTK_TREE_MODEL(gml->search.ls);
	GList            *list = gtk_tree_selection_get_selected_rows(sel, &model);
	GtkTreePath      *path = list->data;
	GtkTreeIter		  selected_row;
	gchar            *artist;

	gtk_tree_model_get_iter(GTK_TREE_MODEL(gml->search.ls),
	                        &selected_row, path);

	gtk_tree_model_get(model, &selected_row, 1, &artist, -1);

	gml_search_do_search(gml, SEARCH_ARTIST, artist);

	g_free(artist);
	g_list_foreach(list, (GFunc)gtk_tree_path_free, NULL);
	g_list_free(list);
}

static void cb_search_popup_search_album_pressed(GtkWidget *menuitem,
                                                  gpointer  userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel = 
		gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->search.list_results));
	GtkTreeModel     *model = GTK_TREE_MODEL(gml->search.ls);
	GList            *list = gtk_tree_selection_get_selected_rows(sel, &model);
	GtkTreePath      *path = list->data;
	GtkTreeIter		  selected_row;
	gchar            *album;

	gtk_tree_model_get_iter(GTK_TREE_MODEL(gml->search.ls),
	                        &selected_row, path);

	gtk_tree_model_get(model, &selected_row, 3, &album, -1);

	/* remove tracknr [xxx] from album name: */
	if (album[strlen(album) - 1] == ']' &&
	    album[strlen(album) - 2 - gml->album_tracknr_digits] == '[')
		album[strlen(album) - 3 - gml->album_tracknr_digits] = '\0';

	gml_search_do_search(gml, SEARCH_ALBUM, album);

	g_free(album);
	g_list_foreach(list, (GFunc)gtk_tree_path_free, NULL);
	g_list_free(list);
}

static int n_media_lib_get_info(xmmsv_t *val, void *arg)
{
	trackinfo  track;
	GtkWidget *dialog;

	trackinfo_update(val, &track);
	dialog = gtrackinfo_new();
	gtrackinfo_set_info(GTRACKINFO(dialog), &track);
	gtk_widget_show(dialog);
	return FALSE;
}

static void search_info_about_selection(GtkTreeModel *model,
                                        GtkTreePath  *path,
                                        GtkTreeIter  *iter,
                                        gpointer      userdata)
{
	guint           id;
	xmmsc_result_t *res;

	gtk_tree_model_get(model, iter, 0, &id, -1);
	res = xmmsc_medialib_get_info(connection, id);
	xmmsc_result_notifier_set(res, n_media_lib_get_info, NULL);
	xmmsc_result_unref(res);
}

static void cb_search_popup_info_pressed(GtkWidget *menuitem,
                                         gpointer   userdata)
{
	GMedialib        *gml = (GMedialib *)userdata;
	GtkTreeSelection *sel;
	sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(gml->search.list_results));
	gtk_tree_selection_selected_foreach(sel, search_info_about_selection, NULL);
}

static void cb_column_id_visible(GtkWidget *menuitem,
                                 gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_search_set_column_id_visible(gml, !gml_search_get_column_id_visible(gml));
}

static void cb_column_artist_visible(GtkWidget *menuitem,
                                     gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_search_set_column_artist_visible(gml, !gml_search_get_column_artist_visible(gml));
}

static void cb_column_track_visible(GtkWidget *menuitem,
                                    gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_search_set_column_track_visible(gml, !gml_search_get_column_track_visible(gml));
}

static void cb_column_album_visible(GtkWidget *menuitem,
                                    gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_search_set_column_album_visible(gml, !gml_search_get_column_album_visible(gml));
}

static void cb_column_rating_visible(GtkWidget *menuitem,
                                     gpointer   userdata)
{
	GMedialib *gml = (GMedialib *)userdata;
	gml_search_set_column_rating_visible(gml, !gml_search_get_column_rating_visible(gml));
}

static void search_view_popup_menu(GtkWidget      *treeview,
                                   GdkEventButton *event,
                                   gpointer        userdata)
{
	GtkWidget        *menu, *menu_columns, *menuitem;
	GtkTreeSelection *selection;
	GMedialib        *gml = (GMedialib *)userdata;
	gboolean          multiple = FALSE, none = FALSE;
	int               i;

	struct submenu
	{
		gchar    *name;
		gboolean (*get_fn)();
		void     (*callback_fn)();
	};

	struct submenu sm_columns[] = 
		{ {"ID", gml_search_get_column_id_visible, cb_column_id_visible},
		  {"Artist", gml_search_get_column_artist_visible, cb_column_artist_visible},
		  {"Track", gml_search_get_column_track_visible, cb_column_track_visible},
		  {"Album", gml_search_get_column_album_visible, cb_column_album_visible},
		  {"Rating", gml_search_get_column_rating_visible, cb_column_rating_visible},
		  { NULL, NULL, NULL } };

	/* Submenu: Visible columns */
	menu_columns = gtk_menu_new();

	i = 0;
	while (sm_columns[i].name != NULL) {
		menuitem = gtk_check_menu_item_new_with_mnemonic(sm_columns[i].name);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menuitem),
		                               sm_columns[i].get_fn(gml));
		gtk_menu_shell_append(GTK_MENU_SHELL(menu_columns), menuitem);
		g_signal_connect(menuitem, "activate",
		                 G_CALLBACK(sm_columns[i].callback_fn), gml);
		i++;
	}


	selection = gtk_tree_view_get_selection(
	            	GTK_TREE_VIEW(gml->search.list_results));
	if (gtk_tree_selection_count_selected_rows(selection) > 1)
		multiple = TRUE;
	if (gtk_tree_selection_count_selected_rows(selection) == 0)
		none = TRUE;

	menu = gtk_menu_new();

	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_ADD, NULL);
	if (none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_search_add_button_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_with_mnemonic("_Insert after selected playlist item");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), 
	                              gtk_image_new_from_stock(GTK_STOCK_ADD, 
	                                                       GTK_ICON_SIZE_SMALL_TOOLBAR));
	if (none || gml_pl_get_selected_pos(gml) == -1)
		gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_search_popup_insert_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_from_stock(GTK_STOCK_DIALOG_INFO, NULL);
	if (multiple || none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_search_popup_info_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_with_mnemonic("Search for _Artist");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), 
	                              gtk_image_new_from_stock(GTK_STOCK_FIND, 
	                                                       GTK_ICON_SIZE_SMALL_TOOLBAR));
	if (multiple || none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_search_popup_search_artist_pressed),
	                 userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_image_menu_item_new_with_mnemonic("Search for Al_bum");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem), 
	                              gtk_image_new_from_stock(GTK_STOCK_FIND, 
	                                                       GTK_ICON_SIZE_SMALL_TOOLBAR));
	if (multiple || none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_search_popup_search_album_pressed),
	                 userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem =  gtk_image_menu_item_new_with_mnemonic("Re_hash");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem),
	                              gtk_image_new_from_stock(GTK_STOCK_EXECUTE,
								                           GTK_ICON_SIZE_SMALL_TOOLBAR));
	
	if (none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
	                 G_CALLBACK(cb_search_rehash_button_pressed), userdata);

	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem =  gtk_image_menu_item_new_with_mnemonic("_Remove from Medialib");
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(menuitem),
	                              gtk_image_new_from_stock(GTK_STOCK_DELETE,
								                           GTK_ICON_SIZE_SMALL_TOOLBAR));
	if (none) gtk_widget_set_sensitive(menuitem, FALSE);

	g_signal_connect(menuitem, "activate",
                     G_CALLBACK(cb_search_remove_button_pressed), userdata);
	
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	menuitem = gtk_menu_item_new_with_mnemonic("_Columns");
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem), menu_columns);
	gtk_menu_shell_append(GTK_MENU_SHELL(menu), menuitem);

	gtk_widget_show_all(menu);

	gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
	               (event != NULL) ? event->button : 0,
	               gdk_event_get_time((GdkEvent*)event));
}

static gboolean cb_search_selection_button_press(GtkWidget      *treeview,
                                                 GdkEventButton *event,
                                                 gpointer        userdata)
{
	gboolean result = FALSE;
	if (event->type == GDK_BUTTON_PRESS  &&  event->button == 3) {
		search_view_popup_menu(treeview, event, userdata);
		result = TRUE;
	}
	return result;
}

static gboolean cb_search_selection_popup_menu(GtkWidget *treeview, 
                                               gpointer   userdata)
{
	search_view_popup_menu(treeview, NULL, userdata);
	return TRUE;
}

static int n_add_result_notified(xmmsv_t *val, void *arguments)
{
	static const char *pref[] = { "client/generic", "server", "plugins/*",
	                              "client/*", "*", NULL };

	struct search_params *arg = (struct search_params*)arguments;
	gchar                *artist = "[Unknown Artist]";
	gchar                *album  = "[Unknown Album]";
	gchar                *title  = "[Unknown Title]";
	gchar                *filename = NULL;
	gint32                id, tracknr = -1, rating = -1;
	static gint           tmp_id = -1;
	GtkTreeIter           iter;

	val = xmmsv_propdict_to_dict(val, pref);
	xmmsv_dict_entry_get_int(val,    "id",      &id);
	xmmsv_dict_entry_get_int(val,    "tracknr", &tracknr);
	xmmsv_dict_entry_get_int(val,    "rating",  &rating);
	rating--;
	if (xmmsv_dict_entry_get_string(val, "title", (const gchar **)&title)) {
		xmmsv_dict_entry_get_string(val, "artist", (const gchar **)&artist);
		xmmsv_dict_entry_get_string(val, "album", (const gchar **)&album);
	} else {
		gchar       *url, *url_utf8;
		xmmsv_dict_entry_get_string(val, "url", (const gchar **)&url);
		if (url) {
			url_utf8 = decode_url(url);
			filename = g_path_get_basename(url_utf8);
			g_free(url_utf8);
			if (filename) {
				artist = filename;
				title = filename;
			}
		}
	}
	if (tmp_id != id) {
		gchar album_with_tracknr[256];
		if (tracknr > 0 && arg->gml->album_tracknr_digits > 0)
			if (arg->gml->album_tracknr_digits >= 3)
				snprintf(album_with_tracknr, 256, "%s [%03d]", album, tracknr);
			else
				snprintf(album_with_tracknr, 256, "%s [%02d]", album, tracknr);
		else
			snprintf(album_with_tracknr, 256, "%s", album);
		tmp_id = id;
		if (arg->gml->search.ls != NULL) {
			gtk_list_store_append(arg->gml->search.ls, &iter);
			gtk_list_store_set(arg->gml->search.ls, &iter, 
			                   0, id,    1, artist, 
			                   2, title, 3, album_with_tracknr, -1);
			if (rating >= 0 && rating <= 4)
				gtk_list_store_set(arg->gml->search.ls, &iter, 
			    	               4, arg->gml->playlist.icon_rating[rating], -1);
		}
	}
	if (filename)
		g_free(filename);
	xmmsv_unref(val);
	return FALSE;
}

static void add_results(GList *n, struct search_params *arg)
{
	for (; n; n = g_list_next(n)) {
		gint32			id = 0;
		xmmsc_result_t 	*res = NULL;

		id = *(gint32 *)(n->data);
		g_free(n->data);

		if (id) {
			res = xmmsc_medialib_get_info(arg->conn, id);
			xmmsc_result_notifier_set(res, n_add_result_notified, arg);
			xmmsc_result_unref(res);
		} else {
			printf("Empty result!\n");
		}
	}
	g_list_free(n);
}

static int n_continue_search(xmmsv_t *val, void *arguments)
{
	struct search_params *arg = (struct search_params*)arguments;
	GList                *n = NULL;
	xmmsv_list_iter_t    *it;
	char const           *xerr;

	if (xmmsv_get_error(val, &xerr)) {
		printf("%s\n", xerr);
	}

	xmmsv_get_list_iter(val, &it);
	for (; xmmsv_list_iter_valid(it); xmmsv_list_iter_next(it)) {
		gint32 *id = g_malloc(sizeof(gint32));
		xmmsv_t *entry;

		xmmsv_list_iter_entry(it, &entry);
		if (!xmmsv_get_int(entry, id)) {
			printf("gxmms2: Broken resultset.\n");
		}

		n = g_list_prepend(n, id);
	}

	add_results(n, arg);
	xmmsc_coll_unref(arg->coll_query);

	gml_set_statusbar_text(arg->gml, "Ready.");
	gtk_widget_set_sensitive(arg->gml->search.button_search,      TRUE);
	gtk_widget_set_sensitive(arg->gml->search.entry_searchstring, TRUE);
	gtk_widget_set_sensitive(arg->gml->search.search_combo,       TRUE);
	gtk_widget_set_sensitive(arg->gml->search.checkbutton_exact,  TRUE);
	return FALSE;
}

gboolean gml_search_get_column_id_visible(GMedialib *gml)
{
	return gml->search.column_id_visible;
}

gboolean gml_search_get_column_artist_visible(GMedialib *gml)
{
	return gml->search.column_artist_visible;
}

gboolean gml_search_get_column_track_visible(GMedialib *gml)
{
	return gml->search.column_track_visible;
}

gboolean gml_search_get_column_album_visible(GMedialib *gml)
{
	return gml->search.column_album_visible;
}

gboolean gml_search_get_column_rating_visible(GMedialib *gml)
{
	return gml->search.column_rating_visible;
}

void gml_search_set_column_id_visible(GMedialib *gml, gboolean visible)
{
	gml->search.column_id_visible = visible;
	gtk_tree_view_column_set_visible(gml->search.column_id, visible);
}

void gml_search_set_column_artist_visible(GMedialib *gml, gboolean visible)
{
	gml->search.column_artist_visible = visible;
	gtk_tree_view_column_set_visible(gml->search.column_artist, visible);
}

void gml_search_set_column_track_visible(GMedialib *gml, gboolean visible)
{
	gml->search.column_track_visible = visible;
	gtk_tree_view_column_set_visible(gml->search.column_track, visible);
}

void gml_search_set_column_album_visible(GMedialib *gml, gboolean visible)
{
	gml->search.column_album_visible = visible;
	gtk_tree_view_column_set_visible(gml->search.column_album, visible);
}

void gml_search_set_column_rating_visible(GMedialib *gml, gboolean visible)
{
	gml->search.column_rating_visible = visible;
	gtk_tree_view_column_set_visible(gml->search.column_rating, visible);
}

gchar *mask_sql_quotes(gchar *string)
{
	int    i, j;
	gchar *result = g_malloc((strlen(string) + 1) * 2 * sizeof(gchar));

	/* mask sql single quotes ' with '' : */
	for (i = 0, j = 0; i < strlen(string); i++) {
		if (string[i] == '\'') {
			result[j] = '\'';
			result[j+1] = '\'';
			j+=2; 
		} else {
			result[j] = string[i];
			j++;
		}
	}
	result[j] = '\0';
	return result;
}

void gmedialib_search(GMedialib *gml,  xmmsc_connection_t *conn, 
                      gchar *category, gchar *search_str, gboolean exact)
{
	xmmsc_result_t              *res;
	static struct search_params  p;
	xmmsc_coll_t                 *coll_query;
	gchar                         pattern[1024];

	if (category == NULL) category = "";

	gml_set_statusbar_text(gml, "Searching...");
	if (!exact)
		snprintf(pattern, 1023, "%s~\"%s\"", category, search_str);
	else
		snprintf(pattern, 1023, "%s:\"%s\"", category, search_str);

	if (!xmmsc_coll_parse(pattern, &coll_query))
		printf("gxmms2: Unable to generate query: %s\n", pattern);
	res = xmmsc_coll_query_ids(conn, coll_query, NULL, 0, 0);

	p.gml = gml;
	p.conn = connection;
	p.category = category;
	p.search_str = search_str;
	p.coll_query = coll_query;

	xmmsc_result_notifier_set(res, n_continue_search, &p);
	xmmsc_result_unref(res);
}

void gmedialib_refresh(GMedialib *gml, xmmsc_connection_t *conn)
{
	gchar *category = map_category(gml->search.last_category);

	gtk_entry_set_text(
		GTK_ENTRY(gml->search.entry_searchstring), gml->search.last_search_str);
	gtk_combo_box_set_active(
		GTK_COMBO_BOX(gml->search.search_combo), gml->search.last_category);
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(gml->search.checkbutton_exact), gml->search.last_exact);
	
	gmedialib_search(gml, conn,	category, gml->search.last_search_str,
	                 gml->search.last_exact);
	
	g_free(category);
}

void gml_create_mlib_search(GMedialib *gml)
{
	GtkCellRenderer  *renderer_pixbuf;
	GtkTreeSelection *selection;
	GtkTooltips      *tooltips;

	gml->search.frame = gtk_frame_new("Media Lib");
	gtk_container_set_border_width(GTK_CONTAINER(gml->search.frame), 2);

	/* table mlib search */
	gml->search.table = gtk_table_new(3, 5, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(gml->search.table), 3);
	gtk_table_set_row_spacings(GTK_TABLE(gml->search.table), 4);
	gtk_table_set_col_spacings(GTK_TABLE(gml->search.table), 4);
	gtk_container_add(GTK_CONTAINER(gml->search.frame), gml->search.table);

	/* ... */
	gml->search.label_search = gtk_label_new("Search for:");
		
	gtk_table_attach(GTK_TABLE(gml->search.table),
		             gml->search.label_search,
		             0, 1, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);

	gml->search.search_combo = gtk_combo_box_new_text();

	gtk_combo_box_append_text(GTK_COMBO_BOX(gml->search.search_combo), 
	                          "Anything");
	gtk_combo_box_append_text(GTK_COMBO_BOX(gml->search.search_combo), 
	                          "Artist");
	gtk_combo_box_append_text(GTK_COMBO_BOX(gml->search.search_combo), 
	                          "Track");
	gtk_combo_box_append_text(GTK_COMBO_BOX(gml->search.search_combo), 
	                          "Album");
	gtk_combo_box_append_text(GTK_COMBO_BOX(gml->search.search_combo), 
	                          "URL");


	gtk_combo_box_set_active(GTK_COMBO_BOX(gml->search.search_combo), 0);

	gtk_table_attach(GTK_TABLE(gml->search.table),
		             gml->search.search_combo,
		             1, 2, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);

	gml->search.entry_searchstring = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(gml->search.table),
		             gml->search.entry_searchstring,
		             2, 3, 0, 1, GTK_FILL | GTK_EXPAND, GTK_SHRINK, 0, 0);

	/* exact button */
	gml->search.checkbutton_exact = gtk_check_button_new_with_mnemonic("e_xact");
	gtk_table_attach(GTK_TABLE(gml->search.table),
		             gml->search.checkbutton_exact,
		             3, 4, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);
	

	/* search button */
	gml->search.button_search = gtk_button_new();
	gml->search.image_search = gtk_image_new_from_stock(GTK_STOCK_FIND, 
	                                                    GTK_ICON_SIZE_MENU);
	gtk_container_add(GTK_CONTAINER(gml->search.button_search), 
	                                gml->search.image_search);

	gtk_table_attach(GTK_TABLE(gml->search.table),
		             gml->search.button_search,
		             4, 5, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* list mlib */
	gml->search.ls = gtk_list_store_new(5, G_TYPE_INT, G_TYPE_STRING, 
	                                    G_TYPE_STRING, G_TYPE_STRING,
	                                    GDK_TYPE_PIXBUF);

	gml->search.list_results = 
		gtk_tree_view_new_with_model(GTK_TREE_MODEL(gml->search.ls));
	gtk_tree_view_set_enable_search(GTK_TREE_VIEW(gml->search.list_results), 
	                                TRUE);
	selection = gtk_tree_view_get_selection(
	            	GTK_TREE_VIEW(gml->search.list_results));
    gtk_tree_selection_set_mode(selection, GTK_SELECTION_MULTIPLE);

	gml->search.scroll_widget = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(gml->search.scroll_widget),
	                               GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_table_attach(GTK_TABLE(gml->search.table),
		             gml->search.scroll_widget,
		             0, 5, 1, 2, 
		             GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, 0, 0);

	gtk_container_add(GTK_CONTAINER(gml->search.scroll_widget), 
	                                gml->search.list_results);

	gml->search.column_id = 
		gtk_tree_view_column_new_with_attributes("ID", gml->renderer_text,
		                                         "text", 0, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->search.list_results), 
	                            gml->search.column_id);
	gtk_tree_view_column_set_sort_column_id(gml->search.column_id, 0);
	gtk_tree_view_column_set_resizable(gml->search.column_id, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->search.column_id,
	                                        gml->renderer_text,
	                                        gml_cell_data_function, gml, NULL);

	gml->search.column_artist = 
		gtk_tree_view_column_new_with_attributes("Artist", gml->renderer_text_ellipsize,
		                                         "text", 1, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->search.list_results),
	                            gml->search.column_artist);
	gtk_tree_view_column_set_sort_column_id(gml->search.column_artist, 1);
	gtk_tree_view_column_set_expand(gml->search.column_artist, TRUE);
	gtk_tree_view_column_set_resizable(gml->search.column_artist, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->search.column_artist,
	                                        gml->renderer_text_ellipsize,
	                                        gml_cell_data_function, gml, NULL);

	gml->search.column_track = 
		gtk_tree_view_column_new_with_attributes("Track", gml->renderer_text_ellipsize,
		                                         "text", 2, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->search.list_results), 
	                            gml->search.column_track);
	gtk_tree_view_column_set_sort_column_id(gml->search.column_track, 2);
	gtk_tree_view_column_set_expand(gml->search.column_track, TRUE);
	gtk_tree_view_column_set_resizable(gml->search.column_track, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->search.column_track,
	                                        gml->renderer_text_ellipsize,
	                                        gml_cell_data_function, gml, NULL);

	gml->search.column_album = 
		gtk_tree_view_column_new_with_attributes("Album [#]", gml->renderer_text_ellipsize,
		                                         "text", 3, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->search.list_results),
	                            gml->search.column_album);
	gtk_tree_view_column_set_sort_column_id(gml->search.column_album, 3);
	gtk_tree_view_column_set_expand(gml->search.column_album, TRUE);
	gtk_tree_view_column_set_resizable(gml->search.column_album, TRUE);
	gtk_tree_view_column_set_cell_data_func(gml->search.column_album,
	                                        gml->renderer_text_ellipsize,
	                                        gml_cell_data_function, gml, NULL);

	renderer_pixbuf = gtk_cell_renderer_pixbuf_new();
	gml->search.column_rating = 
		gtk_tree_view_column_new_with_attributes("Rating", renderer_pixbuf,
		                                         "pixbuf", 4, NULL);
	gtk_tree_view_append_column(GTK_TREE_VIEW(gml->search.list_results),
	                            gml->search.column_rating);
	/*gtk_tree_view_column_set_sort_column_id(column_rating, 4);*/
	gtk_tree_view_column_set_resizable(gml->search.column_rating, FALSE);

	/* add to playlist button */
	gml->search.button_add = gtk_button_new_from_stock(GTK_STOCK_ADD);
	gtk_table_attach(GTK_TABLE(gml->search.table),
		             gml->search.button_add,
		             0, 5, 2, 3,
		             GTK_SHRINK, GTK_SHRINK, 0, 0);

	/* tooltips */
	tooltips = gtk_tooltips_new();
	gtk_tooltips_set_tip(tooltips, gml->search.button_search, 
	                     "Search", NULL);
	gtk_tooltips_set_tip(tooltips, gml->search.entry_searchstring, 
	                     "Enter at least 3 characters (or use exact search)",
	                     NULL);
	gtk_tooltips_set_tip(tooltips, gml->search.button_add, 
	                     "Add selection to playlist", NULL);

	gml_search_set_column_id_visible    (gml, TRUE);
	gml_search_set_column_artist_visible(gml, TRUE);
	gml_search_set_column_track_visible (gml, TRUE);
	gml_search_set_column_album_visible (gml, TRUE);
	gml_search_set_column_rating_visible(gml, TRUE);

    g_signal_connect(G_OBJECT(gml->search.button_search),            "clicked",
	                 G_CALLBACK(cb_search_search_button_pressed), 
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->search.entry_searchstring),      "activate",
	                 G_CALLBACK(cb_search_search_button_pressed), 
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->search.button_add),               "clicked",
	                 G_CALLBACK(cb_search_add_button_pressed), 
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->search.list_results),       "row-activated",
	                 G_CALLBACK(cb_search_on_tree_view_row_activated),
	                 (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->search.list_results),  "button-press-event",
                     G_CALLBACK(cb_search_selection_button_press),
                     (gpointer)gml);
    g_signal_connect(G_OBJECT(gml->search.list_results),          "popup-menu",
                     G_CALLBACK(cb_search_selection_popup_menu),
                     (gpointer)gml);
}
