;;; Copyright (C) 2011 Team GPS.
;;; 
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 2 of the License, or
;;; (at your option) any later version.
;;; 
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;; 
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

(ns twitter_clj.engine_controler
  (:import (java.io InputStreamReader OutputStreamWriter))
  (:require [twitter_clj.common :as common]
            [twitter_clj.engine :as engine]
            [twitter_clj.ki2 :as ki2]
            [twitter_clj.snapshot :as snapshot]
            [twitter_clj.subprocess :as sub]
            [twitter_clj.twitter :as twitter]
            [clojure.string :as str]
            [clojure.contrib.command-line :as cmd]
            [clojure.contrib.logging :as log]))

;; ==================================================
;; Global variables
;; ==================================================


;; ==================================================
;; Functions
;; ==================================================

(defmacro parse-integer
  [key-str ret xs]
  `(let [symbl# (keyword ~key-str)
         ret# ~ret
         xs#  ~xs]
     (recur (merge ret# {symbl# (Integer/parseInt (second xs#))}) (rest (rest xs#)))))

(defn- append-first-move
  "For a map col with :pv, return a new map with :first-move"
  [col]
  (assert (:pv col))
  (letfn [(get-first-move [pv]
            (first (str/split pv #"\s")))]
    (let [pv (:pv col)]
      (assoc col :first-move (get-first-move pv)))))

(defn- parse-info-depth-message
  "Parse the info depth message and return a corresponding map"
  [msg]
  (loop [ret {}
         xs  (str/split msg #"\s")]
    (if-not (seq xs)
      ret
      (condp = (first xs)
        "info"     (recur ret (rest xs))
        "depth"    (parse-integer "depth"    ret xs)
        "seldepth" (parse-integer "seldepth" ret xs)
        "score"    (recur ret (rest xs))
        "cp"       (parse-integer "cp"       ret xs)
        "nodes"    (parse-integer "nodes"    ret xs)
        "nps"      (parse-integer "nps"      ret xs)
        "time"     (parse-integer "time"     ret xs)
        "pv"       (recur (merge ret {:pv (str/join " " (rest xs))})
                          nil)
        (log/error (str "Unknown attributes for InfoDepthMessage: " xs))))))

(defn parse-line
  "Parse a raw line that gpsusi produces."
  [message]
  (try
    (condp (comp seq re-seq) message
      #"^info depth (\d+)$" nil
      #"^info depth (\d+) " (-> message
                                parse-info-depth-message
                                append-first-move)
      nil)
    (catch Exception e
      (log/error (str "Failed to parse message: " message))
      nil)))

(defn start-monitor
  "Start a monitor thread reading what gpsusi produces."
  [stdin]
  (log/debug "Starting a new monitor...")
  (future
    (reset! common/pv [])
    (loop [lines (line-seq stdin)]
      (if-let [line (first lines)]
        (condp (comp seq re-seq) line
          #"warning stop ignored" (common/sort-pv @common/pv) ; base
          #"bestmove (.*)"        :>> #(if (empty? @common/pv) ; base
                                         (reset! common/pv [{:cp 0 :pv (-> % first (nth 1))}])
                                         (common/sort-pv @common/pv))
          (if-let [m (parse-line line)]                       ; else
            (let [current-depth (or (:depth (first @common/pv)) 0)
                  depth (:depth m)]
              (if (< current-depth depth)
                (do
                  (log/debug (format "PV deepened: %s" m))
                  (reset! common/pv [m])
                  (recur (rest lines)))
                (do
                  (swap! common/pv conj m)
                  (recur (rest lines)))))
            (recur (rest lines))))))))


(defn get-nmove
  [line]
  {:pre  [(not-empty line)]
   :post [(pos? %)]}
  (let [m (re-find #"moves (.*)" line)
        _ (assert m)
        n (count (str/split (nth m 1) #" "))]
    n))


(defn interrupt-thread
  [thread]
  {:pre [(not (nil? thread))]}
  (.interrupt thread)
  (.join thread))
  

(defn stop-monitor
  "Stop a monitor (which is a future)."
  [stdout monitor]
  {:pre [(not (nil? monitor))]}
  (sub/write stdout "stop")
  (log/debug "waiting monitor")
  (log/debug (format "Monitor finished: %s" @monitor)))


(defn -main
  [& args]
  (cmd/with-command-line args
    "lein trampoline run [--dry-run]"
    [[dry-run?      "Do not tweet" false]
     [force-think   "Think at least n seconds for each move" 20]
     [force-update? "Update possile exisiting tweets" false]
     [gpsusi        "Path to gpsusi" "../../gpsshogi/bin/gpsusi"]
     remains]
    (swap! common/options assoc :dry-run dry-run?)
    (swap! common/options assoc :force-think (Integer. force-think))
    (swap! common/options assoc :force-update force-update?)
    (swap! common/options assoc :gpsusi gpsusi)
    (let [gpsusi-cmd (str (:gpsusi @common/options) " --extended-usi 1 -N 7") ;; extended mode
          [proc stdin stdout stderr] (engine/start-engine gpsusi-cmd)]   ;; search engine
      (ki2/start-ki2-engine (:gpsusi @common/options)) ;; normal mode
      (twitter/post-version (:id-name @common/options))
      (twitter/post-title)
      (loop [monitor         nil
             snapshot-thread nil
             dummy           nil
             lines           (line-seq (java.io.BufferedReader. *in*))]
        (if (and (seq lines)
                 (not= "resign" (first lines)))
          (let [line (first lines)
                nmove (get-nmove line)]
            (log/info (format ">>> [%d] %s" nmove line))
            (when (and snapshot-thread
                       (pos? (:force-think @common/options)))
              (log/debug (format "Force to think for %d secs..." (:force-think @common/options)))
              (Thread/sleep (* 1000 (:force-think @common/options))))
            (when snapshot-thread
              (interrupt-thread snapshot-thread))
            (when monitor
              (stop-monitor stdout monitor))
            ;; both snapshot-thread and monitor have finished.
            (if (and (twitter/moves-file-exists? nmove)
                     (not (:force-update @common/options)))
              (do
                (log/info "Found a twitter log file for this move. Skip it.")
                (recur nil nil dummy (rest lines)))
              (do
                (log/debug "Sending the move to the engine...")
                (if-not (ki2/is-valid-position line)
                  (do
                    (log/warn (format "Read an invalid position. Skip it: %s" line))
                    (recur nil nil dummy (rest lines)))
                  (do
                    (ki2/set-position line)
                    (sub/write stdout line)
                    (recur (start-monitor stdin)
                           (snapshot/start-snapshot-thread)
                           (sub/write stdout "go infinite")
                           (rest lines)))))))
          (do ; else
            (log/info "Finished reading.")
            (when snapshot-thread
              (interrupt-thread snapshot-thread))
            (when monitor
              (stop-monitor stdout monitor))
            nil))) ; end loop
      (ki2/stop-ki2-engine)
      (engine/stop-engine proc stdin stdout stderr))
    (shutdown-agents)))

