// Copyright 2017 Google Inc. All Rights Reserved.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

// This file supports generating unique IDs so that multiple test executions
// don't interfere with each other, and cleaning up old entities that may
// remain if tests exit early.

package testutil

import (
	"fmt"
	"regexp"
	"strconv"
	"sync"
	"time"
)

var startTime = time.Now().UTC()

// A UIDSpace manages a set of unique IDs distinguished by a prefix.
type UIDSpace struct {
	Prefix string
	re     *regexp.Regexp
	mu     sync.Mutex
	count  int
}

func NewUIDSpace(prefix string) *UIDSpace {
	return &UIDSpace{
		Prefix: prefix,
		re:     regexp.MustCompile("^" + regexp.QuoteMeta(prefix) + `-(\d{4})(\d{2})(\d{2})-(\d+)-\d+$`),
	}

}

// New generates a new unique ID . The ID consists of the UIDSpace's prefix, a
// timestamp, and a counter value. All unique IDs generated in the same test
// execution will have the same timestamp.
//
// Aside from the characters in the prefix, IDs contain only letters, numbers
// and hyphens.
func (s *UIDSpace) New() string { return s.newID(startTime) }

func (s *UIDSpace) newID(t time.Time) string {
	s.mu.Lock()
	c := s.count
	s.count++
	s.mu.Unlock()
	// Write the time as a date followed by nanoseconds from midnight of that date.
	// That makes it easier to see the approximate time of the ID when it is displayed.
	y, m, d := t.Date()
	ns := t.Sub(time.Date(y, m, d, 0, 0, 0, 0, time.UTC))
	// Zero-pad the counter for lexical sort order for IDs with the same timestamp.
	return fmt.Sprintf("%s-%04d%02d%02d-%d-%04d", s.Prefix, y, m, d, ns, c)
}

// Timestamp extracts the timestamp of uid, which must have been generated by
// s. The second return value is true on success, false if there was a problem.
func (s *UIDSpace) Timestamp(uid string) (time.Time, bool) {
	subs := s.re.FindStringSubmatch(uid)
	if subs == nil {
		return time.Time{}, false
	}
	y, err1 := strconv.Atoi(subs[1])
	m, err2 := strconv.Atoi(subs[2])
	d, err3 := strconv.Atoi(subs[3])
	ns, err4 := strconv.Atoi(subs[4])
	if err1 != nil || err2 != nil || err3 != nil || err4 != nil {
		return time.Time{}, false
	}
	return time.Date(y, time.Month(m), d, 0, 0, 0, ns, time.UTC), true
}

// Older reports whether uid was created by m and has a timestamp older than
// the current time by at least d.
func (s *UIDSpace) Older(uid string, d time.Duration) bool {
	ts, ok := s.Timestamp(uid)
	if !ok {
		return false
	}
	return time.Since(ts) > d
}
