------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-statements.adb,v 1.25 2008/07/03 06:22:48 vagul Exp $

with
   Ada.Strings.Maps,
   Ada.Strings.Fixed,
   Ada.Unchecked_Conversion,
   Interfaces.C.Strings,
   OCI.Thread;

package body OCI.Thick.Statements is

   use type SWord;
   use type Ub4;
   use type OCIHandle;

   Dummy_Connect : Connection;

   use Ada.Strings.Maps;

   Dos_Fix_CR : constant Character_Mapping := To_Mapping ("" & ASCII.CR, " ");

   function To_Statement_Type is
      new Ada.Unchecked_Conversion (Ub2, Statement_Type);

   procedure Execute_Internal
     (Stmt              : in out Statement;
      Raise_Exception   : in out Boolean;
      Commit_On_Success : in     Boolean := False);

   function Get_Attr_Ub2 is new Get_Attr_G (Ub2);
   function Get_Attr_Ub4 is new Get_Attr_G (Ub4);

   ----------------
   -- Bind_Names --
   ----------------

   function Bind_Names (Stmt : Statement) return VStrings is
      Names  : aliased Vector_Of_OraText (1 .. 256);
      Lens   : aliased Vector_Of_Ub1     (Names'Range);
      INames : aliased Vector_Of_OraText (Names'Range);
      ILens  : aliased Vector_Of_Ub1     (Names'Range);
      Dups   : aliased Vector_Of_Ub1     (Names'Range);
      Binds  : aliased Vector_Of_OCIBind (Names'Range);
      Found  : aliased Sb4;

      RC : constant SWord
        := OCIStmtGetBindInfo
             (Stmtp    => OCIStmt (Stmt.Handle),
              Errhp    => Thread.Error,
              Size     => Names'Length,
              Startloc => 1,
              Found    => Found'Unchecked_Access,
              Bvnp     => Names  (Names'First)'Unchecked_Access,
              Bvnl     => Lens   (Lens'First)'Unchecked_Access,
              Invp     => INames (INames'First)'Unchecked_Access,
              Inpl     => ILens  (ILens'First)'Unchecked_Access,
              Dupl     => Dups   (Dups'First)'Unchecked_Access,
              Hndl     => Binds  (Binds'First)'Unchecked_Access);
      use Interfaces.C.Strings;
      use type C.size_t;
      use type Ub1;
   begin
      if RC = OCI_NO_DATA then
         return (1 .. 0 => Null_Unbounded_String);
      end if;

      Check_Error (RC);

      declare
         Result : VStrings (1 .. Natural (Found));
         Count : Natural := 0;
      begin
         for i in Result'Range loop
            if Dups (i) = 0 then
               Count := Count + 1;

               Result (Count) := To_Unbounded_String
                 (C.To_Ada (Value (Names (i), C.size_t (Lens (i))), False));
            end if;
         end loop;

         return Result (1 .. Count);
      end;
   end Bind_Names;

   ------------
   -- Cancel --
   ------------

   procedure Cancel (Stmt : in Statement) is
   begin
      Check_Error (OCIStmtFetch
        (OCIStmt (Stmt.Handle), Thread.Error, Nrows => 0));
   end Cancel;

   --------------
   -- Describe --
   --------------

   procedure Describe (Connect : in Connection; Stmt : in out Statement) is
   begin
      Stmt.Connect := Connect;
      Describe (Stmt);
   end Describe;

   --------------
   -- Describe --
   --------------

   procedure Describe (Stmt : in out Statement) is
   begin
      Check_Error (OCIStmtExecute
        (OCISvcCtx (Handle (Stmt.Connect)),
         OCIStmt (Stmt.Handle),
         Thread.Error,
         0,
         Mode => OCI_DESCRIBE_ONLY));
      Stmt.Described := True;
   end Describe;

   --------------
   -- Describe --
   --------------

   procedure Describe (Stmt : in out Statement; Success : out Boolean) is
   begin
      Success
        := OCIStmtExecute
             (OCISvcCtx (Handle (Stmt.Connect)),
              OCIStmt (Stmt.Handle),
              Thread.Error,
              0,
              Mode => OCI_DESCRIBE_ONLY) = OCI_SUCCESS;

      if Success then
         Stmt.Described := True;
      end if;
   end Describe;

   ---------------
   -- Described --
   ---------------

   function Described (Stmt : in Statement) return Boolean is
   begin
      return Stmt.Described;
   end Described;

   -------------
   -- Destroy --
   -------------

   procedure Destroy (Object : in out Statement) is
   begin
      if Object.Handle /= Empty_Handle then
         OCI.Thick.Free (Object.Handle, OCI_HTYPE_STMT);
      end if;
   end Destroy;

   -------------
   -- Execute --
   -------------

   procedure Execute (Connect : in Connection; Stmt : in out Statement) is
   begin
      Stmt.Connect := Connect;
      Execute (Stmt);
   end Execute;

   procedure Execute (Stmt : in out Statement) is
      Raise_Exception : Boolean := True;
   begin
      Execute_Internal (Stmt, Raise_Exception);
   end Execute;

   procedure Execute (Stmt : in out Statement; Success : out Boolean) is
      Raise_Exception : Boolean := False;
   begin
      Execute_Internal (Stmt, Raise_Exception => Raise_Exception);
      Success := not Raise_Exception;
   end Execute;

   -------------
   -- Execute --
   -------------

   function Execute
     (Connect : in Connection; Code : in String) return Statement
   is
      Result : Statement := Prepare (Connect, Code);
   begin
      Execute (Result);
      return Result;
   end Execute;

   procedure Execute (Connect : in Connection; Code : in String) is
      Dummy : Statement := Prepare (Connect, Code);
   begin
      Execute (Dummy);
   end Execute;

   ------------------------
   -- Execute_And_Commit --
   ------------------------

   procedure Execute_And_Commit
     (Connect : in     Connection;
      Stmt    : in out Statement) is
   begin
      Stmt.Connect := Connect;
      Execute_And_Commit (Stmt);
   end Execute_And_Commit;

   procedure Execute_And_Commit (Stmt : in out Statement) is
      Raise_Exception : Boolean := True;
   begin
      Execute_Internal
        (Stmt,
         Raise_Exception   => Raise_Exception,
         Commit_On_Success => True);
   end Execute_And_Commit;

   procedure Execute_And_Commit
     (Stmt    : in out Statement;
      Success :    out Boolean)
   is
      Raise_Exception : Boolean := False;
   begin
      Execute_Internal
        (Stmt,
         Raise_Exception   => Raise_Exception,
         Commit_On_Success => True);
      Success := not Raise_Exception;
   end Execute_And_Commit;

   ----------------------
   -- Execute_Internal --
   ----------------------

   procedure Execute_Internal
     (Stmt              : in out Statement;
      Raise_Exception   : in out Boolean;
      Commit_On_Success : in     Boolean := False)
   is
      Not_Select : constant Boolean := Type_Of_Statement (Stmt) /= Stmt_Select;

      Commit_Mode : constant array (Boolean) of Ub4
         := (False => OCI_DEFAULT,
             True  => OCI_COMMIT_ON_SUCCESS);

      Rc : SWord := OCIStmtExecute
        (OCISvcCtx (Handle (Stmt.Connect)),
         OCIStmt (Stmt.Handle),
         Thread.Error,
         Boolean'Pos (Not_Select),
         Mode => Commit_Mode (Commit_On_Success));
   begin
      if Rc = OCI_ERROR and then Last_Error_Code = 1405 then
         --  Work aroung Oracle bug 1850169
         --  Fixed in 9.2.0.3

         Rc := OCI_SUCCESS;
      end if;

      Stmt.Executing  := Rc = OCI_STILL_EXECUTING;

      if Stmt.Executing then
         Raise_Exception := False;
      elsif Raise_Exception then
         Check_Error (Rc);
         Raise_Exception := False;
      else
         if Rc = OCI_SUCCESS then
            Stmt.Executed := True;
            Stmt.Described := True;
         else
            Raise_Exception := True;
         end if;
      end if;
   end Execute_Internal;

   --------------
   -- Executed --
   --------------

   function Executed (Stmt : in Statement) return Boolean is
   begin
      return Stmt.Executed;
   end Executed;

   -----------
   -- Fetch --
   -----------

   function Fetch (Stmt : in Statement) return Boolean is
      RC : constant SWord
        := OCIStmtFetch (OCIStmt (Stmt.Handle), Thread.Error);
   begin
      case RC is
         when OCI_NO_DATA => return False;
         when OCI_SUCCESS => return True;
         when others => Check_Error (RC);
      end case;

      --  We should not be there.

      raise Program_Error;
   end Fetch;

   --------------------
   -- Get_Connection --
   --------------------

   function Get_Connection (Stmt : in Statement) return Connection is
   begin
      return Stmt.Connect;
   end Get_Connection;

   ------------------
   -- Is_Executing --
   ------------------

   function Is_Executing (Stmt : in Statement) return Boolean is
   begin
      return Stmt.Executing;
   end Is_Executing;

   -----------------------
   -- Number_Of_Columns --
   -----------------------

   function Number_Of_Columns (Stmt : in Statement) return Natural is
   begin
      return Natural (Get_Attr_Ub4 (Stmt.Handle,
                                    OCI_HTYPE_STMT,
                                    OCI_ATTR_PARAM_COUNT));
   end Number_Of_Columns;

   ------------------------
   -- Parse_Error_Offset --
   ------------------------

   function Parse_Error_Offset (Stmt : in Statement) return Natural is
   begin
      return Natural (Get_Attr_Ub2 (Stmt.Handle,
                                    OCI_HTYPE_STMT,
                                    OCI_ATTR_PARSE_ERROR_OFFSET));
   end Parse_Error_Offset;

   -------------
   -- Prepare --
   -------------

   function Prepare
     (Connect : in Connection;
      Code    : in String)
      return  Statement
   is
      H  : OCIHandle := Alloc_Handle (Thread.Environment, OCI_HTYPE_STMT);
      Rc : SWord;
   begin
      Rc := OCIStmtPrepare
              (OCIStmt (H),
               Thread.Error,
               C.To_C (Ada.Strings.Fixed.Translate (Code, Dos_Fix_CR)),
               Code'Length);

      if Rc /= OCI_SUCCESS then
         OCI.Thick.Free (H, OCI_HTYPE_STMT);
         Check_Error (Rc);
      end if;

      return (RF.Controlled_Reference with
               Handle    => H,
               Connect   => Connect,
               Executed  => False,
               Executing => False,
               Described => False);
   end Prepare;

   function Prepare (Code : in String) return Statement is
   begin
      return Prepare (Dummy_Connect, Code);
   end Prepare;

   --------------------
   -- Rows_Processed --
   --------------------

   function Rows_Processed (Stmt : in Statement) return Natural is
   begin
      return Natural (Get_Attr_Ub4 (Stmt.Handle,
                                    OCI_HTYPE_STMT,
                                    OCI_ATTR_ROW_COUNT));
   end Rows_Processed;

   ------------------
   -- Set_Blocking --
   ------------------

   procedure Set_Blocking (Item : in out Statement; Mode : in Boolean) is
   begin
      Connections.Set_Blocking (Item.Connect, Mode);
   end Set_Blocking;

   --------------------
   -- Set_Connection --
   --------------------

   procedure Set_Connection
     (Stmt    : in out Statement;
      Connect : in     Connection) is
   begin
      Stmt.Connect := Connect;
   end Set_Connection;

   -----------------------
   -- Type_Of_Statement --
   -----------------------

   function Type_Of_Statement (Stmt : in Statement) return Statement_Type is
   begin
      return To_Statement_Type (Get_Attr_Ub2 (Stmt.Handle,
                                              OCI_HTYPE_STMT,
                                              OCI_ATTR_STMT_TYPE));
   end Type_Of_Statement;

end OCI.Thick.Statements;
